"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const __1 = require("..");
const common_1 = require("../common");
const util_1 = require("./util");
test('composing projects declaratively', () => {
    const comp = new util_1.TestProject();
    new TestSubproject({ name: 'foo', parent: comp, outdir: path.join('packages', 'foo') });
    comp.synth();
    // THEN
    expect(fs.existsSync(path.join(comp.outdir, 'packages', 'foo', '.gitignore'))).toBeTruthy();
});
test('composing projects synthesizes to subdirs', () => {
    // GIVEN
    const comp = new util_1.TestProject();
    // WHEN
    new TestSubproject({ name: 'foo', parent: comp, outdir: path.join('packages', 'foo') });
    new TestSubproject({ name: 'bar', parent: comp, outdir: path.join('packages', 'bar') });
    comp.synth();
    // THEN
    expect(fs.pathExistsSync(path.join(comp.outdir, 'README.md')));
    expect(fs.pathExistsSync(path.join(comp.outdir, 'packages', 'foo', '.gitignore')));
    expect(fs.pathExistsSync(path.join(comp.outdir, 'packages', 'bar', '.gitignore')));
});
test('errors when paths overlap', () => {
    // GIVEN
    const comp = new util_1.TestProject();
    new TestSubproject({ name: 'foo', parent: comp, outdir: path.join('packages', 'foo') });
    // WHEN/THEN
    expect(() => new TestSubproject({ name: 'foo', parent: comp, outdir: path.join('packages', 'foo') })).toThrowError(/there is already a sub-project with/i);
});
test('multiple levels', () => {
    const root = new util_1.TestProject();
    const child1 = new TestSubproject({ name: 'child1', parent: root, outdir: 'child1' });
    const child2 = new TestSubproject({ name: 'child2', parent: child1, outdir: 'child2' });
    expect(child1.outdir).toEqual(path.join(root.outdir, 'child1'));
    expect(child2.outdir).toEqual(path.join(root.outdir, 'child1', 'child2'));
});
test('subprojects cannot introduce files that override each other', () => {
    const root = new util_1.TestProject();
    const child = new TestSubproject({ name: 'sub-project', parent: root, outdir: 'sub-project' });
    new __1.TextFile(root, 'sub-project/file.txt');
    expect(() => new __1.TextFile(child, 'file.txt')).toThrow(/there is already a file under sub-project(\\|\/)file\.txt/);
});
test('"outdir" for subprojects must be relative', () => {
    const root = new util_1.TestProject();
    expect(() => new TestSubproject({ name: 'foobar', parent: root, outdir: '/foo/bar' })).toThrow(/"outdir" must be a relative path/);
});
test('subproject generated files do not get cleaned up by parent project', () => {
    const root = new util_1.TestProject();
    const child = new PreSynthProject({ parent: root, outdir: 'sub-project' });
    // no files have been generated yet
    expect(fs.existsSync(child.file.absolutePath)).toEqual(false);
    // generate all project files at least once
    root.synth();
    expect(child.fileExistedDuringPresynth).toEqual(false);
    expect(fs.existsSync(child.file.absolutePath)).toEqual(true);
    // resynthesize projects with all generated files already existing
    root.synth();
    expect(child.fileExistedDuringPresynth).toEqual(true);
    expect(fs.existsSync(child.file.absolutePath)).toEqual(true);
});
// a project that depends on generated files during preSynthesize()
class PreSynthProject extends __1.Project {
    constructor(options = {}) {
        super({ name: 'presynth-project', clobber: false, logging: { level: __1.LogLevel.OFF }, ...options });
        this.file = new __1.TextFile(this, 'presynth.txt', { lines: [common_1.PROJEN_MARKER] });
        this.fileExistedDuringPresynth = false;
    }
    preSynthesize() {
        this.fileExistedDuringPresynth = fs.existsSync(this.file.absolutePath);
    }
}
class TestSubproject extends __1.Project {
    constructor(options) {
        super({ logging: { level: __1.LogLevel.OFF }, ...options });
    }
}
//# sourceMappingURL=data:application/json;base64,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