"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
const json_1 = require("../json");
const util_1 = require("./util");
class ChildObjectFile extends __1.ObjectFile {
}
test('json object can be mutated before synthesis', () => {
    const prj = new util_1.TestProject();
    const obj = {
        hello: 'world',
    };
    new ChildObjectFile(prj, 'my/object/file.json', { obj, marker: false });
    // mutate obj (should be reflected in the output)
    obj.anotherField = {
        foo: 1234,
    };
    expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
        hello: 'world',
        anotherField: { foo: 1234 },
    });
});
describe('overrides', () => {
    test('addOverride(p, v) allows assigning arbitrary values to synthesized resource definitions', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', { obj: { initialObj: 'must be nonempty' }, marker: false });
        // WHEN
        file.addOverride('metadata', { key: 12 });
        file.addOverride('use.dot.notation', 'to create subtrees');
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            initialObj: 'must be nonempty',
            use: { dot: { notation: 'to create subtrees' } },
            metadata: { key: 12 },
        });
    });
    test('addOverride(p, undefined) can be used to delete a value', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', {
            obj: {
                hello: {
                    world: {
                        value1: 'Hello',
                        value2: 129,
                    },
                },
            },
            marker: false,
        });
        // WHEN
        file.addOverride('hello.world.value2', undefined);
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            hello: {
                world: {
                    value1: 'Hello',
                },
            },
        });
    });
    test('addOverride(p, undefined) will not create empty trees', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', { obj: { initialObj: 'must be nonempty' }, marker: false });
        // WHEN
        file.addOverride('tree.exists', 42);
        file.addOverride('tree.does.not.exist', undefined);
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            initialObj: 'must be nonempty',
            tree: {
                exists: 42,
            },
        });
    });
    test('addDeletionOverride(p) is sugar for `undefined`', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', {
            obj: {
                hello: {
                    world: {
                        value1: 'Hello',
                        value2: 129,
                        value3: ['foo', 'bar'],
                    },
                },
            },
            marker: false,
        });
        // WHEN
        file.addDeletionOverride('hello.world.value2');
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            hello: {
                world: {
                    value1: 'Hello',
                    value3: ['foo', 'bar'],
                },
            },
        });
    });
    test('addOverride(p, v) will overwrite any non-objects along the path', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', {
            obj: {
                hello: {
                    world: 42,
                },
            },
            marker: false,
        });
        // WHEN
        file.addOverride('override1', ['Hello', 123]);
        file.addOverride('override1.override2', { foo: [1] });
        file.addOverride('hello.world.foo.bar', 42);
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            hello: { world: { foo: { bar: 42 } } },
            override1: {
                override2: { foo: [1] },
            },
        });
    });
    test('addOverride(p, v) will not split on escaped dots', () => {
        // GIVEN
        const prj = new util_1.TestProject();
        const file = new json_1.JsonFile(prj, 'my/object/file.json', { obj: { initialObj: 'cannot be empty' }, marker: false });
        // WHEN
        file.addOverride(String.raw `Hello\.World.Foo\.Bar\.Baz`, 42);
        file.addOverride(String.raw `Single\Back\Slashes`, 42);
        file.addOverride(String.raw `Escaped\\.Back\\.Slashes`, 42);
        file.addOverride(String.raw `DoublyEscaped\\\\Back\\\\Slashes`, 42);
        file.addOverride('EndWith\\', 42); // Raw string cannot end with a backslash
        // THEN
        expect(util_1.synthSnapshot(prj)['my/object/file.json']).toStrictEqual({
            'initialObj': 'cannot be empty',
            'Hello.World': { 'Foo.Bar.Baz': 42 },
            'SingleBackSlashes': 42,
            'Escaped\\': { 'Back\\': { Slashes: 42 } },
            'DoublyEscaped\\\\Back\\\\Slashes': 42,
            'EndWith\\': 42,
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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