"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tests for `projen new`: we run `projen new` for each supported project type
// and compare against a golden snapshot.
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const inventory = require("../inventory");
const util_1 = require("./util");
for (const type of inventory.discover()) {
    test(`projen new ${type.pjid}`, () => {
        const outdir = util_1.mkdtemp();
        try {
            const projectdir = createProjectDir(outdir);
            // execute `projen new PJID --no-synth` in the project directory
            util_1.execProjenCLI(projectdir, ['new', '--no-synth', type.pjid]);
            // compare generated .projenrc.js to the snapshot
            const actual = util_1.directorySnapshot(projectdir, {
                excludeGlobs: [
                    '.git/**',
                ],
            });
            expect(actual).toMatchSnapshot();
        }
        finally {
            fs_extra_1.removeSync(outdir);
        }
    });
}
test('post-synthesis option enabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshotWithPost(project)['.postsynth']).toContain('postsynth');
});
test('post-synthesis option disabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshot(project)['.postsynth']).toBeUndefined();
});
test('projen new --from external', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_1.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project']);
        // compare generated .projenrc.js to the snapshot
        const actual = util_1.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual).toMatchSnapshot();
        expect(actual['schema.graphql']).toBeDefined();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
function createProjectDir(workdir) {
    // create project under "my-project" so that basedir is deterministic
    const projectdir = path_1.join(workdir, 'my-project');
    fs_extra_1.mkdirSync(projectdir);
    const git = (command) => child_process_1.execSync(`git ${command}`, { cwd: projectdir });
    git('init');
    git('remote add origin git@boom.com:foo/bar.git');
    git('config user.name "My User Name"');
    git('config user.email "my@user.email.com"');
    return projectdir;
}
//# sourceMappingURL=data:application/json;base64,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