# This code is part of Quinteng.
#
# (C) Copyright IBM 2018, 2019, 2021
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.
"""
Quinteng App qasm simulator backend.
"""

import copy
import logging
from quinteng.providers.options import Options
from quinteng.providers.models import QasmBackendConfiguration
from quinteng.providers.backend import BackendV2

from ..version import __version__
from .appbackend import AppBackend, AppError
from .backend_utils import (cpp_execute, available_methods,
                            available_devices,
                            MAX_QUBITS_STATEVECTOR,
                            BASIS_GATES)
# pylint: disable=import-error, no-name-in-module
from .controller_wrappers import app_controller_execute

logger = logging.getLogger(__name__)


class AppSimulator(AppBackend):
    """
    Noisy quantum circuit simulator backend.

    **Configurable Options**

    The `AppSimulator` supports multiple simulation methods and
    configurable options for each simulation method. These may be set using the
    appropriate kwargs during initialization. They can also be set of updated
    using the :meth:`set_options` method.

    Run-time options may also be specified as kwargs using the :meth:`run` method.
    These will not be stored in the backend and will only apply to that execution.
    They will also override any previously set options.

    For example, to configure a density matrix simulator with a custom noise
    model to use for every execution

    .. code-block:: python

        noise_model = NoiseModel.from_backend(backend)
        backend = AppSimulator(method='density_matrix',
                                noise_model=noise_model)

    **Simulating an IBMQ Backend**

    The simulator can be automatically configured to mimic an IBMQ backend using
    the :meth:`from_backend` method. This will configure the simulator to use the
    basic device :class:`NoiseModel` for that backend, and the same basis gates
    and coupling map.

    .. code-block:: python

        backend = AppSimulator.from_backend(backend)

    **Returning the Final State**

    The final state of the simulator can be saved to the returned
    ``Result`` object by appending the
    :func:`~quinteng.providers.app.library.save_state` instruction to a
    quantum circuit. The format of the final state will depend on the
    simulation method used. Additional simulation data may also be saved
    using the other save instructions in :mod:`quinteng.provider.app.library`.

    **Simulation Method Option**

    The simulation method is set using the ``method`` kwarg. A list supported
    simulation methods can be returned using :meth:`available_methods`, these
    are

    * ``"automatic"``: Default simulation method. Select the simulation
      method automatically based on the circuit and noise model.

    * ``"statevector"``: A dense statevector simulation that can sample
      measurement outcomes from *ideal* circuits with all measurements at
      end of the circuit. For noisy simulations each shot samples a
      randomly sampled noisy circuit from the noise model.

    * ``"density_matrix"``: A dense density matrix simulation that may
      sample measurement outcomes from *noisy* circuits with all
      measurements at end of the circuit.

    * ``"stabilizer"``: An efficient Clifford stabilizer state simulator
      that can simulate noisy Clifford circuits if all errors in the noise
      model are also Clifford errors.

    * ``"extended_stabilizer"``: An approximate simulated for Clifford + T
      circuits based on a state decomposition into ranked-stabilizer state.
      The number of terms grows with the number of non-Clifford (T) gates.

    * ``"matrix_product_state"``: A tensor-network statevector simulator that
      uses a Matrix Product State (MPS) representation for the state. This
      can be done either with or without truncation of the MPS bond dimensions
      depending on the simulator options. The default behaviour is no
      truncation.

    * ``"unitary"``: A dense unitary matrix simulation of an ideal circuit.
      This simulates the unitary matrix of the circuit itself rather than
      the evolution of an initial quantum state. This method can only
      simulate gates, it does not support measurement, reset, or noise.

    * ``"superop"``: A dense superoperator matrix simulation of an ideal or
      noisy circuit. This simulates the superoperator matrix of the circuit
      itself rather than the evolution of an initial quantum state. This method
      can simulate ideal and noisy gates, and reset, but does not support
      measurement.

    **GPU Simulation**

    By default all simulation methods run on the CPU, however select methods
    also support running on a GPU if quinteng-app was installed with GPU support
    on a compatible NVidia GPU and CUDA version.

    +--------------------------+---------------+
    | Method                   | GPU Supported |
    +==========================+===============+
    | ``automatic``            | Sometimes     |
    +--------------------------+---------------+
    | ``statevector``          | Yes           |
    +--------------------------+---------------+
    | ``density_matrix``       | Yes           |
    +--------------------------+---------------+
    | ``stabilizer``           | No            |
    +--------------------------+---------------+
    | ``matrix_product_state`` | No            |
    +--------------------------+---------------+
    | ``extended_stabilizer``  | No            |
    +--------------------------+---------------+
    | ``unitary``              | Yes           |
    +--------------------------+---------------+
    | ``superop``              | No            |
    +--------------------------+---------------+

    Running a GPU simulation is done using ``device="GPU"`` kwarg during
    initialization or with :meth:`set_options`. The list of supported devices
    for the current system can be returned using :meth:`available_devices`.

    **Additional Backend Options**

    The following simulator specific backend options are supported

    * ``method`` (str): Set the simulation method (Default: ``"automatic"``).
      Use :meth:`available_methods` to return a list of all availabe methods.

    * ``device`` (str): Set the simulation device (Default: ``"CPU"``).
      Use :meth:`available_devices` to return a list of devices supported
      on the current system.

    * ``precision`` (str): Set the floating point precision for
      certain simulation methods to either ``"single"`` or ``"double"``
      precision (default: ``"double"``).

    * ``executor`` (futures.Executor or None): Set a custom executor for
      asynchronous running of simulation jobs (Default: None).

    * ``max_job_size`` (int or None): If the number of run circuits
      exceeds this value simulation will be run as a set of of sub-jobs
      on the executor. If ``None`` simulation of all circuits are submitted
      to the executor as a single job (Default: None).

    * ``max_shot_size`` (int or None): If the number of shots of a noisy
      circuit exceeds this value simulation will be split into multi
      circuits for execution and the results accumulated. If ``None``
      circuits will not be split based on shots. When splitting circuits
      use the ``max_job_size`` option to control how these split circuits
      should be submitted to the executor (Default: None).

      a noise model exceeds this value simulation will be splitted into
      sub-circuits. If ``None``  simulator does noting (Default: None).

    * ``enable_truncation`` (bool): If set to True this removes unnecessary
      qubits which do not affect the simulation outcome from the simulated
      circuits (Default: True).


    * ``zero_threshold`` (double): Sets the threshold for truncating
      small values to zero in the result data (Default: 1e-10).

    * ``validation_threshold`` (double): Sets the threshold for checking
      if initial states are valid (Default: 1e-8).

    * ``max_parallel_threads`` (int): Sets the maximum number of CPU
      cores used by OpenMP for parallelization. If set to 0 the
      maximum will be set to the number of CPU cores (Default: 0).

    * ``max_parallel_experiments`` (int): Sets the maximum number of
      qobj experiments that may be executed in parallel up to the
      max_parallel_threads value. If set to 1 parallel circuit
      execution will be disabled. If set to 0 the maximum will be
      automatically set to max_parallel_threads (Default: 1).

    * ``max_parallel_shots`` (int): Sets the maximum number of
      shots that may be executed in parallel during each experiment
      execution, up to the max_parallel_threads value. If set to 1
      parallel shot execution will be disabled. If set to 0 the
      maximum will be automatically set to max_parallel_threads.
      Note that this cannot be enabled at the same time as parallel
      experiment execution (Default: 0).

    * ``max_memory_mb`` (int): Sets the maximum size of memory
      to store a state vector. If a state vector needs more, an error
      is thrown. In general, a state vector of n-qubits uses 2^n complex
      values (16 Bytes). If set to 0, the maximum will be automatically
      set to the system memory size (Default: 0).

    * ``blocking_enable`` (bool): This option enables parallelization with
      multiple GPUs or multiple processes with MPI (CPU/GPU). This option
      is only available for ``"statevector"``, ``"density_matrix"`` and
      ``"unitary"`` (Default: False).

    * ``blocking_qubits`` (int): Sets the number of qubits of chunk size
      used for parallelizing with multiple GPUs or multiple processes with
      MPI (CPU/GPU). 16*2^blocking_qubits should be less than 1/4 of the GPU
      memory in double precision. This option is only available for
      ``"statevector"``, ``"density_matrix"`` and ``"unitary"``.
      This option should be set when using option ``blocking_enable=True``
      (Default: 0).

    * ``batched_shots_gpu`` (bool): This option enables batched execution
      of multiple shot simulations on GPU devices for GPU enabled simulation
      methods. This optimization is intended for statevector simulations with
      noise models, or statevecor and density matrix simulations with
      intermediate measurements and can greatly accelerate simulation time
      on GPUs. If there are multiple GPUs on the system, shots are distributed
      automatically across available GPUs. Also this option distributes multiple
      shots to parallel processes of MPI (Default: True).

    * ``batched_shots_gpu_max_qubits`` (int): This option sets the maximum
      number of qubits for enabling the ``batched_shots_gpu`` option. If the
      number of active circuit qubits is greater than this value batching of
      simulation shots will not be used. (Default: 16).

    These backend options only apply when using the ``"statevector"``
    simulation method:

    * ``statevector_parallel_threshold`` (int): Sets the threshold that
      the number of qubits must be greater than to enable OpenMP
      parallelization for matrix multiplication during execution of
      an experiment. If parallel circuit or shot execution is enabled
      this will only use unallocated CPU cores up to
      max_parallel_threads. Note that setting this too low can reduce
      performance (Default: 14).

    * ``statevector_sample_measure_opt`` (int): Sets the threshold that
      the number of qubits must be greater than to enable a large
      qubit optimized implementation of measurement sampling. Note
      that setting this two low can reduce performance (Default: 10)

    These backend options only apply when using the ``"stabilizer"``
    simulation method:

    * ``stabilizer_max_snapshot_probabilities`` (int): set the maximum
      qubit number for the
      `~quinteng.providers.app.extensions.SnapshotProbabilities`
      instruction (Default: 32).

    These backend options only apply when using the ``"extended_stabilizer"``
    simulation method:

    * ``extended_stabilizer_sampling_method`` (string): Choose how to simulate
      measurements on qubits. The performance of the simulator depends
      significantly on this choice. In the following, let n be the number of
      qubits in the circuit, m the number of qubits measured, and S be the
      number of shots (Default: resampled_metropolis).

      - ``"metropolis"``: Use a Monte-Carlo method to sample many output
        strings from the simulator at once. To be accurate, this method
        requires that all the possible output strings have a non-zero
        probability. It will give inaccurate results on cases where
        the circuit has many zero-probability outcomes.
        This method has an overall runtime that scales as n^{2} + (S-1)n.

      - ``"resampled_metropolis"``: A variant of the metropolis method,
        where the Monte-Carlo method is reinitialised for every shot. This
        gives better results for circuits where some outcomes have zero
        probability, but will still fail if the output distribution
        is sparse. The overall runtime scales as Sn^{2}.

      - ``"norm_estimation"``: An alternative sampling method using
        random state inner products to estimate outcome probabilites. This
        method requires twice as much memory, and significantly longer
        runtimes, but gives accurate results on circuits with sparse
        output distributions. The overall runtime scales as Sn^{3}m^{3}.

    * ``extended_stabilizer_metropolis_mixing_time`` (int): Set how long the
      monte-carlo method runs before performing measurements. If the
      output distribution is strongly peaked, this can be decreased
      alongside setting extended_stabilizer_disable_measurement_opt
      to True (Default: 5000).

    * ``extended_stabilizer_approximation_error`` (double): Set the error
      in the approximation for the extended_stabilizer method. A
      smaller error needs more memory and computational time
      (Default: 0.05).

    * ``extended_stabilizer_norm_estimation_samples`` (int): The default number
      of samples for the norm estimation sampler. The method will use the
      default, or 4m^{2} samples where m is the number of qubits to be
      measured, whichever is larger (Default: 100).

    * ``extended_stabilizer_norm_estimation_repetitions`` (int): The number
      of times to repeat the norm estimation. The median of these reptitions
      is used to estimate and sample output strings (Default: 3).

    * ``extended_stabilizer_parallel_threshold`` (int): Set the minimum
      size of the extended stabilizer decomposition before we enable
      OpenMP parallelization. If parallel circuit or shot execution
      is enabled this will only use unallocated CPU cores up to
      max_parallel_threads (Default: 100).

    * ``extended_stabilizer_probabilities_snapshot_samples`` (int): If using
      the metropolis or resampled_metropolis sampling method, set the number of
      samples used to estimate probabilities in a probabilities snapshot
      (Default: 3000).

    These backend options only apply when using the ``matrix_product_state``
    simulation method:

    * ``matrix_product_state_max_bond_dimension`` (int): Sets a limit
      on the number of Schmidt coefficients retained at the end of
      the svd algorithm. Coefficients beyond this limit will be discarded.
      (Default: None, i.e., no limit on the bond dimension).

    * ``matrix_product_state_truncation_threshold`` (double):
      Discard the smallest coefficients for which the sum of
      their squares is smaller than this threshold.
      (Default: 1e-16).

    * ``mps_sample_measure_algorithm`` (str): Choose which algorithm to use for
      ``"sample_measure"`` (Default: "mps_apply_measure").

      - ``mps_probabilities``: This method first constructs the probability
        vector and then generates a sample per shot. It is more efficient for
        a large number of shots and a small number of qubits, with complexity
        O(2^n * n * D^2) to create the vector and O(1) per shot, where n is
        the number of qubits and D is the bond dimension.

      - ``mps_apply_measure``: This method creates a copy of the mps structure
        and measures directly on it. It is more efficient for a small number of
        shots, and a large number of qubits, with complexity around
        O(n * D^2) per shot.

    * ``mps_log_data`` (str): if True, output logging data of the MPS
      structure: bond dimensions and values discarded during approximation.
      (Default: False)

    * ``mps_swap_direction`` (str): Determine the direction of swapping the
      qubits when internal swaps are inserted for a 2-qubit gate.
      Possible values are "mps_swap_right" and "mps_swap_left".
      (Default: "mps_swap_left")

    These backend options apply in circuit optimization passes:

    * ``fusion_enable`` (bool): Enable fusion optimization in circuit
      optimization passes [Default: True]
    * ``fusion_verbose`` (bool): Output gates generated in fusion optimization
      into metadata [Default: False]
    * ``fusion_max_qubit`` (int): Maximum number of qubits for a operation generated
      in a fusion optimization [Default: 5]
    * ``fusion_threshold`` (int): Threshold that number of qubits must be greater
      than or equal to enable fusion optimization [Default: 14]
    """

    _BASIS_GATES = BASIS_GATES

    _CUSTOM_INSTR = {
        'statevector': sorted([
            'quantum_channel', 'qerror_loc', 'roerror', 'kraus',
            'snapshot', 'save_expval', 'save_expval_var',
            'save_probabilities', 'save_probabilities_dict',
            'save_amplitudes', 'save_amplitudes_sq',
            'save_density_matrix', 'save_state', 'save_statevector',
            'save_statevector_dict', 'set_statevector',
        ]),
        'density_matrix': sorted([
            'quantum_channel', 'qerror_loc', 'roerror', 'kraus', 'superop', 'snapshot',
            'save_state', 'save_expval', 'save_expval_var',
            'save_probabilities', 'save_probabilities_dict',
            'save_density_matrix', 'save_amplitudes_sq', 'set_density_matrix'
        ]),
        'matrix_product_state': sorted([
            'quantum_channel', 'qerror_loc', 'roerror', 'snapshot', 'kraus',
            'save_expval', 'save_expval_var',
            'save_probabilities', 'save_probabilities_dict',
            'save_state', 'save_matrix_product_state', 'save_statevector',
            'save_density_matrix', 'save_amplitudes', 'save_amplitudes_sq',
            'set_matrix_product_state',
        ]),
        'stabilizer': sorted([
            'quantum_channel', 'qerror_loc', 'roerror', 'snapshot',
            'save_expval', 'save_expval_var',
            'save_probabilities', 'save_probabilities_dict',
            'save_amplitudes_sq', 'save_state', 'save_clifford',
            'save_stabilizer', 'set_stabilizer'
        ]),
        'extended_stabilizer': sorted([
            'quantum_channel', 'qerror_loc', 'roerror', 'snapshot', 'save_statevector'
        ]),
        'unitary': sorted([
            'snapshot', 'save_state', 'save_unitary', 'set_unitary'
        ]),
        'superop': sorted([
            'quantum_channel', 'qerror_loc', 'kraus', 'superop', 'save_state',
            'save_superop', 'set_superop'
        ])
    }

    # Automatic method custom instructions are the union of statevector,
    # density matrix, and stabilizer methods
    _CUSTOM_INSTR[None] = _CUSTOM_INSTR['automatic'] = sorted(
        set(_CUSTOM_INSTR['statevector']).union(
            _CUSTOM_INSTR['stabilizer']).union(
                _CUSTOM_INSTR['density_matrix']).union(
                    _CUSTOM_INSTR['matrix_product_state']).union(
                        _CUSTOM_INSTR['unitary']).union(
                            _CUSTOM_INSTR['superop']))

    _DEFAULT_CONFIGURATION = {
        'backend_name': 'app_simulator',
        'backend_version': __version__,
        'n_qubits': MAX_QUBITS_STATEVECTOR,
        'url': 'https://github.com/Quinteng/quinteng-app',
        'simulator': True,
        'local': True,
        'conditional': True,
        'open_pulse': False,
        'memory': True,
        'max_shots': int(1e6),
        'description': 'A C++ QasmQobj simulator with noise',
        'coupling_map': None,
        'basis_gates': BASIS_GATES['automatic'],
        'custom_instructions': _CUSTOM_INSTR['automatic'],
        'gates': []
    }

    _SIMULATION_METHODS = [
        'automatic', 'statevector', 'density_matrix',
        'stabilizer', 'matrix_product_state', 'extended_stabilizer',
        'unitary', 'superop'
    ]

    _AVAILABLE_METHODS = None

    _SIMULATION_DEVICES = ('CPU', 'GPU', 'Thrust')

    _AVAILABLE_DEVICES = None

    def __init__(self,
                 configuration=None,
                 properties=None,
                 provider=None,
                 **backend_options):

        self._controller = app_controller_execute()

        # Update available methods and devices for class
        if AppSimulator._AVAILABLE_METHODS is None:
            AppSimulator._AVAILABLE_METHODS = available_methods(
                self._controller, AppSimulator._SIMULATION_METHODS)
        if AppSimulator._AVAILABLE_DEVICES is None:
            AppSimulator._AVAILABLE_DEVICES = available_devices(
                self._controller, AppSimulator._SIMULATION_DEVICES)

        # Default configuration
        if configuration is None:
            configuration = QasmBackendConfiguration.from_dict(
                AppSimulator._DEFAULT_CONFIGURATION)

        # Cache basis gates since computing the intersection
        # of noise model, method, and config gates is expensive.
        self._cached_basis_gates = self._BASIS_GATES['automatic']

        super().__init__(configuration,
                         properties=properties,
                         provider=provider,
                         backend_options=backend_options)

    @classmethod
    def _default_options(cls):
        return Options(
            # Global options
            shots=1024,
            method='automatic',
            device='CPU',
            precision="double",
            executor=None,
            max_job_size=None,
            max_shot_size=None,
            enable_truncation=True,
            zero_threshold=1e-10,
            validation_threshold=None,
            max_parallel_threads=None,
            max_parallel_experiments=None,
            max_parallel_shots=None,
            max_memory_mb=None,
            fusion_enable=True,
            fusion_verbose=False,
            fusion_max_qubit=5,
            fusion_threshold=14,
            accept_distributed_results=None,
            memory=None,
            noise_model=None,
            seed_simulator=None,
            # cache blocking for multi-GPUs/MPI options
            blocking_qubits=None,
            blocking_enable=False,
            # multi-shots optimization options (GPU only)
            batched_shots_gpu=True,
            batched_shots_gpu_max_qubits=16,
            # statevector options
            statevector_parallel_threshold=14,
            statevector_sample_measure_opt=10,
            # stabilizer options
            stabilizer_max_snapshot_probabilities=32,
            # extended stabilizer options
            extended_stabilizer_sampling_method='resampled_metropolis',
            extended_stabilizer_metropolis_mixing_time=5000,
            extended_stabilizer_approximation_error=0.05,
            extended_stabilizer_norm_estimation_samples=100,
            extended_stabilizer_norm_estimation_repetitions=3,
            extended_stabilizer_parallel_threshold=100,
            extended_stabilizer_probabilities_snapshot_samples=3000,
            # MPS options
            matrix_product_state_truncation_threshold=1e-16,
            matrix_product_state_max_bond_dimension=None,
            mps_sample_measure_algorithm='mps_heuristic',
            mps_log_data=False,
            mps_swap_direction='mps_swap_left',
            chop_threshold=1e-8,
            mps_parallel_threshold=14,
            mps_omp_threads=1)

    def __repr__(self):
        """String representation of an AppSimulator."""
        display = super().__repr__()
        noise_model = getattr(self.options, 'noise_model', None)
        if noise_model is None or noise_model.is_ideal():
            return display
        pad = ' ' * (len(self.__class__.__name__) + 1)
        return '{}\n{}noise_model={})'.format(display[:-1], pad, repr(noise_model))

    def name(self):
        """Format backend name string for simulator"""
        name = self._configuration.backend_name
        method = getattr(self.options, 'method', None)
        if method not in [None, 'automatic']:
            name += f'_{method}'
        device = getattr(self.options, 'device', None)
        if device not in [None, 'CPU']:
            name += f'_{device}'.lower()
        return name

    @classmethod
    def from_backend(cls, backend, **options):
        """Initialize simulator from backend."""
        if isinstance(backend, BackendV2):
            raise AppError(
                "AppSimulator.from_backend does not currently support V2 Backends."
            )
        # Get configuration and properties from backend
        configuration = copy.copy(backend.configuration())
        properties = copy.copy(backend.properties())

        # Customize configuration name
        name = configuration.backend_name
        configuration.backend_name = 'app_simulator({})'.format(name)

        # Use automatic noise model if none is provided
        if 'noise_model' not in options:
            # pylint: disable=import-outside-toplevel
            # Avoid cyclic import
            from ..noise.noise_model import NoiseModel

            noise_model = NoiseModel.from_backend(backend)
            if not noise_model.is_ideal():
                options['noise_model'] = noise_model

        # Initialize simulator
        sim = cls(configuration=configuration,
                  properties=properties,
                  **options)
        return sim

    def available_methods(self):
        """Return the available simulation methods."""
        return copy.copy(self._AVAILABLE_METHODS)

    def available_devices(self):
        """Return the available simulation methods."""
        return copy.copy(self._AVAILABLE_DEVICES)

    def configuration(self):
        """Return the simulator backend configuration.

        Returns:
            BackendConfiguration: the configuration for the backend.
        """
        config = copy.copy(self._configuration)
        for key, val in self._options_configuration.items():
            setattr(config, key, val)
        # Update basis gates based on custom options, config, method,
        # and noise model
        config.custom_instructions = self._CUSTOM_INSTR[
            getattr(self.options, 'method', 'automatic')]
        config.basis_gates = self._cached_basis_gates + config.custom_instructions
        # Update simulator name
        config.backend_name = self.name()
        return config

    def _execute(self, qobj):
        """Execute a qobj on the backend.

        Args:
            qobj (QasmQobj): simulator input.

        Returns:
            dict: return a dictionary of results.
        """
        return cpp_execute(self._controller, qobj)

    def set_option(self, key, value):
        if key == "custom_instructions":
            self._set_configuration_option(key, value)
            return
        if key == "method":
            if (value is not None and value not in self.available_methods()):
                raise AppError(
                    "Invalid simulation method {}. Available methods"
                    " are: {}".format(value, self.available_methods()))
            self._set_method_config(value)
        super().set_option(key, value)
        if key in ["method", "noise_model", "basis_gates"]:
            self._cached_basis_gates = self._basis_gates()

    def _validate(self, qobj):
        """Semantic validations of the qobj which cannot be done via schemas.

        Warn if no measure or save instructions in run circuits.
        """
        for experiment in qobj.experiments:
            # If circuit does not contain measurement or save
            # instructions raise a warning
            no_data = True
            for op in experiment.instructions:
                if op.name == "measure" or op.name[:5] == "save_":
                    no_data = False
                    break
            if no_data:
                logger.warning(
                    'No measure or save instruction in circuit "%s": '
                    'results will be empty.',
                    experiment.header.name)

    def _basis_gates(self):
        """Return simualtor basis gates.

        This will be the option value of basis gates if it was set,
        otherwise it will be the intersection of the configuration, noise model
        and method supported basis gates.
        """
        # Use option value for basis gates if set
        if 'basis_gates' in self._options_configuration:
            return self._options_configuration['basis_gates']

        # Compute intersection with method basis gates
        method = getattr(self._options, 'method', 'automatic')
        method_gates = self._BASIS_GATES[method]
        config_gates = self._configuration.basis_gates
        if config_gates:
            basis_gates = set(config_gates).intersection(
                method_gates)
        else:
            basis_gates = method_gates

        # Compute intersection with noise model basis gates
        noise_model = getattr(self.options, 'noise_model', None)
        if noise_model:
            noise_gates = noise_model.basis_gates
            basis_gates = basis_gates.intersection(noise_gates)
        else:
            noise_gates = None

        if not basis_gates:
            logger.warning(
                "The intersection of configuration basis gates (%s), "
                "simulation method basis gates (%s), and "
                "noise model basis gates (%s) is empty",
                config_gates, method_gates, noise_gates)
        return sorted(basis_gates)

    def _set_method_config(self, method=None):
        """Set non-basis gate options when setting method"""
        # Update configuration description and number of qubits
        if method == 'statevector':
            description = 'A C++ statevector simulator with noise'
            n_qubits = MAX_QUBITS_STATEVECTOR
        elif method == 'density_matrix':
            description = 'A C++ density matrix simulator with noise'
            n_qubits = MAX_QUBITS_STATEVECTOR // 2
        elif method == 'unitary':
            description = 'A C++ unitary matrix simulator'
            n_qubits = MAX_QUBITS_STATEVECTOR // 2
        elif method == 'superop':
            description = 'A C++ superop matrix simulator with noise'
            n_qubits = MAX_QUBITS_STATEVECTOR // 4
        elif method == 'matrix_product_state':
            description = 'A C++ matrix product state simulator with noise'
            n_qubits = 63  # TODO: not sure what to put here?
        elif method == 'stabilizer':
            description = 'A C++ Clifford stabilizer simulator with noise'
            n_qubits = 10000  # TODO: estimate from memory
        elif method == 'extended_stabilizer':
            description = 'A C++ Clifford+T extended stabilizer simulator with noise'
            n_qubits = 63  # TODO: estimate from memory
        else:
            # Clear options to default
            description = None
            n_qubits = None
        self._set_configuration_option('description', description)
        self._set_configuration_option('n_qubits', n_qubits)
