# Testing Decorator Functionality with Flow Dev

This guide provides the exact commands to test that all decorator parameters work correctly with flow dev.

## Prerequisites

```bash
# Ensure you're in the flow-compute directory
cd /Users/jq/Downloads/flow-compute

# Ensure flow is configured
flow init  # If not already done
```

## Test Sequence

### Step 1: Generate Test Scripts

First, generate all the test wrapper scripts:

```bash
# Generate the simple test
python tests/legacy_root/test_decorator_simple.py

# Generate comprehensive tests
python tests/legacy_root/test_decorator_flow_dev.py
```

This creates the following test files (now under `tests/legacy_root/`):
- `tests/legacy_root/wrapper_test.py` - Simple add function test
- `tests/legacy_root/test_basic.py` - Basic decorator test
- `tests/legacy_root/test_gpu.py` - GPU configuration test
- `tests/legacy_root/test_volumes.py` - Volume mounting test
- `tests/legacy_root/test_environment.py` - Environment variables test
- `tests/legacy_root/test_retries.py` - Retry configuration test
- `tests/legacy_root/test_timeout.py` - Timeout test
- `tests/legacy_root/test_complex_args.py` - Complex argument types test
- `tests/legacy_root/test_all_params.py` - All parameters combined test
- `tests/legacy_root/run_all_tests.sh` - Master test runner

### Step 2: Start Flow Dev VM

```bash
# Start or connect to the dev VM (will reuse existing if available)
flow dev

# If you want to force a new VM:
# flow dev --force-new

# Exit the SSH session when connected (Ctrl+D or exit)
```

### Step 3: Check VM Status

```bash
# Verify the VM is running
flow dev --status
```

### Step 4: Upload Test Files to VM

```bash
# Upload current directory to the VM
flow dev --upload

# Or if you want to be specific:
flow dev --upload --upload-path .
```

### Step 5: Run Individual Tests

Run each test individually to verify specific functionality:

```bash
# Test 1: Simple add function
flow dev -c 'python /root/tests/legacy_root/wrapper_test.py'
flow dev -c 'cat /tmp/flow_result.json'

# Test 2: Basic decorator
flow dev -c 'python /root/tests/legacy_root/test_basic.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'

# Test 3: GPU configuration (will show PyTorch info if image has it)
flow dev -c 'python /root/tests/legacy_root/test_gpu.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'

# Test 4: Environment variables
flow dev -c 'python /root/tests/legacy_root/test_environment.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'

# Test 5: Complex arguments
flow dev -c 'python /root/tests/legacy_root/test_complex_args.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'

# Test 6: Timeout configuration
flow dev -c 'python /root/tests/legacy_root/test_timeout.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'

# Test 7: All parameters combined
flow dev -c 'python /root/tests/legacy_root/test_all_params.py'
flow dev -c 'python -m json.tool /tmp/flow_result.json'
```

### Step 6: Run All Tests at Once

```bash
# Run the master test script
flow dev -c 'bash /root/tests/legacy_root/run_all_tests.sh'
```

### Step 7: Test with Different Images

Test that the image parameter works:

```bash
# Test with Python 3.11 (default)
flow dev -c 'python /root/tests/legacy_root/test_basic.py'

# Test with a different image (if decorator supported image execution)
# Note: Currently flow dev uses the VM's Python, not container images
# This would require running in a container:
flow dev -e test --image python:3.10 -c 'python /root/tests/legacy_root/test_basic.py'
```

### Step 8: Verify All Parameters

Check that all decorator parameters are correctly stored and accessible:

```bash
# Create a verification script
cat << 'EOF' > verify_params.py
from flow import FlowApp
from flow.api.models import Retries
from flow.api.secrets import Secret

app = FlowApp()

# Test all parameters
@app.function(
    gpu="h100:8",
    cpu=(16.0, 32.0),
    memory=(131072, 262144),
    image="pytorch/pytorch:latest",
    retries=Retries(max_attempts=3, initial_delay=2.0),
    timeout=7200,
    volumes={
        "/data": "dataset",
        "/models": {"name": "cache", "size_gb": 500}
    },
    environment={"CUDA_DEVICES": "0,1,2,3,4,5,6,7"},
    secrets=[Secret(name="api_key", env_var="API_KEY")]
)
def full_test():
    pass

# Verify all parameters are stored
print(f"GPU: {full_test.gpu}")
print(f"CPU: {full_test.cpu}")
print(f"Memory: {full_test.memory}")
print(f"Image: {full_test.image}")
print(f"Retries: {full_test.retries}")
print(f"Timeout: {full_test.timeout}")
print(f"Volumes: {full_test.volumes}")
print(f"Environment: {full_test.environment}")
print(f"Secrets: {len(full_test.secrets)} secret(s)")

# Test wrapper generation
wrapper = full_test._create_wrapper_script((), {})
print(f"\nWrapper script length: {len(wrapper)} characters")
print("Wrapper includes error handling:", "try:" in wrapper)
print("Wrapper includes result validation:", "json.dumps(result)" in wrapper)
EOF

# Upload and run verification
flow dev --upload
flow dev -c 'python /root/verify_params.py'
```

### Step 9: Clean Up Tests

```bash
# Remove test results
flow dev -c 'rm -f /tmp/flow_result.json'

# Remove test files if desired
flow dev -c 'rm -f /root/tests/legacy_root/test_*.py /root/tests/legacy_root/wrapper_test.py'
```

### Step 10: Stop Dev VM (Optional)

```bash
# Stop the dev VM when done
flow dev --stop

# Or keep it running for future tests
```

## Expected Results

Each test should produce a JSON result file at `/tmp/flow_result.json` with:

```json
{
  "success": true,
  "result": {
    // Test-specific results
  }
}
```

For failed tests:

```json
{
  "success": false,
  "error": "Error message",
  "error_type": "ExceptionType",
  "traceback": "Full traceback..."
}
```

## Troubleshooting

1. **No result file**: Check if the wrapper script has syntax errors
   ```bash
   flow dev -c 'python -m py_compile /root/tests/legacy_root/test_basic.py'
   ```

2. **Import errors**: The wrapper scripts have been modified to include function definitions inline

3. **Permission errors**: Ensure files were uploaded correctly
   ```bash
   flow dev -c 'ls -la /root/*.py'
   ```

4. **VM not starting**: Check your Flow configuration
   ```bash
   flow status
   ```

## Summary

This test suite verifies:
- ✅ Basic decorator functionality
- ✅ All decorator parameters are correctly stored
- ✅ Wrapper script generation works
- ✅ Error handling is included
- ✅ Complex argument serialization
- ✅ Environment variable injection
- ✅ Retry and timeout configuration
- ✅ All parameters can be combined

The decorator UX is fully functional and ready for use with flow dev!