import argparse
import sys

import flet.version
import flet_cli.commands.build
import flet_cli.commands.create
import flet_cli.commands.doctor  # Adding the doctor command
import flet_cli.commands.pack
import flet_cli.commands.publish
import flet_cli.commands.run
import flet_cli.commands.serve
from flet.version import update_version


# Source https://stackoverflow.com/a/26379693
def set_default_subparser(
    parser: argparse.ArgumentParser, name: str, args: list = None, index: int = 0
):
    """
    Set a default subparser when no subparser is provided.
    This should be called after setting up the argument parser but before
    `parse_args()`.

    Args:
        name: The name of the default subparser to use.
        args: A list of arguments passed to `parse_args()`.
        index: Position in `sys.argv` where the default subparser should be
            inserted.
    """

    # exit if help or version flags are present
    if any(flag in sys.argv[1:] for flag in {"-h", "--help", "-V", "--version"}):
        return

    # all subparser actions
    subparser_actions = [
        action
        for action in parser._subparsers._actions
        if isinstance(action, argparse._SubParsersAction)
    ]

    # all subparser names
    subparser_names = [
        sp_name for action in subparser_actions for sp_name in action._name_parser_map
    ]

    # if an existing subparser is provided, skip setting a default
    if any(arg in subparser_names for arg in sys.argv[1:]):
        return

    # if the default subparser doesn't exist, register it in the first subparser action
    if (name not in subparser_names) and subparser_actions:
        subparser_actions[0].add_parser(name)

    # insert the default subparser into the appropriate argument list
    if args is None:
        if len(sys.argv) > 1:
            sys.argv.insert(index, name)
    else:
        args.insert(index, name)


def get_parser() -> argparse.ArgumentParser:
    """Construct and return the CLI argument parser."""
    parser = argparse.ArgumentParser()

    # add version flag
    parser.add_argument(
        "--version",
        "-V",
        action="version",
        version=flet.version.version if flet.version.version else update_version(),
    )

    sp = parser.add_subparsers(dest="command")

    # register subcommands
    flet_cli.commands.create.Command.register_to(sp, "create")
    flet_cli.commands.run.Command.register_to(sp, "run")
    flet_cli.commands.build.Command.register_to(sp, "build")
    flet_cli.commands.pack.Command.register_to(sp, "pack")
    flet_cli.commands.publish.Command.register_to(sp, "publish")
    flet_cli.commands.serve.Command.register_to(sp, "serve")
    flet_cli.commands.doctor.Command.register_to(sp, "doctor")

    # set "run" as the default subparser
    set_default_subparser(parser, name="run", index=1)

    return parser


def main():
    parser = get_parser()

    # print usage/help if called without arguments
    if len(sys.argv) == 1:
        parser.print_help(sys.stdout)
        sys.exit(1)

    # parse arguments
    args = parser.parse_args()

    # execute command
    args.handler(args)


if __name__ == "__main__":
    main()
