# RaRa Linker

![Py3.10](https://img.shields.io/badge/python-3.10-green.svg)
![Py3.11](https://img.shields.io/badge/python-3.11-green.svg)
![Py3.12](https://img.shields.io/badge/python-3.12-green.svg)

**`rara-norm-linker`** is a  Python library for linking **personal names, organizations, geographical names** and **keywords** with taxonomy entries.

**NB!** Requires access to an **Elasticsearch>=8.0** instance.

---

## ✨ Features  

- Link **personal names, organizations, geographical names**, and **keywords** with taxonomy entries.
- Use **fuzzy matching** for linking.
- Use **vector search** for filtering results.
- Use [**VIAF**](https://viaf.org/en) queries for enrichment.
---


## ⚡ Quick Start  

Get started with `rara-norm-linker` in just a few steps:

1. **Install the Package**  
   Ensure you're using Python 3.10 or above, then run:  
   ```bash
   pip install rara-norm-linker
   ```

2. **Import and Use**  
   Example usage to link entries with default configuration:

   ```python
   from rara_linker.linkers.linker import Linker
   from pprint import pprint
   import logging
   
   # Disables logging, feel free to comment this out
   logging.disable(logging.CRITICAL) 

   # Initialize Linker instance
   linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")
   entity = "Lennart Mere"

   linked_info = linker.link(entity)
   pprint(linked_info.to_dict())
   ```

---



## ⚙️ Installation Guide

Follow the steps below to install the `rara-norm-linker` package, either via `pip` or locally.

---

### Installation via `pip`

<details><summary>Click to expand</summary>

1. **Set Up Your Python Environment**  
   Create or activate a Python environment using Python **3.10** or above.

2. **Install the Package**  
   Run the following command:  
   ```bash
   pip install rara-norm-linker
   ```
</details>

---

### Local Installation

Follow these steps to install the `rara-norm-linker` package locally:  

<details><summary>Click to expand</summary>


1. **Clone the Repository**  
   Clone the repository and navigate into it:  
   ```bash
   git clone <repository-url>
   cd <repository-directory>
   ```

2. **Set Up Python Environment**  
   Create or activate a Python environment using Python 3.10 or above. E.g:
   ```bash
   conda create -n py310 python==3.10
   conda activate py310
   ```

3. **Install Build Package**  
   Install the `build` package to enable local builds:  
   ```bash
   pip install build
   ```

4. **Build the Package**  
   Run the following command inside the repository:  
   ```bash
   python -m build
   ```

5. **Install the Package**  
   Install the built package locally:  
   ```bash
   pip install .
   ```

</details>

---

## 🚀 Testing Guide

Follow these steps to test the `rara-norm-linker` package.


### How to Test

<details><summary>Click to expand</summary>

1. **Clone the Repository**  
   Clone the repository and navigate into it:  
   ```bash
   git clone <repository-url>
   cd <repository-directory>
   ```

2. **Set Up Python Environment**  
   Create or activate a Python environment using Python 3.10 or above.

3. **Install Build Package**  
   Install the `build` package:  
   ```bash
   pip install build
   ```

4. **Build the Package**  
   Build the package inside the repository:  
   ```bash
   python -m build
   ```

5. **Install with Testing Dependencies**  
   Install the package along with its testing dependencies:  
   ```bash
   pip install .[testing]
   ```

6. **Run Tests**  
   Run the test suite from the repository root:  
   ```bash
   python -m pytest -v tests
   ```

---

</details>


## 📝 Documentation

<details><summary>Click to expand</summary>

### 🔍 `Linker` Class

#### Overview

`Linker` class combines 4 different classes (`PersonLinker`, `OrganizationLinker`, `EMSLinker` and `LocationLinker`) into a single workflow. This adds some flexibility to the input: the user doesn't necessarily have to know the type of the entity to link.

#### Importing

```python
from rara_linker.linkers.linker import Linker
```

#### Class Parameters

| Name                 | Type   | Optional | Default | Description                                                          |
| -------------------- | ------ |----------|---------|----------------------------------------------------------------------|
| add_viaf_info        | bool   | True     | False   | If enabled, a query is made to VIAF to enrich the linked information. |
| vectorized_data_path | str    | True     | "./vectorizer_data"| Specifies the directory, where vectorization model's resources are downloaded. |
| per_config           | dict   | True     | rara_linker.config.PER_CONFIG | Configuration of PersonLinker's fields. |
| org_config           | dict   | True     | rara_linker.config.ORG_CONFIG | Configuration of OrganizationLinker's fields   |
| loc_config           | dict   | True     | rara_linker.config.LOC_CONFIG | Configuration of LocationLinker's fields   |
| ems_config           | dict   | True     | rara_linker.config.EMS_CONFIG | Configuration of EMSLinker's fields   |

NB! `per_config`, `org_config`, `loc_config` and `ems_config` have already pre-configured values. However, every single one of them can be overwritten, if necessary. For example, if the same data for linking personal names is uploaded into a new index, it is sufficient to just pass the new index with the configuration:

```python
linker = Linker(
        per_config={"es_index": "my_new_index"}
)
```

All possible configuration parameters are listed in the table below:

##### Configuration Parameters

| Name             | Type   | Required | Description                                    |
| ---------------- | ------ |----------|------------------------------------------------|
| es_host          |  str   | True     | Elasticsearch's URL, e.g. "http://localhost:9200"|
| es_index         |  str   | True     | Elasticsearch's index containing the norm data used for linking. |
| search_field     |  str   | True     | Field in Elasticsearch's index that is used for linking. **NB! The value of the field has to be of type List[str]**, e.g. `["Contra", "Margus Konnula"]`. |
| alt_search_field |  str   | False    | Alternative search field to `search_field`. This is used as a backup, is no results were found from `search_field`. The information is contained in a separate field as somewhat different linking parameters might apply (e.g. acronyms for organizations, where fuzziness is restricted to 0). |
| key_field        |  str   | True     | Field containing the normalized value of the entity. |
| json_field       |  str   | False    | Field containing JSON version of MARC21-I (compatible with Sierra). |
| marc_field       |  str   | False    | Field containing MARC21-I |
| identifier_field |  str   | True     | Field containing the identifier value. |
| vector_field     |  str   | False    | Field containing vectrized data. |
| viaf_field       |  str   | False    | VIAF search field, e.g. `local.personalNames` |


#### Key Functions

##### `link`

###### Parameters

| Name           | Type   | Required | Default | Description    |
| -------------- | ------ | -------- | --------|----------------|
| entity         | str    | True     | -       | Entity to link. |
| entity_type    | str    | False    | -       | Type of the entity. Specifying it will make the linking process a little bit faster and possibly more accurate. Allowed options are: ["PER", "ORG", "LOC", "EMS_KEYWORD"] |
| fuzziness      | int    | False    | 2       | Maximum edit distance ([Levenshtein distance](https://en.wikipedia.org/wiki/Levenshtein_distance)) used for linking. NB! **Allowed values are: 0, 1, 2**. |
| prefix_length  | int    | False    | 1       | Number of prefix symbols that need to match exactly. |
| context        | str    | False    | None    | Some contextual information about the entity. This will be vectorized and compared to the vectors stored in a corresponding Elasticsearch index to select the likeliest match in case multiple entities with the same similarity score are returned. |
| min_similarity | float  | False    | 0.9     | Minimum required [Jaro-Winkler distance](https://en.wikipedia.org/wiki/Jaro%E2%80%93Winkler_distance). The matches not surpassing it, will NOT be returned, even if they pass the fuzziness threshold. |

##### Result 

`link` will return a result of type `LinkingResult`.

###### `LinkingResult` Class

**Attributes:**

| Name              | Type            | Description                                |
| ----------------- | --------------- | -------------------------------------------|
| original_entity   | str             | The original entity passed to the function. |
| entity_type       | str             | Type of the linked entity. One of the following types: ["PER", "ORG", "LOC", "EMS_KEYWORD", "UNKNOWN"] |
| linked_info       | List[LinkedDoc] | List of the linked entities. See `LinkedDoc` for more specific information. |
| linking_config    | dict            | The configuration used for linking. |
| n_linked          | int             | Number of linked entities. |
| similarity_score  | float           | Similarity score of the linked entity/entities. If multiple entities are returned, they all have the same similarity score as lower ones are always filtered out. |

**Funtcions:**

`to_dict()` - Converts all information stored in the class into a dictionary.

###### `LinkedDoc` Class

**Attributes**

| Name             | Type  | Description                                |
| -----------------| ------| -------------------------------------------|
| viaf             | dict  | Information about the entity retrieved from VIAF. |
| json             | dict  | All information about the entity in JSON format (converted directly from MARC21-I). |
| marc             | str   | All information about the entity in MARC21-I format. |
| linked_entity    | str   | Normalized entity. |
| elastic          | dict  | All information about the entity stored in Elasticsearch's index. |
| similarity_score | float | Similarity score of the linked entity.

**Funtcions:**

`to_dict()` - Converts all information stored in the class into a dictionary.


#### Usage Examples

##### Example 1: Simple Usage

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Lennar Mere"

linked = linker.link(entity)
pprint(linked.to_dict())
```

<details><summary>Raw Output</summary>

```
{'entity_type': 'PER',
 'linked_info': [{'elastic': {'birth_year': 1929,
                              'death_year': 2006,
                              'description': 'Eesti riigitegelane, aja- ja '
                                             'kultuuriloolane ning esseist. '
                                             'A-tel 1992-2001 EV president. '
                                             'Tõlkija ja diplomaadi Georg Meri '
                                             '(1900-1983) poeg, Hindrek Meri '
                                             'vend',
                              'identifier': 'a11133193',
                              'identifier_source': 'ErRR',
                              'life_year': '1929-2006',
                              'link_variations': ['meri, lennart',
                                                  'meri, lennart-georg',
                                                  'lennart meri',
                                                  'meri, lennarts',
                                                  'lennart-georg meri',
                                                  'леннарт-георг мери',
                                                  'мери, леннарт-георг',
                                                  'мери, леннарт',
                                                  'леннарт мери',
                                                  'lennarts meri'],
                              'name': 'Meri, Lennart',
                              'name_in_cyrillic': False,
                              'name_specification': '',
                              'name_variations': ['Meri, Lennart-Georg',
                                                  'Meri, Lennarts',
                                                  'Мери, Леннарт',
                                                  'Мери, Леннарт-Георг'],
                              'source': 'Eesti kirjanike leksikon, 2000 ja EE, '
                                        '14. kd., 2000'},
                  'json': {'fields': [{'001': 'a11133193'},
                                      {'003': 'ErRR'},
                                      {'008': '990104|n|adnnnaabn          || '
                                              '|a|      '},
                                      {'040': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'ErRR'},
                                                             {'b': 'est'},
                                                             {'c': 'ErRR'},
                                                             {'d': 'ErTrtKR'}]}},
                                      {'043': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'c': 'ee'}]}},
                                      {'100': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Meri, '
                                                                   'Lennart,'},
                                                             {'d': '1929-2006.'}]}},
                                      {'400': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Meri, '
                                                                   'Lennart-Georg.'}]}},
                                      {'400': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Meri, '
                                                                   'Lennarts.'}]}},
                                      {'400': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Мери, '
                                                                   'Леннарт,'},
                                                             {'d': '1929-2006.'}]}},
                                      {'400': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Мери, '
                                                                   'Леннарт-Георг,'},
                                                             {'d': '1929-2006.'}]}},
                                      {'670': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Eesti '
                                                                   'kirjanike '
                                                                   'leksikon, '
                                                                   '2000 ja '
                                                                   'EE, 14. '
                                                                   'kd., '
                                                                   '2000.'}]}},
                                      {'680': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'i': 'Eesti '
                                                                   'riigitegelane, '
                                                                   'aja- ja '
                                                                   'kultuuriloolane '
                                                                   'ning '
                                                                   'esseist. '
                                                                   'A-tel '
                                                                   '1992-2001 '
                                                                   'EV '
                                                                   'president. '
                                                                   'Tõlkija ja '
                                                                   'diplomaadi '
                                                                   'Georg Meri '
                                                                   '(1900-1983) '
                                                                   'poeg, '
                                                                   'Hindrek '
                                                                   'Meri '
                                                                   'vend.'}]}}],
                           'leader': '00654nz  a2200169n  4500'},
                  'linked_entity': 'Meri, Lennart',
                  'marc': '=LDR  00654nz  a2200169n  4500\n'
                          '=001  a11133193\n'
                          '=003  ErRR\n'
                          '=008  '
                          '990104|n|adnnnaabn\\\\\\\\\\\\\\\\\\\\||\\|a|\\\\\\\\\\\\\n'
                          '=040  \\\\$aErRR$best$cErRR$dErTrtKR\n'
                          '=043  \\\\$cee\n'
                          '=100  1\\$aMeri, Lennart,$d1929-2006.\n'
                          '=400  1\\$aMeri, Lennart-Georg.\n'
                          '=400  1\\$aMeri, Lennarts.\n'
                          '=400  1\\$aМери, Леннарт,$d1929-2006.\n'
                          '=400  1\\$aМери, Леннарт-Георг,$d1929-2006.\n'
                          '=670  \\\\$aEesti kirjanike leksikon, 2000 ja EE, '
                          '14. kd., 2000.\n'
                          '=680  \\\\$iEesti riigitegelane, aja- ja '
                          'kultuuriloolane ning esseist. A-tel 1992-2001 EV '
                          'president. Tõlkija ja diplomaadi Georg Meri '
                          '(1900-1983) poeg, Hindrek Meri vend.\n',
                  'similarity_score': 0.9484848484848484,
                  'viaf': {'message': '/api/search Successfully reached!',
                           'queryResult': {'echoedSearchRetrieveRequest': {'maximumRecords': {'type': 'xsd:nonNegativeInteger',
                                                                                              'value': 50},
                                                                           'query': {'type': 'xsd:string',
                                                                                     'value': 'local.personalNames '
                                                                                              'all '
                                                                                              '"a11133193"'},
                                                                           'recordPacking': {'type': 'xsd:string',
                                                                                             'value': 'xml'},
                                                                           'recordSchema': {'type': 'xsd:string',
                                                                                            'value': 'BriefVIAF'},
                                                                           'sortKeys': {'type': 'xsd:string',
                                                                                        'value': 'holdingscount'},
                                                                           'startRecord': {'type': 'xsd:positiveInteger',
                                                                                           'value': 1},
                                                                           'type': 'ns2:echoedSearchRetrieveRequestType',
                                                                           'version': {'type': 'xsd:string',
                                                                                       'value': 1.1},
                                                                           'xQuery': {'searchClause': {'index': {'type': 'xsd:string',
                                                                                                                 'value': 'local.personalNames'},
                                                                                                       'relation': {'type': 'ns3:relationType',
                                                                                                                    'value': [{'value': 'all'}]},
                                                                                                       'term': {'type': 'xsd:string',
                                                                                                                'value': 'a11133193'},
                                                                                                       'type': 'ns3:searchClauseType'}}},
                                           'extraResponseData': {'extraData': {'databaseTitle': 'VIAF: '
                                                                                                'The '
                                                                                                'Virtual '
                                                                                                'International '
                                                                                                'Authority '
                                                                                                'File'},
                                                                 'type': 'ns4:extraDataType'},
                                           'numberOfRecords': {'type': 'xsd:nonNegativeInteger',
                                                               'value': 1},
                                           'records': {'record': [{'recordData': {'VIAFCluster': {'mainHeadings': {'data': [{'sources': {'s': ['DNB',
                                                                                                                                               'NKC',
                                                                                                                                               'LIH',
                                                                                                                                               'ERRR',
                                                                                                                                               'NUKAT',
                                                                                                                                               'SUDOC',
                                                                                                                                               'LNB',
                                                                                                                                               'PLWABN',
                                                                                                                                               'BNF',
                                                                                                                                               'NTA',
                                                                                                                                               'SZ',
                                                                                                                                               'SELIBR'],
                                                                                                                                         'sid': ['DNB|11930063X',
                                                                                                                                                 'NKC|jo2003190730',
                                                                                                                                                 'LIH|LNB:_b_D_o_;=B_l_',
                                                                                                                                                 'ERRR|a11133193',
                                                                                                                                                 'NUKAT|n '
                                                                                                                                                 '2004098863',
                                                                                                                                                 'SUDOC|031225608',
                                                                                                                                                 'LNB|LNC10-000015183',
                                                                                                                                                 'PLWABN|9810611048005606',
                                                                                                                                                 'BNF|12248233',
                                                                                                                                                 'NTA|070005389',
                                                                                                                                                 'SZ|11930063X',
                                                                                                                                                 'SELIBR|tb4zmtkqrlq5bsk2']},
                                                                                                                             'text': 'Meri, '
                                                                                                                                     'Lennart, '
                                                                                                                                     '1929-2006.'},
                                                                                                                            {'sources': {'s': ['BIBSYS',
                                                                                                                                               'LC',
                                                                                                                                               'RERO',
                                                                                                                                               'NYNYRILM',
                                                                                                                                               'NLA',
                                                                                                                                               'NII'],
                                                                                                                                         'sid': ['BIBSYS|90264114',
                                                                                                                                                 'LC|n  '
                                                                                                                                                 '88622494',
                                                                                                                                                 'RERO|A005679822',
                                                                                                                                                 'NYNYRILM|389279',
                                                                                                                                                 'NLA|000049286517',
                                                                                                                                                 'NII|DA19756873']},
                                                                                                                             'text': 'Meri, '
                                                                                                                                     'Lennart'},
                                                                                                                            {'sources': {'s': ['WKP'],
                                                                                                                                         'sid': ['WKP|Q153149']},
                                                                                                                             'text': 'Lennart-Georg '
                                                                                                                                     'Meri'},
                                                                                                                            {'sources': {'s': ['ISNI'],
                                                                                                                                         'sid': ['ISNI|0000000078320195']},
                                                                                                                             'text': 'Lennart '
                                                                                                                                     'Meri'}]},
                                                                                                  'nameType': 'Personal',
                                                                                                  'titles': {'work': [{'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|12248233']},
                                                                                                                       'title': '1940-1988 '
                                                                                                                                '(&#xA7; '
                                                                                                                                '58)'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Ahvide '
                                                                                                                                'planeet'},
                                                                                                                      {'sources': {'s': ['DNB'],
                                                                                                                                   'sid': ['DNB|11930063X']},
                                                                                                                       'title': 'Baltikum '
                                                                                                                                '- '
                                                                                                                                'Pr&#xFC;fstein '
                                                                                                                                'f&#xFC;r '
                                                                                                                                'die '
                                                                                                                                'Union '
                                                                                                                                'Europas'},
                                                                                                                      {'sources': {'s': ['DNB',
                                                                                                                                         'LC'],
                                                                                                                                   'sid': ['DNB|11930063X',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Botschaften '
                                                                                                                                'und '
                                                                                                                                'Zukunftsvisionen '
                                                                                                                                'Reden '
                                                                                                                                'des '
                                                                                                                                'estnischen '
                                                                                                                                'Pr&#xE4;sidenten'},
                                                                                                                      {'sources': {'s': ['SUDOC'],
                                                                                                                                   'sid': ['SUDOC|031225608']},
                                                                                                                       'title': 'Dans '
                                                                                                                                'le '
                                                                                                                                'silence '
                                                                                                                                'des '
                                                                                                                                'glaces'},
                                                                                                                      {'sources': {'s': ['SUDOC'],
                                                                                                                                   'sid': ['SUDOC|031225608']},
                                                                                                                       'title': 'Deportation '
                                                                                                                                'from '
                                                                                                                                'Estonia '
                                                                                                                                'to '
                                                                                                                                'Russia '
                                                                                                                                ': '
                                                                                                                                'deportation '
                                                                                                                                'in '
                                                                                                                                'March '
                                                                                                                                '1949'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Eesti '
                                                                                                                                'identiteet '
                                                                                                                                'ja '
                                                                                                                                'iseseisvus'},
                                                                                                                      {'sources': {'s': ['NLA',
                                                                                                                                         'LC'],
                                                                                                                                   'sid': ['NLA|000049286517',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Eesti '
                                                                                                                                'kirjanduse '
                                                                                                                                'biog. '
                                                                                                                                'leks., '
                                                                                                                                '1975:'},
                                                                                                                      {'sources': {'s': ['LC'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Eesti '
                                                                                                                                'maailmas '
                                                                                                                                '21. '
                                                                                                                                'sajandi '
                                                                                                                                'k&#xFC;nnisel '
                                                                                                                                ': '
                                                                                                                                'Eesti '
                                                                                                                                'Vabariigi '
                                                                                                                                'presidendi '
                                                                                                                                'Lennart '
                                                                                                                                'Meri '
                                                                                                                                '70. '
                                                                                                                                's&#xFC;nnip&#xE4;evale '
                                                                                                                                'p&#xFC;hendatud '
                                                                                                                                'konverentsi '
                                                                                                                                'kogumik '
                                                                                                                                ': '
                                                                                                                                'Tartus, '
                                                                                                                                '27. '
                                                                                                                                'm&#xE4;rtsil '
                                                                                                                                '1999.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Eesti '
                                                                                                                                'rahva '
                                                                                                                                'elulood.'},
                                                                                                                      {'sources': {'s': ['DNB',
                                                                                                                                         'SUDOC'],
                                                                                                                                   'sid': ['DNB|11930063X',
                                                                                                                                           'SUDOC|031225608']},
                                                                                                                       'title': 'Es '
                                                                                                                                'zog '
                                                                                                                                'uns '
                                                                                                                                'nach '
                                                                                                                                'Kamtschatka'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'LC'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Freedom '
                                                                                                                                'through '
                                                                                                                                'democracy, '
                                                                                                                                'security '
                                                                                                                                'and '
                                                                                                                                'unity '
                                                                                                                                'in '
                                                                                                                                'diversity '
                                                                                                                                ': '
                                                                                                                                'memorable '
                                                                                                                                'words '
                                                                                                                                'of '
                                                                                                                                'Lennart '
                                                                                                                                'Meri, '
                                                                                                                                'President '
                                                                                                                                'of '
                                                                                                                                'the '
                                                                                                                                'Republic '
                                                                                                                                'of '
                                                                                                                                'Estonia, '
                                                                                                                                'from '
                                                                                                                                'his '
                                                                                                                                'speeches '
                                                                                                                                '1992-2001'},
                                                                                                                      {'sources': {'s': ['PLWABN',
                                                                                                                                         'NUKAT'],
                                                                                                                                   'sid': ['PLWABN|9810611048005606',
                                                                                                                                           'NUKAT|n '
                                                                                                                                           '2004098863']},
                                                                                                                       'title': 'Gor&#x105;ce '
                                                                                                                                'wodospady '
                                                                                                                                '/ '
                                                                                                                                'Lennart '
                                                                                                                                'Meri. '
                                                                                                                                '- '
                                                                                                                                'Warszawa, '
                                                                                                                                '1971.'},
                                                                                                                      {'id': 'VIAF|5551155832947733490005',
                                                                                                                       'sources': {'s': ['XR',
                                                                                                                                         'ERRR',
                                                                                                                                         'BIBSYS',
                                                                                                                                         'LC',
                                                                                                                                         'SUDOC',
                                                                                                                                         'LNB',
                                                                                                                                         'WKP',
                                                                                                                                         'BNF',
                                                                                                                                         'NTA',
                                                                                                                                         'NII'],
                                                                                                                                   'sid': ['XR|VIAFWORKLCno2019067182',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'BIBSYS|90264114',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494',
                                                                                                                                           'SUDOC|031225608',
                                                                                                                                           'LNB|LNC10-000015183',
                                                                                                                                           'WKP|Q153149',
                                                                                                                                           'BNF|12248233',
                                                                                                                                           'NTA|070005389',
                                                                                                                                           'NII|DA19756873']},
                                                                                                                       'title': 'H&#xF5;bevalge.'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193']},
                                                                                                                       'title': 'Hopeanvalkea '
                                                                                                                                ': '
                                                                                                                                'matka '
                                                                                                                                'menneeseen '
                                                                                                                                'oppaina '
                                                                                                                                'aurinko, '
                                                                                                                                'fantasia '
                                                                                                                                'ja '
                                                                                                                                'folklore'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Ie&#x161;kant '
                                                                                                                                'prarastos '
                                                                                                                                '&#x161;ypsenos '
                                                                                                                                ': '
                                                                                                                                'dienora&#x161;tis '
                                                                                                                                'apie '
                                                                                                                                'kelion&#x119; '
                                                                                                                                '&#x12F; '
                                                                                                                                '160 '
                                                                                                                                'meridian&#x105;'},
                                                                                                                      {'id': 'VIAF|309479565',
                                                                                                                       'sources': {'s': ['XR'],
                                                                                                                                   'sid': ['XR|VIAFWORKLCn  '
                                                                                                                                           '00025583']},
                                                                                                                       'title': 'Ilmamaa.'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'LNB',
                                                                                                                                         'BNF',
                                                                                                                                         'LC'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'LNB|LNC10-000015183',
                                                                                                                                           'BNF|12248233',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Kaks '
                                                                                                                                'ajalugu, '
                                                                                                                                'seljad '
                                                                                                                                'vastamisi '
                                                                                                                                ': '
                                                                                                                                'aulaloeng, '
                                                                                                                                '14. '
                                                                                                                                'mail '
                                                                                                                                '1996'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193']},
                                                                                                                       'title': 'Kamt&#x161;atka '
                                                                                                                                ': '
                                                                                                                                'tulivuorten '
                                                                                                                                'maa'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': 'K&#x101;vu '
                                                                                                                                'v&#x101;rtos'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Keskp&#xE4;evane '
                                                                                                                                'praam'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193']},
                                                                                                                       'title': 'Kobrade '
                                                                                                                                'ja '
                                                                                                                                'karakurtide '
                                                                                                                                'j&#xE4;lgedes '
                                                                                                                                ': '
                                                                                                                                'Kesk-Aasia '
                                                                                                                                'matkam&#xE4;rkmeid'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Kobr&#xE1;k '
                                                                                                                                '&#xE9;s '
                                                                                                                                'karakurtok '
                                                                                                                                'nyom&#xE1;ban.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Kommunismi '
                                                                                                                                'must '
                                                                                                                                'raamat '
                                                                                                                                ': '
                                                                                                                                'kuriteod, '
                                                                                                                                'terror, '
                                                                                                                                'repressioonid'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'BNF|12248233']},
                                                                                                                       'title': 'K&#xFC;&#xFC;ditamine '
                                                                                                                                'Eestist '
                                                                                                                                'Venemaale '
                                                                                                                                ': '
                                                                                                                                'm&#xE4;rtsik&#xFC;&#xFC;ditamine '
                                                                                                                                '1949.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Laevapoisid '
                                                                                                                                'rohelisel '
                                                                                                                                'ookeanil'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'NLA',
                                                                                                                                         'LC',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'NLA|000049286517',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': 'L&#xE4;henevad '
                                                                                                                                'rannad '
                                                                                                                                ': '
                                                                                                                                'reisid '
                                                                                                                                '130. '
                                                                                                                                'ja '
                                                                                                                                '160. '
                                                                                                                                'meridiaani '
                                                                                                                                'vahel'},
                                                                                                                      {'sources': {'s': ['DNB'],
                                                                                                                                   'sid': ['DNB|11930063X']},
                                                                                                                       'title': 'Lennart '
                                                                                                                                'Meri '
                                                                                                                                '1929 '
                                                                                                                                '- '
                                                                                                                                '2006 '
                                                                                                                                '; '
                                                                                                                                'spezial'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR',
                                                                                                                                         'RERO'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'RERO|A005679822']},
                                                                                                                       'title': 'Lennart '
                                                                                                                                'Meri, '
                                                                                                                                'ein '
                                                                                                                                'Leben '
                                                                                                                                'f&#xFC;r '
                                                                                                                                'Estland '
                                                                                                                                ': '
                                                                                                                                'Dialog '
                                                                                                                                'mit '
                                                                                                                                'dem '
                                                                                                                                'Pr&#xE4;sidenten'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Le '
                                                                                                                                'livre '
                                                                                                                                'noir '
                                                                                                                                'du '
                                                                                                                                'communisme.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Look '
                                                                                                                                'back '
                                                                                                                                'in '
                                                                                                                                'anger '
                                                                                                                                ': '
                                                                                                                                'a '
                                                                                                                                'play '
                                                                                                                                'in '
                                                                                                                                'three '
                                                                                                                                'acts.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Maailm '
                                                                                                                                'ja '
                                                                                                                                'meie'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Mees, '
                                                                                                                                'kes '
                                                                                                                                'k&#xE4;is '
                                                                                                                                'l&#xE4;bi '
                                                                                                                                'seina '
                                                                                                                                ': '
                                                                                                                                '[novellid, '
                                                                                                                                'n&#xE4;idend]'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Meie '
                                                                                                                                'mees '
                                                                                                                                'Havannas'},
                                                                                                                      {'sources': {'s': ['NKC'],
                                                                                                                                   'sid': ['NKC|jo2003190730']},
                                                                                                                       'title': 'Most '
                                                                                                                                'v '
                                                                                                                                'beloje '
                                                                                                                                'bezmolvije'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Our '
                                                                                                                                'man '
                                                                                                                                'in '
                                                                                                                                'Havanna.'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|12248233']},
                                                                                                                       'title': 'peinture '
                                                                                                                                'estonienne '
                                                                                                                                'au '
                                                                                                                                'XX&#xE8;me '
                                                                                                                                'si&#xE8;cle'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Piirideta '
                                                                                                                                'maailm '
                                                                                                                                ': '
                                                                                                                                'valmistumine '
                                                                                                                                '21. '
                                                                                                                                'sajandi '
                                                                                                                                'kapitalismiks'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Piirilinn '
                                                                                                                                'M&#xF5;isak&#xFC;la.'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'La '
                                                                                                                                'plan&#xE8;te '
                                                                                                                                'des '
                                                                                                                                'singes.'},
                                                                                                                      {'sources': {'s': ['NKC',
                                                                                                                                         'ERRR'],
                                                                                                                                   'sid': ['NKC|jo2003190730',
                                                                                                                                           'ERRR|a11133193']},
                                                                                                                       'title': 'Pod '
                                                                                                                                'klenbou '
                                                                                                                                'pol&#xE1;rn&#xED; '
                                                                                                                                'z&#xE1;&#x159;e'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'P&#xF5;hiseadus '
                                                                                                                                'ja '
                                                                                                                                'P&#xF5;hiseaduse '
                                                                                                                                'Assamblee '
                                                                                                                                ': '
                                                                                                                                'koguteos'},
                                                                                                                      {'id': 'VIAF|9345163464740205680007',
                                                                                                                       'sources': {'s': ['ERRR',
                                                                                                                                         'SUDOC',
                                                                                                                                         'LC',
                                                                                                                                         'XR',
                                                                                                                                         'NTA',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'SUDOC|031225608',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494',
                                                                                                                                           'XR|VIAFWORK1269735045',
                                                                                                                                           'NTA|070005389',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': 'Poliitiline '
                                                                                                                                'testament'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'BNF|12248233']},
                                                                                                                       'title': 'Poliitilised '
                                                                                                                                'arreteerimised '
                                                                                                                                'Eestis.'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'BNF|12248233']},
                                                                                                                       'title': 'Political '
                                                                                                                                'arrests '
                                                                                                                                'in '
                                                                                                                                'Estonia'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Population '
                                                                                                                                'ageing '
                                                                                                                                'in '
                                                                                                                                'Estonia.'},
                                                                                                                      {'id': 'VIAF|316341491',
                                                                                                                       'sources': {'s': ['ERRR',
                                                                                                                                         'BNF',
                                                                                                                                         'LC',
                                                                                                                                         'SUDOC',
                                                                                                                                         'XR',
                                                                                                                                         'NTA',
                                                                                                                                         'LNB',
                                                                                                                                         'WKP'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'BNF|12248233',
                                                                                                                                           'LC|n  '
                                                                                                                                           '88622494',
                                                                                                                                           'SUDOC|031225608',
                                                                                                                                           'XR|VIAFWORK243846008',
                                                                                                                                           'NTA|070005389',
                                                                                                                                           'LNB|LNC10-000015183',
                                                                                                                                           'WKP|Q153149']},
                                                                                                                       'title': 'Presidendik&#xF4;ned'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Raudne '
                                                                                                                                'kodu '
                                                                                                                                ': '
                                                                                                                                'Evald '
                                                                                                                                'Tammlaane '
                                                                                                                                'draama '
                                                                                                                                'lauludega '
                                                                                                                                '3-es '
                                                                                                                                'vaatuses '
                                                                                                                                'proloogi '
                                                                                                                                'ja '
                                                                                                                                'epiloogiga'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|12248233']},
                                                                                                                       'title': 'Reisikiri '
                                                                                                                                'suurest '
                                                                                                                                'paugust, '
                                                                                                                                'tuulest '
                                                                                                                                'ja '
                                                                                                                                'muinasluulest'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193']},
                                                                                                                       'title': 'Revontulten '
                                                                                                                                'porteilla'},
                                                                                                                      {'sources': {'s': ['LNB'],
                                                                                                                                   'sid': ['LNB|LNC10-000015183']},
                                                                                                                       'title': 'R&#x12B;gas '
                                                                                                                                'Balss '
                                                                                                                                '15.marts '
                                                                                                                                '(Nr.53), '
                                                                                                                                '2006:'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'NTA'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'NTA|070005389']},
                                                                                                                       'title': 'Riigimured'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Sajandi '
                                                                                                                                'sada '
                                                                                                                                'elulugu '
                                                                                                                                'kahes '
                                                                                                                                'osas.'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'NTA'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'NTA|070005389']},
                                                                                                                       'title': '&#x160;amaan'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'NTA'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'NTA|070005389']},
                                                                                                                       'title': 'Soome-ugri '
                                                                                                                                'rahvaste '
                                                                                                                                'filmients&#xFC;klopeedia '
                                                                                                                                ': '
                                                                                                                                'viis '
                                                                                                                                'dokumentaalfilm, '
                                                                                                                                '1970-1997 '
                                                                                                                                '= '
                                                                                                                                'Encyclopaedia '
                                                                                                                                'cinematographica '
                                                                                                                                'gentium '
                                                                                                                                'Fenno-Ugricarum '
                                                                                                                                ': '
                                                                                                                                'five '
                                                                                                                                'documentaries, '
                                                                                                                                '1970-1997'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Soome-ugri '
                                                                                                                                'rahvaste '
                                                                                                                                'VI '
                                                                                                                                'folkloorifestival '
                                                                                                                                'Eestis '
                                                                                                                                '17.-21. '
                                                                                                                                'VII '
                                                                                                                                '1997 '
                                                                                                                                '= '
                                                                                                                                'VI '
                                                                                                                                '&#x444;&#x438;&#x43D;&#x43D;&#x43E;-&#x443;&#x433;&#x43E;&#x440;&#x441;&#x43A;&#x438;&#x439; '
                                                                                                                                '&#x444;&#x43E;&#x43B;&#x44C;&#x43A;&#x43B;&#x43E;&#x440;&#x43D;&#x44B;&#x439; '
                                                                                                                                '&#x444;&#x435;&#x441;&#x442;&#x438;&#x432;&#x430;&#x43B;&#x44C; '
                                                                                                                                '&#x432; '
                                                                                                                                '&#x42D;&#x441;&#x442;&#x43E;&#x43D;&#x438;&#x438;'},
                                                                                                                      {'id': 'VIAF|6246156497273917740008',
                                                                                                                       'sources': {'s': ['LC'],
                                                                                                                                   'sid': ['LC|n '
                                                                                                                                           '2019043521']},
                                                                                                                       'title': 'Speeches. '
                                                                                                                                'Selections'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Tallinna '
                                                                                                                                'saladused'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Teise '
                                                                                                                                'mehe '
                                                                                                                                'pea '
                                                                                                                                ': '
                                                                                                                                'n&#xE4;idend '
                                                                                                                                '4 '
                                                                                                                                'vaatuses'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'NTA'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'NTA|070005389']},
                                                                                                                       'title': 'Toorumi '
                                                                                                                                'pojad '
                                                                                                                                'Hantide '
                                                                                                                                'karupeied '
                                                                                                                                ': '
                                                                                                                                '[dokumentaalfilm '
                                                                                                                                'hantide '
                                                                                                                                'rahvakommetest]'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'T&#xF5;otan '
                                                                                                                                'ustavaks '
                                                                                                                                'j&#xE4;&#xE4;da... '
                                                                                                                                ': '
                                                                                                                                'Eesti '
                                                                                                                                'Vabariigi '
                                                                                                                                'Valitsus '
                                                                                                                                '1940-1992'},
                                                                                                                      {'sources': {'s': ['LC'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '88622494']},
                                                                                                                       'title': 'Tri '
                                                                                                                                'ba&#x12D;darki '
                                                                                                                                'v '
                                                                                                                                'zelenom '
                                                                                                                                'okeane.'},
                                                                                                                      {'id': 'VIAF|309516186',
                                                                                                                       'sources': {'s': ['ERRR',
                                                                                                                                         'PLWABN',
                                                                                                                                         'LNB',
                                                                                                                                         'XR'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'PLWABN|9810611048005606',
                                                                                                                                           'LNB|LNC10-000015183',
                                                                                                                                           'XR|VIAFWORKLCn '
                                                                                                                                           '2006019349']},
                                                                                                                       'title': 'Tulem&#xE4;gede '
                                                                                                                                'maale.'},
                                                                                                                      {'id': 'VIAF|307144592',
                                                                                                                       'sources': {'s': ['XR'],
                                                                                                                                   'sid': ['XR|VIAFWORK77318117']},
                                                                                                                       'title': 'Tulem&#xE4;gede '
                                                                                                                                'maale; '
                                                                                                                                'reisip&#xE4;evik '
                                                                                                                                '160. '
                                                                                                                                'meridiaanilt'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Tulen '
                                                                                                                                'maasta, '
                                                                                                                                'jonka '
                                                                                                                                'nimi '
                                                                                                                                'on '
                                                                                                                                'Viro'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Tuli '
                                                                                                                                'ei '
                                                                                                                                'kustu'},
                                                                                                                      {'sources': {'s': ['NKC'],
                                                                                                                                   'sid': ['NKC|jo2003190730']},
                                                                                                                       'title': "Udivitel'nyj "
                                                                                                                                '&#x10D;elovek '
                                                                                                                                ': '
                                                                                                                                'Kniga '
                                                                                                                                'pute&#x161;estvij'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#xDC;ks '
                                                                                                                                'p&#xE4;ev '
                                                                                                                                'Ivan '
                                                                                                                                'Denissovit&#x161;i '
                                                                                                                                'elus '
                                                                                                                                ': '
                                                                                                                                '[jutustus]'},
                                                                                                                      {'sources': {'s': ['NKC',
                                                                                                                                         'NUKAT'],
                                                                                                                                   'sid': ['NKC|jo2003190730',
                                                                                                                                           'NUKAT|n '
                                                                                                                                           '2004098863']},
                                                                                                                       'title': 'V '
                                                                                                                                'poiskah '
                                                                                                                                'poter&#xE2;nnoj '
                                                                                                                                'ulybki'},
                                                                                                                      {'sources': {'s': ['NTA'],
                                                                                                                                   'sid': ['NTA|070005389']},
                                                                                                                       'title': 'Veelinnurahvas'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Viimne '
                                                                                                                                'reliikvia'},
                                                                                                                      {'id': 'VIAF|307013366',
                                                                                                                       'sources': {'s': ['SUDOC',
                                                                                                                                         'ERRR',
                                                                                                                                         'XR',
                                                                                                                                         'BIBSYS',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['SUDOC|031225608',
                                                                                                                                           'ERRR|a11133193',
                                                                                                                                           'XR|VIAFWORK249662780',
                                                                                                                                           'BIBSYS|90264114',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': 'Virmaliste '
                                                                                                                                'v&#xE4;raval'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': 'Virolais-suomalaiset '
                                                                                                                                'laulu- '
                                                                                                                                'ja '
                                                                                                                                'soittojuhlat '
                                                                                                                                'Tallinnassa'},
                                                                                                                      {'sources': {'s': ['PLWABN'],
                                                                                                                                   'sid': ['PLWABN|9810611048005606']},
                                                                                                                       'title': 'Z '
                                                                                                                                'g&#xF3;ry '
                                                                                                                                'wi&#x119;cej '
                                                                                                                                'wida&#x107;'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': '&#x412; '
                                                                                                                                '&#x43F;&#x43E;&#x438;&#x441;&#x43A;&#x430;&#x445; '
                                                                                                                                '&#x43F;&#x43E;&#x442;&#x435;&#x440;&#x44F;&#x43D;&#x43D;&#x43E;&#x439; '
                                                                                                                                '&#x443;&#x43B;&#x44B;&#x431;&#x43A;&#x438; '
                                                                                                                                ': '
                                                                                                                                '(&#x434;&#x43D;&#x435;&#x432;&#x43D;&#x438;&#x43A; '
                                                                                                                                '&#x43F;&#x443;&#x442;&#x435;&#x448;&#x435;&#x441;&#x442;&#x432;&#x438;&#x44F; '
                                                                                                                                '&#x43A; '
                                                                                                                                '160-&#x43C;&#x443; '
                                                                                                                                '&#x43C;&#x435;&#x440;&#x438;&#x434;&#x438;&#x430;&#x43D;&#x443;)'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#x418;&#x441;&#x441;&#x43B;&#x435;&#x434;&#x43E;&#x432;&#x430;&#x43D;&#x438;&#x435; '
                                                                                                                                '&#x43F;&#x440;&#x438;&#x440;&#x43E;&#x434;&#x44B; '
                                                                                                                                '&#x414;&#x430;&#x43B;&#x44C;&#x43D;&#x435;&#x433;&#x43E; '
                                                                                                                                '&#x412;&#x43E;&#x441;&#x442;&#x43E;&#x43A;&#x430;'},
                                                                                                                      {'sources': {'s': ['ERRR',
                                                                                                                                         'LNB'],
                                                                                                                                   'sid': ['ERRR|a11133193',
                                                                                                                                           'LNB|LNC10-000015183']},
                                                                                                                       'title': '&#x41C;&#x43E;&#x441;&#x442; '
                                                                                                                                '&#x432; '
                                                                                                                                '&#x431;&#x435;&#x43B;&#x43E;&#x435; '
                                                                                                                                '&#x431;&#x435;&#x437;&#x43C;&#x43E;&#x43B;&#x432;&#x438;&#x435;'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#x41E;&#x434;&#x438;&#x43D; '
                                                                                                                                '&#x434;&#x435;&#x43D;&#x44C; '
                                                                                                                                '&#x418;&#x432;&#x430;&#x43D;&#x430; '
                                                                                                                                '&#x414;&#x435;&#x43D;&#x438;&#x441;&#x43E;&#x432;&#x438;&#x447;&#x430;.'},
                                                                                                                      {'sources': {'s': ['LNB'],
                                                                                                                                   'sid': ['LNB|LNC10-000015183']},
                                                                                                                       'title': '&#x420;&#x435;&#x441;&#x43F;&#x443;&#x431;&#x43B;&#x438;&#x43A;&#x430; '
                                                                                                                                '(&#x412;&#x438;&#x43B;&#x44C;&#x43D;&#x44E;&#x441;), '
                                                                                                                                '(28 '
                                                                                                                                '&#x438;&#x44E;&#x43D;&#x44F; '
                                                                                                                                '1999):'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#x421;&#x430;&#x43C;&#x43E;&#x43E;&#x43F;&#x440;&#x435;&#x434;&#x435;&#x43B;&#x435;&#x43D;&#x438;&#x435; '
                                                                                                                                '&#x438; '
                                                                                                                                '&#x43D;&#x435;&#x437;&#x430;&#x432;&#x438;&#x441;&#x438;&#x43C;&#x43E;&#x441;&#x442;&#x44C; '
                                                                                                                                '&#x42D;&#x441;&#x442;&#x43E;&#x43D;&#x438;&#x438;'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#x421;&#x435;&#x440;&#x435;&#x431;&#x440;&#x438;&#x441;&#x442;&#x43E;-&#x431;&#x435;&#x43B;&#x44B;&#x439; '
                                                                                                                                '&#x43F;&#x443;&#x442;&#x44C; '
                                                                                                                                '&#x41B;&#x435;&#x43D;&#x43D;&#x430;&#x440;&#x442;&#x430; '
                                                                                                                                '&#x41C;&#x435;&#x440;&#x438; '
                                                                                                                                '&#x43F;&#x443;&#x442;&#x435;&#x432;&#x44B;&#x435; '
                                                                                                                                '&#x437;&#x430;&#x43C;&#x435;&#x442;&#x43A;&#x438;. '
                                                                                                                                '&#x412;&#x44B;&#x441;&#x442;&#x443;&#x43F;&#x43B;&#x435;&#x43D;&#x438;&#x44F; '
                                                                                                                                '&#x438; '
                                                                                                                                '&#x438;&#x43D;&#x442;&#x435;&#x440;&#x432;&#x44C;&#x44E;. '
                                                                                                                                '&#x41B;&#x435;&#x43D;&#x43D;&#x430;&#x440;&#x442; '
                                                                                                                                '&#x41C;&#x435;&#x440;&#x438; '
                                                                                                                                '&#x433;&#x43B;&#x430;&#x437;&#x430;&#x43C;&#x438; '
                                                                                                                                '&#x434;&#x440;&#x443;&#x437;&#x435;&#x439;'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a11133193']},
                                                                                                                       'title': '&#x423;&#x434;&#x438;&#x432;&#x438;&#x442;&#x435;&#x43B;&#x44C;&#x43D;&#x44B;&#x439; '
                                                                                                                                '&#x447;&#x435;&#x43B;&#x43E;&#x432;&#x435;&#x43A; '
                                                                                                                                ': '
                                                                                                                                '&#x43A;&#x43D;&#x438;&#x433;&#x430; '
                                                                                                                                '&#x43F;&#x443;&#x442;&#x435;&#x448;&#x435;&#x441;&#x442;&#x432;&#x438;&#x439; '
                                                                                                                                ': '
                                                                                                                                '&#x43F;&#x440;&#x43E;&#x438;&#x437;&#x432;&#x435;&#x434;&#x435;&#x43D;&#x438;&#x44F; '
                                                                                                                                '&#x412;&#x43E;&#x43B;&#x44C;&#x434;&#x435;&#x43C;&#x430;&#x440;&#x430; '
                                                                                                                                '&#x41F;&#x430;&#x43D;&#x441;&#x43E; '
                                                                                                                                '&#x438; '
                                                                                                                                '&#x41B;&#x435;&#x43D;&#x43D;&#x430;&#x440;&#x442;&#x430; '
                                                                                                                                '&#x41C;&#x435;&#x440;&#x438; '
                                                                                                                                '&#x432; '
                                                                                                                                '&#x43F;&#x435;&#x440;&#x435;&#x432;&#x43E;&#x434;&#x435; '
                                                                                                                                '&#x441; '
                                                                                                                                '&#x44D;&#x441;&#x442;&#x43E;&#x43D;&#x441;&#x43A;&#x43E;&#x433;&#x43E; '
                                                                                                                                '&#x412;. '
                                                                                                                                '&#x420;&#x443;&#x431;&#x435;&#x440;'}]},
                                                                                                  'viafID': 84153775},
                                                                                  'type': 'ns1:stringOrXmlFragment'},
                                                                   'recordPacking': {'type': 'xsd:string',
                                                                                     'value': 'xml'},
                                                                   'recordPosition': {'type': 'xsd:positiveInteger',
                                                                                      'value': 1},
                                                                   'recordSchema': {'type': 'xsd:string',
                                                                                    'value': 'http://viaf.org/BriefVIAFCluster'},
                                                                   'type': 'ns1:recordType'}],
                                                       'type': 'ns1:recordsType'},
                                           'resultSetIdleTime': {'type': 'xsd:positiveInteger',
                                                                 'value': 1},
                                           'schemaLocation': 'http://www.loc.gov/zing/srw/ '
                                                             'http://www.loc.gov/standards/sru/sru1-1archive/xml-files/srw-types.xsd',
                                           'version': {'type': 'xsd:string',
                                                       'value': 1.1}}}}],
 'linking_config': {'add_viaf_info': True,
                    'context': None,
                    'entity': 'Lennar Mere',
                    'fuzziness': 2,
                    'min_similarity': 0.9,
                    'prefix_length': 1},
 'n_linked': 1,
 'original_entity': 'Lennar Mere',
 'similarity_score': 0.9484848484848484}
```
</details>

##### Example 2: Linker returns multiple matches

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Paul Keres"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print(f"Description: {entity_info.elastic['description']}")
   print()
```

**Output:**

```
Original entity: Paul Keres
Entity type: PER
Number of matches: 2
Similarity: 1.0

Linked entity: Keres, Paul
Description: Eesti maletaja ja maleteoreetik


Linked entity: Keres, Paul
Description: Eesti advokaat
```

##### Example 3: Add some context for vector-based filtering

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Paul Keres"
context = "Viimsis selgusid 53. maleturniiri medalistid."

linked = linker.link(entity, context=context)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print(f"Description: {entity_info.elastic['description']}")
   print()

```

**Output:**

```
Original entity: Paul Keres
Entity type: PER
Number of matches: 1
Similarity: 1.0

Linked entity: Keres, Paul
Description: Eesti maletaja ja maleteoreetik
```

##### Example 4: Link a geographical name

```python

from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Reval"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print()

```

**Output:**

```
Original entity: Reval
Entity type: LOC
Number of matches: 1
Similarity: 1.0

Linked entity: Tallinn
```

##### Example 5: Link an organization

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "NASA"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print(f"Description: {entity_info.elastic['description']}")
   print()
```

**Output:**
```
Original entity: NASA
Entity type: ORG
Number of matches: 1
Similarity: 1.0

Linked entity: United States National Aeronautics and Space Administration
Description: USA riiklik kosmose uurimise ja kosmonautika arendamise organisatsioon. Asutatud 1958. a., keskus Washingtonis
```

##### Example 6: Link a keyword

```python

from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "cinema"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print()
```

**Output:**
```
Original entity: cinema
Entity type: EMS_KEYWORD
Number of matches: 1
Similarity: 0.9714285714285714

Linked entity: kinod
```

##### Example 7: Link one-word pseudonyms / stage names

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Shakira"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print(f"Description: {entity_info.elastic['description']}")
   print()
```

**Output:**

```
Original entity: Shakira
Entity type: PER
Number of matches: 1
Similarity: 1.0

Linked entity: Shakira
Description: Colombia laulja ja laulukirjutaja. Täisnimi Shakira Isabel Mebarak Ripoll
```

##### Example 8: Link a single surname

```python
from rara_linker.linkers.linker import Linker
from pprint import pprint
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

# As linking only first names / surnames is not supported,
# no matches should be returned
entity = "Bulgakov"
linked = linker.link(entity)

# Output formatting:
print(f"Original entity: {linked.original_entity}")
print(f"Entity type: {linked.entity_type}")
print(f"Number of matches: {linked.n_linked}")
print(f"Similarity: {linked.similarity_score}")

for entity_info in linked.linked_info:
   print()
   print(f"Linked entity: {entity_info.linked_entity}")
   print(f"Description: {entity_info.elastic['description']}")
   print()
```

**Output:**

```
Original entity: Bulgakov
Entity type: UNKNOWN
Number of matches: 0
Similarity: 0
```
</details>

## 🔍 Usage Examples

The following function is used to help formatting output:

<details><summary>Click to expand</summary>

```python
from rara_linker.linkers.linking_result import LinkingResult
from typing import NoReturn

def format_output(linked: LinkingResult) -> NoReturn:
   print(f"Original entity: {linked.original_entity}")
   print(f"Entity type: {linked.entity_type}")
   print(f"Number of matches: {linked.n_linked}")
   print(f"Similarity: {linked.similarity_score}")

   for entity_info in linked.linked_info:
      print()
      print(f"Linked entity: {entity_info.linked_entity}")
      description = entity_info.elastic.get("description", "")
      if description:
         print(f"Description: {description}")
      print()

```
</details>

### Example 1: Simple linking

```python
from rara_linker.linkers.linker import Linker
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "Damon Albarn"
linked = linker.link(entity)

format_output(linked)
```

**Output:**

```
Original entity: Damon Albarn
Entity type: PER
Number of matches: 1
Similarity: 1.0

Linked entity: Albarn, Damon
Description: Inglise muusik ja laulukirjutaja
```

```python
from pprint import pprint
# Code for displaying the raw output of the same linking result:
pprint(linked.to_dict())
```

<details><summary>Raw output</summary>

```
{'entity_type': 'PER',
 'linked_info': [{'elastic': {'birth_year': 1968,
                              'death_year': None,
                              'description': 'Inglise muusik ja laulukirjutaja',
                              'identifier': 'a12660826',
                              'identifier_source': 'ErRR',
                              'life_year': '1968-',
                              'link_variations': ['albran, damon',
                                                  'damon albran',
                                                  'damon albarn',
                                                  'albarn, damon'],
                              'name': 'Albarn, Damon',
                              'name_in_cyrillic': False,
                              'name_specification': '',
                              'name_variations': ['Albran, Damon'],
                              'source': 'Vikipeedia'},
                  'json': {'fields': [{'001': 'a12660826'},
                                      {'003': 'ErRR'},
                                      {'008': '240418|||aznnnaabn          || '
                                              '|||      '},
                                      {'040': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'ErRR'},
                                                             {'b': 'est'},
                                                             {'c': 'ErRR'},
                                                             {'e': 'rda'}]}},
                                      {'043': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'c': 'uk'}]}},
                                      {'046': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'f': '1968'}]}},
                                      {'075': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'persoon'}]}},
                                      {'100': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Albarn, '
                                                                   'Damon,'},
                                                             {'d': '1968-'}]}},
                                      {'372': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'rockmuusika'},
                                                             {'a': 'elektronmuusika'},
                                                             {'a': 'hip hop'},
                                                             {'a': 'Britpop.'}]}},
                                      {'374': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'laulukirjutaja'},
                                                             {'a': 'laulja'},
                                                             {'a': 'muusik.'}]}},
                                      {'400': {'ind1': '1',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Albran, '
                                                                   'Damon,'},
                                                             {'d': '1968-'}]}},
                                      {'670': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'a': 'Vikipeedia'},
                                                             {'u': 'https://en.wikipedia.org/wiki/Damon_Albarn.'}]}},
                                      {'680': {'ind1': ' ',
                                               'ind2': ' ',
                                               'subfields': [{'i': 'Inglise '
                                                                   'muusik ja '
                                                                   'laulukirjutaja.'}]}}],
                           'leader': '00529nz  a2200181n  4500'},
                  'linked_entity': 'Albarn, Damon',
                  'marc': '=LDR  00529nz  a2200181n  4500\n'
                          '=001  a12660826\n'
                          '=003  ErRR\n'
                          '=008  '
                          '240418|||aznnnaabn\\\\\\\\\\\\\\\\\\\\||\\|||\\\\\\\\\\\\\n'
                          '=040  \\\\$aErRR$best$cErRR$erda\n'
                          '=043  \\\\$cuk\n'
                          '=046  \\\\$f1968\n'
                          '=075  \\\\$apersoon\n'
                          '=100  1\\$aAlbarn, Damon,$d1968-\n'
                          '=372  \\\\$arockmuusika$aelektronmuusika$ahip '
                          'hop$aBritpop.\n'
                          '=374  \\\\$alaulukirjutaja$alaulja$amuusik.\n'
                          '=400  1\\$aAlbran, Damon,$d1968-\n'
                          '=670  '
                          '\\\\$aVikipeedia$uhttps://en.wikipedia.org/wiki/Damon_Albarn.\n'
                          '=680  \\\\$iInglise muusik ja laulukirjutaja.\n',
                  'similarity_score': 1.0,
                  'viaf': {'message': '/api/search Successfully reached!',
                           'queryResult': {'echoedSearchRetrieveRequest': {'maximumRecords': {'type': 'xsd:nonNegativeInteger',
                                                                                              'value': 50},
                                                                           'query': {'type': 'xsd:string',
                                                                                     'value': 'local.personalNames '
                                                                                              'all '
                                                                                              '"a12660826"'},
                                                                           'recordPacking': {'type': 'xsd:string',
                                                                                             'value': 'xml'},
                                                                           'recordSchema': {'type': 'xsd:string',
                                                                                            'value': 'BriefVIAF'},
                                                                           'sortKeys': {'type': 'xsd:string',
                                                                                        'value': 'holdingscount'},
                                                                           'startRecord': {'type': 'xsd:positiveInteger',
                                                                                           'value': 1},
                                                                           'type': 'ns2:echoedSearchRetrieveRequestType',
                                                                           'version': {'type': 'xsd:string',
                                                                                       'value': 1.1},
                                                                           'xQuery': {'searchClause': {'index': {'type': 'xsd:string',
                                                                                                                 'value': 'local.personalNames'},
                                                                                                       'relation': {'type': 'ns3:relationType',
                                                                                                                    'value': [{'value': 'all'}]},
                                                                                                       'term': {'type': 'xsd:string',
                                                                                                                'value': 'a12660826'},
                                                                                                       'type': 'ns3:searchClauseType'}}},
                                           'extraResponseData': {'extraData': {'databaseTitle': 'VIAF: '
                                                                                                'The '
                                                                                                'Virtual '
                                                                                                'International '
                                                                                                'Authority '
                                                                                                'File'},
                                                                 'type': 'ns4:extraDataType'},
                                           'numberOfRecords': {'type': 'xsd:nonNegativeInteger',
                                                               'value': 1},
                                           'records': {'record': [{'recordData': {'VIAFCluster': {'mainHeadings': {'data': [{'sources': {'s': ['DNB',
                                                                                                                                               'KRNLK',
                                                                                                                                               'PLWABN',
                                                                                                                                               'LIH',
                                                                                                                                               'BNF',
                                                                                                                                               'BNE',
                                                                                                                                               'NKC',
                                                                                                                                               'BIBSYS',
                                                                                                                                               'NUKAT',
                                                                                                                                               'ERRR',
                                                                                                                                               'SUDOC'],
                                                                                                                                         'sid': ['DNB|135275245',
                                                                                                                                                 'KRNLK|KAC2020M4718',
                                                                                                                                                 'PLWABN|9810618563005606',
                                                                                                                                                 'LIH|LNB:B2HO;=_u_Y',
                                                                                                                                                 'BNF|14025704',
                                                                                                                                                 'BNE|XX1502205',
                                                                                                                                                 'NKC|xx0042289',
                                                                                                                                                 'BIBSYS|6096767',
                                                                                                                                                 'NUKAT|n '
                                                                                                                                                 '2009143303',
                                                                                                                                                 'ERRR|a12660826',
                                                                                                                                                 'SUDOC|168995603']},
                                                                                                                             'text': 'Albarn, '
                                                                                                                                     'Damon, '
                                                                                                                                     '1968-....'},
                                                                                                                            {'sources': {'s': ['NLA',
                                                                                                                                               'ISNI',
                                                                                                                                               'LC',
                                                                                                                                               'SIMACOB',
                                                                                                                                               'RERO',
                                                                                                                                               'NSK',
                                                                                                                                               'DBC',
                                                                                                                                               'J9U'],
                                                                                                                                         'sid': ['NLA|000041317329',
                                                                                                                                                 'ISNI|0000000108754251',
                                                                                                                                                 'LC|n  '
                                                                                                                                                 '97085620',
                                                                                                                                                 'SIMACOB|213734755',
                                                                                                                                                 'RERO|A002915097',
                                                                                                                                                 'NSK|000760001',
                                                                                                                                                 'DBC|87097969360297',
                                                                                                                                                 'J9U|987011827765305171']},
                                                                                                                             'text': 'Albarn, '
                                                                                                                                     'Damon'},
                                                                                                                            {'sources': {'s': ['WKP'],
                                                                                                                                         'sid': ['WKP|Q272069']},
                                                                                                                             'text': 'Damon '
                                                                                                                                     'Albarn '
                                                                                                                                     'English '
                                                                                                                                     'musician, '
                                                                                                                                     'singer-songwriter, '
                                                                                                                                     'and '
                                                                                                                                     'record '
                                                                                                                                     'producer'}]},
                                                                                                  'nameType': 'Personal',
                                                                                                  'titles': {'work': [{'sources': {'s': ['DNB',
                                                                                                                                         'WKP',
                                                                                                                                         'RERO',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['DNB|135275245',
                                                                                                                                           'WKP|Q272069',
                                                                                                                                           'RERO|A002915097',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': '101 '
                                                                                                                                'Reykjav&#xED;k'},
                                                                                                                      {'sources': {'s': ['NUKAT'],
                                                                                                                                   'sid': ['NUKAT|n '
                                                                                                                                           '2009143303']},
                                                                                                                       'title': 'Anna '
                                                                                                                                'and '
                                                                                                                                'the '
                                                                                                                                'moods'},
                                                                                                                      {'sources': {'s': ['BIBSYS'],
                                                                                                                                   'sid': ['BIBSYS|6096767']},
                                                                                                                       'title': 'Anna '
                                                                                                                                'g&#xE5;r '
                                                                                                                                'i '
                                                                                                                                'svart'},
                                                                                                                      {'sources': {'s': ['NUKAT'],
                                                                                                                                   'sid': ['NUKAT|n '
                                                                                                                                           '2009143303']},
                                                                                                                       'title': 'Anna '
                                                                                                                                'i '
                                                                                                                                'humorki'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Apple '
                                                                                                                                'carts '
                                                                                                                                '(2 '
                                                                                                                                'min '
                                                                                                                                '36 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['DBC'],
                                                                                                                                   'sid': ['DBC|87097969360297']},
                                                                                                                       'title': 'Arbejdsnarkoman '
                                                                                                                                'uden '
                                                                                                                                'en '
                                                                                                                                'plan'},
                                                                                                                      {'sources': {'s': ['DNB'],
                                                                                                                                   'sid': ['DNB|135275245']},
                                                                                                                       'title': 'ballad '
                                                                                                                                'of '
                                                                                                                                'Darren'},
                                                                                                                      {'sources': {'s': ['DBC',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['DBC|87097969360297',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Bananaz'},
                                                                                                                      {'sources': {'s': ['RERO'],
                                                                                                                                   'sid': ['RERO|A002915097']},
                                                                                                                       'title': 'Broken'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Call '
                                                                                                                                'me '
                                                                                                                                'up'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Cathedrals '
                                                                                                                                '(3 '
                                                                                                                                'min)'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Coffee '
                                                                                                                                '& '
                                                                                                                                'TV'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'com&#xE9;die'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Coronation '
                                                                                                                                '(1 '
                                                                                                                                'min '
                                                                                                                                '10 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['SIMACOB'],
                                                                                                                                   'sid': ['SIMACOB|213734755']},
                                                                                                                       'title': 'Cracker '
                                                                                                                                'Island'},
                                                                                                                      {'sources': {'s': ['SUDOC'],
                                                                                                                                   'sid': ['SUDOC|168995603']},
                                                                                                                       'title': 'Damon '
                                                                                                                                'Albarn '
                                                                                                                                ': '
                                                                                                                                'Blur, '
                                                                                                                                'Gorillaz '
                                                                                                                                'and '
                                                                                                                                'other '
                                                                                                                                'fables'},
                                                                                                                      {'sources': {'s': ['PLWABN'],
                                                                                                                                   'sid': ['PLWABN|9810618563005606']},
                                                                                                                       'title': 'Damoniczny '
                                                                                                                                '&#x15B;wiat'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'dancing '
                                                                                                                                'king '
                                                                                                                                '(3 '
                                                                                                                                'min '
                                                                                                                                '21 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Dare'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Democrazy'},
                                                                                                                      {'sources': {'s': ['LC',
                                                                                                                                         'DNB'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '97085620',
                                                                                                                                           'DNB|135275245']},
                                                                                                                       'title': 'Doctor '
                                                                                                                                'Dee'},
                                                                                                                      {'id': 'VIAF|7806173669165707660003',
                                                                                                                       'sources': {'s': ['DNB',
                                                                                                                                         'RERO',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF',
                                                                                                                                         'LC',
                                                                                                                                         'NYNYRILM'],
                                                                                                                                   'sid': ['DNB|135275245',
                                                                                                                                           'RERO|A002915097',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704',
                                                                                                                                           'LC|n  '
                                                                                                                                           '97085620',
                                                                                                                                           'NYNYRILM|95370']},
                                                                                                                       'title': 'Dr '
                                                                                                                                'Dee: '
                                                                                                                                'An '
                                                                                                                                'English '
                                                                                                                                'opera'},
                                                                                                                      {'sources': {'s': ['PLWABN'],
                                                                                                                                   'sid': ['PLWABN|9810618563005606']},
                                                                                                                       'title': 'Drapie&#x17C;cy'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Dude, '
                                                                                                                                'Where&#x2019;s '
                                                                                                                                'My '
                                                                                                                                'Car?'},
                                                                                                                      {'sources': {'s': ['ERRR'],
                                                                                                                                   'sid': ['ERRR|a12660826']},
                                                                                                                       'title': 'Euro '
                                                                                                                                'dance '
                                                                                                                                '1999'},
                                                                                                                      {'sources': {'s': ['DNB',
                                                                                                                                         'SIMACOB',
                                                                                                                                         'RERO',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['DNB|135275245',
                                                                                                                                           'SIMACOB|213734755',
                                                                                                                                           'RERO|A002915097',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Everyday '
                                                                                                                                'robots'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Fatti, '
                                                                                                                                'strafatti '
                                                                                                                                'e '
                                                                                                                                'strafighe'},
                                                                                                                      {'sources': {'s': ['LC',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '97085620',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Film '
                                                                                                                                'of '
                                                                                                                                'life'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Girls '
                                                                                                                                '& '
                                                                                                                                'Boys'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Give '
                                                                                                                                'it '
                                                                                                                                'to '
                                                                                                                                'the '
                                                                                                                                'people'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Give '
                                                                                                                                'me'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'golden '
                                                                                                                                'dawn'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'good, '
                                                                                                                                'the '
                                                                                                                                'bad '
                                                                                                                                '&the '
                                                                                                                                'queen '
                                                                                                                                'Herculean'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Heavy '
                                                                                                                                'seas '
                                                                                                                                'of '
                                                                                                                                'love '
                                                                                                                                '(3 '
                                                                                                                                'min '
                                                                                                                                '44 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'history '
                                                                                                                                'of '
                                                                                                                                'a '
                                                                                                                                'cheating '
                                                                                                                                'heart '
                                                                                                                                '(4 '
                                                                                                                                'min)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Hollow '
                                                                                                                                'ponds '
                                                                                                                                '(4 '
                                                                                                                                'min '
                                                                                                                                '59 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Honest '
                                                                                                                                'Jons '
                                                                                                                                'sampler'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Hostiles '
                                                                                                                                '(4 '
                                                                                                                                'min '
                                                                                                                                '09 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['LC'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '97085620']},
                                                                                                                       'title': 'The '
                                                                                                                                'isle '
                                                                                                                                'of '
                                                                                                                                'view'},
                                                                                                                      {'sources': {'s': ['SIMACOB'],
                                                                                                                                   'sid': ['SIMACOB|213734755']},
                                                                                                                       'title': 'Live '
                                                                                                                                'forever  '
                                                                                                                                'the '
                                                                                                                                'rise '
                                                                                                                                'and '
                                                                                                                                'fall '
                                                                                                                                'of '
                                                                                                                                'Brit '
                                                                                                                                'pop'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Lonely '
                                                                                                                                'press '
                                                                                                                                'play '
                                                                                                                                '(3 '
                                                                                                                                'min '
                                                                                                                                '42 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['DNB',
                                                                                                                                         'RERO',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['DNB|135275245',
                                                                                                                                           'RERO|A002915097',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Mali '
                                                                                                                                'music'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'man '
                                                                                                                                'of '
                                                                                                                                'England '
                                                                                                                                '(3 '
                                                                                                                                'min '
                                                                                                                                '17 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'marvelous '
                                                                                                                                'dream'},
                                                                                                                      {'sources': {'s': ['BIBSYS'],
                                                                                                                                   'sid': ['BIBSYS|6096767']},
                                                                                                                       'title': 'Me '
                                                                                                                                'and '
                                                                                                                                'the '
                                                                                                                                'devil'},
                                                                                                                      {'sources': {'s': ['DNB'],
                                                                                                                                   'sid': ['DNB|135275245']},
                                                                                                                       'title': 'Merrie '
                                                                                                                                'land'},
                                                                                                                      {'id': 'VIAF|210959931',
                                                                                                                       'sources': {'s': ['LC',
                                                                                                                                         'DNB',
                                                                                                                                         'WKP'],
                                                                                                                                   'sid': ['LC|n  '
                                                                                                                                           '97085620',
                                                                                                                                           'DNB|135275245',
                                                                                                                                           'WKP|Q272069',
                                                                                                                                           'DNB|300912528']},
                                                                                                                       'title': 'Monkey'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Moon'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Mr '
                                                                                                                                'Tembo '
                                                                                                                                '(3 '
                                                                                                                                'min '
                                                                                                                                '43 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['DBC'],
                                                                                                                                   'sid': ['DBC|87097969360297']},
                                                                                                                       'title': 'No '
                                                                                                                                'distance '
                                                                                                                                'left '
                                                                                                                                'to '
                                                                                                                                'run'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'One '
                                                                                                                                'day'},
                                                                                                                      {'sources': {'s': ['BNE',
                                                                                                                                         'WKP',
                                                                                                                                         'NUKAT',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['BNE|XX1502205',
                                                                                                                                           'WKP|Q272069',
                                                                                                                                           'NUKAT|n '
                                                                                                                                           '2009143303',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Ordinary '
                                                                                                                                'Decent '
                                                                                                                                'Criminal'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Parakeet '
                                                                                                                                '(43 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Parklife '
                                                                                                                                '(canci&#xF3;n)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Perdu'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Un '
                                                                                                                                'perfetto '
                                                                                                                                'criminale'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Photographs'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Point '
                                                                                                                                'star'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Preparation'},
                                                                                                                      {'sources': {'s': ['WKP',
                                                                                                                                         'NUKAT'],
                                                                                                                                   'sid': ['WKP|Q272069',
                                                                                                                                           'NUKAT|n '
                                                                                                                                           '2009143303']},
                                                                                                                       'title': 'Przyzwoity '
                                                                                                                                'przest&#x119;pca'},
                                                                                                                      {'sources': {'s': ['PLWABN',
                                                                                                                                         'BNE',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['PLWABN|9810618563005606',
                                                                                                                                           'BNE|XX1502205',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Ravenous'},
                                                                                                                      {'sources': {'s': ['BNE'],
                                                                                                                                   'sid': ['BNE|XX1502205']},
                                                                                                                       'title': 'Scott '
                                                                                                                                'Walker, '
                                                                                                                                '30 '
                                                                                                                                'Century '
                                                                                                                                'Man'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'selfish '
                                                                                                                                'giant '
                                                                                                                                '(4 '
                                                                                                                                'min '
                                                                                                                                '47 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Seven '
                                                                                                                                'high '
                                                                                                                                '(1 '
                                                                                                                                'min)'},
                                                                                                                      {'id': 'VIAF|7146522416032391727',
                                                                                                                       'sources': {'s': ['DNB',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['DNB|135275245',
                                                                                                                                           'DNB|1102560960',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Songs '
                                                                                                                                'from '
                                                                                                                                'Wonder.land'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Stary, '
                                                                                                                                'gdzie '
                                                                                                                                'moja '
                                                                                                                                'bryka?'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Swim '
                                                                                                                                'The '
                                                                                                                                'Channel'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Tavaline '
                                                                                                                                'kurjategija'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Tavallisen '
                                                                                                                                'rehti '
                                                                                                                                'rikollinen'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Temptation '
                                                                                                                                'comes '
                                                                                                                                'in '
                                                                                                                                'the '
                                                                                                                                'afternoon '
                                                                                                                                '(2 '
                                                                                                                                'min '
                                                                                                                                '05 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'To '
                                                                                                                                'the '
                                                                                                                                'end'},
                                                                                                                      {'sources': {'s': ['NLA',
                                                                                                                                         'PLWABN',
                                                                                                                                         'WKP',
                                                                                                                                         'BNE',
                                                                                                                                         'LC',
                                                                                                                                         'BNF',
                                                                                                                                         'RERO',
                                                                                                                                         'SUDOC'],
                                                                                                                                   'sid': ['NLA|000041317329',
                                                                                                                                           'PLWABN|9810618563005606',
                                                                                                                                           'WKP|Q272069',
                                                                                                                                           'BNE|XX1502205',
                                                                                                                                           'LC|n  '
                                                                                                                                           '97085620',
                                                                                                                                           'BNF|14025704',
                                                                                                                                           'RERO|A002915097',
                                                                                                                                           'SUDOC|168995603']},
                                                                                                                       'title': 'Trainspotting'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': 'Traukini&#x173; '
                                                                                                                                '&#x17E;ym&#x117;jimas'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Tree '
                                                                                                                                'of '
                                                                                                                                'beauty '
                                                                                                                                '(2 '
                                                                                                                                'min)'},
                                                                                                                      {'sources': {'s': ['PLWABN'],
                                                                                                                                   'sid': ['PLWABN|9810618563005606']},
                                                                                                                       'title': 'Twarz'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Twentieth '
                                                                                                                                'century '
                                                                                                                                'blues '
                                                                                                                                'the '
                                                                                                                                'songs '
                                                                                                                                'of '
                                                                                                                                'No&#xEB;l '
                                                                                                                                'Coward'},
                                                                                                                      {'sources': {'s': ['RERO',
                                                                                                                                         'SUDOC',
                                                                                                                                         'BNF'],
                                                                                                                                   'sid': ['RERO|A002915097',
                                                                                                                                           'SUDOC|168995603',
                                                                                                                                           'BNF|14025704']},
                                                                                                                       'title': 'Vorace'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Warrior'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'Watching '
                                                                                                                                'the '
                                                                                                                                'fire '
                                                                                                                                'that '
                                                                                                                                'waltzed '
                                                                                                                                'away '
                                                                                                                                '(2 '
                                                                                                                                'min '
                                                                                                                                '37 '
                                                                                                                                's)'},
                                                                                                                      {'id': 'VIAF|22146522418032391851',
                                                                                                                       'sources': {'s': ['DNB'],
                                                                                                                                   'sid': ['DNB|1102560677']},
                                                                                                                       'title': 'Wonder.land'},
                                                                                                                      {'sources': {'s': ['BNF'],
                                                                                                                                   'sid': ['BNF|14025704']},
                                                                                                                       'title': 'You '
                                                                                                                                '& '
                                                                                                                                'me '
                                                                                                                                '(7 '
                                                                                                                                'min '
                                                                                                                                '05 '
                                                                                                                                's)'},
                                                                                                                      {'sources': {'s': ['NSK'],
                                                                                                                                   'sid': ['NSK|000760001']},
                                                                                                                       'title': '&#x17D;ivio '
                                                                                                                                'album!'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x414;&#x435; '
                                                                                                                                '&#x43C;&#x43E;&#x44F; '
                                                                                                                                '&#x442;&#x430;&#x447;&#x43A;&#x430;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x41E;&#x431;&#x44B;&#x43A;&#x43D;&#x43E;&#x432;&#x435;&#x43D;&#x43D;&#x44B;&#x439; '
                                                                                                                                '&#x43F;&#x440;&#x435;&#x441;&#x442;&#x443;&#x43F;&#x43D;&#x438;&#x43A;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x422;&#x440;&#x435;&#x439;&#x43D;&#x441;&#x43F;&#x43E;&#x442;&#x438;&#x43D;&#x433;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x5D4;&#x5D0;&#x5E8;&#x5D9; '
                                                                                                                                '&#x5D4;&#x5DE;&#x5D6;&#x5D5;&#x5D4;&#x5DD;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x645;&#x62C;&#x631;&#x645; '
                                                                                                                                '&#x646;&#x62C;&#x6CC;&#x628; '
                                                                                                                                '&#x639;&#x627;&#x62F;&#x6CC;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x99F;&#x9CD;&#x9B0;&#x9C7;&#x9A8;&#x9B8;&#x9CD;&#x9AA;&#x99F;&#x9BF;&#x982;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#xE0B;&#xE2D;&#xE07;&#xE17;&#xE39;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#xB0B4; '
                                                                                                                                '&#xCC28; '
                                                                                                                                '&#xBD24;&#xB0D0;?'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#xD2B8;&#xB808;&#xC778;&#xC2A4;&#xD3EC;&#xD305;'},
                                                                                                                      {'sources': {'s': ['KRNLK'],
                                                                                                                                   'sid': ['KRNLK|KAC2020M4718']},
                                                                                                                       'title': '&#xD398;&#xC774;&#xC2A4;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x79C1;&#x304C;&#x611B;&#x3057;&#x305F;&#x30AE;&#x30E3;&#x30F3;&#x30B0;&#x30B9;&#x30BF;&#x30FC;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x731C;&#x706B;&#x8ECA;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x8C6C;&#x982D;&#xFF0C;&#x6211;&#x7684;&#x8ECA;&#x54A7;&#xFF1F;'},
                                                                                                                      {'sources': {'s': ['WKP'],
                                                                                                                                   'sid': ['WKP|Q272069']},
                                                                                                                       'title': '&#x738B;&#x724C;&#x7F6A;&#x72AF;'}]},
                                                                                                  'viafID': 17421456},
                                                                                  'type': 'ns1:stringOrXmlFragment'},
                                                                   'recordPacking': {'type': 'xsd:string',
                                                                                     'value': 'xml'},
                                                                   'recordPosition': {'type': 'xsd:positiveInteger',
                                                                                      'value': 1},
                                                                   'recordSchema': {'type': 'xsd:string',
                                                                                    'value': 'http://viaf.org/BriefVIAFCluster'},
                                                                   'type': 'ns1:recordType'}],
                                                       'type': 'ns1:recordsType'},
                                           'resultSetIdleTime': {'type': 'xsd:positiveInteger',
                                                                 'value': 1},
                                           'schemaLocation': 'http://www.loc.gov/zing/srw/ '
                                                             'http://www.loc.gov/standards/sru/sru1-1archive/xml-files/srw-types.xsd',
                                           'version': {'type': 'xsd:string',
                                                       'value': 1.1}}}}],
 'linking_config': {'add_viaf_info': True,
                    'context': None,
                    'entity': 'Damon Albarn',
                    'fuzziness': 0,
                    'min_similarity': 0.9,
                    'prefix_length': 1},
 'n_linked': 1,
 'original_entity': 'Damon Albarn',
 'similarity_score': 1.0}

```
</details>

### Example 2: Multiple matches are returned

```python
from rara_linker.linkers.linker import Linker
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

# The input contains some typos, 
# we are actually trying to find
# Paul Keres, the chess grandmaster
entity = "Paul Keers"
linked = linker.link(entity)

format_output(linked)
```

**Output:**

```
Original entity: Paul Keers
Entity type: PER
Number of matches: 3
Similarity: 0.98

Linked entity: Kees, Paul
Description: Eesti pedagoogikateadlane ja tõlkija

Linked entity: Keres, Paul
Description: Eesti maletaja ja maleteoreetik

Linked entity: Keres, Paul
Description: Eesti advokaat
```

### Example 3: Using vector search for additional filtering

```python
from rara_linker.linkers.linker import Linker
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

# The input contains some typos, 
# we are actually trying to find
# Paul Keres, the chess grandmaster
entity = "Paul Keers"

# The context can be any short text that
# might bare some contextual resemblance to 
# the entity. In practice, it will most likely
# be a title or a short paragraph,
# where the name was mentioned, let's try
# something similar:
context = "Viljandis selgusid 56. maleturniiri võitjad"
linked = linker.link(entity, context=context)

format_output(linked)
```

**Output:**

```
Original entity: Paul Keers
Entity type: PER
Number of matches: 1
Similarity: 0.98

Linked entity: Keres, Paul
Description: Eesti maletaja ja maleteoreetik
```

### Example 4: Link a keyword / subject index

```python
from rara_linker.linkers.linker import Linker
import logging

logging.disable(logging.CRITICAL)

linker = Linker(add_viaf_info=True, vectorizer_or_dir_path="./vectorizer_data")

entity = "alajahtumine"
linked = linker.link(entity)

format_output(linked)
```

**Output:**

```
Original entity: alajahtumine
Entity type: EMS_KEYWORD
Number of matches: 1
Similarity: 1.0

Linked entity: hüpotermia
```