"""
Transformers for all the navigation events
"""
import json


def edx_course_student_notes_edited(current_event, caliper_event):
    """
    The browser emits edx.course.student_notes.edited events when a
    learner edits a note in course.

    :param current_event: default event log generated.
    :param caliper_event: caliper_event log having some basic attributes.
    :return: updated caliper_event.
    """

    object_extensions = json.loads(current_event['event'])

    object_extensions.update({
        'course_id': current_event['context']['course_id'],
        'org_id': current_event['context']['org_id'],
    })

    caliper_object = {
        'id': current_event['referer'],
        'type': 'Document',
        'extensions': object_extensions
    }

    caliper_event.update({
        'type': 'Event',
        'action': 'Modified',
        'object': caliper_object
    })

    caliper_event['referrer']['type'] = 'WebPage'

    caliper_event['actor'].update({
        'name': current_event['username'],
        'type': 'Person'
    })

    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
    })

    return caliper_event


def edx_course_student_notes_added(current_event, caliper_event):
    """
    The server emits an edx.course.student_notes.added event when the student
    add notes on a textual content present in course.

    :param current_event: default log
    :param caliper_event: log containing both basic and default attribute
    :return: final created log
    """
    caliper_event.update(
        {
            'type': 'AnnotationEvent',
            'action': 'Tagged',
            'object': {
                'id': current_event['referer'],
                'type': 'DigitalResource',
                'extensions': json.loads(current_event['event'])
            }
        }
    )
    caliper_event['object']['extensions']['course_id'] = \
        current_event['context']['course_id']
    caliper_event['actor'].update({
        'name': current_event['username'],
        'type': 'Person'
    })
    caliper_event['referrer']['type'] = 'WebPage'
    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
        'path': current_event['context']['path'],
    })
    return caliper_event


def edx_course_student_notes_viewed(current_event, caliper_event):
    """
    The server emits an edx.course.student_notes.viewed event when the student
    views notes on a textual content present in course.

    :param current_event: default logs generated by edx itself
    :param caliper_event: log containing both basic and default attribute
    :return: final created log
    """
    caliper_object = {
        'id': current_event['referer'],
        'type': 'Document',
        'extensions': json.loads(current_event['event'])
    }

    caliper_object['extensions'].update({
        'course_id': current_event['context']['course_id'],
        'org_id': caliper_event['extensions']['extra_fields'].pop('org_id')
    })

    caliper_event.update({
        'type': 'ViewEvent',
        'action': 'Viewed',
        'object': caliper_object,
    })

    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
        'event_source': current_event['event_source'],
    })

    caliper_event['actor'].update({
        'type': 'Person',
        'name': current_event['username']
    })

    caliper_event['referrer']['type'] = 'WebPage'

    return caliper_event


def edx_course_student_notes_deleted(current_event, caliper_event):
    """
    The browser emits edx.course.student_notes.deleted events when a
    learner deletes a note in course.

    :param current_event: default event log generated.
    :param caliper_event: caliper_event log having some basic attributes.
    :return: updated caliper_event.
    """

    object_extensions = json.loads(current_event['event'])

    object_extensions.update({
        'course_id': current_event['context']['course_id'],
        'org_id': caliper_event['extensions']['extra_fields'].pop('org_id'),
    })

    caliper_object = {
        'id': current_event['referer'],
        'type': 'Document',
        'extensions': object_extensions
    }

    caliper_event.update({
        'type': 'Event',
        'action': 'Deleted',
        'object': caliper_object
    })

    caliper_event['referrer']['type'] = 'WebPage'

    caliper_event['actor'].update({
        'name': current_event['username'],
        'type': 'Person'
    })

    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
    })

    return caliper_event


def edx_course_student_notes_searched(current_event, caliper_event):
    """
    The server emits an edx.course.student_notes.searched event when the student
    searches notes on a textual content present in course.

    :param current_event: default logs generated by edx itself
    :param caliper_event: log containing both basic and default attribute
    :return: final created log
    """
    caliper_object = {
        'id': current_event['referer'],
        'type': 'Document',
        'extensions': json.loads(current_event['event'])
    }

    caliper_object['extensions'].update({
        'course_id': current_event['context']['course_id'],
        'org_id': caliper_event['extensions']['extra_fields'].pop('org_id'),
        'user_id': caliper_event['extensions']['extra_fields'].pop('user_id')
    })

    caliper_event.update({
        'type': 'Event',
        'action': 'Searched',
        'object': caliper_object,
    })

    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
        'event_source': current_event['event_source'],
    })

    caliper_event['actor'].update({
        'type': 'Person',
        'name': current_event['username']
    })

    caliper_event['referrer']['type'] = 'WebPage'

    return caliper_event


def edx_course_student_notes_used_unit_link(current_event, caliper_event):
    """
    This event is generated when the user clicks on the note link.

    :param current_event: default logs generated by edx itself
    :param caliper_event: log containing both basic and default attribute
    :return: final created log
    """
    caliper_object = {
        'id': current_event['referer'],
        'type': 'WebPage',
        'extensions': json.loads(current_event['event'])
    }

    caliper_object['extensions'].update({
        'course_id': current_event['context']['course_id'],
        'org_id': caliper_event['extensions']['extra_fields'].pop('org_id')
    })

    caliper_event.update({
        'type': 'NavigationEvent',
        'action': 'NavigatedTo',
        'object': caliper_object,
    })

    caliper_event['extensions']['extra_fields'].update({
        'ip': current_event['ip'],
        'event_source': current_event['event_source']
    })

    caliper_event['actor'].update({
        'type': 'Person',
        'name': current_event['username']
    })

    caliper_event['referrer']['type'] = 'WebPage'

    return caliper_event
