!> \file voronov_atomic_rates.F90

!> \brief Module that stores the rate fits from   
!! "A practical fit formula for ionization rate coefficients of atoms and ions 
!! by electron impact: Z = 1-28",  G. S. Voronov (1997, Atomic Data and Nuclear 
!! Data Tables, 65, 1).



module voronov_atomic_rates_mod
use myf90_mod
implicit none

contains


function Vor_HI_col_ion(T) result(rate)
  real(r8b) :: T, rate
  call cfit(1,1,T,rate)
  return
end function Vor_HI_col_ion

function Vor_HeI_col_ion(T) result(rate)
  real(r8b) :: T, rate
  call cfit(2,2,T,rate)
  return
end function Vor_HeI_col_ion

function Vor_HeII_col_ion(T) result(rate)
  real(r8b) :: T, rate
  call cfit(2,1,T,rate)
  return
end function Vor_HeII_col_ion


subroutine cfit(iz,in,t,c)
!* Version 2, March 24, 1997
!******************************************************************************
!*** This subroutine calculates rates of direct collisional ionization 
!*** for all ionization stages of all elements from H to Ni (Z=28)
!*** by use of the fits from G. S. Voronov, 1997, ADNDT, 65, 1
!*** Input parameters:  iz - atomic number 
!***                    in - number of electrons from 1 to iz 
!***                    t  - temperature, K
!*** Output parameter:  c  - rate coefficient, cm^3 s^(-1)
!******************************************************************************
integer :: iz,in
real(r8b) :: c, t

real :: CF(5,28,28)
real :: te, u


  CF(:, 1, 1) = (/   13.6,0.,2.91E-08,0.2320,0.39 /)
  CF(:, 2, 2) = (/   24.6,0.,1.75E-08,0.1800,0.35 /)
  CF(:, 2, 1) = (/   54.4,1.,2.05E-09,0.2650,0.25 /)
  CF(:, 3, 3) = (/    5.4,0.,1.39E-07,0.4380,0.41 /)
  CF(:, 3, 2) = (/   75.6,1.,2.01E-09,0.2090,0.23 /)
  CF(:, 3, 1) = (/  122.4,1.,9.60E-10,0.5820,0.17 /)
  CF(:, 4, 4) = (/    9.3,0.,1.02E-07,0.3750,0.27 /)
  CF(:, 4, 3) = (/   18.2,1.,2.08E-08,0.4390,0.21 /)
  CF(:, 4, 2) = (/  153.9,0.,2.67E-09,0.6120,0.27 /)
  CF(:, 4, 1) = (/  217.7,1.,4.27E-10,0.6580,0.15 /)
  CF(:, 5, 5) = (/    8.3,0.,6.49E-08,0.2000,0.26 /)
  CF(:, 5, 4) = (/   25.2,1.,1.24E-08,0.2670,0.22 /)
  CF(:, 5, 3) = (/   37.9,1.,3.27E-09,0.2950,0.23 /)
  CF(:, 5, 2) = (/  259.4,1.,4.95E-10,0.4890,0.09 /)
  CF(:, 5, 1) = (/  340.2,1.,2.19E-10,0.6570,0.15 /)
  CF(:, 6, 6) = (/   11.3,0.,6.85E-08,0.1930,0.25 /)
  CF(:, 6, 5) = (/   24.4,1.,1.86E-08,0.2860,0.24 /)
  CF(:, 6, 4) = (/   47.9,1.,6.35E-09,0.4270,0.21 /)
  CF(:, 6, 3) = (/   64.5,1.,1.50E-09,0.4160,0.13 /)
  CF(:, 6, 2) = (/  392.1,1.,2.99E-10,0.6660,0.02 /)
  CF(:, 6, 1) = (/  490.0,1.,1.23E-10,0.6200,0.16 /)
  CF(:, 7, 7) = (/   14.5,0.,4.82E-08,0.0652,0.42 /)
  CF(:, 7, 6) = (/   29.6,0.,2.98E-08,0.3100,0.30 /)
  CF(:, 7, 5) = (/   47.5,1.,8.10E-09,0.3500,0.24 /)
  CF(:, 7, 4) = (/   77.5,1.,3.71E-09,0.5490,0.18 /)
  CF(:, 7, 3) = (/   97.9,0.,1.51E-09,0.0167,0.74 /)
  CF(:, 7, 2) = (/  552.1,0.,3.71E-10,0.5460,0.29 /)
  CF(:, 7, 1) = (/  667.0,1.,7.77E-11,0.6240,0.16 /)
  CF(:, 8, 8) = (/   13.6,0.,3.59E-08,0.0730,0.34 /)
  CF(:, 8, 7) = (/   35.1,1.,1.39E-08,0.2120,0.22 /)
  CF(:, 8, 6) = (/   54.9,1.,9.31E-09,0.2700,0.27 /)
  CF(:, 8, 5) = (/   77.4,0.,1.02E-08,0.6140,0.27 /)
  CF(:, 8, 4) = (/  113.9,1.,2.19E-09,0.6300,0.17 /)
  CF(:, 8, 3) = (/  138.1,0.,1.95E-09,0.3600,0.54 /)
  CF(:, 8, 2) = (/  739.3,0.,2.12E-10,0.3960,0.35 /)
  CF(:, 8, 1) = (/  871.4,1.,5.21E-11,0.6290,0.16 /)
  CF(:, 9, 9) = (/   17.4,1.,7.00E-08,0.1780,0.29 /)
  CF(:, 9, 8) = (/   35.0,0.,5.41E-08,0.5710,0.27 /)
  CF(:, 9, 7) = (/   62.7,1.,9.37E-09,0.3190,0.20 /)
  CF(:, 9, 6) = (/   87.1,1.,4.92E-09,0.3230,0.24 /)
  CF(:, 9, 5) = (/  114.2,0.,7.06E-09,0.6840,0.27 /)
  CF(:, 9, 4) = (/  157.2,1.,1.28E-09,0.6480,0.16 /)
  CF(:, 9, 3) = (/  185.2,1.,5.61E-10,0.7380,0.16 /)
  CF(:, 9, 2) = (/  953.9,0.,1.66E-10,0.5420,0.29 /)
  CF(:, 9, 1) = (/ 1103.1,1.,3.74E-11,0.6590,0.15 /)
  CF(:,10,10) = (/   21.6,1.,1.50E-08,0.0329,0.43 /)
  CF(:,10, 9) = (/   41.0,0.,1.98E-08,0.2950,0.20 /)
  CF(:,10, 8) = (/   63.5,1.,7.03E-09,0.0677,0.39 /)
  CF(:,10, 7) = (/   97.1,1.,4.24E-09,0.0482,0.58 /)
  CF(:,10, 6) = (/  126.2,1.,2.79E-09,0.3050,0.25 /)
  CF(:,10, 5) = (/  157.9,0.,3.45E-09,0.5810,0.28 /)
  CF(:,10, 4) = (/  207.3,1.,9.56E-10,0.7490,0.14 /)
  CF(:,10, 3) = (/  239.1,1.,4.73E-10,0.9920,0.04 /)
  CF(:,10, 2) = (/ 1196.0,1.,3.92E-11,0.2620,0.20 /)
  CF(:,10, 1) = (/ 1360.6,1.,2.77E-11,0.6610,0.13 /)
  CF(:,11,11) = (/    5.1,1.,1.01E-07,0.2750,0.23 /)
  CF(:,11,10) = (/   47.3,1.,7.35E-09,0.0560,0.35 /)
  CF(:,11, 9) = (/   71.6,1.,8.10E-09,0.1480,0.32 /)
  CF(:,11, 8) = (/   98.9,0.,1.14E-08,0.5530,0.28 /)
  CF(:,11, 7) = (/  138.4,1.,2.63E-09,0.2300,0.29 /)
  CF(:,11, 6) = (/  172.2,1.,1.85E-09,0.3630,0.22 /)
  CF(:,11, 5) = (/  208.5,0.,2.82E-09,0.6740,0.27 /)
  CF(:,11, 4) = (/  264.2,1.,6.72E-10,0.7520,0.14 /)
  CF(:,11, 3) = (/  299.9,1.,2.80E-10,0.7810,0.15 /)
  CF(:,11, 2) = (/ 1465.1,1.,4.63E-11,0.5580,0.16 /)
  CF(:,11, 1) = (/ 1648.7,1.,2.16E-11,0.7430,0.13 /)
  CF(:,12,12) = (/    7.6,0.,6.21E-07,0.5920,0.39 /)
  CF(:,12,11) = (/   15.2,0.,1.92E-08,0.0027,0.85 /)
  CF(:,12,10) = (/   80.1,1.,5.56E-09,0.1070,0.30 /)
  CF(:,12, 9) = (/  109.3,1.,4.35E-09,0.1590,0.31 /)
  CF(:,12, 8) = (/  141.3,0.,7.10E-09,0.6580,0.25 /)
  CF(:,12, 7) = (/  186.5,1.,1.70E-09,0.2420,0.28 /)
  CF(:,12, 6) = (/  224.9,1.,1.22E-09,0.3430,0.23 /)
  CF(:,12, 5) = (/  266.0,0.,2.20E-09,0.8970,0.22 /)
  CF(:,12, 4) = (/  328.2,1.,4.86E-10,0.7510,0.14 /)
  CF(:,12, 3) = (/  367.5,1.,2.35E-10,1.0300,0.10 /)
  CF(:,12, 2) = (/ 1761.8,1.,2.06E-11,0.1960,0.25 /)
  CF(:,12, 1) = (/ 1962.7,1.,1.75E-11,0.8350,0.11 /)
  CF(:,13,13) = (/    6.0,1.,2.28E-07,0.3870,0.25 /)
  CF(:,13,12) = (/   18.8,0.,1.18E-07,2.2100,0.25 /)
  CF(:,13,11) = (/   28.5,1.,4.40E-09,0.1060,0.24 /)
  CF(:,13,10) = (/  120.0,0.,1.75E-08,0.8720,0.22 /)
  CF(:,13, 9) = (/  153.8,1.,2.61E-09,0.1590,0.31 /)
  CF(:,13, 8) = (/  198.5,1.,1.85E-09,0.1520,0.36 /)
  CF(:,13, 7) = (/  241.4,1.,1.14E-09,0.2280,0.29 /)
  CF(:,13, 6) = (/  284.6,1.,8.00E-10,0.4170,0.16 /)
  CF(:,13, 5) = (/  390.2,1.,5.83E-10,0.4970,0.23 /)
  CF(:,13, 4) = (/  399.4,0.,4.93E-10,0.7060,0.16 /)
  CF(:,13, 3) = (/  442.0,1.,9.77E-11,0.2780,0.17 /)
  CF(:,13, 2) = (/ 2086.6,0.,3.94E-11,0.2860,0.36 /)
  CF(:,13, 1) = (/ 2304.1,1.,1.38E-11,0.8350,0.11 /)
  CF(:,14,14) = (/    8.2,1.,1.88E-07,0.3760,0.25 /)
  CF(:,14,13) = (/   16.4,1.,6.43E-08,0.6320,0.20 /)
  CF(:,14,12) = (/   33.5,1.,2.01E-08,0.4730,0.22 /)
  CF(:,14,11) = (/   54.0,1.,4.94E-09,0.1720,0.23 /)
  CF(:,14,10) = (/  166.8,1.,1.76E-09,0.1020,0.31 /)
  CF(:,14, 9) = (/  205.3,1.,1.74E-09,0.1800,0.29 /)
  CF(:,14, 8) = (/  246.5,1.,1.23E-09,0.5180,0.07 /)
  CF(:,14, 7) = (/  303.5,1.,8.27E-10,0.2390,0.28 /)
  CF(:,14, 6) = (/  351.1,1.,6.01E-10,0.3050,0.25 /)
  CF(:,14, 5) = (/  401.4,1.,4.65E-10,0.6660,0.04 /)
  CF(:,14, 4) = (/  476.4,1.,2.63E-10,0.6660,0.16 /)
  CF(:,14, 3) = (/  523.5,1.,1.18E-10,0.7340,0.16 /)
  CF(:,14, 2) = (/ 2437.7,0.,3.36E-11,0.3360,0.37 /)
  CF(:,14, 1) = (/ 2673.2,1.,1.19E-11,0.9890,0.08 /)
  CF(:,15,15) = (/   10.5,1.,1.99E-07,0.5350,0.24 /)
  CF(:,15,14) = (/   19.8,1.,5.88E-08,0.5370,0.21 /)
  CF(:,15,13) = (/   30.2,1.,2.96E-08,0.8650,0.16 /)
  CF(:,15,12) = (/   51.4,1.,1.01E-08,0.5460,0.20 /)
  CF(:,15,11) = (/   65.0,1.,2.36E-09,0.1920,0.17 /)
  CF(:,15,10) = (/  220.4,0.,6.66E-09,1.0000,0.18 /)
  CF(:,15, 9) = (/  263.2,1.,1.24E-09,0.2150,0.26 /)
  CF(:,15, 8) = (/  309.4,0.,2.27E-09,0.7340,0.23 /)
  CF(:,15, 7) = (/  371.7,1.,6.14E-10,0.2560,0.27 /)
  CF(:,15, 6) = (/  424.5,1.,4.69E-10,0.3420,0.23 /)
  CF(:,15, 5) = (/  479.6,0.,6.14E-10,0.3340,0.39 /)
  CF(:,15, 4) = (/  560.4,0.,3.22E-10,0.8500,0.12 /)
  CF(:,15, 3) = (/  611.9,1.,9.32E-11,0.7340,0.16 /)
  CF(:,15, 2) = (/ 2816.9,0.,3.79E-11,0.8050,0.22 /)
  CF(:,15, 1) = (/ 3069.9,1.,9.73E-12,0.9910,0.08 /)
  CF(:,16,16) = (/   10.4,1.,5.49E-08,0.1000,0.25 /)
  CF(:,16,15) = (/   23.3,1.,6.81E-08,0.6930,0.21 /)
  CF(:,16,14) = (/   34.8,1.,2.14E-08,0.3530,0.24 /)
  CF(:,16,13) = (/   47.3,1.,1.66E-08,1.0300,0.14 /)
  CF(:,16,12) = (/   72.6,1.,6.12E-09,0.5800,0.19 /)
  CF(:,16,11) = (/   88.1,1.,1.33E-09,0.0688,0.35 /)
  CF(:,16,10) = (/  280.9,0.,4.93E-09,1.1300,0.16 /)
  CF(:,16, 9) = (/  328.2,1.,8.73E-10,0.1930,0.28 /)
  CF(:,16, 8) = (/  379.1,0.,1.35E-09,0.4310,0.32 /)
  CF(:,16, 7) = (/  447.1,1.,4.59E-10,0.2420,0.28 /)
  CF(:,16, 6) = (/  504.8,1.,3.49E-10,0.3050,0.25 /)
  CF(:,16, 5) = (/  564.7,0.,5.23E-10,0.4280,0.35 /)
  CF(:,16, 4) = (/  651.6,0.,2.59E-10,0.8540,0.12 /)
  CF(:,16, 3) = (/  707.2,1.,7.50E-11,0.7340,0.16 /)
  CF(:,16, 2) = (/ 3223.9,0.,2.67E-11,0.5720,0.28 /)
  CF(:,16, 1) = (/ 3494.2,1.,6.32E-12,0.5850,0.17 /)
  CF(:,17,17) = (/   13.0,1.,1.69E-07,0.4300,0.24 /)
  CF(:,17,16) = (/   23.8,1.,6.96E-08,0.6700,0.20 /)
  CF(:,17,15) = (/   39.6,1.,3.40E-08,0.8650,0.18 /)
  CF(:,17,14) = (/   53.5,1.,1.10E-08,0.3280,0.25 /)
  CF(:,17,13) = (/   67.8,1.,1.11E-08,1.3700,0.10 /)
  CF(:,17,12) = (/   97.0,1.,3.17E-09,0.3300,0.24 /)
  CF(:,17,11) = (/  114.2,1.,1.01E-09,0.1960,0.16 /)
  CF(:,17,10) = (/  348.3,0.,2.11E-09,0.3130,0.37 /)
  CF(:,17, 9) = (/  400.1,1.,6.32E-10,0.1730,0.30 /)
  CF(:,17, 8) = (/  455.6,0.,9.48E-10,0.3440,0.36 /)
  CF(:,17, 7) = (/  529.3,1.,3.69E-10,0.2730,0.26 /)
  CF(:,17, 6) = (/  592.0,1.,2.85E-10,0.3430,0.23 /)
  CF(:,17, 5) = (/  656.7,0.,4.81E-10,0.6580,0.27 /)
  CF(:,17, 4) = (/  749.8,1.,1.31E-10,0.6230,0.16 /)
  CF(:,17, 3) = (/  809.4,1.,6.13E-11,0.7360,0.16 /)
  CF(:,17, 2) = (/ 3658.4,0.,1.90E-11,0.3790,0.36 /)
  CF(:,17, 1) = (/ 3946.3,1.,5.14E-12,0.5530,0.18 /)
  CF(:,18,18) = (/   15.8,1.,5.99E-08,0.1360,0.26 /)
  CF(:,18,17) = (/   27.6,1.,6.07E-08,0.5440,0.21 /)
  CF(:,18,16) = (/   40.9,1.,3.43E-08,0.8340,0.17 /)
  CF(:,18,15) = (/   52.3,0.,3.00E-08,1.0300,0.25 /)
  CF(:,18,14) = (/   75.0,1.,8.73E-09,0.3660,0.31 /)
  CF(:,18,13) = (/   91.0,1.,5.78E-09,0.3140,0.34 /)
  CF(:,18,12) = (/  124.3,1.,2.98E-09,0.7030,0.16 /)
  CF(:,18,11) = (/  143.5,1.,7.25E-10,0.2070,0.15 /)
  CF(:,18,10) = (/  422.4,1.,1.40E-09,0.6960,0.13 /)
  CF(:,18, 9) = (/  478.7,1.,4.78E-10,0.1640,0.31 /)
  CF(:,18, 8) = (/  539.0,0.,8.02E-10,0.4390,0.32 /)
  CF(:,18, 7) = (/  618.3,1.,2.88E-10,0.2590,0.27 /)
  CF(:,18, 6) = (/  686.1,1.,2.32E-10,0.3620,0.22 /)
  CF(:,18, 5) = (/  755.7,0.,3.33E-10,0.4120,0.36 /)
  CF(:,18, 4) = (/  854.8,1.,1.27E-10,0.9100,0.13 /)
  CF(:,18, 3) = (/  918.0,1.,5.21E-11,0.7810,0.15 /)
  CF(:,18, 2) = (/ 4120.7,0.,1.66E-11,0.4350,0.33 /)
  CF(:,18, 1) = (/ 4426.2,1.,4.32E-12,0.5540,0.18 /)
  CF(:,19,19) = (/    4.3,1.,2.02E-07,0.2720,0.31 /)
  CF(:,19,18) = (/   31.6,1.,4.01E-08,0.3710,0.22 /)
  CF(:,19,17) = (/   45.8,1.,1.50E-08,0.4330,0.21 /)
  CF(:,19,16) = (/   60.9,1.,1.94E-08,0.8890,0.16 /)
  CF(:,19,15) = (/   82.7,1.,6.95E-09,0.4940,0.18 /)
  CF(:,19,14) = (/   99.4,1.,4.11E-09,0.5400,0.17 /)
  CF(:,19,13) = (/  117.6,1.,2.23E-09,0.5190,0.16 /)
  CF(:,19,12) = (/  154.7,1.,2.15E-09,0.8280,0.14 /)
  CF(:,19,11) = (/  175.8,0.,1.61E-09,0.6420,0.13 /)
  CF(:,19,10) = (/  504.0,1.,1.07E-09,0.6950,0.13 /)
  CF(:,19, 9) = (/  564.7,1.,3.78E-10,0.1730,0.30 /)
  CF(:,19, 8) = (/  629.4,0.,6.24E-10,0.4180,0.33 /)
  CF(:,19, 7) = (/  714.6,1.,2.29E-10,0.2450,0.28 /)
  CF(:,19, 6) = (/  786.6,1.,1.86E-10,0.3440,0.23 /)
  CF(:,19, 5) = (/  861.1,0.,2.69E-10,0.3960,0.37 /)
  CF(:,19, 4) = (/  968.0,1.,1.06E-10,0.9120,0.13 /)
  CF(:,19, 3) = (/ 1053.4,1.,4.24E-11,0.7370,0.16 /)
  CF(:,19, 2) = (/ 4610.9,0.,1.38E-11,0.4160,0.34 /)
  CF(:,19, 1) = (/ 4934.1,1.,3.67E-12,0.5550,0.18 /)
  CF(:,20,20) = (/    6.1,0.,4.40E-07,0.8480,0.33 /)
  CF(:,20,19) = (/   11.9,0.,5.22E-08,0.1510,0.34 /)
  CF(:,20,18) = (/   50.9,1.,2.06E-08,0.4180,0.20 /)
  CF(:,20,17) = (/   67.3,1.,1.72E-08,0.6380,0.19 /)
  CF(:,20,16) = (/   84.5,1.,1.26E-08,1.0100,0.14 /)
  CF(:,20,15) = (/  108.8,1.,4.72E-09,0.5260,0.17 /)
  CF(:,20,14) = (/  127.2,1.,2.89E-09,0.5480,0.17 /)
  CF(:,20,13) = (/  147.2,1.,1.64E-09,0.5520,0.15 /)
  CF(:,20,12) = (/  188.3,1.,1.57E-09,0.7990,0.14 /)
  CF(:,20,11) = (/  211.3,1.,4.32E-10,0.2320,0.14 /)
  CF(:,20,10) = (/  591.9,0.,9.47E-10,0.3110,0.38 /)
  CF(:,20, 9) = (/  657.2,1.,2.98E-10,0.1630,0.31 /)
  CF(:,20, 8) = (/  726.6,0.,4.78E-10,0.3590,0.36 /)
  CF(:,20, 7) = (/  817.6,1.,1.86E-10,0.2440,0.28 /)
  CF(:,20, 6) = (/  894.5,1.,1.56E-10,0.3640,0.22 /)
  CF(:,20, 5) = (/  974.0,0.,2.16E-10,0.3570,0.39 /)
  CF(:,20, 4) = (/ 1087.0,1.,7.70E-11,0.6550,0.15 /)
  CF(:,20, 3) = (/ 1157.0,1.,3.58E-11,0.7360,0.16 /)
  CF(:,20, 2) = (/ 5128.9,0.,1.28E-11,0.5200,0.30 /)
  CF(:,20, 1) = (/ 5469.9,1.,3.08E-12,0.5280,0.19 /)
  CF(:,21,21) = (/    6.6,1.,3.16E-07,0.2040,0.28 /)
  CF(:,21,20) = (/   12.8,1.,8.61E-08,0.1810,0.25 /)
  CF(:,21,19) = (/   24.8,1.,5.08E-08,0.3570,0.24 /)
  CF(:,21,18) = (/   73.5,1.,1.00E-08,0.4530,0.15 /)
  CF(:,21,17) = (/   91.9,1.,6.76E-09,0.4600,0.15 /)
  CF(:,21,16) = (/  110.7,1.,5.27E-09,0.5610,0.17 /)
  CF(:,21,15) = (/  138.0,1.,3.40E-09,0.5600,0.16 /)
  CF(:,21,14) = (/  158.1,1.,2.18E-09,0.6120,0.15 /)
  CF(:,21,13) = (/  180.0,1.,1.26E-09,0.6100,0.14 /)
  CF(:,21,12) = (/  225.1,1.,1.24E-09,0.8520,0.13 /)
  CF(:,21,11) = (/  249.8,1.,3.62E-10,0.3490,0.05 /)
  CF(:,21,10) = (/  687.4,1.,5.52E-10,0.3750,0.28 /)
  CF(:,21, 9) = (/  756.7,1.,5.64E-10,0.8730,0.15 /)
  CF(:,21, 8) = (/  830.8,1.,4.50E-10,1.0500,0.13 /)
  CF(:,21, 7) = (/  927.5,1.,2.73E-10,0.8660,0.15 /)
  CF(:,21, 6) = (/ 1009.0,1.,1.56E-10,0.7150,0.17 /)
  CF(:,21, 5) = (/ 1094.0,0.,1.81E-10,1.1400,0.36 /)
  CF(:,21, 4) = (/ 1213.0,1.,4.29E-11,0.7840,0.15 /)
  CF(:,21, 3) = (/ 1288.0,0.,2.21E-11,0.0270,0.82 /)
  CF(:,21, 2) = (/ 5674.9,1.,4.51E-12,0.9180,0.04 /)
  CF(:,21, 1) = (/ 6033.8,0.,2.03E-12,0.0170,0.70 /)
  CF(:,22,22) = (/    6.8,1.,1.60E-07,0.3600,0.28 /)
  CF(:,22,21) = (/   13.6,0.,2.14E-07,0.8800,0.28 /)
  CF(:,22,20) = (/   27.5,1.,2.85E-08,0.2270,0.21 /)
  CF(:,22,19) = (/   43.3,1.,3.48E-08,0.3900,0.23 /)
  CF(:,22,18) = (/   99.3,1.,1.00E-08,0.5790,0.18 /)
  CF(:,22,17) = (/  119.5,1.,7.01E-09,0.6380,0.17 /)
  CF(:,22,16) = (/  140.8,1.,4.95E-09,0.7170,0.16 /)
  CF(:,22,15) = (/  170.4,1.,2.99E-09,0.6930,0.17 /)
  CF(:,22,14) = (/  192.1,1.,2.10E-09,0.7220,0.16 /)
  CF(:,22,13) = (/  215.9,1.,1.62E-09,0.7650,0.14 /)
  CF(:,22,12) = (/  265.0,1.,1.11E-09,0.8850,0.12 /)
  CF(:,22,11) = (/  291.5,0.,9.09E-10,0.9720,0.06 /)
  CF(:,22,10) = (/  787.8,1.,4.41E-10,0.3590,0.29 /)
  CF(:,22, 9) = (/  863.1,1.,4.39E-10,0.7810,0.17 /)
  CF(:,22, 8) = (/  941.9,1.,3.73E-10,1.0500,0.13 /)
  CF(:,22, 7) = (/ 1044.0,1.,2.28E-10,0.8580,0.15 /)
  CF(:,22, 6) = (/ 1131.0,1.,1.34E-10,0.7570,0.16 /)
  CF(:,22, 5) = (/ 1221.0,0.,1.55E-10,1.1500,0.36 /)
  CF(:,22, 4) = (/ 1346.0,1.,3.80E-11,0.8350,0.14 /)
  CF(:,22, 3) = (/ 1426.0,0.,1.89E-11,0.0280,0.82 /)
  CF(:,22, 2) = (/ 6249.1,1.,4.01E-12,0.9680,0.03 /)
  CF(:,22, 1) = (/ 6625.0,1.,1.62E-12,0.6570,0.14 /)
  CF(:,23,23) = (/    6.7,0.,8.82E-07,0.3590,0.32 /)
  CF(:,23,22) = (/   14.7,0.,3.11E-07,0.4320,0.29 /)
  CF(:,23,21) = (/   29.3,1.,3.50E-08,0.2470,0.25 /)
  CF(:,23,20) = (/   46.7,0.,5.32E-08,1.1100,0.16 /)
  CF(:,23,19) = (/   65.3,1.,8.98E-09,0.1400,0.37 /)
  CF(:,23,18) = (/  128.1,1.,5.87E-09,0.5170,0.17 /)
  CF(:,23,17) = (/  150.6,1.,5.11E-09,0.6790,0.16 /)
  CF(:,23,16) = (/  173.4,1.,3.71E-09,0.7610,0.15 /)
  CF(:,23,15) = (/  205.8,1.,2.24E-09,0.7110,0.17 /)
  CF(:,23,14) = (/  230.5,1.,1.65E-09,0.7640,0.15 /)
  CF(:,23,13) = (/  256.0,1.,1.26E-09,0.7620,0.14 /)
  CF(:,23,12) = (/  308.0,1.,8.86E-10,0.8860,0.12 /)
  CF(:,23,11) = (/  336.3,0.,3.89E-10,0.1420,0.39 /)
  CF(:,23,10) = (/  896.0,1.,3.80E-10,0.4090,0.27 /)
  CF(:,23, 9) = (/  976.0,0.,4.84E-10,0.1730,0.57 /)
  CF(:,23, 8) = (/ 1060.0,1.,2.49E-10,0.6500,0.14 /)
  CF(:,23, 7) = (/ 1168.0,0.,5.91E-10,1.6100,0.18 /)
  CF(:,23, 6) = (/ 1260.0,0.,5.02E-10,2.1200,0.15 /)
  CF(:,23, 5) = (/ 1355.0,1.,5.38E-11,0.1370,0.40 /)
  CF(:,23, 4) = (/ 1486.0,1.,5.56E-11,0.7080,0.10 /)
  CF(:,23, 3) = (/ 1571.0,0.,2.84E-11,0.0240,0.79 /)
  CF(:,23, 2) = (/ 6851.3,0.,2.54E-11,2.9200,0.09 /)
  CF(:,23, 1) = (/ 7246.1,0.,1.32E-11,3.5100,0.07 /)
  CF(:,24,24) = (/    6.8,1.,1.03E-07,0.2170,0.27 /)
  CF(:,24,23) = (/   16.5,0.,2.45E-07,0.3810,0.32 /)
  CF(:,24,22) = (/   31.0,0.,1.09E-07,0.5180,0.27 /)
  CF(:,24,21) = (/   49.1,1.,1.52E-08,0.1820,0.30 /)
  CF(:,24,20) = (/   69.5,0.,3.25E-08,1.3600,0.13 /)
  CF(:,24,19) = (/   90.6,1.,5.50E-09,0.1430,0.37 /)
  CF(:,24,18) = (/  160.2,1.,5.13E-09,0.6570,0.16 /)
  CF(:,24,17) = (/  184.7,1.,3.85E-09,0.7220,0.15 /)
  CF(:,24,16) = (/  209.3,1.,2.81E-09,0.7590,0.15 /)
  CF(:,24,15) = (/  244.4,1.,1.76E-09,0.7320,0.16 /)
  CF(:,24,14) = (/  271.0,1.,1.30E-09,0.7640,0.15 /)
  CF(:,24,13) = (/  298.0,1.,1.02E-09,0.8100,0.13 /)
  CF(:,24,12) = (/  354.8,1.,7.19E-10,0.8870,0.12 /)
  CF(:,24,11) = (/  384.2,1.,1.61E-10,0.1500,0.22 /)
  CF(:,24,10) = (/ 1011.0,1.,4.64E-10,0.9710,0.12 /)
  CF(:,24, 9) = (/ 1097.0,1.,3.31E-10,0.9240,0.14 /)
  CF(:,24, 8) = (/ 1185.0,1.,2.49E-10,0.9310,0.15 /)
  CF(:,24, 7) = (/ 1299.0,1.,1.68E-10,0.9100,0.14 /)
  CF(:,24, 6) = (/ 1396.0,1.,1.01E-10,0.8050,0.15 /)
  CF(:,24, 5) = (/ 1496.0,0.,1.17E-10,1.2100,0.35 /)
  CF(:,24, 4) = (/ 1634.0,1.,2.91E-11,0.8840,0.13 /)
  CF(:,24, 3) = (/ 1721.0,0.,1.45E-11,0.0350,0.80 /)
  CF(:,24, 2) = (/ 7482.0,1.,3.07E-12,0.9670,0.03 /)
  CF(:,24, 1) = (/ 7894.8,1.,1.46E-12,0.1830,0.39 /)
  CF(:,25,25) = (/    7.4,1.,8.56E-08,0.1320,0.26 /)
  CF(:,25,24) = (/   15.6,0.,1.18E-07,0.3590,0.19 /)
  CF(:,25,23) = (/   33.7,0.,8.54E-08,0.3970,0.32 /)
  CF(:,25,22) = (/   51.2,1.,1.80E-08,0.2720,0.18 /)
  CF(:,25,21) = (/   72.4,1.,8.22E-09,0.1610,0.32 /)
  CF(:,25,20) = (/   95.0,0.,2.15E-08,1.5400,0.11 /)
  CF(:,25,19) = (/  119.3,1.,3.65E-09,0.1470,0.37 /)
  CF(:,25,18) = (/  194.5,1.,3.91E-09,0.6990,0.15 /)
  CF(:,25,17) = (/  221.8,1.,2.92E-09,0.7190,0.15 /)
  CF(:,25,16) = (/  248.3,1.,2.23E-09,0.8060,0.14 /)
  CF(:,25,15) = (/  286.0,1.,1.39E-09,0.7350,0.16 /)
  CF(:,25,14) = (/  314.4,1.,1.04E-09,0.7610,0.15 /)
  CF(:,25,13) = (/  343.6,1.,8.28E-10,0.8090,0.13 /)
  CF(:,25,12) = (/  403.0,1.,5.60E-10,0.7870,0.14 /)
  CF(:,25,11) = (/  435.2,1.,1.52E-10,0.2990,0.08 /)
  CF(:,25,10) = (/ 1133.0,1.,4.03E-10,1.0400,0.11 /)
  CF(:,25, 9) = (/ 1244.0,1.,2.74E-10,0.9230,0.14 /)
  CF(:,25, 8) = (/ 1317.0,1.,2.18E-10,0.9900,0.14 /)
  CF(:,25, 7) = (/ 1437.0,1.,1.49E-10,0.9680,0.13 /)
  CF(:,25, 6) = (/ 1539.0,1.,8.70E-11,0.8020,0.15 /)
  CF(:,25, 5) = (/ 1644.0,0.,1.02E-10,1.2200,0.35 /)
  CF(:,25, 4) = (/ 1788.0,1.,2.54E-11,0.8830,0.13 /)
  CF(:,25, 3) = (/ 1880.0,0.,1.28E-11,0.0330,0.81 /)
  CF(:,25, 2) = (/ 8141.0,1.,2.77E-12,1.0100,0.02 /)
  CF(:,25, 1) = (/ 8571.9,1.,1.32E-12,0.2190,0.37 /)
  CF(:,26,26) = (/    7.9,0.,2.52E-07,0.7010,0.25 /)
  CF(:,26,25) = (/   16.2,1.,2.21E-08,0.0330,0.45 /)
  CF(:,26,24) = (/   30.6,0.,4.10E-08,0.3660,0.17 /)
  CF(:,26,23) = (/   54.8,0.,3.53E-08,0.2430,0.39 /)
  CF(:,26,22) = (/   75.0,1.,1.04E-08,0.2850,0.17 /)
  CF(:,26,21) = (/   99.0,1.,1.23E-08,0.4110,0.21 /)
  CF(:,26,20) = (/  125.0,1.,9.47E-09,0.4580,0.21 /)
  CF(:,26,19) = (/  151.1,1.,4.71E-09,0.2800,0.28 /)
  CF(:,26,18) = (/  233.6,1.,3.02E-09,0.6970,0.15 /)
  CF(:,26,17) = (/  262.1,1.,2.34E-09,0.7640,0.14 /)
  CF(:,26,16) = (/  290.0,1.,1.76E-09,0.8050,0.14 /)
  CF(:,26,15) = (/  331.0,1.,1.14E-09,0.7730,0.15 /)
  CF(:,26,14) = (/  361.0,1.,8.66E-10,0.8050,0.14 /)
  CF(:,26,13) = (/  392.0,1.,6.61E-10,0.7620,0.14 /)
  CF(:,26,12) = (/  457.0,1.,4.41E-10,0.6980,0.16 /)
  CF(:,26,11) = (/  489.3,1.,1.18E-10,0.2110,0.15 /)
  CF(:,26,10) = (/ 1262.0,1.,3.61E-10,1.1600,0.09 /)
  CF(:,26, 9) = (/ 1360.0,1.,2.45E-10,0.9780,0.13 /)
  CF(:,26, 8) = (/ 1470.0,1.,1.87E-10,0.9880,0.14 /)
  CF(:,26, 7) = (/ 1582.0,1.,1.33E-10,1.0300,0.12 /)
  CF(:,26, 6) = (/ 1690.0,1.,7.84E-11,0.8480,0.14 /)
  CF(:,26, 5) = (/ 1800.0,0.,8.90E-11,1.2000,0.35 /)
  CF(:,26, 4) = (/ 1960.0,1.,2.29E-11,0.9360,0.12 /)
  CF(:,26, 3) = (/ 2046.0,0.,1.12E-11,0.0340,0.81 /)
  CF(:,26, 2) = (/ 8828.0,1.,2.46E-12,1.0200,0.02 /)
  CF(:,26, 1) = (/ 9277.7,1.,9.79E-13,0.6640,0.14 /)
  CF(:,27,27) = (/    7.9,1.,8.89E-08,0.1270,0.24 /)
  CF(:,27,26) = (/   17.1,1.,5.65E-08,0.1940,0.23 /)
  CF(:,27,25) = (/   33.5,1.,3.06E-08,0.2010,0.22 /)
  CF(:,27,24) = (/   51.3,0.,2.27E-08,0.5740,0.10 /)
  CF(:,27,23) = (/   79.5,0.,1.93E-08,0.1950,0.42 /)
  CF(:,27,22) = (/  102.0,0.,1.27E-08,0.1260,0.47 /)
  CF(:,27,21) = (/  129.0,1.,3.58E-09,0.1940,0.29 /)
  CF(:,27,20) = (/  158.0,0.,1.17E-08,1.9800,0.07 /)
  CF(:,27,19) = (/  186.1,1.,1.78E-09,0.1120,0.42 /)
  CF(:,27,18) = (/  275.0,1.,2.41E-09,0.7390,0.14 /)
  CF(:,27,17) = (/  305.0,1.,1.86E-09,0.7620,0.14 /)
  CF(:,27,16) = (/  336.0,1.,1.41E-09,0.8040,0.14 /)
  CF(:,27,15) = (/  379.0,1.,9.54E-10,0.8130,0.14 /)
  CF(:,27,14) = (/  411.0,1.,7.12E-10,0.8030,0.14 /)
  CF(:,27,13) = (/  444.0,1.,5.34E-10,0.7180,0.15 /)
  CF(:,27,12) = (/  512.0,1.,3.62E-10,0.6580,0.17 /)
  CF(:,27,11) = (/  546.6,1.,1.05E-10,0.2520,0.12 /)
  CF(:,27,10) = (/ 1397.0,1.,3.10E-10,1.1700,0.09 /)
  CF(:,27, 9) = (/ 1486.0,1.,1.56E-10,0.5720,0.15 /)
  CF(:,27, 8) = (/ 1603.0,1.,1.32E-10,0.6820,0.13 /)
  CF(:,27, 7) = (/ 1735.0,1.,9.08E-11,0.5110,0.17 /)
  CF(:,27, 6) = (/ 1846.0,0.,3.45E-10,2.8400,0.11 /)
  CF(:,27, 5) = (/ 1962.0,1.,5.01E-11,0.7140,0.11 /)
  CF(:,27, 4) = (/ 2119.0,1.,1.92E-11,0.1170,0.42 /)
  CF(:,27, 3) = (/ 2219.0,1.,1.95E-11,1.2000,0.00 /)
  CF(:,27, 2) = (/ 9544.0,0.,1.68E-11,3.5200,0.05 /)
  CF(:,27, 1) = (/10012.1,1.,1.45E-12,0.6350,0.15 /)
  CF(:,28,28) = (/    7.6,0.,1.65E-07,0.4520,0.28 /)
  CF(:,28,27) = (/   18.2,0.,8.42E-08,0.6190,0.16 /)
  CF(:,28,26) = (/   35.3,1.,1.89E-08,0.2200,0.21 /)
  CF(:,28,25) = (/   54.9,1.,1.48E-08,0.2160,0.21 /)
  CF(:,28,24) = (/   76.0,0.,1.13E-08,0.5180,0.09 /)
  CF(:,28,23) = (/  108.0,0.,1.16E-08,0.1850,0.44 /)
  CF(:,28,22) = (/  133.0,0.,8.68E-09,0.1380,0.46 /)
  CF(:,28,21) = (/  162.0,1.,2.45E-09,0.1630,0.32 /)
  CF(:,28,20) = (/  193.0,0.,9.24E-09,2.2500,0.05 /)
  CF(:,28,19) = (/  225.0,0.,2.41E-09,0.0270,0.79 /)
  CF(:,28,18) = (/  321.0,1.,1.92E-09,0.7380,0.14 /)
  CF(:,28,17) = (/  352.0,1.,1.50E-09,0.7610,0.14 /)
  CF(:,28,16) = (/  384.0,1.,1.16E-09,0.8030,0.14 /)
  CF(:,28,15) = (/  430.0,1.,8.08E-10,0.8560,0.13 /)
  CF(:,28,14) = (/  464.0,1.,6.09E-10,0.8500,0.13 /)
  CF(:,28,13) = (/  499.0,1.,4.48E-10,0.7180,0.15 /)
  CF(:,28,12) = (/  571.3,1.,3.00E-10,0.6220,0.18 /)
  CF(:,28,11) = (/  607.0,1.,7.90E-11,0.1600,0.19 /)
  CF(:,28,10) = (/ 1541.0,1.,2.78E-10,1.2500,0.08 /)
  CF(:,28, 9) = (/ 1648.0,1.,1.92E-10,1.0400,0.12 /)
  CF(:,28, 8) = (/ 1756.0,1.,1.51E-10,1.1100,0.12 /)
  CF(:,28, 7) = (/ 1894.0,1.,1.05E-10,1.0900,0.11 /)
  CF(:,28, 6) = (/ 2011.0,1.,6.04E-11,0.8490,0.14 /)
  CF(:,28, 5) = (/ 2131.0,0.,6.91E-11,1.2100,0.35 /)
  CF(:,28, 4) = (/ 2295.0,1.,1.84E-11,0.9910,0.11 /)
  CF(:,28, 3) = (/ 2399.0,0.,9.03E-12,0.0420,0.79 /)
  CF(:,28, 2) = (/10290.0,1.,2.61E-12,0.5680,0.16 /)
  CF(:,28, 1) = (/10775.3,1.,1.39E-12,0.7290,0.13 /)


  c = 0.0
  if (iz.lt.1 .or. iz.gt.28) return
  if (in.lt.1 .or. in.gt.iz) return
  te = t * 8.617385e-05
  u = CF(1,iz,in) / te
  if ( u.gt.80.0) return
  c = CF(3,iz,in) * ( 1.0 + CF(2,iz,in) * sqrt(u) ) / ( CF(4,iz,in)+u ) * u**CF(5,iz,in) * exp(-u)
  return
  
  !*********************************************
  
  
end subroutine cfit


end module voronov_atomic_rates_mod




