# KeyCloakAuthenticator

Authenticates users via SSO using OIDC. 

This authenticator implements a refresh mechanism, ensuring that the tokens stored in the user dict are always up-to-date (if the update is not possible, it forces a re-authentication of the user). It also allows exchanging the user token for tokens that can be used to authenticate against other (external) services.

This Authenticator is built on top of [OAuthenticator](https://github.com/jupyterhub/oauthenticator) and should be possible to use some of its configuration values.


## Requirements

* Jupyterhub
* oauthenticator
* PyJWT[crypto]
* openssl\_devel (see below)

## Installation

```bash
pip install keycloakauthenticator
```

If you enable check\_signature, you also need the `openssl_devel` (or equivalent in your distribution) package.

## Usage

In your JupyterHub config file, set the authenticator and configure it:

```python
# Enable the authenticator
c.JupyterHub.authenticator_class = 'keycloakauthenticator.KeyCloakAuthenticator'
c.KeyCloakAuthenticator.username_key = 'preferred_username'
c.KeyCloakAuthenticator.logout_redirect_uri = 'https://cern.ch/swan'
c.KeyCloakAuthenticator.oauth_callback_url = 'https://swan.cern.ch/hub/oauth_callback'

# Specify the issuer url, to get all the endpoints automatically from .well-known/openid-configuration
c.KeyCloakAuthenticator.oidc_issuer = 'https://auth.cern.ch/auth/realms/cern'

# If you need to set a different scope, like adding the offline option for longer lived refresh token
c.KeyCloakAuthenticator.scope = ['profile', 'email', 'offline_access']
# Only allow users with this specific roles (none, to allow all)
c.KeyCloakAuthenticator.accepted_roles = set()
# Specify the role to set a user as admin
c.KeyCloakAuthenticator.admin_role = 'swan-admin'
# Request access tokens for other services by passing their id's (this uses the token exchange mechanism)
c.KeyCloakAuthenticator.exchange_tokens = ['eos-service', 'cernbox-service']

# If your authenticator needs extra configurations, set them in the pre-spawn hook
def pre_spawn_hook(authenticator, spawner, auth_state):
    spawner.environment['ACCESS_TOKEN'] = auth_state['exchanged_tokens']['eos-service']
    spawner.environment['OAUTH_INSPECTION_ENDPOINT'] = authenticator.userdata_url.replace('https://', '')
    spawner.user_roles = authenticator.get_roles_for_token(auth_state['access_token'])
    spawner.user_uid = auth_state['oauth_user']['cern_uid']
c.KeyCloakAuthenticator.pre_spawn_hook = pre_spawn_hook

#Configure token signature verification
c.KeyCloakAuthenticator.check_signature=True
c.KeyCloakAuthenticator.jwt_signing_algorithms = ["HS256", "RS256"]
```

It's also necessary to configure the Client ID and secret. One way of doing this is by setting the following environment variables:

```bash
OAUTH_CLIENT_ID=my_id
OAUTH_CLIENT_SECRET=my_secret
```
