# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['openhasp_config_manager',
 'openhasp_config_manager.model',
 'openhasp_config_manager.processing',
 'openhasp_config_manager.validation']

package_data = \
{'': ['*']}

install_requires = \
['Jinja2', 'click', 'paho-mqtt', 'py-range-parse', 'pyyaml', 'requests']

entry_points = \
{'console_scripts': ['openhasp-config-manager = '
                     'openhasp_config_manager.cli:cli']}

setup_kwargs = {
    'name': 'openhasp-config-manager',
    'version': '0.0.2',
    'description': 'A tool to manage all of your openHASP device configs in a centralized place.',
    'long_description': '# openhasp-config-manager\n\nA tool to manage all of your [openHASP](https://github.com/HASwitchPlate/openHASP) device configs in a centralized place.\n\n# Features\n\n* [x] manage as many devices as you like\n* [x] share configuration files between devices\n* [x] jsonl preprocessing, which allows for\n  * [x] `//` comments within jsonl files\n  * [x] line breaks wherever you like\n  * [x] jinja2 templating within object values\n* [x] output validation\n  * [x] no more weird behavior due to invalid "id" range\n* [x] one click configuration upload to the device\n  * [x] automatic diffing to only update changed configuration files\n* [x] execute commands directly from the CLI (still needs a connection to the MQTT broker)\n\n# How to use\n\n## Installation\n\nSince openhasp-config-manager needs some dependencies (see [here](/pyproject.toml)) it is\nrecommended to install it inside of a virtualenv:\n\n```\npython3 -m venv ~/path/to/new/virtual/environment\nsource ~/path/to/new/virtual/environment/bin/activate\npip3 install openhasp-config-manager\n```\n\n## Configuration\n\nopenhasp-config-manager is first and foremost a configuration\nmanagement system. Simply follow the basic folder structure and\nconfig deployment will become trivial.\n\n* `devices`: In the root directory of your configuration, a folder called\n  `devices` is expected.\n  * In there you can create as many subfolders as\n    you like, naming them according to the physical devices that you\n    want to manage.\n    * Within those device subfolders you can then create\n      `*.jsonl` and `*.cmd` files.\n    * You must also provide a `config.json` file, see [config.json](#config.json)\n      for more info on how to set it.\n* `common`: The `common` directory can be used to put files\n  that should be included on _all_ device.\n\nYou are not limited to a folder depth of one. However, the files\non OpenHasp devices cannot be put into subfolders. Therefore, if you put\n`.json` or `.cmd` files into subfolders, the name of the\nresulting file on the OpenHasp device will be a concatenation of\nthe full subpath using an underscore (`_`) as a separator. So f.ex.\nthe file in the following structure:\n\n```text\nopenhasp-configs\n└── devices\n    └── touch_down_1\n        └── 0_home\n            └── 0_header.jsonl\n```\n\nwould only be uploaded to the `touch_down_1` device and named:\n`0_home_0_header.jsonl`\n\nA more advanced configuration layout could look something like this:\n\n```text\nopenhasp-configs\n├── common\n│\xa0\xa0 ├── content\n│\xa0\xa0 │\xa0\xa0 └── card.jsonl\n│\xa0\xa0 ├── dialog\n│\xa0\xa0 │\xa0\xa0 ├── connected.jsonl\n│\xa0\xa0 │\xa0\xa0 └── offline.jsonl\n│\xa0\xa0 ├── navigation_bar.jsonl\n│\xa0\xa0 └── page_header.jsonl\n└── devices\n    └── touch_down_1\n        ├── 0_home\n        │\xa0\xa0 ├── 0_header.jsonl\n        │\xa0\xa0 ├── 1_content.jsonl\n        │\xa0\xa0 └── page.cmd\n        ├── 5_about\n        │\xa0\xa0 ├── 0_header.jsonl\n        │\xa0\xa0 ├── 1_content.jsonl\n        │\xa0\xa0 └── page.cmd\n        ├── boot.cmd\n        ├── config.json\n        ├── offline.cmd\n        └── online.cmd\n```\n\n### config.json\n\nopenhasp-config-manager makes use of the `config.json` on your plate. It can use information \nto detect things like screen orientation, and also allows you to deploy config changes to your\nplate when you make changes in the config.json file. Since [the official API does not support\nuploading the full file](https://github.com/HASwitchPlate/openHASP/issues/363), only settings \nwhich can also be set through the web ui on the plate itself are currently supported.\n\nTo retrieve the initial version of the `config.json` file you can use the\nbuilt-in file browser integrated into the webserver of your OpenHASP plate.\n\nThe official `config.json` file doesn\'t provide enough info for openhasp-config-manager\nto enable all of its features though. To fix that openhasp-config-manager looks for an \nadditional section within the file which is not present by default:\n\n```json\n{\n  "openhasp_config_manager": {\n    "device": {\n      "ip": "192.168.5.134",\n      "screen": {\n        "width": 320,\n        "height": 480\n      }\n    }\n  },\n  "wifi": {\n    "ssid": "Turris IoT",\n    ...\n  }\n```\n\nSimply add this section to the `config.json` after you have retrieved it from\nthe plate.\n\n### Preprocessing\n\nopenhasp-config-manager runs all configuration files through a preprocessor, which allows us to use\nfeatures the original file format doesn\'t support, like f.ex. templating.\n\n#### Templating\n\nYou can use Jinja2 templates inside of values. You can access each of the objects using the\n`pXbY` syntax established by OpenHASP, where `X` is the `page` of an object and `Y` is its `id`.\n\nYou can use the full functionality of Jinja2 like f.ex. math operations, function calls or type conversions.\n\n```yaml\n{\n  "page": 1,\n  "id": 1,\n  "x": 0,\n  "y": 0,\n  ...\n}\n\n{\n  "page": 1,\n  "id": 2,\n  "x": "{{ p1b1.x }}",\n  "y": "{{ p1b1.y + 10 }}",\n  ...\n}\n```\n\n#### Variables\n\nBesides accessing other objects, you can also define custom variables yourself, which can then\nbe used inside of templates.\n\n##### Global\n\nGlobal variables can be specified by creating `*.yaml` files inside of the `common` folder.\n\nExample:\n\n`common/global.vars.yaml`\n\n```yaml\nabout:\n  page_title: "About"\n```\n\n`common/about_page.jsonl`\n\n```json lines\n{\n  "page": 9,\n  "id": 1,\n  ...\n  "title": "{{ about.page_title }}",\n  ...\n}\n```\n\n##### Device specific\n\nDevice specific variables can be specified by creating `*.yaml` files inside any of the sub-folders\nof the `device` folder.\n\n> **Note**\n>\n> Device specific variables will override global variables, given the same name.\n\nExample:\n\n`device/my_device/device.vars.yaml`\n\n```yaml\npage_title: "My Device"\n```\n\n`device/my_device/some_folder/some_page.jsonl`\n\n```json lines\n{\n  "page": 1,\n  "id": 1,\n  ...\n  "title": "{{ page_title }}",\n  ...\n}\n```\n\n`device/my_device/some_other_folder/some_page.jsonl`\n\n```json lines\n{\n  "page": 2,\n  "id": 1,\n  ...\n  "title": "{{ page_title }}",\n  ...\n}\n```\n\n## Deployment\n\nTo deploy your configurations to the already connected OpenHASP devices, simply use the\ncommand line tool `openhasp-config-manager`:\n\n```shell\n> openhasp-config-manager                                                         0 (0.604s) < 02:11:38\nUsage: openhasp-config-manager [OPTIONS] COMMAND [ARGS]...\n\nOptions:\n  --version   Show the version and exit.\n  -h, --help  Show this message and exit.\n\nCommands:\n  cmd       Sends a command request to a device.\n  deploy    Combines the generation and upload of a configuration.\n  generate  Generates the output files for all devices in the given...\n  upload    Uploads the previously generated configuration to their...\n```\n\n## Run commands\n\nOpenHasp allows you to run commands on a device by issuing MQTT messages.\nWhile openhasp-config-manager is first and foremost a config management system,\nit also allows you to run commands on a device without the need to install a separate\nMQTT client first. Note that the MQTT _server_ does need to be running and also has to\nbe reachable from your local machine.\n\nSee: https://openhasp.haswitchplate.com/latest/commands/\n\n```shell\n> openhasp-config-manager cmd -c ./openhasp-configs -d plate35 -C backlight -p "{\\"state\\":\\"on\\",\\"brightness\\":128}"\n```\n\n# Contributing\n\nGitHub is for social coding: if you want to write code, I encourage contributions\nthrough pull requests from forks of this repository. Create GitHub tickets for\nbugs and new features and comment on the ones that you are interested in.\n\n# License\n\n```text\nopenhasp-config-manager is free software: you can redistribute it and/or modify\nit under the terms of the GNU AFFERO GENERAL PUBLIC LICENSE as published by\nthe Free Software Foundation, either version 3 of the License, or\n(at your option) any later version.\n\nThis program is distributed in the hope that it will be useful,\nbut WITHOUT ANY WARRANTY; without even the implied warranty of\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\nGNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License\nalong with this program.  If not, see <https://www.gnu.org/licenses/>.\n```\n',
    'author': 'Markus Ressel',
    'author_email': 'mail@markusressel.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/markusressel/openhasp-config-manager',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
