# Copyright 2016 Clearpath Robotics Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
import time
import traceback
import yaml

try:
    # Python3
    from queue import Queue
except ImportError:
    # Python2
    from Queue import Queue

from catkin_pkg.packages import find_packages
from catkin_pkg.topological_order import topological_order_packages
from catkin_pkg.package import parse_package

from catkin_tools.common import log
from catkin_tools.common import wide_log
from catkin_tools.common import get_cached_recursive_build_depends_in_workspace

from catkin_tools.execution.controllers import ConsoleStatusController
from catkin_tools.execution.executor import execute_jobs
from catkin_tools.execution.executor import run_until_complete
from catkin_tools.execution.jobs import Job
from catkin_tools.execution.stages import CommandStage
from catkin_tools.execution.stages import FunctionStage
from catkin_tools.terminal_color import fmt

from catkin_tools.jobs.utils import makedirs

from catkin_tools.verbs.catkin_build.build import determine_packages_to_be_built
from catkin_tools.verbs.catkin_build.build import verify_start_with_option

from catkin_tools_document import builders
from .messages import generate_messages
from .messages import generate_services
from .messages import generate_package_summary
from .messages import generate_overall_summary
from .util import which


def create_package_job(context, package, package_path, deps):
    docs_space = os.path.realpath(os.path.join(context.build_space_abs, '..', 'docs', package.name))
    docs_build_space = os.path.realpath(os.path.join(context.build_space_abs, 'docs', package.name))
    package_path_abs = os.path.realpath(os.path.join(context.source_space_abs, package_path))

    # Load rosdoc config, if it exists.
    rosdoc_yaml_path = os.path.join(package_path_abs, 'rosdoc.yaml')
    for export in package.exports:
        if export.tagname == "rosdoc":
            config = export.attributes.get('config', '')
            if config:
                rosdoc_yaml_path_temp = os.path.join(package_path_abs, config)
                if os.path.exists(rosdoc_yaml_path_temp):
                    # Stop if configuration is found which exists
                    rosdoc_yaml_path = rosdoc_yaml_path_temp
                    break

    if os.path.exists(rosdoc_yaml_path):
        with open(rosdoc_yaml_path) as f:
            rosdoc_conf = yaml.full_load(f)
    else:
        if os.path.exists(os.path.join(package_path_abs, 'src')) or \
            os.path.exists(os.path.join(package_path_abs, 'include')):
            rosdoc_conf = [{'builder': 'doxygen'}]
        else:
            rosdoc_conf = []

    stages = []

    # Create package docs spaces.
    stages.append(FunctionStage('mkdir_docs_build_space', makedirs, path=docs_build_space))

    # Generate msg/srv/action docs with package summary page.
    stages.append(FunctionStage('generate_messages', generate_messages,
                                package=package, package_path=package_path,
                                output_path=docs_build_space))
    stages.append(FunctionStage('generate_services', generate_services,
                                package=package, package_path=package_path,
                                output_path=docs_build_space))
    stages.append(FunctionStage('generate_package_summary', generate_package_summary,
                                package=package, package_path=package_path_abs,
                                rosdoc_conf=rosdoc_conf, output_path=docs_build_space))

    # Add steps to run native doc generators, as appropriate. This has to happen after
    # the package summary generates, as we're going to override the subdirectory index.html
    # files generated by that sphinx run.
    for conf in rosdoc_conf:
        try:
            stages.extend(getattr(builders, conf['builder'])(
                conf, package, deps, docs_space, package_path_abs, docs_build_space))
        except AttributeError:
            log(fmt("[document] @!@{yf}Warning:@| Skipping unrecognized rosdoc builder [%s] for package [%s]" %
                (conf['builder'], package.name)))

    return Job(jid=package.name, deps=deps, env={}, stages=stages)


def create_summary_job(context, package_names):
    docs_space = os.path.join(context.build_space_abs, '..', 'docs')
    docs_build_space = os.path.join(context.build_space_abs, 'docs')

    stages = []

    stages.append(FunctionStage('generate_overall_summary', generate_overall_summary,
                                output_path=docs_build_space))

    # Run Sphinx for the package summary.
    stages.append(CommandStage(
        'summary_sphinx',
        [which('sphinx-build'), '-j8', '-E', '.', docs_space],
        cwd=docs_build_space
    ))

    return Job(jid='summary', deps=package_names, env={}, stages=stages)


def document_workspace(
    context,
    packages=None,
    start_with=None,
    no_deps=False,
    n_jobs=None,
    force_color=False,
    quiet=False,
    interleave_output=False,
    no_status=False,
    limit_status_rate=10.0,
    no_notify=False,
    continue_on_failure=False,
    summarize_build=None
):
    pre_start_time = time.time()

    # Get all the packages in the context source space
    # Suppress warnings since this is a utility function
    workspace_packages = find_packages(context.source_space_abs, exclude_subspaces=True, warnings=[])

    # If no_deps is given, ensure packages to build are provided
    if no_deps and packages is None:
        log(fmt("[document] @!@{rf}Error:@| With no_deps, you must specify packages to build."))
        return

    # Find list of packages in the workspace
    packages_to_be_documented, packages_to_be_documented_deps, all_packages = determine_packages_to_be_built(
        packages, context, workspace_packages)

    if not no_deps:
        # Extend packages to be documented to include their deps
        packages_to_be_documented.extend(packages_to_be_documented_deps)

    # Also re-sort
    try:
        packages_to_be_documented = topological_order_packages(dict(packages_to_be_documented))
    except AttributeError:
        log(fmt("[document] @!@{rf}Error:@| The workspace packages have a circular "
                "dependency, and cannot be documented. Please run `catkin list "
                "--deps` to determine the problematic package(s)."))
        return

    # Check the number of packages to be documented
    if len(packages_to_be_documented) == 0:
        log(fmt('[document] No packages to be documented.'))

    # Assert start_with package is in the workspace
    verify_start_with_option(
        start_with,
        packages,
        all_packages,
        packages_to_be_documented + packages_to_be_documented_deps)

    # Remove packages before start_with
    if start_with is not None:
        for path, pkg in list(packages_to_be_documented):
            if pkg.name != start_with:
                wide_log(fmt("@!@{pf}Skipping@|  @{gf}---@| @{cf}{}@|").format(pkg.name))
                packages_to_be_documented.pop(0)
            else:
                break

    # Get the names of all packages to be built
    packages_to_be_documented_names = [p.name for _, p in packages_to_be_documented]
    packages_to_be_documeted_deps_names = [p.name for _, p in packages_to_be_documented_deps]

    jobs = []

    # Construct jobs
    for pkg_path, pkg in all_packages:
        if pkg.name not in packages_to_be_documented_names:
            continue

        # Get actual execution deps
        deps = [
            p.name for _, p
            in get_cached_recursive_build_depends_in_workspace(pkg, packages_to_be_documented)
        ]

        jobs.append(create_package_job(context, pkg, pkg_path, deps))

    # Special job for post-job summary sphinx step.
    jobs.append(create_summary_job(context, package_names=packages_to_be_documented_names))

    # Queue for communicating status
    event_queue = Queue()

    try:
        # Spin up status output thread
        status_thread = ConsoleStatusController(
            'document',
            ['package', 'packages'],
            jobs,
            n_jobs,
            [pkg.name for _, pkg in context.packages],
            [p for p in context.whitelist],
            [p for p in context.blacklist],
            event_queue,
            show_notifications=not no_notify,
            show_active_status=not no_status,
            show_buffered_stdout=not quiet and not interleave_output,
            show_buffered_stderr=not interleave_output,
            show_live_stdout=interleave_output,
            show_live_stderr=interleave_output,
            show_stage_events=not quiet,
            show_full_summary=(summarize_build is True),
            pre_start_time=pre_start_time,
            active_status_rate=limit_status_rate)
        status_thread.start()

        # Block while running N jobs asynchronously
        try:
            all_succeeded = run_until_complete(execute_jobs(
                'document',
                jobs,
                None,
                event_queue,
                context.log_space_abs,
                max_toplevel_jobs=n_jobs,
                continue_on_failure=continue_on_failure,
                continue_without_deps=False))

        except Exception:
            status_thread.keep_running = False
            all_succeeded = False
            status_thread.join(1.0)
            wide_log(str(traceback.format_exc()))
        status_thread.join(1.0)

        return 0 if all_succeeded else 1

    except KeyboardInterrupt:
        wide_log("[document] Interrupted by user!")
        event_queue.put(None)

        return 130  # EOWNERDEAD
