# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyfunds']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyfunds',
    'version': '1.1.0',
    'description': 'Python Functional Data Structures',
    'long_description': '# pyfunds\nPython Functional Data Structures\n\nThis repository contains an implementation from scratch of simple functional data structures in Python. \n\nThe following structures are implemented and tested:\n - Either\n - Option\n - Try\n\nThis library is inspired by Scala\'s implementation of these structures.\n\nFeel free to open an issue or [send me an email](mailto:lucaruzzola@gmail.com) \nin case you\'d like to contribute or if you see something that can be improved.\n\n## Installation\nThis project is published on [PyPi](https://pypi.org/project/pyfunds/) as `pyfunds` so you can easily install it with `pip` as:\n```shell\npip install pyfunds\n```\nor with `poetry` as:\n```shell\npoetry add pyfunds\n```\n\n## Setup\n\n### Poetry\nThis project uses [poetry](https://github.com/python-poetry/poetry) to manage its dependencies.\nPlease refer to [poetry\'s official doc](https://python-poetry.org/docs/) for more info.\n\n\n## Usage Examples\n\n### Either\n\n`Either` represents a value that can assume one of two types.\n\nConcrete instances are of type `Left` or `Right`.\n\nAs an example, let\'s consider the case of making HTTP calls which might return a \nstatus code representing an error as the url is user-defined. \nIf a call is successful, we want to return the JSON from the response, but if it\'s\nnot we\'ll map it to an internal error message.\n\nThe examples use this [example server](https://jsonplaceholder.typicode.com).\n\n```python\nimport requests\nfrom pyfunds.either import Left, Right, Either\nfrom typing import Dict, Any\n\ndef map_response_to_msg(response: requests.models.Response):\n    return f"The {response.request.method} request to {response.url} couldn\'t be completed " \\\n    f"and returned a {response.status_code} status_code"\n\ndef call_and_check(url: str) -> Either[str, Dict[Any, Any]]:\n    response = requests.get(url)\n    return Right(response.json()) if response.ok else Left(map_response_to_msg(response))\n```\n\nUsers of this method will then be able to further chain operations which can result in 2 different results easily,\nkeeping track of the error message identifying the step that returned something unexpected in the chain.\n\n```python\nbase_url = "https://jsonplaceholder.typicode.com"\nusers_json = call_and_check(f"{base_url}/users")\nposts = users_json.flat_map(lambda json: call_and_check(f"{base_url}/posts?userId={json[0][\'id\']}"))\n```\n\nLastly, we\'ll log the content of the `Either`at the appropriate level in each case; the contained string in the `Left` \ncase at `warn`, or the `msg` field of the JSON dictionary in the `Right` case at `info`.\n\n```python\nfrom logging import getLogger\n\nlogger = getLogger()\n\nposts.fold(logger.warning, lambda x: logger.info(x[0]["title"]))\n```\n\nThe above example enters the `Right` branch of the `Either`, change the `base_url` to `$base_url/pizza` to get a `Left` at the first stage.\n\nPlease note that this is different from the case where an `Exception` is raised, which better fits the `Try` structure \ndescribed below.\n\n### Option\n\n`Option` represents an optional value, its concrete instances are \nof type `Nothing` or `Some`.\n\nAs an example, let\'s consider the case of checking for a variable in a dictionary.\nNormally, a default value of `None` is returned if the request key is not present in the dictionary,\nhowever this requires the user of method returning such a value to check explicitly the content of the return variable.\n\nFurther, multiple calls of this type cannot be chained together, and the value needs to be checked every time.\nUsing `Option` we can instead reason using the type directly, and demanding to it the checking steps.\n\n```python\nfrom pyfunds.option import Option\n\nd = {"food": "Pizza"}\n\nresult = Option.apply(d.get("another_key"))\n\nawesomize = lambda x: x + "is awesome" \n\nmsg = result.map(awesomize)\n```\n\nThis way we didn\'t need to check whether the key was present in the dictionary or not.\nFinally, we can get a default value to go from an `Option` to a `str`.\n\n```python\nmsg.fold("Pizza is incredible anyways!", lamdba x: x + ", but fries are good too!")\n```\n\nThe final `msg` will be `Pizza is incredible anyways!`.\n\nIf instead we had looked for the `food` key, `msg` would have been `Pizza is awesome, but fries are good too!`\n\n### Try\n\n`Try` represents a computation that can either fail (raising an Exception) or return the resulting value.\n\nConcrete instances are of type `Failure` or `Success`.\n\nAs an example, let\'s see the case of a function that can raise an Exception:\n``` python\nimport math\n\ndef unsafe_computation(value: int):\n    return math.log(value)  # this throws an Exception if value is <= 0\n```\n\nUpon calling this function with `value` <= 0 we\'ll see:\n\n```python\nunsafe_computation(0)\n```\n```shell\nTraceback (most recent call last):\n  File "<stdin>", line 1, in <module>\nValueError: math domain error\n```\n\nTo make this computation safe, even for `value` <= 0, we\'ll wrap its execution with Try:\n```python\nfrom pyfunds.try_ import Try\n\nsafe_result = Try.apply(unsafe_computation, 0)\n```\n\n`safe_result` will be of type `Failure`, containing the Exception.\nIn case it was called on proper input:\n```python\nsafe_result = Try.apply(unsafe_computation, 1)\n```\n\n`safe_result` will be of type Success and it will contain the proper result.\n\nPlease notice that you need to pass the function and any function arguments, named and not, as arguments to \n`Try.apply()` rather than passing `f(args)`.\n\nAlternatively, you can use this syntax:\n```python\nsafe_result = Try.apply(lambda: unsafe_computation(0))\n```\n\nUsing `Try`, an appropriate return type can be used for methods that might fail and raise an `Exeception`, \nleaving the user in charge of easily dealing with the subsequent behavior, for example:\n\n```python\nTry.apply(unsafe_computation, 1).map(lambda x: x + 1)\n```',
    'author': 'Luca Ruzzola',
    'author_email': 'lucaruzzola@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/lucaruzzola/pyfunds',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
