"""
Classifies: CHEBI:24866 salt
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_salt(smiles: str):
    """
    Determines if a molecule is a salt (assembly of cations and anions)
    
    Args:
        smiles (str): SMILES string of the molecule
        
    Returns:
        bool: True if molecule is a salt, False otherwise
        str: Reason for classification
    """
    try:
        mol = Chem.MolFromSmiles(smiles)
        if mol is None:
            return False, "Invalid SMILES string"
            
        # Split into fragments
        fragments = Chem.GetMolFrags(mol, asMols=True)
        
        if len(fragments) < 1:
            return False, "No molecular fragments found"
            
        # Check formal charges
        cations = []
        anions = []
        neutral = []
        
        # Track total system charge
        total_system_charge = 0
        
        for frag in fragments:
            total_charge = sum(atom.GetFormalCharge() for atom in frag.GetAtoms())
            total_system_charge += total_charge
            
            # Check for zwitterions
            has_positive = any(atom.GetFormalCharge() > 0 for atom in frag.GetAtoms())
            has_negative = any(atom.GetFormalCharge() < 0 for atom in frag.GetAtoms())
            
            if total_charge > 0 or (has_positive and not has_negative):
                cations.append(frag)
            elif total_charge < 0 or (has_negative and not has_positive):
                anions.append(frag)
            elif has_positive and has_negative:
                # Zwitterion - treat as neutral
                neutral.append(frag)
            else:
                neutral.append(frag)

        # Special cases
        if len(fragments) == 1:
            if len(cations) == 0 and len(anions) == 0:
                return False, "Single neutral molecule"
            elif len(cations) > 0 and len(anions) > 0:
                return False, "Single zwitterionic molecule"

        # Check for acid/base pairs
        if len(neutral) > 0 and (len(cations) == 0 or len(anions) == 0):
            # Look for acid/base characteristics in neutral fragments
            for frag in neutral:
                for atom in frag.GetAtoms():
                    if atom.GetSymbol() in ['N', 'O'] and atom.GetTotalNumHs() > 0:
                        # Potential acid/base participant
                        if len(cations) == 0:
                            cations.append(frag)
                        elif len(anions) == 0:
                            anions.append(frag)

        # Final classification
        if len(cations) > 0 and len(anions) > 0:
            msg = f"Salt containing {len(cations)} cation(s) and {len(anions)} anion(s)"
            if len(neutral) > 0:
                msg += f" and {len(neutral)} neutral species"
            return True, msg
        elif len(cations) == 0 and len(anions) == 0:
            return False, "No charged species present"
        elif len(cations) == 0:
            return False, "Only anions present - not a salt"
        else:
            return False, "Only cations present - not a salt"
                
    except Exception as e:
        return None, f"Error analyzing structure: {str(e)}"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:24866',
                          'name': 'salt',
                          'definition': 'A salt is an assembly of cations and '
                                        'anions.',
                          'parents': ['CHEBI:37577']},
    'config': {   'llm_model_name': 'lbl/claude-sonnet',
                  'f1_threshold': 0.8,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0.5941644562334217 is too low.\n'
               'True positives: '
               "[('[I-].C[N+]1(C)CCC(CC1)OC(=O)C(c1ccccc1)c1ccccc1', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[Na+].Oc1cc(c2ccc3c(cc(c4ccc1c2c34)S([O-])(=O)=O)S([O-])(=O)=O)S([O-])(=O)=O', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('[O-]Cl(=O)(=O)=O.[H]C(=C([H])c1oc2ccccc2[n+]1CCCCCCCCCCCCCCCCCC)C([H])=C1Oc2ccccc2N1CCCCCCCCCCCCCCCCCC', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[K+].C1=CC=CC2=C1C(=C(C(O2)=O)[C@@H](CC(C)=O)C3=CC=CC=C3)[O-]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[S--].[Fe+3].[As-]', 'Salt containing 1 cation(s) and 2 "
               "anion(s)'), "
               "('[Na+].C=1(C(=CC=CC1)NC=2C=CC(=CC2)\\\\C(=C/3\\\\C=C/C(/C=C3)=N/C=4C=CC=CC4)\\\\C=5C=C(C(=CC5)N)C)S([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('C1(C(C[NH2+]CCCC)O)=CC=C(C=C1)O.S(=O)(=O)([O-])[O-].C1(C(C[NH2+]CCCC)O)=CC=C(C=C1)O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[O]N1C(C([N+](=C1C=2C=CC(=CC2)C(=O)[O-])[O-])(C)C)(C)C.[K+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)C(C([O-])=O)c1ccccc1)C([O-])=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Cl-].[Cl-].[H][C@]12C[C@@]3([H])[C@@]4(CC[N@@+]3(CC=C)C\\\\C1=C\\\\CO)c1ccccc1N1\\\\C=C3\\\\[C@@]5([H])C[C@@]6([H])[C@@]7(CC[N@@+]6(CC=C)C\\\\C5=C\\\\CO)c5ccccc5N(\\\\C=C2/[C@@]41[H])[C@@]37[H]', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[H+].[H+].O[C@H]([C@@H](O)C([O-])=O)C([O-])=O.C[C@H](N)[C@H](O)c1cccc(O)c1', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 1 neutral "
               "species'), ('[Co++].CC([O-])=O.CC([O-])=O', 'Salt containing 1 "
               "cation(s) and 2 anion(s)'), "
               "('[Na+].[Na+].[O-]C(=O)CC([O-])=O', 'Salt containing 2 "
               "cation(s) and 1 anion(s)'), "
               "('[K+].CC[N+]1=C(/C=C/C=C/C=C2\\\\N(C)c3ccc(NC(=O)CI)cc3C2(C)C)C(C)(C)c2c1ccc1c(cc(cc21)S([O-])(=O)=O)S([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@]2(NC(=O)C([H])(C([O-])=O)c1ccsc1)OC)C([O-])=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[Na+].OC(=O)CN(CCN(CC([O-])=O)CC([O-])=O)CC([O-])=O', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('O.O.O.O.O.[Mn+2].S([O-])([O-])(=O)=O', 'Salt containing 1 "
               "cation(s) and 1 anion(s) and 5 neutral species'), "
               "('[K+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2N1C(=O)[C@H](NC1(C)C)c1ccccc1)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].COC1=CC(OC)=NC(OC2=CC=CC(OC3=NC(OC)=CC(OC)=N3)=C2C([O-])=O)=N1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[H]C(=C([H])c1cc([o+]c2cc3N(CC)C(C)(C)C=C(C)c3cc12)C(C)(C)C)C([H])=C1N(CCCS([O-])(=O)=O)c2ccc(cc2C1(C)CCCC(O)=O)S([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[Ca++].[O-]C(=O)CN(CCN(CC([O-])=O)CC([O-])=O)CC([O-])=O', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('[As-](F)(F)(F)(F)(F)F.[Na+]', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), ('[Be++].[O-]S([O-])(=O)=O', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)[C@H](NC(=O)N1CCN(C1=O)S(C)(=O)=O)c1ccccc1)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[H+].[H+].[O-]C(=O)\\\\C=C/C([O-])=O.CCCCCNC(=N)N\\\\N=C\\\\c1c[nH]c2ccc(OC)cc12', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 1 neutral "
               "species'), ('[Cl-].[Cl-].[Ca++]', 'Salt containing 1 cation(s) "
               "and 2 anion(s)'), "
               "('[Na+].[Na+].[O-][Cr](=O)(=O)O[Cr]([O-])(=O)=O', 'Salt "
               "containing 2 cation(s) and 1 anion(s)'), "
               "('[Cr+3].[Cr+3].[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 3 anion(s)'), "
               "('[H+].[Cl-].NC1=NCC2N1c1ccccc1Cc1ccccc21', 'Salt containing 1 "
               "cation(s) and 1 anion(s) and 1 neutral species'), "
               "('[Cl-].[H][N+](C)(C)CCCN1c2ccccc2CCc2ccccc12', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Cl-].[H][C@@]1(C[C@]23CC[C@]1(OC)[C@@H]1Oc4c(O)ccc5C[C@H]2[NH+](CC[C@@]31c45)CC1CC1)[C@](C)(O)C(C)(C)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[K+].O(C(C)C([O-])=O)CCOCC', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), "
               "('[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.CC(C)CCCCC(=O)N[C@@H](CC[NH3+])C(=O)N[C@@H]([C@@H](C)O)C(=O)N[C@@H](CC[NH3+])C(=O)N[C@H]1CCNC(=O)[C@@H](NC(=O)[C@H](CC[NH3+])NC(=O)[C@H](CC[NH3+])NC(=O)[C@H](CC(C)C)NC(=O)[C@@H](CC(C)C)NC(=O)[C@H](CC[NH3+])NC1=O)[C@@H](C)O.CC(C)CCCCC(=O)N[C@@H](CC[NH3+])C(=O)N[C@@H]([C@@H](C)O)C(=O)N[C@@H](CC[NH3+])C(=O)N[C@H]1CCNC(=O)[C@@H](NC(=O)[C@H](CC[NH3+])NC(=O)[C@H](CC[NH3+])NC(=O)[C@H](CC(C)C)NC(=O)[C@@H](CC(C)C)NC(=O)[C@H](CC[NH3+])NC1=O)[C@@H](C)O', "
               "'Salt containing 2 cation(s) and 5 anion(s)'), "
               "('[Na+].[Na+].[H][C@@]12CCC3=CC(=O)CC[C@]3(C)[C@@]1([H])[C@@H](O)C[C@@]1(C)[C@@]2([H])CC[C@]1(O)C(=O)COP([O-])([O-])=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Ca++].CC(=O)NCCCS([O-])(=O)=O.CC(=O)NCCCS([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('O.O.O.O.[Ca++].[O-][N+]([O-])=O.[O-][N+]([O-])=O', 'Salt "
               "containing 1 cation(s) and 2 anion(s) and 4 neutral species'), "
               "('[O-]S([O-])(=O)=O.CCNc1cc2[o+]c3cc(ccc3nc2c2ccccc12)N(CC)CC.CCNc1cc2[o+]c3cc(ccc3nc2c2ccccc12)N(CC)CC', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)Cc1ccccc1)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[H+].[H+].CS([O-])(=O)=O.CS([O-])(=O)=O.CN1CCN(CCCN2c3ccccc3Sc3ccc(Cl)cc23)CC1', "
               "'Salt containing 2 cation(s) and 2 anion(s) and 1 neutral "
               "species'), ('S(S([O-])=O)([O-])(=O)=O.[Na+].[Na+]', 'Salt "
               "containing 2 cation(s) and 1 anion(s)'), "
               "('[H+].[Cl-].CCN(CC)CCNC(=O)c1ccc(N)cc1', 'Salt containing 1 "
               "cation(s) and 1 anion(s) and 1 neutral species'), "
               "('[Ca++].[H][C@@]1(C[C@H]2CO[C@@H](C\\\\C(C)=C\\\\C(=O)OCCCCCCCCC([O-])=O)[C@H](O)[C@@H]2O)O[C@@]1([H])[C@@H](C)[C@H](C)O.[H][C@@]1(C[C@H]2CO[C@@H](C\\\\C(C)=C\\\\C(=O)OCCCCCCCCC([O-])=O)[C@H](O)[C@@H]2O)O[C@@]1([H])[C@@H](C)[C@H](C)O', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Na+].[H][C@]1(C[NH2+][C@@]([H])(C1)C(=O)Nc1cccc(c1)C([O-])=O)SC1=C(N2C(=O)[C@]([H])([C@@H](C)O)[C@@]2([H])[C@H]1C)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH4+].[NH4+].[S-]C(=S)NCCNC([S-])=S', 'Salt containing 2 "
               "cation(s) and 1 anion(s)'), "
               "('[Al+3].[Al+3].[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 3 anion(s)'), "
               "('[Na+].[O-]B=O', 'Salt containing 1 cation(s) and 1 "
               "anion(s)'), "
               "('[I-].[I-].[I-].[I-].[H]C(=CC([H])=C1Oc2ccccc2N1C)c1cc[n+](CCC(CCCC(CC[n+]2ccc(C([H])=CC([H])=C3Oc4ccccc4N3C)c3ccccc23)=[N+](C)C)=[N+](C)C)c2ccccc12', "
               "'Salt containing 1 cation(s) and 4 anion(s)'), "
               "('[Na+].CC(=O)N[C@H]1[C@H](O)O[C@H](CO)[C@@H](O)[C@@H]1O[C@@H]1O[C@H](COS([O-])(=O)=O)[C@@H](O)[C@H](O)[C@H]1O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH4+].[NH4+].[Ni++].[O-]S([O-])(=O)=O.[O-]S([O-])(=O)=O', "
               "'Salt containing 3 cation(s) and 2 anion(s)'), "
               "('C1[C@@]2([C@]3(CC[C@]4([C@]([C@@]3([C@@H](C[C@@]2(C[C@@H](C1)O)[H])O)[H])(CC[C@@]4([C@@H](CCCS([O-])(=O)=O)C)[H])[H])C)[H])C.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[Na+].[Na+].[Na+].[Na+].[Na+].[Na+].[Na+].[Na+].CO[C@H]1O[C@H](COS([O-])(=O)=O)[C@@H](O[C@@H]2O[C@H]([C@@H](O[C@H]3O[C@H](COS([O-])(=O)=O)[C@@H](O[C@@H]4O[C@@H]([C@@H](O[C@H]5O[C@H](COS([O-])(=O)=O)[C@@H](O)[C@H](O)[C@H]5NS([O-])(=O)=O)[C@H](O)[C@H]4O)C(O)=O)[C@H](OS([O-])(=O)=O)[C@H]3NS([O-])(=O)=O)[C@H](O)[C@H]2OS([O-])(=O)=O)C(O)=O)[C@H](O)[C@H]1NS([O-])(=O)=O', "
               "'Salt containing 10 cation(s) and 1 anion(s)'), "
               "('[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)c1c(C)onc1-c1c(Cl)cccc1Cl)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('O.[Mn++].[O-]S([O-])(=O)=O', 'Salt containing 1 cation(s) "
               "and 1 anion(s) and 1 neutral species'), "
               "('[Na+].[Na+].[O-]S([O-])(=O)=O', 'Salt containing 2 cation(s) "
               "and 1 anion(s)'), "
               "('[Na+].[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)[C@]([H])(NC(=O)c1cnc2cccnc2c1O)c1ccccc1)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Ca++].CCCCCCCCCCCCCCCCCC([O-])=O.CCCCCCCCCCCCCCCCCC([O-])=O', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Na+].[Na+].N[C@@H](Cc1ccc([O-])cc1)C([O-])=O', 'Salt "
               "containing 2 cation(s) and 1 anion(s)'), "
               "('[Cl-].CCC[NH2+]C(C)C(=O)Nc1ccccc1C', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].Cl[Pt--](Cl)(Cl)(Cl)(Cl)Cl', 'Salt containing 2 "
               "cation(s) and 1 anion(s)'), ('[Li+].F[P-](F)(F)(F)(F)F', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Zn++].[O-][N+]([O-])=O.[O-][N+]([O-])=O', 'Salt containing "
               "1 cation(s) and 2 anion(s)'), ('Cl[O-].[Ca+2].Cl[O-]', 'Salt "
               "containing 1 cation(s) and 2 anion(s)'), "
               "('[Cl-].CC(C)([NH3+])Cc1ccccc1', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), "
               "('[H+].[H+].[H+].[H+].[O-]C(=O)\\\\C=C/C([O-])=O.[O-]C(=O)\\\\C=C/C([O-])=O.CN1CCN(CC1)c1nc2cc(ccc2n2cccc12)C(F)(F)F', "
               "'Salt containing 4 cation(s) and 2 anion(s) and 1 neutral "
               "species'), "
               "('[Na+].[Na+].[H][C@@]12C[C@H](C)[C@](O)(C(=O)COP([O-])([O-])=O)[C@@]1(C)C[C@H](O)[C@@]1(F)[C@@]2([H])CCC2=CC(=O)C=C[C@]12C', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Cl-].CN(C)c1ccc2c(-c3ccccc3C(O)=O)c3ccc(cc3[o+]c2c1)N(C)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('O.O.O.O.O.O.O.[Cr+3].[O-][N+]([O-])=O.[O-][N+]([O-])=O.[O-][N+]([O-])=O', "
               "'Salt containing 1 cation(s) and 3 anion(s) and 7 neutral "
               "species'), ('O.[Na+].CC(=O)[N-]S(=O)(=O)c1ccc(N)cc1', 'Salt "
               "containing 1 cation(s) and 1 anion(s) and 1 neutral species'), "
               "('[H+].[H+].[Cl-].[Cl-].COc1ccc2nc3cc(Cl)ccc3c(NC(C)CCCN(CCCl)CCCl)c2c1', "
               "'Salt containing 2 cation(s) and 2 anion(s) and 1 neutral "
               "species'), "
               "('[Na+].[H][C@]1(O[C@H](C)[C@@H](O[C@]2([H])O[C@H](CO)[C@@H](O)[C@H](O[C@]3([H])O[C@H](CO)[C@@H](O)[C@H](OC)[C@H]3O)[C@H]2O)[C@H](O)[C@H]1O)O[C@@H]1[C@@H](O)[C@@H](CO[C@@]1([H])O[C@H]1CC[C@]2(C)C3=C[C@H](O)C45C(=O)O[C@@](C)(C(=O)CCC(C)(C)O)[C@@]4(O)CC[C@@]5(C)[C@]3([H])CC[C@@]2([H])C1(C)C)OS([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('O(CC(CCCCCC)CCCC)S([O-])(=O)=O.[Na+]', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), "
               "('[Cl-].Oc1cc(O)c2ccc([o+]c2c1)-c1ccc(O)c(O)c1', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[H+].[H+].[H+].[H+].[O-]C(=O)\\\\C=C/C([O-])=O.[O-]C(=O)\\\\C=C/C([O-])=O.CCSc1ccc2Sc3ccccc3N(CCCN3CCN(C)CC3)c2c1', "
               "'Salt containing 4 cation(s) and 2 anion(s) and 1 neutral "
               "species'), ('[K+].[I-]', 'Salt containing 1 cation(s) and 1 "
               "anion(s)'), "
               "('[Na+].C1=CC=CC2=C1C(=C(C(O2)=O)[C@H](CC(C)=O)C3=CC=CC=C3)[O-]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Br-].CC[n+]1c(-c2ccccc2)c2cc(N)ccc2c2ccc(N)cc12', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), ('[Na+].[O-]C(=O)CF', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('Clc1ccc(Cn2c(C[NH+]3CCCC3)nc3ccccc23)cc1.[H][C@]12SC(C)(C)[C@@H](N1C(=O)[C@H]2NC(=O)Cc1ccccc1)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Li+].[Li+].Nc1ccc2c(-c3cc(ccc3C([O-])=O)C(O)=O)c3ccc(N)c(c3[o+]c2c1S([O-])(=O)=O)S([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Na+].CC(C)(O)c1ccccc1CC[C@@H](SCC1(CC1)CC([O-])=O)c1cccc(\\\\C=C\\\\c2ccc3ccc(Cl)cc3n2)c1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[Na+].CC(\\\\C=C\\\\C=C1\\\\N(CCS([O-])(=O)=O)c2ccc(cc2C1(C)C)S([O-])(=O)=O)=C/C=C/C1=[N+](CCS([O-])(=O)=O)c2ccc(cc2C1(C)C)S([O-])(=O)=O', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('[Cl-].CCCCC(O)(C(=O)NCCCCCC[NH3+])C(=O)N(Nc1ccccc1)c1ccccc1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[O-][Se]([O-])=O', 'Salt containing 2 cation(s) "
               "and 1 anion(s)'), ('C(C(=O)[O-])(=O)[O-].[Na+].[Na+]', 'Salt "
               "containing 2 cation(s) and 1 anion(s)'), "
               "('[H+].[H+].[O-]C(=O)\\\\C=C\\\\C([O-])=O.OCCOCCN1CCN(CC1)C1=Nc2ccccc2Sc2ccccc12.OCCOCCN1CCN(CC1)C1=Nc2ccccc2Sc2ccccc12', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 2 neutral "
               "species'), ('O.O.O=[V++].[O-]S([O-])(=O)=O', 'Salt containing "
               "1 cation(s) and 1 anion(s) and 2 neutral species'), "
               "('[Na+].OC[C@H](N=[N+]=[N-])[C@H]1O[C@@](C[C@@H](O)[C@H]1O)(OCC=C)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Br-].CCCCCCCCCC[N+](C)(C)C', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), "
               "('[H+].[Cl-].CN\\\\C(NCCSCc1ccc(CN(C)C)o1)=C/[N+]([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s) and 1 neutral "
               "species'), ('[K+].[O-]Br(=O)=O', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), ('[F-].CCCC[N+](CCCC)(CCCC)CCCC', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Mg++].CC([O-])=O.CC([O-])=O', 'Salt containing 1 cation(s) "
               "and 2 anion(s)'), "
               "('[Cl-].ClCc1ccc(cc1)C1=c2cc3CCC[N+]4=c3c(CCC4)c2Oc2c3CCCN4CCCc(cc12)c34', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].O[B-]1(O)OO[B-](O)(O)OO1', 'Salt containing 2 "
               "cation(s) and 1 anion(s)'), "
               "('O.O.O.O.O.O.O.[Na+].[Na+].Nc1nc(=O)c2c(CCc3ccc(cc3)C(=O)N[C@@H](CCC([O-])=O)C([O-])=O)c[nH]c2[nH]1', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 7 neutral "
               "species'), "
               "('[Na+].[Na+].[Na+].[Na+].Cc1cc(ccc1\\\\N=N\\\\c1c(O)c2c(N)cc(cc2cc1S([O-])(=O)=O)S([O-])(=O)=O)-c1ccc(\\\\N=N\\\\c2c(O)c3c(N)cc(cc3cc2S([O-])(=O)=O)S([O-])(=O)=O)c(C)c1', "
               "'Salt containing 4 cation(s) and 1 anion(s)'), "
               "('[K+].[K+].[K+].[K+].[K+].[K+].[O-]C(=O)CN(CC([O-])=O)c1ccccc1OCCOc1cc(NC(=O)c2ccc3c(c2)C(=O)OC32c3cc(Cl)c([O-])cc3Oc3cc([O-])c(Cl)cc23)ccc1N(CC([O-])=O)CC([O-])=O', "
               "'Salt containing 6 cation(s) and 1 anion(s)'), "
               "('S([O-])([O-])(=O)=O.S([O-])([O-])(=O)=O.[Al+3].O.O.O.O.O.O.O.O.O.O.O.O.[NH4+]', "
               "'Salt containing 2 cation(s) and 2 anion(s) and 12 neutral "
               "species'), "
               "('[Cu++].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[O-]S([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s) and 5 neutral "
               "species'), "
               "('[Na+].[Na+].[H][C@@]12CCC3C[C@H](OS([O-])(=O)=O)[C@H](C[C@]3(C)[C@@]1([H])CC[C@]1(C)[C@@]([H])([C@H](C)C(OC(=O)CCC)C(OC(=O)CCC)C(CC)C(C)C)[C@@H](O)[C@H](OC(C)=O)[C@@]21[H])OS([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)')]\n"
               'False positives: '
               "[('O.[Na+].COc1c(cc(cc1C(C)(C)C)-n1ccc(=O)[nH]c1=O)-c1ccc2cc([N-]S(C)(=O)=O)ccc2c1', "
               "'Salt containing 1 cation(s) and 1 anion(s) and 1 neutral "
               "species'), ('P(=O)([O-])([O-])C1OC1C.[Na+].[Na+]', 'Salt "
               "containing 2 cation(s) and 1 anion(s)'), "
               "('[H+].[H+].[H+].[H+].[H+].[H+].[Ga+3].CC(C)(C)[Si]([18F])(C1=CC=C(C=C1)C(=O)NC[C@@H](NC(=O)CC[C@H](N1CCN(CC([O-])=O)CCN(CC([O-])=O)CCN(CC([O-])=O)CC1)C([O-])=O)C(=O)N[C@H](CCCCNC(=O)CCC(=O)NCCC[C@@H](NC(=O)CC[C@H](NC(=O)N[C@@H](CCC([O-])=O)C([O-])=O)C([O-])=O)C([O-])=O)C([O-])=O)C(C)(C)C', "
               "'Salt containing 7 cation(s) and 1 anion(s)'), "
               "('[C@]1(C(N=C(N1)C2=C(C=CC=N2)C(=O)[O-])=O)(C)C(C)C.C(C)([NH3+])C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[O-]Cl(=O)(=O)=O.CN(C)c1ccc2cc3ccc(cc3[n+](CCCC(=O)ON3C(=O)CCC3=O)c2c1)N(C)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('O.O.O.O.[Na+].[Na+].OP([O-])(=O)C(Cl)(Cl)P(O)([O-])=O', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 4 neutral "
               "species'), "
               "('[O-]C(=O)C(F)(F)F.[O-]C(=O)C(F)(F)F.NC(=[NH2+])NCCCc1[nH+]ccc2c3ccc(Br)cc3[nH]c12', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Zn+2].S1C(=N[C@](C1)(C(=O)[O-])C)C([C@@H](O)[C@H]2N([C@@H]([C@@H]3N=C(C4=C([O-])C=CC=C4CCCCC)SC3)SC2)C)(C)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Gd+3].COCCNC(=O)CN(CCN(CCN(CC([O-])=O)CC(=O)NCCOC)CC([O-])=O)CC([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('CC[NH+](CC)CC.NC1=C(Cl)C(=NC(=N1)C1CC1)C([O-])=O', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('C=1(C2=C(N=CN1)N(C=N2)[C@H]3[C@H](OC(CCC)=O)[C@H]4[C@@H](COP(O4)(=O)[O-])O3)NC(=O)CCC.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('C=1(C(=CC=2C=CC=CC2N1)C([O-])=O)C=3NC([C@@](N3)(C)C(C)C)=O.[NH4+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), ('[OH-].[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Br-].CC[N+](CCOC(C(C1=CC=CC=C1)(C=2C=CC=CC2)O)=O)(CC)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[K+].[K+].C[C@]12CC[C@@](C)(C[C@H]1C1=CC(=O)[C@@H]3[C@@]4(C)CC[C@H](O[C@H]5O[C@@H]([C@@H](O)[C@H](O)[C@H]5O[C@@H]5O[C@@H]([C@@H](O)[C@H](O)[C@H]5O)C([O-])=O)C([O-])=O)C(C)(C)[C@@H]4CC[C@@]3(C)[C@]1(C)CC2)C(O)=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[I-].[I-].C=12[C@@]3([N+](C)(CCC1C=C(C(=C2OC4=CC=C(C=C4)C[C@@]5([N+](CCC6=C5C=C(OC=7C(OC)=CC=C(C3)C7)C(=C6)OC)(C)C)[H])OC)OC)C)[H]', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Ca+2].O=C(C=C([O-])C(CC(CC=CC(C(O)C(C(O)CC1OC(C2OC(C(O)C)(C)CC2)(C)CC1)C)C)C)C)C(CC(CC(CCC(=O)[O-])C)C)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('CC[NH3+].CC([O-])=O', 'Salt containing 1 cation(s) and 1 "
               "anion(s)'), ('[Ag+].Nc1ccc(cc1)S(=O)(=O)[N-]c1ncccn1', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[C@H]1(O[C@@H]([C@@H]([C@@H]([C@H]1O)O[C@@]23O[C@]([C@@H]([C@H](C2)O)NC3=O)([C@@H]([C@@H](CO)O)O)[H])O)CO)O[C@H]4[C@@H]([C@H]([C@@H](O[C@@H]4COS(=O)(=O)[O-])O[C@@H]5[C@H]([C@@H](O[C@@H]([C@@H]5O)CO)O[C@H]6[C@@H]([C@H]([C@@H](O[C@@H]6CO)OC[C@@H](CCCCCCCCCCCCCC)CCCCCCCCCCCCCC)O)O)O)NC(=O)C)O[C@@H]7O[C@H]([C@H]([C@H]([C@@H]7O)O)O)C.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH4+].[H]C([O-])=O', 'Salt containing 1 cation(s) and 1 "
               "anion(s)'), ('[H+].OC(CCC([O-])=O)C([O-])=O', 'Salt containing "
               "1 cation(s) and 1 anion(s)'), "
               "('[Se+4].CCN(CC)C([S-])=S.CCN(CC)C([S-])=S.CCN(CC)C([S-])=S.CCN(CC)C([S-])=S', "
               "'Salt containing 1 cation(s) and 4 anion(s)'), "
               "('O.O.[Cl-].[Cl-].C[N+](C)(C)CCOC(=O)CCC(=O)OCC[N+](C)(C)C', "
               "'Salt containing 1 cation(s) and 2 anion(s) and 2 neutral "
               "species'), ('O.O.O.O.O.O.O.[Co+2].S([O-])([O-])(=O)=O', 'Salt "
               "containing 1 cation(s) and 1 anion(s) and 7 neutral species'), "
               "('[Cl-].CCCCCCCCCC[N+](C)(C)CCCCCCCCCC', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), "
               "('[Al+3].N1=C2[N-]C(=NC3=NC(=NC4=NC(=NC=5[N-]C1=C6C5C(=C(C(=C6[H])[H])[H])[H])C=7C4=C(C(=C(C7[H])[H])[H])[H])C=8C3=C(C(=C(C8[H])[H])[H])[H])C=9C2=C(C(=C(C9[H])[H])[H])[H]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].O=C1N2C(C3=NC4=CC=CC=C4C=C3C2)=CC(=C1CO)[C@@](O)(C(=O)[O-])CC', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Li+].[Li+].[Li+].[O-]B([O-])[O-]', 'Salt containing 3 "
               "cation(s) and 1 anion(s)'), ('S=C1N([O-])C=CC=C1.[Na+]', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('C(CC(=O)[O-])C(C(=O)[O-])O.[Na+].[Na+]', 'Salt containing 2 "
               "cation(s) and 1 anion(s)'), ('[NH4+].NS([O-])(=O)=O', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('C1(C(NC([S-])=NC1=O)=O)(C(CCC)C)CC=C.[Na+]', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Cl-].CC(C[N+](C)(C)C)OC(C)=O', 'Salt containing 1 cation(s) "
               "and 1 anion(s)'), ('[I-].[S+](CC[C@H](N)C(O)=O)(C)C', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Ca+2].OC(C(CO)(C)C)C(=O)NCCC([O-])=O.OC(C(CO)(C)C)C(=O)NCCC([O-])=O', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Cl-].O1[C@@H](OC=2C(=[O+]C=3C(C2)=C(O)C=C(O)C3)C4=CC(OC)=C(O)C=C4)[C@H](O)[C@@H](O)[C@@H](O)C1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Cl-].COc1ccc2cc3-c4cc5OCOc5cc4CC[n+]3cc2c1OC', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('C1CC1C2=NC3=CC=CC=C3C(=C2C=C[C@H](C[C@H](CC(=O)[O-])O)O)C4=CC=C(C=C4)F.C1CC1C2=NC3=CC=CC=C3C(=C2C=C[C@H](C[C@H](CC(=O)[O-])O)O)C4=CC=C(C=C4)F.[Ca+2]', "
               "'Salt containing 1 cation(s) and 2 anion(s)'), "
               "('[Na+].[O-]c1ccccc1', 'Salt containing 1 cation(s) and 1 "
               "anion(s)'), "
               "('S(O[C@H]1[C@H](O)C(C(=C(C1)C)\\\\C=C\\\\C(\\\\C)=C\\\\C=C\\\\C(\\\\C)=C\\\\C=C\\\\C=C(\\\\C=C\\\\C=C(\\\\C=C\\\\C=2C([C@@H](O)[C@H](O)C(=O)C2C)(C)C)/C)/C)(C)C)([O-])(=O)=O.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[H][B-]([H])([H])[H].[H][N+]([H])([H])[H]', 'Salt containing "
               "1 cation(s) and 1 anion(s)'), ('[Na+].O=C(C(C)C)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('CC[NH+](CC)CC.CC[NH+](CC)CC.CN1c2cc3Oc4cc5=[N+](C)C(C)(C)C=C(CS([O-])(=O)=O)c5cc4=C(c3cc2C(CS([O-])(=O)=O)=CC1(C)C)c1c(Cl)c(SCC(=O)NCCCCCC(=O)ON2C(=O)CCC2=O)c(Cl)c(Cl)c1C([O-])=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Ca++].[I-].[I-]', 'Salt containing 1 cation(s) and 2 "
               "anion(s)'), ('[Na+].COc1cc(OC)nc(Sc2cccc(Cl)c2C([O-])=O)n1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH4+].C1(=NC=C(C=C1C(=O)[O-])COC)C=2NC([C@@](N2)(C)C(C)C)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('F[B-](F)(F)F.[O+]1=C(C(C)(C)C)C=C(C=C1C(C)(C)C)C2=CC=C(OC)C=C2', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[O-]C(=O)CS[Sb]1OC(=O)CS1', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), "
               "('[Ti-2]12(CC(C(C1)=O)C)(CC(C(C2)C)=O)(O)O.[NH4+].[NH4+]', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('S(OCC(\\\\C=C\\\\[C@H]([C@@]1([C@@]2([C@]([C@@]3(O)[C@@]([C@@]4([C@@]([C@H](O)C3)(C[C@@H](O[C@@H]5OC[C@@H](O)C(O)[C@@H]5O)CC4)[H])C)(CC2)[H])([C@@H](O)C1)[H])C)[H])C)C)([O-])(=O)=O.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[H][C@@]12CC[C@]3(C)[C@@H](CCC4=CC(=O)OC4)\\\\C(CC[C@]3([H])[C@@]1(C)CCCC2(C)C)=C\\\\OS([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].C=1(S(NC=2C=CC=CC2C(=O)NCC([O-])=O)(=O)=O)C=CC(OC(C(C)(C)C)=O)=CC1.O.O.O.O', "
               "'Salt containing 1 cation(s) and 1 anion(s) and 4 neutral "
               "species'), "
               "('P(OC[C@@H](O)[C@@H](O)[C@H](O)C(=O)COP([O-])([O-])=O)(O)([O-])=O.[Na+].[Na+].[Na+]', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('S(O[C@H](CCCCCCCCC(=O)CCCCCC)[C@@](N)(CO)C(O)=O)([O-])(=O)=O.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NaH].[Na+].Cc1cc(C)c(cc1C)\\\\N=N\\\\c1c(O)c(cc2cc(ccc12)S([O-])(=O)=O)S([O-])(=O)=O', "
               "'Salt containing 1 cation(s) and 1 anion(s) and 1 neutral "
               "species'), "
               "('S([O-])(=O)(=O)C1=CC=C(\\\\N=N\\\\C=2C=3C(C=CC2O)=CC=CC3)C=C1.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('S1C2N(C(C(=O)[O-])=C(C1)C=CC=3SC=NC3C)C(=O)C2NC(=O)C(=NOC)C=4N=C(N)SC4.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Fe+3].[Fe+3].S(=O)([O-])(=O)[O-].S(=O)([O-])(=O)[O-].S(=O)([O-])(=O)[O-].O', "
               "'Salt containing 2 cation(s) and 3 anion(s) and 1 neutral "
               "species'), "
               "('C1=C(C=C2C(=C1)[N-]C(=N2)S(CC3=C(C(=C(C=N3)C)OC)C)=O)OC.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[H+].OC(C(CC([O-])=O)C([O-])=O)C([O-])=O', 'Salt containing "
               "1 cation(s) and 1 anion(s)'), "
               "('O.[Cl-].[Cl-].CCO.[H][C@@]12[C@@H](C)c3cccc(O)c3C(=O)C1=C(O)[C@]1(O)C(=O)C(C(N)=O)=C(O)[C@@H]([NH+](C)C)[C@]1([H])[C@H]2O.[H][C@@]12[C@@H](C)c3cccc(O)c3C(=O)C1=C(O)[C@]1(O)C(=O)C(C(N)=O)=C(O)[C@@H]([NH+](C)C)[C@]1([H])[C@H]2O', "
               "'Salt containing 2 cation(s) and 2 anion(s) and 2 neutral "
               "species'), ('[NH4+].[NH4+].OP([O-])([O-])=O', 'Salt containing "
               "2 cation(s) and 1 anion(s)'), "
               "('[Br-].O(C1C2([C@](C3[C@@](C4([C@@](CC3)(C[C@H](OC(=O)C)C([N+]5(CCCCC5)C)C4)[H])C)(CC2)[H])(CC1[N+]6(CCCCC6)C)[H])C)C(=O)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].CO\\\\N=C(/C(=O)N[C@H]1[C@H]2SCC([C@@H]3CCCO3)=C(N2C1=O)C([O-])=O)c1csc(N)n1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Fe+3].O=C1N[C@H](C(=O)NC(CN[C@H](C(=O)NCC(=O)NC(C(=O)NCC(OC[C@@H](C(NC(C(NCC(N[C@@H]1C(C)C)=O)=O)CCCN([O-])C(=O)C)=O)NC(=O)C/C=C\\\\CCCCCC)=O)CCCN([O-])C(=O)C)C)CCCN([O-])C(=O)C)CO', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH4+].[NH4+].[NH4+].N#C[Fe-3](C#N)(C#N)(C#N)(C#N)C#N', "
               "'Salt containing 3 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[H]O[H].[O-]S([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 10 neutral "
               "species'), ('[Cl-].O=C(CCC[NH+]1CCCC1)C2=C(OC)C=C(OC)C=C2OC', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[K+].[K+].[H]O[H].[H]O[H].[H]O[H].O=C1O[Sb-]23OC1C1O[Sb-]4(OC(C(O2)C(=O)O3)C(=O)O4)OC1=O', "
               "'Salt containing 2 cation(s) and 1 anion(s) and 3 neutral "
               "species'), "
               "('P(=O)(O)(CCC(N)C(=O)NC(C([O-])=NC(C(=O)O)C)C)C.[Na+]', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[C@]12(N(C(=C(CS1)CSC3=NN=NN3C)C(=O)[O-])C([C@]2(NC(=O)[C@](NC(N4C(C(N(CC4)CC)=O)=O)=O)([C@@H](O)C)[H])OC)=O)[H].[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[NH3+]CCO.CC(Cc1cc(C)cc(=O)n1[O-])CC(C)(C)C', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[O-]C(=O)[C@@H]([C@H]([C@H](O)[C@@H](CO)O)O)O.[K+]', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('C(CN(C(/C=C(/CCO)\\\\C)=O)[O-])C[C@@H](C(=O)OCC/C(/C)=C/C(=O)N([O-])CCC[C@@]1(NC([C@](CCCN(C(/C=C(/CCO)\\\\C)=O)[O-])(NC1=O)[H])=O)[H])N.[Fe+3]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].OC(=O)c1nc(c[nH]1)C(=O)N[C@@H](C(=O)N[C@H]1[C@H]2SCC(C[n+]3ccc(CCS([O-])(=O)=O)cc3)=C(N2C1=O)C([O-])=O)c1ccccc1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('S(O[C@@H]1[C@@]2([C@@]([C@]([C@H]1O)([C@@H](CCC[C@@H](CO)C)C)[H])(CC[C@]3([C@@]2(O)C[C@@H](O)C=4[C@@]3(CC[C@H](O[C@@H]5OC[C@@H](O)C(O)[C@@H]5OC)C4)C)[H])C)[H])([O-])(=O)=O.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].[Na+].[O-]C(=O)[C@H]1N(\\\\[N+](\\\\[O-])=N\\\\[O-])CCC1', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Cl-].CC[n+]1c(-c2ccccc2)c2cc(N\\\\N=N\\\\c3cccc(c3)C(N)=N)ccc2c2ccc(N)cc12', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[H+].O[C@@H]([C@@H](O)[C@H](O)C([O-])=O)[C@H](O)C([O-])=O', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), ('[Cl-].[Au+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Ca++].[O-]C=O.[O-]C=O', 'Salt containing 1 cation(s) and 2 "
               "anion(s)'), "
               "('[H+].[Na+].CC1CC(C)(C)Nc2c1cc1c(Oc3c(c4=NC(C)(C)CC(C)c4cc3=C1c1c(Cl)c(SCC(=O)NCCCCCC(=O)ON3C(=O)CCC3=O)c(Cl)c(Cl)c1C(O)=O)S([O-])(=O)=O)c2S([O-])(=O)=O', "
               "'Salt containing 2 cation(s) and 1 anion(s)'), "
               "('[Zn+2].S(O[H])(=O)(=O)C1=C(C2=C(C=3[N-]C2=NC4=NC(=NC=5[N-]C(N=C6N=C(N3)C7=C6C(=C(C(S(O[H])(=O)=O)=C7[H])[H])[H])=C8C5C(=C(C(S(O[H])(=O)=O)=C8[H])[H])[H])C9=C4C(=C(C(S(O[H])(=O)=O)=C9[H])[H])[H])C(=C1[H])[H])[H]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[Na+].Cc1conc1[N-]S(=O)(=O)c1ccc(N)cc1', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), "
               "('[O-]Cl(=O)(=O)=O.CN(C)c1ccc2C=c3cc4CCC[N+](CCCC(=O)NCCN5C(=O)C=CC5=O)=c4cc3C(C)(C)c2c1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('O.O.O.[Ca++].CC(C)c1c(C(=O)Nc2ccccc2)c(-c2ccccc2)c(-c2ccc(F)cc2)n1CC[C@@H](O)C[C@@H](O)CC([O-])=O.CC(C)c1c(C(=O)Nc2ccccc2)c(-c2ccccc2)c(-c2ccc(F)cc2)n1CC[C@@H](O)C[C@@H](O)CC([O-])=O', "
               "'Salt containing 1 cation(s) and 2 anion(s) and 3 neutral "
               "species'), "
               "('[Cl-].O1[C@@H](OC=2C(=[O+]C=3C(C2)=C(O)C=C(O)C3)C4=CC(O)=C(O)C(O)=C4)[C@H](O)[C@@H](O)[C@@H](O)C1', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('P(=O)([O-])([O-])[O-].[Mg+2].[NH4+].O', 'Salt containing 2 "
               "cation(s) and 1 anion(s) and 1 neutral species'), "
               "('[Cl-].O(CC[N+](CC1=CC=CC=C1)(C)C)CCOC2=C(C=C(C(CC(C)(C)C)(C)C)C=C2)C', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('OC[C@H]([C@@H](O)[C@H](O)[C@@H](O)C([O-])=O)O.[Fe+3].OC[C@H]([C@@H](O)[C@H](O)[C@@H](O)C([O-])=O)O.OC[C@H]([C@@H](O)[C@H](O)[C@@H](O)C([O-])=O)O', "
               "'Salt containing 1 cation(s) and 3 anion(s)'), "
               "('[Bi+3].OC(CC([O-])=O)(CC([O-])=O)C([O-])=O', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('[Al+3].CCCCCCCCCCCCCCCCCC([O-])=O.CCCCCCCCCCCCCCCCCC([O-])=O.CCCCCCCCCCCCCCCCCC([O-])=O', "
               "'Salt containing 1 cation(s) and 3 anion(s)'), "
               "('[V](=O)(=O)[O-].[Na+].O', 'Salt containing 1 cation(s) and 1 "
               "anion(s) and 1 neutral species'), "
               "('[Mg++].OC[C@H](O)[C@H]1OC(=O)C(OP([O-])([O-])=O)=C1O', 'Salt "
               "containing 1 cation(s) and 1 anion(s)'), "
               "('S(OP(OC[C@H]1O[C@@H](N2C3=NC=NC(N)=C3N=C2)[C@H](O)[C@@H]1O)(O)=O)([O-])(=O)=O.[Na+]', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('CC[NH+](CC)CC.CCCCCCCCCCCCCCCC(=O)OCC(COP([O-])(=O)OCCNS(=O)(=O)c1ccc(c(c1)S([O-])(=O)=O)C1=c2cc3CCC[N+]4=c3c(CCC4)c2Oc2c3CCCN4CCCc(cc12)c34)OC(=O)CCCCCCCCCCCCCCC', "
               "'Salt containing 1 cation(s) and 1 anion(s)'), "
               "('[K+].[O-]C(=O)c1nc(=O)[nH]c(=O)[nH]1', 'Salt containing 1 "
               "cation(s) and 1 anion(s)'), ('O.O.O.O.O.O.[Cl-].[Cl-].[Co++]', "
               "'Salt containing 1 cation(s) and 2 anion(s) and 6 neutral "
               "species'), ('[Br-].C1CCCC1C(C(OCC[N+](CC)(CC)CC)=O)C2CCCC2', "
               "'Salt containing 1 cation(s) and 1 anion(s)')]\n"
               'False negatives: '
               "[('CC(=O)NCC(O)=O.CC(=O)NCC(O)=O.NC(=N)c1ccc(NN=Nc2ccc(cc2)C(N)=N)cc1', "
               "'No charged species present'), "
               "('Cl.Cc1c(oc2c(cccc2c1=O)C(=O)OCCN1CCCCC1)-c1ccccc1', 'No "
               "charged species present'), "
               "('CS(O)(=O)=O.C1CCC(CC1)c1ccc2n3CCN[C@@H]4CCCc(c34)c2c1', 'No "
               "charged species present'), "
               "('CS(O)(=O)=O.N[C@H]1[C@@H]2CN(C[C@H]12)c1nc2n(cc(C(O)=O)c(=O)c2cc1F)-c1ccc(F)cc1F', "
               "'No charged species present'), "
               "('Cl.Cc1ncc(CNC(=O)N(CCCl)N=O)c(N)n1', 'No charged species "
               "present'), ('NCCNCCN.C1C(CCl)O1.Cl', 'No charged species "
               "present'), "
               "('CC(O)=O.CCNC(=O)[C@@H]1CCCN1C(=O)[C@H](CCCNC(N)=N)NC(=O)[C@H](CC(C)C)NC(=O)[C@@H](CC(C)C)NC(=O)[C@H](Cc1ccc(O)cc1)NC(=O)[C@H](CO)NC(=O)[C@H](Cc1c[nH]c2ccccc12)NC(=O)[C@H](Cc1c[nH]cn1)NC(=O)[C@@H]1CCC(=O)N1', "
               "'No charged species present'), ('[Li]C#CC', 'Single molecular "
               "entity - not a salt'), ('Cl.COc1ccc(OC)c(c1)C(O)CNC(=O)CN', "
               "'No charged species present'), "
               "('O[N+]([O-])=O.Clc1ccc([C@H](Cn2ccnc2)OCc2ccc(Sc3ccccc3)cc2)c(Cl)c1', "
               "'No charged species present'), "
               "('Cl.C1CCC(CC1)c1ccc2n3CCN[C@H]4CCCc(c34)c2c1', 'No charged "
               "species present'), "
               "('OC(=O)\\\\C=C/C(O)=O.COc1ccc(cn1)-c1ccc2ncc3n(C)c(=O)n(-c4ccc(N5CCNCC5)c(c4)C(F)(F)F)c3c2c1', "
               "'No charged species present'), "
               "('Cl.N(c1ccccc1)c1ccc(cc1)C(=C1C=CC(C=C1)=Nc1ccccc1)c1ccc(Nc2ccccc2)cc1', "
               "'No charged species present'), "
               "('Cl.COC1=CC2=C(CCCC2C(=O)N(CC2=CC=C(C=C2)N(C)C)C2=CC=C(C=C2)C(C)C)C=C1', "
               "'No charged species present'), "
               "('C1[C@@]2(N3CC[C@@]42[C@]5(N(C=6C4=CC=CC6)C(C[C@]7([C@@]5([C@@]1(C(=CCO7)C3)[H])[H])[H])=O)[H])[H].Cl', "
               "'No charged species present'), "
               "('Cl.C[C@@H](NCCCc1cccc(c1)C(F)(F)F)c1cccc2ccccc12', 'No "
               "charged species present'), "
               "('OC(=O)CC(O)(CC(O)=O)C(O)=O.C[C@@H]1CCN(C[C@@H]1N(C)c1ncnc2[nH]ccc12)C(=O)CC#N', "
               "'No charged species present'), "
               "('CCCCCCCCCCCCOS(O)(=O)=O.CC[C@H]1OC(=O)[C@H](C)[C@@H](O[C@H]2C[C@@](C)(OC)[C@@H](O)[C@H](C)O2)[C@H](C)[C@@H](O[C@@H]2O[C@H](C)C[C@@H]([C@H]2OC(=O)CC)N(C)C)[C@](C)(O)C[C@@H](C)C(=O)[C@H](C)[C@@H](O)[C@]1(C)O', "
               "'No charged species present'), "
               "('Cc1ccc(cc1)S(O)(=O)=O.N[C@@H]1CCN(C1)c1nc2n(cc(C(O)=O)c(=O)c2cc1F)-c1ccc(F)cc1F', "
               "'No charged species present'), "
               "('O[N+]([O-])=O.Clc1ccc([C@H](Cn2ccnc2)OCc2c(Cl)cccc2Cl)c(Cl)c1', "
               "'No charged species present'), "
               "('Cl.CNCC[C@H](Oc1ccc(cc1)C(F)(F)F)c1ccccc1', 'No charged "
               "species present'), "
               "('Br.CN1CCc2c(cc(O)c(O)c2Cl)C(C1)c1cccc(C)c1', 'No charged "
               "species present'), "
               "('Cl.Cl.C\\\\C(\\\\C=N\\\\NC(N)=N)=N/NC(N)=N', 'No charged "
               "species present'), "
               "('Cl.CN1CCCC(CC1)n1nc(Cc2ccc(Cl)cc2)c2ccccc2c1=O', 'No charged "
               "species present'), ('Cl.COC1=C(OC)C=C2CNCCC2=C1', 'No charged "
               "species present'), "
               "('Cl.CN(C)C(=O)Oc1cc(OC(=O)N(C)C)cc(c1)C(O)CNC(C)(C)C', 'No "
               "charged species present'), "
               "('Cl.CC[C@@]1(O)C[C@H](O[C@H]2C[C@@H]([C@H](O)[C@H](C)O2)N(C)C)c2c(O)c3C(=O)c4c(O)cccc4C(=O)c3cc2[C@H]1C(=O)OC', "
               "'No charged species present'), "
               "('Cl.CN(C)C(=O)Oc1cc(OC(=O)N(C)C)cc(c1)[C@H](O)CNC(C)(C)C', "
               "'No charged species present'), "
               "('Br.CC(Cc1ccc(O)cc1)NCC(O)c1cc(O)cc(O)c1', 'No charged "
               "species present'), "
               "('Cl.O=C1N(C[C@H]2CCCc3cccc1c23)[C@@H]1CN2CCC1CC2', 'No "
               "charged species present'), "
               "('Cl.CN1CCN(CC1)C(c1ccccc1)c1ccccc1', 'No charged species "
               "present'), "
               "('OS(=O)(=O)c1ccccc1.CCOC(=O)C1=C(COCCN)NC(C)=C(C1c1ccccc1Cl)C(=O)OC', "
               "'No charged species present'), "
               "('Cl.COc1ccc(CCN(C)CCC[C@](C#N)(C(C)C)c2ccc(OC)c(OC)c2)cc1OC', "
               "'No charged species present'), "
               "('Cl.Clc1ccc2N=C3NC(=O)CN3Cc2c1Cl', 'No charged species "
               "present'), ('C1=2C(C=CC=C1)=NC(=CC2)/C=C/C=3C(=CC=CC3)O.Cl', "
               "'No charged species present'), "
               "('Cl.CCN1CCC(C1C)=C1C2=CC=CC=C2CCC2=CC=CC=C12', 'No charged "
               "species present'), "
               "('Br.COc1ccccc1N1CCN(CCCCN2C(=O)c3ccccc3C2=O)CC1', 'No charged "
               "species present'), ('Br.CC1NCCC2=CC(O)=C(O)C=C12', 'No charged "
               "species present'), "
               "('Cl.OC(=O)c1cn(C2CC2)c2cc(N3CCNCC3)c(F)cc2c1=O', 'No charged "
               "species present'), ('Cl.CN1CCC(CC1)=C1c2ccccc2C=Cc2ccccc12', "
               "'No charged species present'), "
               "('Cl.C1CCC(CC1)c1ccc2n3CCN[C@@H]4CCCc(c34)c2c1', 'No charged "
               "species present'), ('Cl.NCCCC[C@H](N)C(O)=O', 'No charged "
               "species present'), "
               "('NC(CO)(CO)CO.NC(CO)(CO)CO.OC(=O)\\\\C=C/C(O)=O', 'No charged "
               "species present'), "
               "('Cl.CCCCCC[C@@H]([C@@H](C)O)n1cnc2c(N)ncnc12', 'No charged "
               "species present'), "
               "('C=1C(=CC=C(C1)N)C(=C2C=CC(=N)C=C2)C=3C=CC(=CC3)N.Cl', 'No "
               "charged species present'), "
               "('OS(O)(=O)=O.NC[C@H]1O[C@H](O[C@@H]2[C@@H](N)C[C@@H](N)[C@H](O[C@H]3O[C@H](CO)[C@@H](O)[C@H](N)[C@H]3O)[C@H]2O)[C@H](N)[C@@H](O)[C@@H]1O', "
               "'No charged species present'), ('Cl.NOCC(O)=O.NOCC(O)=O', 'No "
               "charged species present'), "
               "('CC(O)=O.CC(C)C[C@H](NC(=O)[C@@H](CC1=CC2=C(C=CC=C2)C=C1)NC(=O)[C@H](CC1=CC=C(O)C=C1)NC(=O)[C@H](CO)NC(=O)[C@H](CC1=CNC2=C1C=CC=C2)NC(=O)[C@H](CC1=CNC=N1)NC(=O)[C@@H]1CCC(=O)N1)C(=O)N[C@@H](CCCNC(N)=N)C(=O)N1CCC[C@H]1C(=O)NCC(N)=O', "
               "'No charged species present'), "
               "('O[N+]([O-])=O.Clc1ccc([C@@H](Cn2ccnc2)OCc2csc3c(Cl)cccc23)c(Cl)c1', "
               "'No charged species present'), "
               "('C=1C(=CC(=C(C1)N)C)C(=C2C=CC(=N)C=C2)C=3C=C(C(=CC3)N)C.Cl', "
               "'No charged species present'), "
               "('C(=C\\\\SC)(\\\\C1=C(OCCCCC)C=CC=C1)/N2C=CN=C2.Cl', 'No "
               "charged species present'), "
               "('CS(O)(=O)=O.CCCCCCOC(=O)\\\\N=C(\\\\N)c1ccc(NCc2nc3cc(ccc3n2C)C(=O)N(CCC(=O)OCC)c2ccccn2)cc1', "
               "'No charged species present'), ('O.O.O.O.Cl[Mn]Cl', 'No "
               "charged species present')]",
    'attempt': 1,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 143,
    'num_false_positives': 100,
    'num_true_negatives': 14008,
    'num_false_negatives': 22,
    'num_negatives': None,
    'precision': 0.588477366255144,
    'recall': 0.8666666666666667,
    'f1': 0.7009803921568626,
    'accuracy': 0.9914523926294402}