"""
Classifies: CHEBI:15734 primary alcohol
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_primary_alcohol(smiles: str):
    """
    Determines if a molecule contains a primary alcohol group (-CH2OH).
    A primary alcohol has a hydroxy group (-OH) attached to a saturated carbon atom 
    which has either:
    - Three hydrogen atoms attached to it (CH3OH)
    - One other carbon atom and two hydrogen atoms attached to it (-CH2OH)

    Args:
        smiles (str): SMILES string of the molecule

    Returns:
        bool: True if molecule contains a primary alcohol group, False otherwise
        str: Reason for classification
    """
    # Handle special cases first
    if smiles == "*CO" or smiles == "CO":
        return True, "Contains primary alcohol group (-CH2OH)"
        
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return False, "Invalid SMILES string"

    # SMARTS pattern for primary alcohols
    # [CH2X4]-[OX2H] matches -CH2OH
    # [CH3X4]-[OX2H] matches CH3OH (methanol)
    primary_alcohol_pattern = Chem.MolFromSmarts('[CH2X4,CH3X4]-[OX2H]')
    
    if mol.HasSubstructMatch(primary_alcohol_pattern):
        matches = mol.GetSubstructMatches(primary_alcohol_pattern)
        
        for match in matches:
            c_atom = mol.GetAtomWithIdx(match[0])
            o_atom = mol.GetAtomWithIdx(match[1])
            
            # Verify oxygen is not part of a larger functional group
            if len(o_atom.GetNeighbors()) != 1:
                continue
                
            # Verify carbon is sp3 hybridized
            if c_atom.GetHybridization() != Chem.HybridizationType.SP3:
                continue
                
            # Count number of hydrogens and carbon neighbors
            h_count = c_atom.GetTotalNumHs()
            c_neighbors = len([n for n in c_atom.GetNeighbors() if n.GetAtomicNum() == 6])
            
            # Check conditions for primary alcohol:
            # Either methanol (3 H's) or -CH2OH (2 H's and 1 C neighbor)
            if (h_count == 3) or (h_count == 2 and c_neighbors == 1):
                return True, "Contains primary alcohol group (-CH2OH)"
    
    return False, "No primary alcohol group found"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:15734',
                          'name': 'primary alcohol',
                          'definition': 'A primary alcohol is a compound in '
                                        'which a hydroxy group, -OH, is '
                                        'attached to a saturated carbon atom '
                                        'which has either three hydrogen atoms '
                                        'attached to it or only one other '
                                        'carbon atom and two hydrogen atoms '
                                        'attached to it.',
                          'parents': ['CHEBI:30879']},
    'config': {   'llm_model_name': 'lbl/claude-sonnet',
                  'f1_threshold': 0.8,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0.662251655629139 is too low.\n'
               "True positives: [('OCC(CC(C)C)=O', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('CN1C(=O)C(F)=C(NC2=C(F)C=C(I)C=C2)C2=C1N=CN(C[C@@H](O)CO)C2=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[H][C@]12CC=C(CO)CC[C@]1(C)CC[C@H]2C(C)(C)O', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('OCC(N)\\\\C=C\\\\CCCCCCCCC/C=C\\\\CCC', 'Contains primary "
               "alcohol group (-CH2OH)'), ('OCCCCCCCCCCCCCC\\\\C=C\\\\C(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1N2[C@@]3([C@@]4([C@]5([N+](CC4)(C\\\\C(\\\\[C@](C5)(C3=CC1)[H])=C\\\\CO)[O-])[H])C=6C2=CC=CC6)[H]', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(C)CCCCCCCCCCCCCCCCCO', 'Contains primary alcohol group "
               "(-CH2OH)'), ('*C(NCCO)=O', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('[H][C@@]12CC[C@](O)(C(=O)CO)[C@@]1(C)CC(=O)[C@@]1([H])[C@@]2([H])C[C@@H](O)C2=CC(=O)C=C[C@]12C', "
               "'Contains primary alcohol group (-CH2OH)'), ('OCCN1CCCCC1', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCC(CO)CCCC', 'Contains primary alcohol group "
               "(-CH2OH)'), ('OCCCCCCCCCCCC[C@@H](O)CC(O)=O', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('C(CCCO)C/C=C\\\\CC1C(C/C=C\\\\C/C=C\\\\CCCC(O)=O)O1', "
               "'Contains primary alcohol group (-CH2OH)'), ('CCCC(=O)NCCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(O)CCCCCCCCCCO', 'Contains primary alcohol group "
               "(-CH2OH)'), ('CCCCCCCCc1ccc(CCC(N)(CO)COP(O)(O)=O)cc1', "
               "'Contains primary alcohol group (-CH2OH)'), ('C(/C=C/CCCCC)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC(O)=O', 'Contains primary "
               "alcohol group (-CH2OH)'), ('CCCCCCCCCO', 'Contains primary "
               "alcohol group (-CH2OH)'), ('Nc1nc(=O)[nH]cc1CO', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('[H][C@@]12CCC3=C(CC[C@]4(C)[C@]([H])(CC[C@@]34[H])[C@H](C)CCC=C(C)C)[C@@]1(C)CC[C@H](O)C2CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C(NCCO)CCCCCC=CCC=CCC=CCC=CCC', 'Contains primary alcohol "
               "group (-CH2OH)'), ('CO', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('C=1(C(=CC(=CC1O)CCCCCO)O)[C@H]2[C@](CCC(=C2)C)([H])C(C)=C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCC(=C(CC)[H])[H]', 'Contains primary alcohol group "
               "(-CH2OH)'), ('OCC1=C2[C@@H]([C@H](C)CC[C@H]1CO)CC(C2)(C)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1[C@]([C@@]2([C@@](CC1)(C=3C(CC2)=C(OC)C(=CC3O)C(C)C)C)[H])(CO)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[C@]123[C@@]([C@]4(CC[C@]5([C@]([C@@]4(C[C@H]1O2)[H])(CC[C@@]5([C@H](C)[C@@]6(OC(=O)C(=C(C6)C)CO)[H])O)[H])C)[H])(C(=O)C=C[C@@H]3O)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('COC1=CC2=NC3=C(OC2=C(OC)C1=O)C=CC(CO)=C3', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('OC[C@@H]1C(C2=CC(C)(C)C[C@H]2[C@H](CC1)C)=C', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('[H][C@]12C[C@]([H])(C[C@]11[C@H](C)[C@@H](O)[C@H](O)C[C@]21[H])C(=C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[C@]1([C@@](C(OC1)=O)([C@H](C*)O)[H])(CO)[H]', 'Contains "
               "primary alcohol group (-CH2OH)'), ('CCCCCCCC(O)CCCCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC(O)=O', 'Contains primary "
               "alcohol group (-CH2OH)'), ('OCCNC([*])=O', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('OCC1=C2[C@@H]([C@H](C)CC[C@H]1CO)C[C@](C2)(CO)C', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('CC(CO)CCC[C@@H](C)CCC[C@@H](C)CCC[C@]1(C)CCc2c(C)c(O)c(C)c(C)c2O1', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCCCCC(O)CO', 'Contains primary alcohol group "
               "(-CH2OH)'), ('OCCCCCCCCCCC(O)=O', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('OCC1=CC(=O)C=2[C@]1([C@]3(OC(=O)[C@@H](C)[C@@]3([C@@H](O)CC2C)[H])[H])[H]', "
               "'Contains primary alcohol group (-CH2OH)'), ('C(/C=C/C=C/C)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[H][C@]1(CC[C@]2(CO)CC[C@]3(C)[C@]([H])(CC[C@]4([H])[C@@]5(C)CCC(=O)C(C)(C)[C@]5([H])CC[C@@]34C)[C@@]12[H])[C@@H](C)C=O', "
               "'Contains primary alcohol group (-CH2OH)'), ('OCC(CO)(CO)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCC1(C2C(C(C(=CC2)CO)CO)(CCC1)C)C', 'Contains primary "
               "alcohol group (-CH2OH)'), ('OCC(CC)(CO)CO', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('CC(=O)NC1=CC(=O)[C@]2(CO)O[C@@H]2[C@H]1O', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('COc1cc(cc(OC)c1O)[C@@H]1OC[C@@H]2[C@H]1CO[C@H]2c1cc(OC)c(O[C@@H](CO)[C@H](O)c2cccc(O)c2OC)c(OC)c1', "
               "'Contains primary alcohol group (-CH2OH)'), ('OCC=CCl', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@@H](C[C@H]([C@@H](O1)C)O)O)[C@@H](CCCCCCCCCCCCCCCCCCCC/C=C(/C(=O)NCCO)\\\\C)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCC\\\\C=C/CCCCCCCCCCO', 'Contains primary alcohol "
               "group (-CH2OH)'), ('CCC(CC)(CO)CO', 'Contains primary alcohol "
               "group (-CH2OH)'), ('C1(CO)=CC[C@@](C=2CC(CCC2)(C)C)(CC1)[H]', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCCCCCCCCCCCC(=O)NCCO', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('C/1(\\\\C[C@]2(O[C@]2(C(=O)OCC3=CCN4CC[C@@H](OC1=O)[C@@]34[H])C)CO)=C/C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC[C@H]1CN2CCc3c(O)c(OC)c(OC)cc3[C@@H]2C[C@@H]1CCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[H][C@]1(O[C@](O)(CO)C[C@H](O)[C@H]1NC(C)=O)[C@H](O)[C@H](O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCC1=C2[C@H]([C@@H](C)CC[C@@H]1CO)C[C@](C2)(CO)C', 'Contains "
               "primary alcohol group (-CH2OH)'), ('C(CCCCCC)O', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('C[C@@H](CCO)\\\\C=C(C)\\\\C=C\\\\C1=CC2=CC3=C(C(C)=O)C(=O)O[C@@]3(C)[C@H](O)C2=CO1', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CC(O)=C2C(=O)c3c(O[C@]2(CO)[C@@H]1OC(C)=O)ccc(c3O)-c1ccc2O[C@]3(CO)[C@H](OC(C)=O)[C@H](C)CC(O)=C3C(=O)c2c1O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('N(C(*)=O)(CCO)[H]', 'Contains primary alcohol group "
               "(-CH2OH)'), ('[H][C@]1(CO)CN(C(=O)O1)C1=CC(C)=CC=C1', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCCCCCCCCCCCCCCCCCC\\\\C=C\\\\C(O)=O', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('CC1(C)[C@]2([C@](CC([C@@]1(C2)[H])=O)(CO)[H])[H]', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('[C@]1([C@@](C(OC1)=O)(C(C*)=O)[H])(CO)[H]', 'Contains "
               "primary alcohol group (-CH2OH)'), ('OCCCN', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('OCCCCCCCCCCCCCCCCCC[C@@H](O)CC(O)=O', 'Contains primary "
               "alcohol group (-CH2OH)'), ('CCCCCC\\\\C=C/CCCCCCCC(=O)NCCO', "
               "'Contains primary alcohol group (-CH2OH)'), ('OCC(O)CC(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC/C=C\\\\CCCCCO', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('C1[C@@]2([C@@]([C@@]3(C([C@@H]1O)=CC(CC3)=O)C)(C(C[C@@]4([C@@](CC[C@@]24[H])(O)C(CO)=O)C)=O)[H])[H]', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[H][C@@]12C[C@@]1(C[C@H](O)[C@H]2CO)n1cc(C)c(=O)[nH]c1=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(C)CCC[C@@H](C)CCC[C@@H](C)CCC\\\\C(C)=C\\\\CO', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('OC[C@@H]1[C@@]2(N(C=3C([C@@]42[C@H]5N(CC4)C\\\\C(\\\\[C@@]1(C5)[H])=C\\\\C)=CC=CC3)C(C)=O)[H]', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[H][C@@]12C[C@]3([H])\\\\C(=C/C[C@@]4([H])[C@H](CC[C@]4(C)C[C@]3([H])[C@@]1(O)CO)C(C)C)C(=O)O2', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O1[C@H]([C@H]1C/C=C\\\\C/C=C\\\\CCCCC)C/C=C\\\\CCCC(=O)NCCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCc1cc(ccc1O)[C@H](O)CNCCCCCCOCCCCc1ccccc1', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('C1CC(CC2C1C3C2C4C3CC4)CCCCCCO', 'Contains primary alcohol "
               "group (-CH2OH)'), ('OCCCCCCC[C@@H](O)CC(O)=O', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('Cc1coc2C(=O)c3c(ccc4c3CCCC4(C)CO)C(=O)c12', 'Contains "
               "primary alcohol group (-CH2OH)'), ('OCC(C)C=C', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('COc1cccc2C(=O)c3c(O)c4C[C@](O)(C[C@H](O[C@H]5C[C@H](N)[C@H](I)[C@H](C)O5)c4c(O)c3C(=O)c12)C(=O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCCCCCCCCCC(O)CC(O)=O', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('[H][C@@]12CCC(=O)[C@H](C)[C@@]1(C)CC[C@@]1([H])[C@@]2(C)CC[C@@]2(C)[C@]3([H])CC(C)(C)CC[C@]3(CO)[C@H](O)C[C@]12C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC[C@H]([C@H](CC)C)[NH3+]', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('C12=C(C=3C(=CC(=CC3C=C1C[C@H](OC2=O)C[C@@H](C[C@@H](CCCCCO)O)O)OC)O)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1C=C2[C@@]3([C@]([C@]4([C@@](CC3)(CC[C@](C4)(C(=O)O)C)C)[H])(CC[C@]2(C=5C1=C(C(OC)=C(O)C5)CO)C)C)C', "
               "'Contains primary alcohol group (-CH2OH)'), ('N(C(*)=O)CCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCC(C)CCC(CO)C(C)C', 'Contains primary alcohol group "
               "(-CH2OH)'), "
               "('CN[C@@H]1Cc2c[nH]c3cccc([C@H]1\\\\C=C(/C)CO)c23', 'Contains "
               "primary alcohol group (-CH2OH)'), "
               "('C1OC(C2COC(C12)C3=CC(=C(C(=C3)OC)O)OC)C4=CC=C(C(=C4)OC)OC(C(C=5C=C(C(=CC5)O)OC)O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCC[C@H](O)\\\\C=C\\\\[C@H]1[C@H]2C[C@H](OO2)[C@@H]1C\\\\C=C/CCCC(=O)NCCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCCCCCCCCO', 'Contains primary alcohol group "
               "(-CH2OH)'), ('C(=C/CO)\\\\CCCCCC', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('CCCCC\\\\C=C/C\\\\C=C/C\\\\C=C/C\\\\C=C/CCCCCC(=O)NCCO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCC(=O)COS(O)(=O)=O', 'Contains primary alcohol group "
               "(-CH2OH)'), ('CC#CC#CC#C[C@@H]1O[C@H]1CO', 'Contains primary "
               "alcohol group (-CH2OH)'), "
               "('CC(\\\\C=C\\\\C=C(C)\\\\C=C\\\\C1=C(C)CCCC1(C)C)=C\\\\CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OCC(CO)(CCl)CCl', 'Contains primary alcohol group "
               "(-CH2OH)'), ('C=1C=C(C(=O)[H])NC1CO', 'Contains primary "
               "alcohol group (-CH2OH)')]\n"
               'False positives: '
               "[('COC1=CC=C(C=C1)S(=O)(=O)N[C@H]2CC[C@@H](O[C@H]2CO)CC(=O)NCC3=CC=CC=C3F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCC#CC1=CC=C(C=C1)[C@H]2[C@@H]3CN(CC(=O)N3[C@H]2CO)C(=O)NC4=CC(=CC=C4)F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC[C@H]1O[C@H]([C@H](O)[C@@H]1O)n1c(cc(=O)[nH]c1=O)C(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)C3=CC=C(C=C3)C)O[C@H]1CN(C)C(=O)CC4CC4)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[Cl-].O1[C@@H]([C@@H](O)[C@H](O)[C@@H](O)[C@@H]1OC=2C=C3C(O[C@@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4O)CO)=CC(O)=CC3=[O+]C2C5=CC(O)=C(O)C=C5)CO[C@@H]6O[C@H]([C@H](O)[C@@H](O)[C@H]6O)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)CC2=C(C=CC(=C2)NC(=O)CN3C=NN=N3)O[C@H]1CN(C)CC4=CC=C(C=C4)C(F)(F)F)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H](C[C@H]([C@@H](C(C)(C)O[C@H]1[C@@H]([C@H]([C@@H]([C@@H](CO)O1)O)O)O)O)O)[C@]2(CC[C@@]3(C)[C@@]4(CC=C5[C@](CC[C@@H](C5(C)C)O[C@H]6[C@@H](C([C@@H]([C@@H](CO[C@H]7[C@@H]([C@H]([C@@H]([C@@H](CO)O7)O)O)O)O6)O)O)O)([H])[C@]4(C)CC[C@]23C)[H])[H]', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1O[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@@H](O[C@@H]3O[C@@H]([C@H](O)[C@H](O[C@]4(O[C@H]([C@H](NC(=O)C)[C@@H](O)C4)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]3O)CO)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O)[C@@H]1O[C@@H]5O[C@@H]([C@@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O)[C@H]6O)CO[C@]7(O[C@H]([C@H](NC(=O)C)[C@@H](O)C7)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]5NC(=O)C)CO)CO)[C@H]8[C@H](O)[C@H](O[C@@H](O[C@H]9[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]9CO)O[C@@H]([C@H](O)[C@@H](NC(=O)C)CO)[C@H](O)CO[C@@H]%10O[C@H]([C@@H](O)[C@@H](O)[C@@H]%10O)C)[C@H]8O)CO[C@H]%11O[C@@H]([C@@H](O)[C@H](O)[C@@H]%11O[C@@H]%12O[C@@H]([C@@H](O[C@@H]%13O[C@@H]([C@H](O)[C@H](O[C@]%14(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%14)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]%13O)CO)[C@H](O)[C@H]%12NC(=O)C)CO)CO[C@@H]%15O[C@@H]([C@@H](O[C@@H]%16O[C@@H]([C@H](O)[C@H](O[C@]%17(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%17)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]%16O)CO)[C@H](O)[C@H]%15NC(=O)C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC=CC1=CC=C2[C@H]3[C@@H](CN2C1=O)[C@@H]([C@H](N3CCC(F)(F)F)C(=O)N(C)C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1O[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@@H](O[C@@H]3O[C@@H]([C@H](O)[C@H](O[C@]4(O[C@H]([C@H](NC(=O)C)[C@@H](O)C4)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]3O)CO)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O)[C@@H]1O[C@@H]5O[C@@H]([C@@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O)[C@H]6O)CO[C@]7(O[C@H]([C@H](NC(=O)CO)[C@@H](O)C7)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]5NC(=O)C)CO)CO)[C@@H]8[C@H](O)[C@@H](O[C@@H]([C@H]8O)CO[C@H]9O[C@@H]([C@@H](O)[C@H](O)[C@@H]9O[C@@H]%10O[C@@H]([C@@H](O[C@@H]%11O[C@@H]([C@H](O)[C@H](O)[C@H]%11O)CO[C@]%12(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%12)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%10NC(=O)C)CO)CO)O[C@H]%13[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]%13CO)O[C@H]%14[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]%14CO[C@@H]%15O[C@H]([C@@H](O)[C@@H](O)[C@@H]%15O)C)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1N2[C@H](C(=O)N[C@@H]3C(=O)N([C@H](C(=O)N(C(C(=O)O[C@@H]([C@H](N(C)C)C(N[C@H](C(N[C@H](C(N4[C@H](C(N([C@H](C(NC1=C(C5OC5)CO)=O)C(C)C)C)=O)CCC4)=O)C6=CC=CC=C6)=O)C(C)C)=O)C)=CC7=CC=C(OC=8C=C(C3O)C=CC8OC)C=C7)C)C(C)C)C)CCC2', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CN1[C@H]([C@@H]2CCN([C@@H]2C3=C1C=CC(=C3)Br)S(=O)(=O)C4=CC=CC(=C4)F)CO', "
               "'Contains primary alcohol group (-CH2OH)'), ('OCC=C=CC#CC#C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C=CC(=C2)NC(=O)C3CC3)O[C@H]1CN(C)S(=O)(=O)C4=CC=C(C=C4)F)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCC(=O)N[C@@H](CCO)C(O)=O', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('CCCCCCCCCCCCCCC\\\\C=C\\\\[C@@H](O)[C@H](CO)NC([*])=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C=1(N=C(C(=C(N1)N)C)C(=O)NC(C(NC(C(O)C(C(=O)NC(C(S*)=O)C(O)(C)C)C)CCO)=O)CC=2N=CNC2)C(NCC(C(=O)N)N)CC(N)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC[C@@H]1O[C@H](OP([O-])(=O)OP([O-])(=O)OC[C@H]2O[C@H]([C@H](O)[C@@H]2O)n2ccc(=O)[nH]c2=O)[C@H](O)[C@H]1O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC1=CC=CC=C1S(=O)(=O)NCC[C@H]2CC[C@@H]([C@H](O2)CO)NS(=O)(=O)C3=CC=CC=C3C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC1=CC=C(C=C1)S(=O)(=O)N(C[C@H](C)[C@@H](CN(C)C(=O)C2=CC3=C(C=C2)OCO3)OC)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@@H]1[C@H](O)[C@@H](O[C@@H]([C@H]1O)CO[C@H]2O[C@@H]([C@@H](O)[C@H](O)[C@@H]2O[C@@H]3O[C@@H]([C@@H](O[C@@H]4O[C@@H]([C@H](O)[C@H](O[C@@H]5O[C@@H]([C@H](O)[C@H](O)[C@H]5O)CO)[C@H]4O)CO)[C@H](O)[C@H]3NC(=O)C)CO)CO)O[C@H]6[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]6CO)O[C@H]7[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]7CO)O)[C@H]8O[C@@H]([C@@H](O)[C@H](O)[C@@H]8O[C@@H]9O[C@@H]([C@@H](O[C@@H]%10O[C@@H]([C@H](O)[C@H](O[C@@H]%11O[C@@H]([C@@H](O[C@@H]%12O[C@@H]([C@H](O)[C@H](O[C@@H]%13O[C@@H]([C@@H](O[C@@H]%14O[C@@H]([C@H](O)[C@H](O[C@@H]%15O[C@@H]([C@@H](O[C@@H]%16O[C@@H]([C@H](O)[C@H](O)[C@H]%16O)CO)[C@H](O)[C@H]%15NC(=O)C)CO)[C@H]%14O)CO)[C@H](O)[C@H]%13NC(=O)C)CO)[C@H]%12O)CO)[C@H](O)[C@H]%11NC(=O)C)CO)[C@H]%10O)CO)[C@H](O)[C@H]9NC(=O)C)CO)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H](CN([C@@H](C)CO)S(=O)(=O)C1=CC(=CC=C1)Cl)[C@@H](CN(C)S(=O)(=O)C2=CC=C(C=C2)Cl)OC', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@@H]1O[C@@H]([C@@H](O)[C@H](O[C@@H]2O[C@@H]([C@H](O)[C@H](O[C@]3(O[C@H]([C@H](NC(=O)C)[C@@H](O)C3)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]2O)CO)[C@H]1NC(=O)C)CO[C@]4(O[C@H]([C@H](NC(=O)C)[C@@H](O)C4)[C@H](O)[C@H](O)CO)C(O)=O)[C@@H]5[C@@H](O[C@@H]6[C@H](O)[C@@H](O[C@@H]([C@H]6O)CO[C@H]7O[C@@H]([C@@H](O)[C@H](O)[C@@H]7O[C@@H]8O[C@@H]([C@@H](O[C@@H]9O[C@@H]([C@H](O)[C@H](O[C@]%10(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%10)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]9O)CO)[C@H](O)[C@H]8NC(=O)C)CO)CO)O[C@H]%11[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]%11CO)O[C@H]%12[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]%12CO[C@@H]%13O[C@H]([C@@H](O)[C@@H](O)[C@@H]%13O)C)O)O[C@@H]([C@@H](O)[C@@H]5O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)Br)O[C@@H]1CN(C)C(=O)CCN3CCCCC3)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC=CC1=CC2=C(C=C1)S(=O)(=O)N(C[C@H]([C@@H](O2)CN(C)CC3=CC=C(C=C3)C(=O)O)C)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CC[C@H]2[C@@H]1[C@H](O[C@@H]1O[C@H](CO)[C@@H](O)[C@H](O)[C@H]1O)OC=C2C=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@@H]1[C@@H](O)[C@H](O[C@@H]2[C@@H](O)[C@@H](O[C@@H]([C@H]2O)CO)O[C@H]3[C@H](O)[C@H](OC(O)[C@@H]3O)CO[C@@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4O)CO)O[C@@H]([C@H]1O)CO)[C@@H]5O[C@@H]([C@@H](O)[C@H](O[C@@H]6O[C@@H]([C@@H](O)[C@H](O)[C@H]6O)CO)[C@H]5O)CO[C@@H]7O[C@@H]([C@@H](O)[C@H](O)[C@H]7O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)S(=O)(=O)CC4=CC=CC=C4)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CCCCO[C@H]([C@@H](CN(C(=O)C2=C(O1)C=CC(=C2)N(C)C)[C@@H](C)CO)C)CN(C)C(=O)CCC(F)(F)F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC(CCCCCCCCCCCCCC)C(=O)NC(C(O)/C=C/CC\\\\C=C(\\\\CCCCCCCCC)/C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1OCC2OC(C=3NC(CC=C(C4C(OC(C=C1CC(C)C)=O)C(C(=O)O4)(CO)C)C)=CC3)=NC2/C=C\\\\5/C(=C(/CCCO)\\\\C)/COC5CC(C)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@H](O)[C@H](O[C@@H](O[C@H]2[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]2CO)O[C@@H]([C@H](O)[C@@H](NC(=O)C)CO)[C@H](O)CO[C@@H]3O[C@H]([C@@H](O)[C@@H](O)[C@@H]3O)C)[C@H]1O)CO[C@H]4O[C@@H]([C@@H](O)[C@H](O)[C@@H]4O[C@@H]5O[C@@H]([C@@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O)[C@H]6O)CO)[C@H](O)[C@H]5NC(=O)C)CO)CO[C@@H]7O[C@@H]([C@@H](O[C@@H]8O[C@@H]([C@H](O)[C@H](O)[C@H]8O)CO)[C@@H](O)[C@H]7NC(=O)C)CO)[C@H]9O[C@@H]([C@@H](O[C@@H]%10O[C@@H]([C@@H](O[C@@H]%11O[C@@H]([C@H](O)[C@H](O)[C@H]%11O)CO)[C@H](O)[C@H]%10NC(=O)C)CO)[C@H](O)[C@@H]9O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[C@@]1(O[C@H]2[C@H]([C@H](O[C@H]([C@@H]2O)O[C@@H]3[C@H](O[C@@H](OC[C@@H]([C@@H](*)O)NC(=O)*)[C@@H]([C@H]3O)O)CO)CO)O[C@H]4[C@@H]([C@H]([C@@H](O)[C@H](O4)CO)O)NC(C)=O)(O[C@H]([C@H](NC(=O)CO)[C@H](C1)O)[C@@H]([C@H](O)CO)O)C([O-])=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@H](O)[C@@H](O)[C@H](O[C@@H]1CO)O[C@H]2[C@H](O)[C@H](OC(O)[C@@H]2O)CO)[C@H]3O[C@@H]([C@@H](O)[C@H](O[C@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4O)CO)[C@H]3O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)CC2=C(C=CC(=C2)NC(=O)CCCN(C)C)O[C@@H]1CN(C)CC3=CC=C(C=C3)OC4=CC=CC=C4)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)C2=C(C(=CC=C2)NC(=O)NC3=CC=CC4=CC=CC=C43)O[C@H]1CN(C)S(=O)(=O)C5=CC=CC=C5F)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)C3=CC=C(C=C3)F)O[C@@H]1CN(C)CC4=CC=CC=C4C(=O)O)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O1[C@@H](O[C@@H]2[C@@H](O)[C@H](O[C@@H]3[C@@H](NC(=O)C)[C@H](O[C@@H]4[C@@H](O)[C@H](O[C@H]5[C@@H](O)[C@H](OC(O)[C@@H]5NC(=O)C)CO[C@@H]6O[C@@H]([C@@H](O)[C@H](O[C@@H]7O[C@@H]([C@H](O)[C@H](O)[C@H]7O)CO)[C@H]6NC(=O)C)CO)O[C@@H]([C@@H]4O)CO)O[C@@H]([C@H]3O)CO)O[C@@H]([C@@H]2O)CO)[C@H](NC(=O)C)[C@@H](O[C@@H]8O[C@@H]([C@H](O)[C@H](O)[C@H]8O[C@@H]9O[C@H]([C@@H](O)[C@@H](O)[C@@H]9O)C)CO)[C@H](O[C@@H]%10O[C@H]([C@@H](O)[C@@H](O)[C@@H]%10O)C)[C@H]1CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CN(C[C@@H]1[C@H]([C@@H](N1C(=O)CC2=CN=CC=C2)CO)C3=CC=CC=C3)C(=O)COC', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@@H](O)[C@H](O[C@@H](O[C@@H]2[C@@H](NC(=O)C)[C@@H](O[C@@H]([C@H]2O)CO)O[C@@H]3[C@@H](O)[C@@H](O[C@@H]([C@@H]3O)CO)O[C@H]4[C@H](O)[C@@H](O)[C@@H](O[C@@H]4CO)O)[C@@H]1O)CO[C@@H]5O[C@@H]([C@@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O[C@H]7O[C@@H]([C@H](O)[C@H](O)[C@H]7NC(=O)C)CO)[C@H]6O[C@@H]8O[C@H]([C@@H](O)[C@@H](O)[C@@H]8O)C)CO)[C@H](O)[C@H]5NC(=O)C)CO)[C@@H]9O[C@@H]([C@@H](O)[C@H](O[C@@H]%10O[C@@H]([C@H](O)[C@H](O[C@H]%11O[C@@H]([C@@H](O)[C@H](O)[C@H]%11NC(=O)C)CO)[C@H]%10O[C@@H]%12O[C@H]([C@@H](O)[C@@H](O)[C@@H]%12O)C)CO)[C@H]9NC(=O)C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)OC)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('Nc1nc2n(COC(CO)CO)cnc2c(=O)[nH]1', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('O[C@@H]([C@H](N)C(=O)N[C@@H](CC(=O)N)C(=O)N[C@@H](CO)C(O)=O)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC[C@H]1O[C@H]([C@H](O)[C@@H](O)[C@@H]1O)c1c(O)cc(O)c2c1oc1cc(O)c(O)cc1c2=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)C2=C(N=CC(=C2)C3=CC=CC(=C3)C#N)O[C@@H]1CN(C)C(=O)C4=NC=CN=C4)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1C[C@H]([C@@H](O[C@H]1CCNC(=O)C2=CC=CC=C2F)CO)NC(=O)NC3=CC=C(C=C3)C(F)(F)F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@H](O)[C@@H](O)[C@H](O[C@@H]1CO)OC[C@H]2O[C@H](O[C@H]3[C@H](O)[C@@H](O)C(O[C@@H]3CO)O)[C@H](O)[C@@H](O)[C@@H]2O)[C@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4O)CO[C@H]5O[C@@H]([C@@H](O[C@H]6O[C@@H]([C@@H](O)[C@H](O)[C@H]6O)CO[C@H]7O[C@@H]([C@@H](O)[C@H](O)[C@H]7O)CO)[C@H](O)[C@H]5O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('Cl[C@@H]1C=C2C(=O)C3=C(O)C4=C(O)C=C3C([C@]2(CC=C(CC=C(C4)CO)C)OC1(C)C)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC(=O)[C@@H](NC(=O)[C@@H](NC(=O)[C@@H](N)CO)CC(O)=O)[C@H](CC)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C=12C(=C(C=C(C1C(C=C(O2)C3=CC=C(C(=C3)O)O)=O)O)[O-])[C@H]4[C@@H](C([C@@H]([C@H](O4)CO)O)=O)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C=CC(=C2)NC(=O)C3CCCCC3)O[C@H]1CN(C)CC4=CC=C(C=C4)OC)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCNC(=O)N1C[C@@H]2[C@@H]([C@@H](N2C(=O)CN3CCOCC3)CO)C4=CC=CC=C41', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('S(O[C@H]1[C@H](O[C@H]2[C@H](O)[C@H](O[C@H]2CO)CO)O[C@H]([C@@H](O)[C@@H]1O)C(O)=O)(O)(=O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(C)C[C@H](NC(=O)[C@@H](N)CC(O)=O)C(=O)N[C@@H](CC(C)C)C(=O)N[C@@H](CO)C(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@@H]([C@H]([C@@H]([C@H](O1)CO)O)O[C@H]2[C@@H]([C@H]([C@H]([C@H](O2)CO)O)O[C@@H]3[C@@H]([C@H]([C@H]([C@H](O3)CO)O)O)O)O[C@@H]4O[C@H]([C@H]([C@H]([C@@H]4O)O)O)C)NC(=O)C)[C@@H]5[C@H]([C@@H](O[C@@H]([C@@H]5O)CO)O[C@@H]6[C@H](O[C@@H](OC[C@@H]([C@@H]([C@@H](CCCCCCCCCCCCCC)O)O)NC(*)=O)[C@@H]([C@H]6O)O)CO)O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CCCCO[C@@H]([C@@H](CN(C(=O)C2=C(O1)C=CC(=C2)NC(=O)C3=CC=NC=C3)[C@H](C)CO)C)CN(C)CC4CCCCC4', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O[C@]12[C@@]([C@](CC1)([C@@](O)([C@H](O)CCC(C)C)C)[H])(CC[C@@]3([C@@]4([C@@](C[C@@H](O[C@@H]5O[C@@H]([C@@H](O)[C@H](O)[C@H]5O)CO)[C@@H](O)C4)(C(=O)C=C23)[H])C)[H])C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)CCCN2C=C(CO[C@H]1CN(C)C(=O)NC3=CC4=C(C=C3)OCCO4)N=N2)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCNC(=O)N(C)C[C@H]1[C@@H](CN(S(=O)(=O)C2=C(O1)C=C(C=C2)C3=CC=CC=C3F)[C@@H](C)CO)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1[C@H]2[C@H]([C@@H](N2C(=O)CN1C(=O)NC3=C(C=CC(=C3)F)F)CO)C4=CC=C(C=C4)C=CC5=CC=CC=C5', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C(=CC=C2)NC(=O)NC3=C(ON=C3C)C)O[C@@H]1CN(C)CC4=CC5=C(C=C4)OCO5)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCCCCCCCCCCCCCC(=O)N[C@@H](CO[C@@H]1O[C@H](CO)[C@@H](O[C@@H]2O[C@H](CO)[C@H](O[C@H]3O[C@H](CO)[C@H](O)[C@H](O)[C@H]3O)[C@H](O)[C@H]2O)[C@H](O)[C@H]1O)[C@H](O)\\\\C=C\\\\CCCCCCCCCCCCC', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C(=C/CC/C=C/CCCCCCCCC)\\\\[C@@H](O)[C@@H](NC(*)=O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCCNC(=O)C[C@H]1C[C@@H]2[C@H]([C@H](O1)CO)OC3=C2C=C(C=C3)NS(=O)(=O)C4=CC=C(C=C4)F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('S(CC[C@H](N)C(=O)N[C@H](C(=O)N[C@@H](CC1=CC=C(O)C=C1)C(O)=O)CO)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[C@@H]1([C@@H]([C@H]([C@H](O[C@@H]2O[C@@H]([C@@H]([C@@H]([C@H]2O)O[C@@H]3O[C@@H]([C@@H]([C@@H]([C@H]3O)O[C@@H]4O[C@@H]([C@H]([C@@H]([C@H]4O)O)O[C@H]5O[C@@H]([C@H]([C@@H]([C@H]5NS(=O)(O)=O)OS(O)(=O)=O)O[C@@H]6O[C@@H]([C@@H]([C@H]([C@@H]6OS(=O)(=O)O)O)O[C@H]7O[C@@H]([C@H]([C@@H]([C@H]7NC(=O)C)O)O)CO)C(O)=O)CO)C(O)=O)O)CO)O)CO)CO1)O)O)*', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)C2=CC=CC=C2C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)NC4=CC=CC=C4OC)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@@H]1[C@@H](O)[C@H](O)[C@H](O[C@H]1OC2=CC=C([C@H]3OC4=C(C(=O)C3)C=CC(O)=C4)C=C2)CO)[C@@H]5OC[C@@](O)([C@H]5O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1C[C@H]([C@@H](O[C@H]1CCN2C=C(N=N2)C3CC3)CO)NC(=O)C4=CN=CC=C4', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('COCCNC(=O)C[C@H]1C[C@H]2[C@@H]([C@@H](O1)CO)OC3=C2C=C(C=C3)NS(=O)(=O)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('OC[C@@H]1O[C@@H](O[C@@H]2[C@H](Oc3cc(O)cc(O)c3C2=O)c2ccc(O)c(O)c2)[C@H](O)[C@H]1O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O1[C@@H]([C@H](O)C(O)C(O)[C@@H]1OC[C@H](OC(=O)CCCC/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CC)COC(=O)C/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CC)CO[C@H]2O[C@@H]([C@H](O)C(O)C2O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](N(CC34CCN(CC4)CC5=NC=CS5)S(=O)(=O)C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1N[C@H](C(=O)N[C@H](C(=O)N2[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N3[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)O)CC(=O)N)CCC(=O)O)CC(C)C)CCCNC(=N)N)CCCNC(=N)N)CO)CC(C)C)C(C)C)CCC3)CC=4NC=NC4)CCC(=O)O)CCC(=O)O)CO)CC(=O)N)CCC(=O)N)CCC2)CC(NCC(N[C@H]1[C@H](O)C)=O)=O)[C@H](CC)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1N2[C@H](C(=O)N[C@H](C(=O)N[C@@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N3[C@@H](CCC3)C(O[C@@H](CC(N[C@@H](C(N[C@@H](C(N[C@H](C(N[C@@H]1C(C)C)=O)CO)=O)CC4=CC=CC=C4)=O)CC(=O)O)=O)CCCCCC(C)C)=O)CCCCN)CC5=CC=CC=C5)CC(C)C)CO)CCC2', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)C2=C(N=CC(=C2)C3=CC=C(C=C3)F)O[C@H]1CN(C)C(=O)CC4=CC=CC=N4)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)C4=CC=CC(=C4)C#N)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@H]1[C@@H](O)[C@H](O[C@@H](O[C@H]([C@@H](NC(=O)C)CO)[C@@H](O)[C@H](O)CO)[C@@H]1O)CO[C@@H]2O[C@@H]([C@@H](O)[C@H](O)[C@H]2NC(=O)C)CO)[C@@H]3O[C@@H]([C@@H](O)[C@H](O[C@@H]4O[C@@H]([C@H](O)[C@H](O)[C@H]4O)CO)[C@H]3NC(=O)C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1C[C@H]([C@H](O[C@H]1CC(=O)NCCC2=CC=NC=C2)CO)NS(=O)(=O)C3=CC=C(C=C3)F', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)C3=CC=NC=C3)O[C@@H]1CN(C)C(=O)CN(C)C)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(=O)N(O)CCC[C@H](N)C(=O)N[C@@H](CCCN(O)C(C)=O)C(=O)N[C@@H](CCCN(O)C(C)=O)C(=O)N[C@@H](CO)C(=O)N[C@H]([C@H](O)[C@H]1S[C@H]([C@H](O)[C@H]1O)n1ccc(=N)n(C)c1=O)C(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C(N[C@@H](CCCCN)C(O)=O)[C@@H](NC(=O)[C@@H](N)CO)CC(C)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)CCCN2C=C(CO[C@H]1CN(C)C(=O)OC3=CC=C(C=C3)C)N=N2)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('N(C(=O)CCCCCCCCCCCCCCC)[C@H]([C@@H](/C=C/CCCCCCCCCCCCC)O)CO[C@@H]1O[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@H](O[C@@H]3[C@@H]([C@@H](O[C@H]4[C@@H]([C@H]([C@@H](O)[C@H](O4)CO)O)NC(C)=O)[C@H]([C@@H](CO)O3)O)O)[C@@H]([C@H]2O)O)CO)[C@@H]([C@H]1O)O)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)CC2=C(C=CC(=C2)NC(=O)C3CC3)O[C@H]1CN(C)CC4=CC=C(C=C4)C5=CC=CC=C5)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1[C@H](O[C@H]([C@@H]2[C@H]1C3=C(O2)C=CC(=C3)NC(=O)NC4=CC=C(C=C4)C(F)(F)F)CO)CC(=O)NCC5=CN=CC=C5', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O([C@@H]1[C@H](O)[C@@H](O[C@@H]([C@H]1O)CO[C@H]2O[C@@H]([C@@H](O)[C@H](O)[C@@H]2O[C@@H]3O[C@@H]([C@@H](O[C@@H]4O[C@@H]([C@H](O)[C@H](O)[C@H]4O)CO[C@]5(O[C@H]([C@H](NC(=O)C)[C@@H](O)C5)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]3NC(=O)C)CO)CO)O[C@H]6[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]6CO)O[C@H]7[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]7CO)O)[C@H]8O[C@@H]([C@@H](O)[C@H](O)[C@@H]8O[C@@H]9O[C@@H]([C@@H](O[C@@H]%10O[C@@H]([C@H](O)[C@H](O)[C@H]%10NC(=O)C)CO[C@]%11(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%11)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]9NC(=O)C)CO)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CC(C)(O)C1Cc2cc(C3COc4c([C@@H]5O[C@H](CO)[C@@H](O)[C@H](O)[C@H]5O)c(O)cc(O)c4C3=O)c(O)cc2O1', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('CCNC(=O)[C@@H]1[C@H]([C@@H]2CN3C(=CC=C(C3=O)C4=CC=CC=C4)[C@H]1N2CCC(F)(F)F)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('[C@@]1(O[C@H]2[C@H]([C@H](O[C@H]([C@@H]2O)O[C@@H]3[C@H](O[C@@H](OC[C@@H]([C@@H]([C@@H](CCCCCCCCCCCCCC)O)O)NC(=O)*)[C@@H]([C@H]3O)O)CO)CO)O[C@H]4[C@@H]([C@H]([C@@H](O)[C@H](O4)CO)O[C@@H]5O[C@@H]([C@@H]([C@@H]([C@H]5O)O[C@]6(O[C@]([C@@H]([C@H](C6)O)NC(CO)=O)([C@@H]([C@@H](CO)O)O)[H])C([O-])=O)O)CO)NC(C)=O)(O[C@]([C@H](NC(=O)CO)[C@H](C1)O)([C@@H]([C@H](O)CO)O)[H])C([O-])=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)C#CCC(C)C)O[C@H]1CN(C)C(=O)C3=NC=CN=C3)[C@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('Nc1nc2n(cnc2c(=O)[nH]1)[C@@H]1O[C@H](COP([O-])(=O)OP([O-])(=O)O[C@H]2O[C@@H](CO)[C@@H](O)[C@@H](O)[C@@H]2O)[C@@H](O)[C@H]1O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C(=O)(*)[C@@H](N*)C[C@@](C)(CO)O', 'Contains primary alcohol "
               "group (-CH2OH)'), "
               "('O=C1N[C@@H](C(=O)N[C@H](C(=O)N[C@@H](C(=O)O)CC(=O)N[C@@H](CCCN=C(N)N)C(N[C@H]([C@@H](C(N[C@H](CCC(N[C@H]1CO)=O)C(=O)O)=O)C)/C=C/C(=C/[C@@H]([C@@H](OC)CC2=CC=CC=C2)C)/C)=O)CC(C)C)C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C1C[C@]2([C@]([C@]([C@H]1O)(C)CO)(CC[C@@]3([C@@]2(CC=C4[C@]3(CC[C@@]5([C@]4(CC(CC5)(C)C)[H])C([O-])=O)C)[H])C)[H])C', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@@H]1CN(C(=O)C2=CC=CC=C2C3=CC=CC=C3CO[C@@H]1CN(C)CC4=CC(=CC=C4)F)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O1[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@@H](O)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O[C@H]3O[C@@H]([C@@H](O[C@@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4NC(=O)C)CO)[C@H](O)[C@@H]3O)CO)[C@H](O)[C@@H]1O[C@H]5[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]5CO)O[C@H]6[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]6CO)O)CO[C@H]7O[C@@H]([C@@H](O)[C@H](O)[C@@H]7O[C@@H]8O[C@@H]([C@@H](O)[C@H](O)[C@H]8NC(=O)C)CO)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('C[C@H]1CN(C(=O)C2=C(N=CC(=C2)C3=CC=CC=C3F)O[C@H]1CN(C)C(=O)COC)[C@@H](C)CO', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C(N[C@@H](CC(=O)N)C(O)=O)[C@@H](NC(=O)[C@@H](N)CO)CCC(O)=O', "
               "'Contains primary alcohol group (-CH2OH)'), "
               "('O=C1O[C@@H]([C@H](NC(=O)[C@@H](NC(=O)/C(=C/C)/C)CCC(=O)N)C(=O)N[C@H](C(=O)N[C@H]2CC[C@H](N([C@H](C(N([C@H](C(N[C@H]1C)=O)CC3=CC=C(O)C=C3)C)=O)CC4=CC=CC=C4)C2=O)OC)CCC(=O)NCCO)C', "
               "'Contains primary alcohol group (-CH2OH)')]\n"
               "False negatives: [('[H][C@]1(CCC(C)=CC1)C(C)(C)O', 'No primary "
               "alcohol group found'), ('*CO', 'No primary alcohol group "
               "found')]",
    'attempt': 1,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 101,
    'num_false_positives': 100,
    'num_true_negatives': 284,
    'num_false_negatives': 1,
    'num_negatives': None,
    'precision': 0.5024875621890548,
    'recall': 0.9901960784313726,
    'f1': 0.6666666666666667,
    'accuracy': 0.7921810699588477}