"""
Classifies: CHEBI:25029 leukotriene
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_leukotriene(smiles: str):
    """
    Determines if a molecule is a leukotriene based on structural characteristics.
    
    Args:
        smiles (str): SMILES string of the molecule
        
    Returns:
        bool: True if molecule is a leukotriene, False otherwise
        str: Reason for classification
    """
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return None, "Invalid SMILES string"

    # Check for carboxylic acid group
    carboxylic_pattern = Chem.MolFromSmarts('C(=O)[OH]')
    if not mol.HasSubstructMatch(carboxylic_pattern):
        return False, "Missing carboxylic acid group"

    # Count carbon atoms
    carbon_count = len([atom for atom in mol.GetAtoms() if atom.GetSymbol() == 'C'])
    if carbon_count < 20:
        return False, "Less than 20 carbons"

    # Count conjugated double bonds
    conjugated_pattern = Chem.MolFromSmarts('C=CC=CC=C')
    if not mol.HasSubstructMatch(conjugated_pattern):
        return False, "Missing required conjugated double bond system"

    # Count total double bonds
    double_bond_pattern = Chem.MolFromSmarts('C=C')
    double_bond_count = len(mol.GetSubstructMatches(double_bond_pattern))
    if double_bond_count < 4:
        return False, "Insufficient number of double bonds"

    # Define specific leukotriene patterns
    patterns = [
        'O=C(O)CCC[CH]C=CC=CC=CC',  # Basic core
        'O=C(O)CCC[CH]C=CC=CC=C[CH]', # With hydroxyl group positions
        'SC[CH](NC(=O))C(=O)', # Cysteinyl leukotriene specific part
    ]

    for pattern in patterns:
        patt = Chem.MolFromSmarts(pattern)
        if mol.HasSubstructMatch(patt):
            # Check if molecule contains sulfur (cysteinyl leukotrienes)
            contains_sulfur = any(atom.GetSymbol() == 'S' for atom in mol.GetAtoms())
            if contains_sulfur:
                return True, "Cysteinyl leukotriene"
            
            # Check for hydroxyl groups
            hydroxyl_pattern = Chem.MolFromSmarts('[CH]O')
            if mol.HasSubstructMatch(hydroxyl_pattern):
                return True, "Hydroxylated leukotriene"
            
            return True, "Basic leukotriene structure"

    return False, "Does not match leukotriene structural requirements"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:25029',
                          'name': 'leukotriene',
                          'definition': 'Any icosanoid from that family of C20 '
                                        'polyunsaturated fatty acids and their '
                                        'derivatives generated by leukocytes '
                                        'from arachidonic acid, each member '
                                        'having four double bonds of which '
                                        'three are conjugated.',
                          'parents': ['CHEBI:23899']},
    'config': {   'llm_model_name': 'lbl/claude-sonnet',
                  'f1_threshold': 0.8,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0.07792207792207793 is too low.\n'
               'True positives: '
               "[('O[C@@H](CCCC(O)=O)/C=C/C=C\\\\C=C\\\\[C@@H](O)C/C=C\\\\CCCCC', "
               "'Leukotriene with core structure'), "
               "('C(C(/C=C/C=C/C=C/[C@H](CCCC(O)=O)O)=O)/C=C\\\\CCCCC', "
               "'Leukotriene with core structure'), "
               "('CCCCC\\\\C=C/CC(=O)\\\\C=C\\\\C=C\\\\C=C/[C@@H](O)CCCC(O)=O', "
               "'Leukotriene with core structure')]\n"
               'False positives: '
               "[('C[C@H]1C=CC=CCCC=CC=CC=CC=C[C@@H](C[C@H]2[C@@H]([C@H](C[C@](O2)(C[C@H]([C@@H](CC[C@H](C[C@H](C[C@H](CC(=O)O[C@H]([C@@H]([C@@H]1O)C)C)O)O)O)O)O)O)O)C(=O)O)O[C@H]3[C@H]([C@H]([C@@H]([C@H](O3)C)O)N)O', "
               "'Leukotriene with core structure'), "
               "('S(CCNC(=O)CCNC(=O)C(O)C(COP(OP(OCC1OC(N2C3=NC=NC(N)=C3N=C2)C(O)C1OP(O)(O)=O)(O)=O)(O)=O)(C)C)C(=O)CCC=CCC(O)C=CC=CC=CC(O)CCCC(O)=O', "
               "'Cysteinyl leukotriene'), "
               "('C[C@@H]1[C@H]([C@@H]([C@@H]([C@@H](O1)OC2CC(C(C(CC(=O)CC(C(CCC(CC(CC(CC(=O)OC(C(C(C(C=CC=CCCC=CC=CC=CC=C2)C)O)C)C)O)O)O)O)O)O)C(=O)O)O)O)N)O', "
               "'Leukotriene with core structure'), "
               "('[H][C@@]12C[C@H](O)C[C@]3(O)C[C@H](O)[C@@H](C(O)=O)[C@]([H])(C[C@@H](O[C@@H]4O[C@H](C)[C@@H](O)[C@H](N)[C@@H]4O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C[C@@H](C)OC(=O)\\\\C=C\\\\[C@H]1O2)O3', "
               "'Leukotriene with core structure'), "
               "('C(\\\\[C@H](CCCC(O)=O)O)=C\\\\C=C\\\\C=C\\\\[C@@H](C/C=C\\\\C=C\\\\C(CC)=O)O', "
               "'Leukotriene with core structure'), "
               "('OC(CCCC(O)=O)C=CC=CC=CC(O)CC=CCCC(O)=O', 'Leukotriene with "
               "core structure'), "
               "('O[C@H](CCCC(O)=O)/C=C\\\\C=C\\\\C=C\\\\C(=O)C/C=C\\\\CCCCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('O[C@@H](CCCC(O)=O)\\\\C=C/C=C/C=C/[C@H](O)C\\\\C=C/CCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('O[C@@H](CCCC(O)=O)\\\\C=C/C=C/C=C/C(=O)C/C=C\\\\C=C\\\\[C@@H](O)CC', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(C=CC=CC=CC=C[C@@H](O[C@@H]2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)C[C@H]3O[C@](C[C@@H](C[C@@H]4[C@H](C=C1)O4)O)(O)C[C@@H](O)[C@@H]3C(=O)O)C)C', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(CC)C)CC=CC=CC=CC=CC=CC(O[C@@H]2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CCCC(CC(C1)O)=O)O)(O)CC(O)C3C(=O)O', "
               "'Leukotriene with core structure'), "
               "('CCCCC[C@H](O)[C@H](O)\\\\C=C\\\\C=C\\\\C=C/C=C/[C@@H](O)CCCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('O[C@@H](CCCC(O)=O)/C=C\\\\C=C\\\\C=C\\\\[C@H](O)C/C=C\\\\C=C\\\\[C@@H](O)C(O)C', "
               "'Leukotriene with core structure'), "
               "('O=C1O[C@H]([C@@H](C=CC=CC=CC=CC=CC=CC=C[C@H](O[C@H]2O[C@@H]([C@H](O)[C@H]([C@H]2O)N)C)C[C@@H]3O[C@@](C[C@@H](C[C@@H](C[C@@H](C[C@@H](CC(C[C@H](C1)O)=O)O)O)O)O)(O)C[C@H](O)[C@H]3C(=O)O)C)[C@@H](CC[C@@H](O)CC(=O)C4=CC=C(N)C=C4)C', "
               "'Leukotriene with core structure'), "
               "('[H]C(CCC=C([H])C=C([H])C=C([H])CCCCC(O)=O)=CCCCCC', "
               "'Leukotriene with core structure'), "
               "('[H][C@]12C[C@@H](O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\[C@H](C)[C@@H](O)[C@@H](C)[C@H](C)OC(=O)C[C@H](O)C[C@H](O)CC[C@@H](O)[C@H](O)C[C@H](O)C[C@](O)(C[C@H](O)[C@H]1C(O)=O)O2', "
               "'Leukotriene with core structure'), "
               "('CC1C\\\\C(C)=C(C)/C=C/C2=CC=C(C)\\\\C(O2)=C2OC(C)(C(C(O)=O)C1=O)C(O)=C\\\\2O', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(CC)C)CC=CC=CC=CC=CC=CC(O[C@@H]2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(C(CCC(CC(C1)O)=O)O)O)(O)CC(O)C3C(=O)O', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(C=CC=CC=CC=CC(OC2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CC(C(C=C1)O)O)O)(O)CC(O)C3C(=O)O)CC)C', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(C=CC=CC=CC=CC=CC=CC=CC(OC2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CC(CC(CCCC(CC(C1)O)O)O)O)O)(O)CC(O)C3C(=O)O)C)C(CCC(O)CC(=O)C4=CC=C(NC)C=C4)C', "
               "'Leukotriene with core structure'), "
               "('O[C@@H]([C@H](O)/C=C/C=C/C=C\\\\C=C\\\\[C@H](O)CCCC(O)=O)C/C=C\\\\CC', "
               "'Leukotriene with core structure'), "
               "('O1C(C1/C=C/C=C/C=C\\\\C/C=C\\\\C/C=C\\\\C=C\\\\[C@H](O)C/C=C\\\\CC)CCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('OC(CCCC(O)=O)\\\\C=C\\\\C=C\\\\C=C\\\\C(O)C\\\\C=C/C=C\\\\C(O)CC', "
               "'Leukotriene with core structure'), "
               "('C1(CC2(OC(C1C(=O)O)CC(O[C@H]3[C@H]([C@H]([C@@H]([C@H](O3)C)O)N)O)C=CC=CC=CC=CCC(CCCC)OC(C=CC4C(O4)CC(C2)O)=O)O)O', "
               "'Leukotriene with core structure'), "
               "('O[C@@H](CCCC(O)=O)/C=C\\\\C=C\\\\C=C\\\\[C@H](O)C/C=C\\\\C=C\\\\[C@@H](O)CCO', "
               "'Leukotriene with core structure'), "
               "('CCC[C@@H]1C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\[C@@H](C[C@@H]2O[C@@](O)(C[C@H](O)[C@H]2C(O)=O)C[C@@H](O)CCCC(=O)C[C@@H](O)[C@H](CC)C(=O)O1)O[C@@H]1O[C@H](C)[C@@H](O)[C@H](N)[C@@H]1O', "
               "'Leukotriene with core structure'), "
               "('O=C1N([C@H](CCC(=O)O)C(C1=C(O)/C=C/C=C/C=C/C=C\\\\C=C/C(=C/[C@H]([C@@H](O[C@@H]2O[C@@H]([C@H](O)[C@@H](C2)O)C)C(C)C)C)/C)=O)C', "
               "'Leukotriene with core structure'), "
               "('O=C(C(=O)/C=C/C(CC)C)/C=C/C=C/C=C/C=C/C=C/C=C/C1OC(C(=O)O)C(C1)O', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(CC=CC=CC=CC=CC(OC2OC(C(N)C(C2O)O)C)CC3OC(CC(CC4C(C=C1)O4)O)(O)CC(O)C3C(=O)O)CCC', "
               "'Leukotriene with core structure'), "
               "('CC(CCC(O)CC(=O)c1ccc(N)cc1)C1OC(=O)CC(=O)CCCC(O)CC(O)CC(O)CC(O)CC2(O)CC(O)C(C(CC(O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C1C)O2)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('O=C1N[C@@H](CCC(=O)O)C(C1=C(O)/C=C/C=C/[C@H]2[C@H](C=C[C@@H]3[C@@H]2C[C@@H](C)[C@H]3C(=O)C)/C=C/C=C/C(=O)N)=O', "
               "'Leukotriene with core structure'), "
               "('[H][C@]12C[C@@H](O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\CC\\\\C=C\\\\C=C\\\\[C@H](C)[C@@H](O)[C@@H](C)[C@H](C)OC(=O)C[C@H](O)C[C@H](O)C[C@H](O)CC[C@@H](O)[C@H](O)C[C@](O)(C[C@H](O)[C@H]1C(O)=O)O2', "
               "'Leukotriene with core structure'), "
               "('OC(CCCCC)C(O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C(O)CCCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('O(O)C(CCCC(O)=O)/C=C/C=C/C=C/C(OO)C/C=C\\\\CCCCC', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(CC=CC=CC=CC=CC(O[C@@H]2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CC(CC(CC(C1C)O)O)O)O)(O)CC(O)C3C(=O)O)CCCC', "
               "'Leukotriene with core structure'), "
               "('P(OC[C@H](OC(=O)CCCCCCCCCCCCCCCCCCC)COC(=O)CCC[C@H](O)\\\\C=C/C=C/C=C/[C@H](O)C/C=C\\\\CCCCC)(OC[C@H](N)C(O)=O)(O)=O', "
               "'Leukotriene with core structure'), "
               "('CC[C@H]1[C@H](O)CC(=O)CCC[C@H](O)C[C@]2(O)C[C@H](O)[C@H]([C@H](C[C@@H](O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C[C@@H](C)OC1=O)O2)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('C(C(O)=O)C[C@@H]([C@@H](/C=C/C=C/C=C\\\\C/C=C\\\\C=C\\\\[C@H](C/C=C\\\\CC)O)O)O', "
               "'Leukotriene with core structure'), "
               "('O=C1N[C@@H](CCC(=O)O)C(C1=C(O)/C=C/C=C/[C@H]2[C@H](C=C[C@@H]3[C@@H]2C[C@@H](C)[C@H]3CC)/C=C/C=C/C(=O)N)=O', "
               "'Leukotriene with core structure'), "
               "('C(\\\\[C@H](CCCC(O)=O)O)=C\\\\C=C\\\\C=C\\\\[C@@H](C\\\\C=C/C=C/[C@@H](CC)O)O', "
               "'Leukotriene with core structure'), "
               "('C(\\\\C/C=C\\\\C=C\\\\C=C\\\\C1O[C@H]1CCC(=O)O)=C\\\\C=C\\\\[C@H](O)C/C=C\\\\CC', "
               "'Leukotriene with core structure'), "
               "('O1C2(O)CC(O)C(C1CC(O[C@H]3O[C@H]([C@H](O)[C@@H](N)[C@H]3O)C)C=CC=CC=CC=CC=CC=CC=CC(C(OC(=O)CC(O)CC(=O)CC(O)CC(O)CC(O)CC(O)C2)C(CCC(O)CC(=O)C4=CC=C(N)C=C4)C)C)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('C(C(O)=O)C[C@@H]([C@@H](/C=C/C=C/C=C\\\\C\\\\C=C/C=C/[C@@H](C/C=C\\\\CC)O)O)O', "
               "'Leukotriene with core structure'), "
               "('C[C@H]1C=CC=CC=CC=CC=CC=CC=C[C@@H](C[C@@H]2[C@@H]([C@H](C[C@](O2)(C[C@H](C[C@H]([C@@H](CC[C@H](C[C@H](CC(=O)O[C@H]([C@@H]([C@@H]1O)C)C)O)O)O)O)O)O)O)C(=O)O)O[C@H]3[C@H]([C@H]([C@@H]([C@H](O3)C)O)N)O', "
               "'Leukotriene with core structure'), "
               "('O=C1C(C2=CC=C3[C@]4([C@]([C@@H]([C@H](C(=O)O)CCC(=C)C(C)C)[C@@H](C4)O)(C)CC=C3[C@@]2(C)CC1)C)(C)C', "
               "'Leukotriene with core structure'), "
               "('CC(CCC(O)CC(=O)c1ccc(N)cc1)C1OC(=O)CC(=O)CC([*])CC(O)CC([*])CC(O)CC(O)CC2(O)CC(O)C(C(CC(O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C1C)O2)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('O1[C@@]2(O)C[C@H](O)[C@H]([C@@H]1C[C@@H](O[C@@H]3O[C@@H]([C@@H](O)[C@H](N)[C@@H]3O)C)C=CC=CC=CC=CC=CC=CC=C[C@@H]([C@@H](O)[C@@H]([C@@H](OC(=O)C[C@H](O)C[C@H](O)[C@@H](O)CC[C@H](O)C[C@H](O)C2)C)C)C)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('C[C@@H]1[C@H]([C@@H]([C@@H]([C@@H](O1)OC2CC3C(C(CC(O3)(CC(CC4C(O4)C=CC(=O)OC(CC=CC=CC=CC=C2)C)O)O)O)C(=O)O)O)N)O', "
               "'Leukotriene with core structure'), "
               "('C(C(/C=C/C=C/C=C\\\\[C@H](CCCC(O)=O)O)=O)/C=C\\\\CCCCC=O', "
               "'Leukotriene with core structure'), "
               "('P(OC[C@H](OC(=O)CCC[C@@H](O)/C=C\\\\C=C\\\\C=C\\\\[C@@H](O)C/C=C\\\\CCCCC)COC(=O)CCCCCCCCCCC/C=C\\\\C/C=C\\\\CCCCC)(OC[C@H](N)C(O)=O)(O)=O', "
               "'Leukotriene with core structure'), "
               "('O=C1O/C(=C(\\\\C2=CC=C(O)C=C2)/C(=O)O)/C(=C1C3=CC(=O)[C@@]4([C@H]3[C@H]5C(=O)O[C@@H]6[C@@]5(C=7C(=O)O\\\\C(\\\\C7O)=C(/C8=CC=C(O)C=C8)\\\\C(=O)O)[C@H]4C(=O)C6)O)O', "
               "'Leukotriene with core structure'), "
               "('O=C1O[C@@H]([C@H](C=CC=CC=CC=CC=CC=CC=C[C@@H](O[C@@H]2O[C@@H]([C@@H](O)[C@@H]([C@H]2O)N)C)C[C@H]3O[C@](C[C@@H](C[C@@H](C[C@H](C[C@@H](CC(C[C@H](C1)O)=O)O)O)O)O)(O)C[C@@H](O)[C@H]3C(=O)O)C)[C@@H](CC[C@@H](O)CC(=O)C4=CC=C(NC)C=C4)C', "
               "'Leukotriene with core structure'), "
               "('OCCCCC\\\\C=C/C[C@@H](O)\\\\C=C\\\\C=C\\\\C=C/[C@@H](O)CCCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('C(\\\\C=C/C=C/C=C/[C@H]([C@H](CCCCCO)O)O)=C/[C@H](CCCC(O)=O)O', "
               "'Leukotriene with core structure'), "
               "('O[C@H](CCCC(O)=O)/C=C\\\\C=C\\\\C=C\\\\[C@@H](O)C/C=C\\\\CCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('CC1C=CC=CC=CC=CC=CC=CC=CC(CC2C(C(CC(O2)(CC(CC(C(CCC(CC(CC(=O)OC(C(C1O)C)C)O)O)O)O)O)O)O)C(=O)O)O[C@@H]3C(C(C(C(O3)C)O)N)O', "
               "'Leukotriene with core structure'), "
               "('CCCCC[C@@H](O)[C@H](O)\\\\C=C\\\\C=C\\\\C=C/C=C/[C@@H](O)CCCC(O)=O', "
               "'Leukotriene with core structure'), "
               "('C[C@H]1C=CC=CCCC=CC=CC=CC=CC(CC2C(C(C[C@](O2)(CC(C(CCC(CC(CC(CC(=O)O[C@H](C([C@@H]1O)C)C)O)O)O)O)O)O)O)C(=O)O)OC3[C@H]([C@H]([C@@H]([C@H](O3)C)O)N)O', "
               "'Leukotriene with core structure'), "
               "('[H][C@]12C[C@@H](O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\CC\\\\C=C\\\\C=C\\\\[C@H](C)[C@@H](OC3C[C@@H](O)[C@@H](O)[C@H](C)O3)[C@@H](C)[C@H](C)OC(=O)C[C@H](O)C[C@H](O)C[C@H](O)CC[C@@H](O)[C@H](O)C[C@](O)(C[C@H](O)[C@H]1C(O)=O)O2', "
               "'Leukotriene with core structure'), "
               "('C(\\\\[C@H](CCCC(O)=O)O)=C\\\\C=C\\\\C=C\\\\[C@@H](C\\\\C=C/C=C/[C@H](CC)O)O', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(C=CC=CC=CC=CC=CC=CC=CC(OC2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CC(CC(CCCC(CC(C1)O)O)O)O)O)(O)CC(O)C3C(=O)O)C)C(CCC(O)CC(=O)C4=CC=C(N)C=C4)C', "
               "'Leukotriene with core structure'), "
               "('O=C(CCCC(O)=O)/C=C/C=C\\\\C=C\\\\[C@H](C/C=C\\\\CCCCC)O', "
               "'Leukotriene with core structure'), "
               "('[H][C@]12C[C@@H](O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\CC\\\\C=C\\\\C=C\\\\[C@H](C)[C@@H](O)[C@@H](C)[C@H](C)OC(=O)C[C@H](O)C[C@H](O)C[C@H](O)CCC[C@H](O)C[C@](O)(C[C@H](O)[C@H]1C(O)=O)O2', "
               "'Leukotriene with core structure'), "
               "('CC(CC(C)C1OC(=O)CC(=O)CCCC(=O)CC(O)CC(O)CC(O)CC(=O)CC(O)C(C(O)CC(OC2O[C@H](C)[C@@H](O)[C@H](N)[C@@H]2O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C1C)C(O)=O)C(O)CC(=O)c1ccc(N)cc1', "
               "'Leukotriene with core structure'), "
               "('CC(CCC(O)CC(=O)c1ccc(N)cc1)C1OC(=O)CC(=O)CC(O)CC(O)CCCC(O)CC(O)CC2(O)CC(O)C(C(CC(O[C@@H]3O[C@H](C)[C@@H](O)[C@H](N)[C@@H]3O)\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C=C\\\\C1C)O2)C(O)=O', "
               "'Leukotriene with core structure'), "
               "('OC(CCCC(\\\\C=C/C=C/C=C/C(C\\\\C=C/C=C/[C@@H](CC)O)O)O)=O', "
               "'Leukotriene with core structure'), "
               "('O=C1OC(C(C=CC=CC=CC=CC=CC=CC=CC(OC2O[C@@H]([C@@H](O)[C@@H]([C@@H]2O)N)C)CC3OC(CC(CC(CC(CCCC(CC(C1)O)O)O)O)O)(O)CC(O)C3C(=O)O)C)C(CCC(O)CC(=O)/C=C/C=C/C)C', "
               "'Leukotriene with core structure'), "
               "('O[C@H](CCCC(O)=O)/C=C/C=C\\\\C=C\\\\[C@@H](O)C/C=C\\\\C=C\\\\[C@H](O)CC', "
               "'Leukotriene with core structure')]\n"
               'False negatives: '
               "[('C([C@H](NC(CC[C@@H](C(=O)O)N)=O)C(NCC(=O)O)=O)S[C@H](\\\\C=C\\\\C=C\\\\C[C@@H](C/C=C\\\\CCCCC)O)[C@@H](O)CCCC(=O)O', "
               "'Missing required conjugated double bond system'), "
               "('C(/C=C\\\\CCCCC(=O)O)\\\\C=C/C=C/C=C/[C@H]([C@@H](O)CCCC(=O)O)SC[C@H](N)C(=O)O', "
               "'Does not match leukotriene structural requirements'), "
               "('C(/C=C\\\\CCCCC)\\\\C=C/C=C/C=C/[C@H]([C@@H](O)CCCC(=O)OC)SC[C@H](NC(=O)CC[C@H](N)C(O)=O)C(NCC(=O)O)=O', "
               "'Does not match leukotriene structural requirements')]",
    'attempt': 4,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 4,
    'num_false_positives': 73,
    'num_true_negatives': 183797,
    'num_false_negatives': 2,
    'num_negatives': None,
    'precision': 0.05194805194805195,
    'recall': 0.6666666666666666,
    'f1': 0.0963855421686747,
    'accuracy': 0.9995921164262873}