"""
Classifies: CHEBI:24828 indoles
"""
from rdkit import Chem
from rdkit.Chem import AllChem
from rdkit.Chem import Descriptors
from rdkit.Chem import rdMolDescriptors

def is_indoles(smiles: str):
    """
    Determines if a molecule contains an indole skeleton.
    
    Args:
        smiles (str): SMILES string of the molecule

    Returns:
        bool: True if molecule contains indole skeleton, False otherwise
        str: Reason for classification
    """
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return False, "Invalid SMILES string"

    # SMARTS patterns for indole skeleton variations
    patterns = [
        'c12ccccc1[nH]cc2',  # Basic indole
        'c12ccccc1n(*)cc2',  # N-substituted indole
        'C1=C2C=CC=CC2=CN1',  # Alternative representation
        'c1ccc2[nH]ccc2c1',   # Another representation
        'C1=CC=C2C(=C1)C=CN2', # Yet another representation
        '[nH]1ccc2ccccc12',    # Reversed representation
        'c1ccc2c(c1)cc[nH]2',  # Another variation
        'C1=CC2=C(C=C1)NC=C2'  # Additional variation
    ]

    for pattern in patterns:
        patt = Chem.MolFromSmarts(pattern)
        if patt and mol.HasSubstructMatch(patt):
            # Check if it's N-substituted
            n_sub_pattern = Chem.MolFromSmarts('c12ccccc1n(*)cc2')
            if n_sub_pattern and mol.HasSubstructMatch(n_sub_pattern):
                return True, "Contains N-substituted indole skeleton"
            return True, "Contains indole skeleton"

    # Check for fused indole systems
    fused_patterns = [
        'c12c3ccccc3[nH]c1cc2',  # Fused indole
        'c1cc2c(cc1)[nH]c1ccccc12',  # Another fused system
        'C1=CC2=C(C=C1)C1=C(N2)C=CC=C1'  # Additional fused system
    ]

    for pattern in fused_patterns:
        patt = Chem.MolFromSmarts(pattern)
        if patt and mol.HasSubstructMatch(patt):
            return True, "Contains indole skeleton"

    # Check for reduced indole (indoline) patterns
    reduced_patterns = [
        'C1CN2c3ccccc3CC2=C1',  # Indoline
        'C1Cc2ccccc2N1',        # Another indoline representation
        'C1=CC2=C(C=C1)NC=C2'   # Additional indoline pattern
    ]

    for pattern in reduced_patterns:
        patt = Chem.MolFromSmarts(pattern)
        if patt and mol.HasSubstructMatch(patt):
            return True, "Contains indole skeleton"

    return False, "No indole skeleton found"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:24828',
                          'name': 'indoles',
                          'definition': 'Any compound containing an indole '
                                        'skeleton.',
                          'parents': ['CHEBI:22728']},
    'config': {   'llm_model_name': 'lbl/claude-sonnet',
                  'f1_threshold': 0.8,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0.6958250497017893 is too low.\n'
               "True positives: [('N1C2=C(C=CC(=C2)C)C=C1', 'Contains indole "
               "skeleton'), ('C1=CC=C2C(=C1)C=CC3=C2C=C(N3)C(=O)O', 'Contains "
               "indole skeleton'), "
               "('O=C(NCCC1=CC=C(O)C=C1)CC=2C3=C(C=CC=C3)NC2', 'Contains "
               "indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)CCC(F)(F)F)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('FCCCCCN1C=C(C(=O)NC23CC4CC(C2)CC(C3)C4)C=5C1=CC=CC5', "
               "'Contains N-substituted indole skeleton'), "
               "('CC(=O)NCCC1=C(NC2=C1C=C(C=C2)OC)C3=CC=CC=C3', 'Contains "
               "indole skeleton'), ('O[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O', "
               "'Contains indole skeleton'), "
               "('Cl[C@H]1[C@](C=C)([C@@H]2NC(=O)S[C@@]23C=4C=5C(=CC=CC5C([C@H]3C1)(C)C)NC4)C', "
               "'Contains indole skeleton'), "
               "('NC(=O)c1cc2cc(ccc2o1)N1CCN(CCCCc2c[nH]c3ccc(cc23)C#N)CC1', "
               "'Contains indole skeleton'), "
               "('FC1=CC=2NC=C(CC(N)C(O)=O)C2C=C1', 'Contains indole "
               "skeleton'), ('C=1C2=C(C=C(C(ON3C4=C(N=N3)C=CC=C4)=O)C1)C=CN2', "
               "'Contains indole skeleton'), ('N1C=2C(=C(CN)C=CC2)C=C1', "
               "'Contains indole skeleton'), ('CN(C)CCc1c[nH]c2ccc(O)cc12', "
               "'Contains indole skeleton'), "
               "('C1=CC=C2C(=C1)C(=CN2)CCN=C(N)NC(=O)C3=CC=C(C=C3)Cl', "
               "'Contains indole skeleton'), "
               "('CC(=O)N1C2=CC=CC=C2C(=C1Cl)NC(=O)CCl', 'Contains "
               "N-substituted indole skeleton'), "
               "('N1C(=C(C(=C1C(=O)O)C=2C3=CC=CC=C3NC2)C=4C5=CC=CC=C5NC4)C(O)=O', "
               "'Contains indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)C4=CC=NC=C4)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('NC1=C(F)C(=NC(C(=O)OCC#N)=C1Cl)C1=C(F)C2=C(C=CN2)C=C1', "
               "'Contains indole skeleton'), "
               "('CN1C=C(C2=CC=CC=C21)C=C3C4=C(C=CC(=C4)S(=O)(=O)N)NC3=O', "
               "'Contains N-substituted indole skeleton'), "
               "('CN1CCC2=CC=CC=C2CC3=C(CC1)C4=CC=CC=C4N3', 'Contains indole "
               "skeleton'), "
               "('O=C1N[C@@H](Cc2c[nH]c3ccccc23)C(=O)N[C@H]1Cc1ccccc1', "
               "'Contains indole skeleton'), ('COC1=CC=CC2=C1C(=CN2)CN=C=S', "
               "'Contains indole skeleton'), "
               "('ClC1=CC=C(C(=O)N2C(=C(C3=C2C=CC(OC)=C3)CC(OCCCCCCC)=O)C)C=C1', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)NC4=CC(=CC=C4)OC)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('N(CCC=1C=2C(NC1)=CC=CC2)(CC)CC', 'Contains indole "
               "skeleton'), "
               "('Cl.C1[C@@H](C(C2=CN(C3=C2C=CC=C3)C)=O)CC4=C(C1)N=CN4', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)CC2=C(C=CC(=C2)NC(=O)CC3=CN(C4=CC=CC=C43)C)O[C@H]1CNC)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O(N1C=2C(C(=C1)C(OC)=O)=CC=CC2)C', 'Contains N-substituted "
               "indole skeleton'), ('OC1=CC=2C(=C(NC2C=C1)C)CC(O)=O', "
               "'Contains indole skeleton'), "
               "('C(\\\\[C@H](CC(C)C)NC([C@H](CC(C)C)NC([C@H](CC=1C2=CC=CC=C2NC1)NC(=O)CN3CCOCC3)=O)=O)=C/S(C)(=O)=O', "
               "'Contains indole skeleton'), "
               "('S1C(=NC=C1)C(=O)C=2C3=C(C=CC=C3)NC2', 'Contains indole "
               "skeleton'), "
               "('O(C1=CC=C(C(C=2C3=C(C=CC=C3)NC2)C=4C5=C(C=CC=C5)NC4)C=C1)CCC', "
               "'Contains indole skeleton'), "
               "('S(C=1C2=C(NC1)C=CC=C2)C3=NCCN3', 'Contains indole "
               "skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)NC(C)C)C4=CC=CC=C4N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('CC1=CC2=C(C=C1)C(=CN2)CCN', 'Contains indole skeleton'), "
               "('COC1=CC=CC2=C1C(=C(N2)C(=O)NN)N', 'Contains indole "
               "skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)C4=CC=C(C=C4)OC)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C1=2C(=CNC1=CC=CC2)CC#N', 'Contains indole skeleton'), "
               "('O=C(OCCC=1NC=2C=CC=CC2C1)CC=3C4=C(C=CC=C4)NC3', 'Contains "
               "indole skeleton'), ('O(C(CC=1C=2C(NC1)=CC=CC2)C(O)=O)C', "
               "'Contains indole skeleton'), "
               "('[H][C@@]12CCCN1C(=O)[C@H](Cc1c[nH]c3ccccc13)NC2=O', "
               "'Contains indole skeleton'), "
               "('C(CC1CCNCC1)C=2C=3C(NC2)=CC=CC3', 'Contains indole "
               "skeleton'), ('N1C=C(C2=C1C=CC(=C2Cl)Br)O', 'Contains indole "
               "skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)NC(C)C)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C1[C@@]23C(=O)NC([C@@H]2[C@H](C)[C@@]4([C@H]([C@@H]3C=CC[C@@H](CC([C@H]5[C@H]1O5)=O)C)O4)C)CC=6C7=C(C=CC=C7)NC6', "
               "'Contains indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)S(=O)(=O)C4=CSC=C4)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C=1(NC(=NC1)C(=O)C=2C=3C=CC(=CC3NC2)O)C=4C=5C=CC(=CC5NC4)Br', "
               "'Contains indole skeleton'), "
               "('C1=CC=C2C(=C1)C(=CN2)C(=O)C(=O)NC3=NC=CS3', 'Contains indole "
               "skeleton'), ('C1=C2C(=CNC2=CC=C1C[C@@H]3NC(=O)OC3)CCN(C)C', "
               "'Contains indole skeleton'), "
               "('C12=C(NC(=C1)C)C=CC=C2OC[C@@H](OC(C3=CC=CC=C3)=O)CNC(C)(C)C', "
               "'Contains indole skeleton'), "
               "('O1C2(C1C3C4(C(C2C)C(NC4=O)C(C=5C=6C(NC5)=CC=CC6)C)C(=O)CCC(=O)C(=O)C(=CC(CC=C3)C)C)C', "
               "'Contains indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)NC4=C(C=C(C=C4)OC)OC)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(C(O)CC=1C2=C(C=CC=C2)NC1)C(CC)C', 'Contains indole "
               "skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)CC4=CC=CC=C4)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('OC1=C(C2=C(N1)C=CC(=C2)C#N)C1=NC=C(CN2CCOCC2)C=C1', "
               "'Contains indole skeleton'), "
               "('C1=C2NC=C(C2=CC=C1)CCNC(CC[NH3+])=O', 'Contains indole "
               "skeleton'), ('CCOC(=O)C1=C(C2=C(N1)C=CC(=C2)OC)C', 'Contains "
               "indole skeleton'), "
               "('FC=1C=C(C(=O)N2CC(C3=C(NC=4C3=CC=CC4)C(=C2)C(OC(C)C)=O)(C)C)C=CC1F', "
               "'Contains indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)N[C@H](C)C(C)C)C4=CC=CC=C4N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(C1C(C1(C)C)(C)C)C=2C=3C(NC2)=CC=CC3', 'Contains indole "
               "skeleton'), "
               "('C[C@H]1C\\\\C=C\\\\[C@H]2[C@H](O)C(C)=C(C)[C@H]3[C@H](Cc4c[nH]c5ccccc45)NC(=O)[C@@]23C(=O)CCC(=O)C(=O)\\\\C(C)=C\\\\1', "
               "'Contains indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)NC4=CC(=CC=C4)F)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('CNS(=O)(=O)Cc1ccc2[nH]cc(CCN(C)C)c2c1', 'Contains indole "
               "skeleton'), ('O=C(O)C=1C2=C(C=CC(=C2)CC=C(C)C)NC1', 'Contains "
               "indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)C4=CC=C(C=C4)Cl)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C1=2C(NC=C1CCN(C)C)=CC=C(C2)CN3N=CN=C3.C=1(C=CC=CC1)C(O)=O', "
               "'Contains indole skeleton'), "
               "('ClC1=CC=2C([P@@](OC)(=O)C3=CC(=CC(=C3)/C=C/C#N)C)=C(NC2C=C1)C(=O)N', "
               "'Contains indole skeleton'), "
               "('C=1C=C2C(=CC1)N(C=C2C[C@@H](C(CC(C)C)=O)O)C', 'Contains "
               "N-substituted indole skeleton'), "
               "('COC(=O)C1=C(\\\\N=N\\\\N2CC3C[C@H](C2)C2=CC=CC(=O)N2C3)C2=C(N1)C=CC=C2', "
               "'Contains indole skeleton'), "
               "('OC(=O)CNC(=O)CC1=CNC2=CC=C(O)C=C12', 'Contains indole "
               "skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C=CC(=C2)NC(=O)CC3=CN(C4=CC=CC=C43)C)O[C@@H]1CN(C)CC5=CC6=C(C=C5)OCO6)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C1=CC=CC=2C(=CNC12)CC=[N+](O)[O-]', 'Contains indole "
               "skeleton'), "
               "('C12=CC=CC=C1C(=CN2)[C@@H](C)[C@]3(C(N=C(O3)N)=O)[H]', "
               "'Contains indole skeleton'), "
               "('O(C1=CC=2C(CC(N)C(O)=O)=CNC2C=C1)C', 'Contains indole "
               "skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)N[C@@H](C)C(C)C)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('CC1=C(C2=CC=CC=C2N1C)P(=O)(C3=CC=CC=C3)C4=CC=CC=C4', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)C4=CC=CC(=C4)C)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(OCCC=1C2=C(C=CC=C2)NC1)CC=3C4=C(C=CC=C4)NC3', 'Contains "
               "indole skeleton'), "
               "('O=C(N[C@H](C(=O)NCCC=1C2=C(C=CC=C2)NC1)C(C)C)C(=O)C(C)C', "
               "'Contains indole skeleton'), "
               "('CN1C=C(C2=CC=CC=C21)C=NNC(=O)CCCCCCC(=O)NC3=CC=CC=C3O', "
               "'Contains N-substituted indole skeleton'), "
               "('CC(C)COC(=O)N[C@@H](CC1=CNC2=CC=CC=C21)C(=O)NNC=C3C=C(C(=O)C(=C3)OC)[N+](=O)[O-]', "
               "'Contains indole skeleton'), "
               "('O=C(O)CC=1C2=C(C=CC=C2)N(C1)C3OC(C(O)C(C3O)O)CO', 'Contains "
               "N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)S(=O)(=O)C4=CC=CC=C4OC)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C12=C3C(=C(C(=C1NC(=C2)C(SC)=O)OC)O)N(CC3)C(=O)C4=CC5=C(N4)C(=CC6=C5C(=CN6C(=O)C7=CC8=C(N7)C=C(C(=C8)O)OC)C)O', "
               "'Contains indole skeleton'), "
               "('C[C@@H]1O[C@@H](OCCCCCCCCCCC(O)=O)[C@H](O)C[C@H]1OC(=O)c1c[nH]c2ccccc12', "
               "'Contains indole skeleton'), "
               "('Cn1ccc2cc(NC(=O)Nc3cccnc3)ccc12', 'Contains N-substituted "
               "indole skeleton'), "
               "('C12=CC=CC=C1NC(=C2)C(=O)N[C@@H](CC=3C=CC=C(C3)F)C(N[C@H](C(=O)[H])C[C@H]4C(NCC4)=O)=O', "
               "'Contains indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)S(=O)(=O)C4=C(ON=C4C)C)C5=CC=CC=C5N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C([C@@H](O)CC=1C2=C(C=CC=C2)NC1)CC3=CC=C(O)C=C3', "
               "'Contains indole skeleton'), "
               "('CCOC(=O)C1=C(N(C2=CC(=C(C=C21)O)Br)C)CNC3=CC=NC=C3', "
               "'Contains N-substituted indole skeleton'), "
               "('O[C@@H]1[C@@]2([C@]([C@H](C)CC1)(CCC([C@H]2CC=3C4=C(C=CC=C4)NC3)=C)C)CCC=C(C)C', "
               "'Contains indole skeleton'), "
               "('C=1(C=2NC=C3C[C@@H](CO)NC([C@H](C(C)C)N(C)C(C32)=CC1)=O)[C@](CCC=C(C)C)(C=C)C', "
               "'Contains indole skeleton'), "
               "('O=C(NCCC=1C=2C(NC1)=CC=CC2)CCCCCCCCCCCCCCCCCCCCCCC', "
               "'Contains indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C)C4=CC=CC=C4N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)C(=O)OCC(C)C)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@H]1CN(C)S(=O)(=O)C4=CC=C(C=C4)Cl)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C[C@@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)OCC(C)C)C4=CC=CC=C4N2C)[C@@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C1C(O)=C(C(=O)C(=C1C=2C3=C(C=C(/C=C/C(=C)C)C=C3)NC2)O)C=4C5=C(C=C(/C=C/C(=C)C)C=C5)NC4', "
               "'Contains indole skeleton'), ('O(C1=CC2=C(NC=C2C=O)C=C1)C', "
               "'Contains indole skeleton'), "
               "('BrC1=CC=2NC=C(C2C=C1)CCNC(=O)CC', 'Contains indole "
               "skeleton')]\n"
               'False positives: '
               "[('CCC(C)C1=CC=CC=C1NC(=O)C2=CC3=C(N2)C(=CC=C3)[N+](=O)[O-]', "
               "'Contains indole skeleton'), "
               "('CC(O)C(O)=O.Cc1[nH]c2ccccc2c1CCNCc1ccc(\\\\C=C\\\\C(=O)NO)cc1', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC=3NC=NC3)CC(O)=O', "
               "'Contains indole skeleton'), "
               "('O1[C@@H]([C@@H](O)[C@H](O)[C@@H](O)[C@@H]1OC=2C3=C(NC=C3CC#N)C=CC2)CO[C@@H]4O[C@@H]([C@@H](O)[C@H](O)[C@H]4O)CO', "
               "'Contains indole skeleton'), "
               "('O(C(C)(C)C)C(=O)C1=CC=C(CN2CCN(CC2)C=3C4=C(NC=C4)C=CC3)C=C1', "
               "'Contains indole skeleton'), "
               "('CCN1C2=CC=CC=C2C3=C1N=C(N=N3)SCC(=O)C4=CC=CC=C4', 'Contains "
               "N-substituted indole skeleton'), "
               "('CO[C@H]1[C@@H](C[C@@H]2CN3CCc4c([nH]c5ccccc45)[C@H]3C[C@@H]2[C@@H]1C(=O)OC)OC(=O)c1cc(OC)c(OC)c(OC)c1', "
               "'Contains indole skeleton'), "
               "('O=C(NCC(=O)N[C@@H](CC1=CC=CC=C1)C(O)=O)[C@@H](N)CC=2C=3C(NC2)=CC=CC3', "
               "'Contains indole skeleton'), "
               "('O[C@]12C=3C4=C5C=CC=C4NC3C(C)(C)C([C@]1([N+]#[C-])[C@@](C=C)(CC[C@@H]2C5(C)C)C)=C', "
               "'Contains indole skeleton'), "
               "('C1CC(C2=C(C1)C3=C(N2)C=CC(=C3)Cl)C(=O)N4CCN(CC4)C(=O)C5=CC=CO5', "
               "'Contains indole skeleton'), "
               "('O=C1N[C@H](C(=O)N[C@H](C(=O)O)CCCNC(=N)N)CC(=O)NCCCC[C@@H]2NC(=O)[C@@H](NC([C@H]([C@H](OC(C[C@H]3C(N[C@H](C(N[C@H]1CCC(=O)OC[C@H](NC(=O)[C@H]4C(C(=O)[C@@H](NC2=O)CC5=CC=CC=C5)CCC4)C(=O)N3)=O)CC(C)C)=O)=O)C)NC(=O)[C@@H](NC(=O)[C@H]6N(C(=O)[C@@H](N)C(C)C)CCC6)CC=7C8=C(C=CC=C8)NC7)=O)CC9=CC=C(O)C=C9', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(O)=O)CC=3NC=NC3', "
               "'Contains indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](NCC34CCN(CC4)C(=O)NC5=CC6=C(C=C5)OCO6)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('CSCC[C@H](NC(=O)[C@@H](N)CC(N)=O)C(=O)N[C@@H](Cc1c[nH]c2ccccc12)C(=O)N[C@@H](CC(N)=O)C(O)=O', "
               "'Contains indole skeleton'), "
               "('O=C/1N[C@@](O)(C(=O)N\\\\C1=C/C=2C3=C(C=CC=C3)NC2C(C=C)(C)C)C', "
               "'Contains indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](N(CC34CCN(CC4)C(=O)C5CCCC5)S(=O)(=O)C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(N[C@@H](CCC(=O)N)C(=O)N[C@@H]([C@H](O)C)C(O)=O)[C@@H](N)CC=1C=2C(NC1)=CC=CC2', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](C(C)C)C(O)=O)[C@@H](NC(=O)[C@@H](N)CCCCN)CC=1C=2C(NC1)=CC=CC2', "
               "'Contains indole skeleton'), "
               "('O=C1O[C@@H]([C@H](NC(=O)[C@@H](NC(=O)[C@H](NC(=O)[C@@H](NC(=O)CCCCCCCCC)CC=2C3=C(C=CC=C3)NC2)CCC(=O)O)[C@@H](O)C(=O)N)C(=O)N(CC(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](CCCCN)C(N[C@H](C(NCC(N[C@@H](C(N[C@H](C(N[C@H]1[C@H](CC)C)=O)CCC(=O)O)=O)CC(=O)N)=O)=O)[C@@H](OC)C(=O)O)=O)CC(=O)O)C)C)C', "
               "'Contains indole skeleton'), "
               "('C(=C/C(SCCNC(CCNC(=O)[C@@H](C(COP(OP(OC[C@H]1O[C@@H](N2C3=C(C(=NC=N3)N)N=C2)[C@@H]([C@@H]1OP([O-])([O-])=O)O)(=O)[O-])(=O)[O-])(C)C)O)=O)=O)\\\\CCCC[C@H](O[C@@H]4O[C@@H](C)[C@@H](C[C@H]4O)OC(C=5C=6C=CC=CC6NC5)=O)C', "
               "'Contains indole skeleton'), "
               "('O=C(N1C(C(O)=NC(C(O)=NC(C(O)=NC(C(O)=NC(C(O)=NC(C(O)=NC(CO)CC=2C3=C(C=CC=C3)NC2)CCC(O)=N)CCC(O)=N)C(CC)C)(C)C)C(C)C)CCC1)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C(N=C(O)C)(C)C)C)C)(C)C)(C)C)CCC(O)=N)(C)C)(C)C)(C)C)CO)CC(C)C)(C)C', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC=1NC=NC1)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(=O)N)CC=2C=3C(NC2)=CC=CC3', "
               "'Contains indole skeleton'), "
               "('OC[C@H](O)Cc1nccc2c3ccccc3[nH]c12', 'Contains indole "
               "skeleton'), ('ON(O)[C@@H](Cc1c[nH]c2ccccc12)C(O)=O', 'Contains "
               "indole skeleton'), "
               "('C=C[C@]1(CN2CCC=3C=4C=CC(=C([C@]5(CCCN5C)[H])C4NC3[C@@]2(C[C@@]1(C[C@]6(C7=C(CCN6C)C8=CC=CC=C8N7)[H])[H])[H])O)[H]', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@H](C(=O)N[C@@H](CC(O)=O)C(O)=O)CO)[C@@H](N)CC=1C=2C(NC1)=CC=CC2', "
               "'Contains indole skeleton'), "
               "('CC(C(NC(CC=1C2=C(C=C(C=C2)O)NC1)=O)C(O)=O)C', 'Contains "
               "indole skeleton'), "
               "('OCC1C2N3C(CC1C(C3)=CC)C=4NC5=C(C4C2)C=CC(OC)=C5', 'Contains "
               "indole skeleton'), "
               "('COC(=O)[C@@]1(O)C[C@H]2O[C@]1(C)n1c3ccccc3c3c4CNC(=O)c4c4c5ccccc5n2c4c13', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(=O)N[C@@H](CCCN=C(N)N)C(O)=O)[C@H]3NCCC3', "
               "'Contains indole skeleton'), "
               "('O=C1N2[C@H](C(=O)N3[C@@]1(O)C(=O)C=4C5=C(C=C(OC)C=C5)N(C4[C@@H]3C=C(C)C)CC=C(C)C)CCC2', "
               "'Contains N-substituted indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](N(CC34CCN(CC4)CC5=CC=CC=N5)C(=O)NC6=CC=C(C=C6)F)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O(C=1C=2NC3=C(C2C=C(C1)C=O)C=C(OC)C=C3)C', 'Contains indole "
               "skeleton'), "
               "('O=C1N2[C@@H](C3=C4C(NC=C4C[C@H]2C(=C1C(=O)C)O)=CC=C3)C=C(C)C', "
               "'Contains indole skeleton'), "
               "('O=C(NC=CC1=CC=C(O[C@@H]2O[C@@H]([C@H](NC(=O)C)[C@@H]([C@H]2O)O)C)C=C1)[C@@H](N[C@@H](C(=O)NCCC3=CC=CC=C3)CC4=CC=CC=C4)CC=5C6=C(C=CC=C6)N(C5)C', "
               "'Contains N-substituted indole skeleton'), "
               "('[H]C1C[C@H](C)OC(=O)C[C@@H](NC(=O)[C@@H](Cc2c(Br)[nH]c3ccccc23)N(C)C(=O)[C@H](C)NC(=O)[C@@H](C)C\\\\C(C)=C\\\\1)c1ccc(O)cc1', "
               "'Contains indole skeleton'), "
               "('S([C@]1(N(C(=O)[C@@](OC)(CC=2C3=C(C=CC=C3)NC2)NC1=O)C)CO)C', "
               "'Contains indole skeleton'), "
               "('ClC1=CC=2NC(Cl)=C(C2C=C1)C[C@H]3NC(=O)[C@H](NC(=O)[C@H](NC(=O)[C@H](CO)NC([C@H](NC([C@@H](NC3=O)[C@@H](CC)C)=O)CCC(=O)O)=O)C(C)C)CC=4C5=C(C=CC=C5)NC4', "
               "'Contains indole skeleton'), "
               "('O=C1N2[C@H](C(=O)N3[C@]1(O)[C@@H](OC)C=4C5=C(C=C(OC)C=C5)NC4[C@@H]3C=C(C)C)CCC2', "
               "'Contains indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@@H](N(CC34CCN(CC4)CC5CC5)CC6=CC=CC=C6OC)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C1=C(C(=CC=2NC=CC12)O[C@@H]3[C@H]([C@@H]([C@H]([C@@H](O3)C(O)=O)O)O)O)O', "
               "'Contains indole skeleton'), "
               "('O[C@]12[C@@]([C@@]3([C@@](CC1)(CC4=C3NC=5C4=C(C(C)(C)C=C)C=CC5)[H])C)(CC[C@]67O[C@](C(O6)(C)C)(C(=O)C=C27)[H])C', "
               "'Contains indole skeleton'), "
               "('O[C@H](Cc1c[nH]c2ccc(OCc3ccccc3)cc12)C(O)=O', 'Contains "
               "indole skeleton'), "
               "('S(CC[C@H](NC(=O)[C@@H](N)CC=1C=2C(NC1)=CC=CC2)C(=O)N[C@@H](CCC(=O)N)C(O)=O)C', "
               "'Contains indole skeleton'), "
               "('O=C([O-])[C@@H](NC(CC1=CNC2=CC=CC=C12)=O)CC(=O)[O-]', "
               "'Contains indole skeleton'), "
               "('CC1=CC=C(C=C1)NC(=O)CSC2=NC3=C(C(=O)N2C)NC4=CC=CC=C43', "
               "'Contains indole skeleton'), "
               "('[H]\\\\C(C)=C1/C[NH2+][C@@]2([H])CC3=C(NC4=CC=CC=C34)C(=O)C[C@]1([H])[C@]2([H])C(=O)OC', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](C(C)C)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(C)C)CC=1C=2C(NC1)=CC=CC2', "
               "'Contains indole skeleton'), "
               "('S(CC[C@H](NC(=O)[C@@H](NC(=O)[C@@H](N)[C@H](CC)C)CC=1C=2C(NC1)=CC=CC2)C(O)=O)C', "
               "'Contains indole skeleton'), "
               "('COCC(=O)N1CC2(C1)CN([C@@H](C3=C2C4=C(N3)C=C(C=C4)OC)CO)CC5=CC=CC=C5F', "
               "'Contains indole skeleton'), "
               "('O=C(NCC(=O)N[C@@H](CC1=CC=C(O)C=C1)C(O)=O)[C@@H](N)CC=2C=3C(NC2)=CC=CC3', "
               "'Contains indole skeleton'), "
               "('CC1=C(C2=C(N1)C=CC(=C2)F)CCN(CC3=CC=NC=C3)C(=S)NC4=CC=C(C=C4)F', "
               "'Contains indole skeleton'), "
               "('CCC1CCCCN1C(=O)CSC2=NC3=C(C(=O)N2C)NC4=CC=CC=C43', 'Contains "
               "indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@@H](N(CC34CCN(CC4)CC5=CN=CC=C5)C(=O)COC)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C1N[C@H](C(=O)N[C@@H]1C(C)C)CC=2C3=C(C(=CC(=C3)CC=C(C)C)CC=C(C)C)NC2C(C=C)(C)C', "
               "'Contains indole skeleton'), "
               "('O=C1N(C(C(=O)NC(C(=O)NCCCCC(C(NC(C(NC1CC=2C3=C(C=CC=C3)NC2)=O)C(CC)C)=O)NC(=O)NC(C(=O)O)C(C)C)CC4=CC=CC=C4)C)C', "
               "'Contains indole skeleton'), "
               "('CC[C@H](C)[C@@H](C(=O)N[C@@]1(CCCC=CCCC[C@](NC(=O)[C@@H](NC(=O)[C@@H](NC(=O)[C@@H](NC1=O)CC(C)C)CS)CCCNC(=N)N)(C)C(=O)N[C@@H](CC2=CNC=N2)C(=O)N[C@@H](CC3=CNC=N3)C(=O)N[C@@H](CO)C(=O)N[C@@H]([C@@H](C)O)C(=O)N)C)NC(=O)[C@H](CCCNC(=N)N)NC(=O)[C@H](CCCNC(=N)N)NC(=O)[C@H](CCCNC(=N)N)NC(=O)[C@H](CC(C)C)NC(=O)[C@H](CCCNC(=N)N)NC(=O)[C@H](CCC(=O)O)NC(=O)[C@H](CC4=CNC5=CC=CC=C54)NC(=O)CCNC(=O)C', "
               "'Contains indole skeleton'), "
               "('C[C@H]1CN(C(=O)C2=C(C3=CC=CC=C3CO[C@@H]1CN(C)C(=O)[C@H](CC4=CC=CC=C4)N(C)C)C5=CC=CC=C5N2C)[C@H](C)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CCCN=C(N)N)CCCCN', "
               "'Contains indole skeleton'), "
               "('[O-]C(=O)C(F)(F)F.[O-]C(=O)C(F)(F)F.NC(=[NH2+])NCCCc1[nH+]ccc2c3ccc(Br)cc3[nH]c12', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@H](C(=O)O)CC=1C2=C(C=CC=C2)NC1)C3N(C(=O)[C@@H](NC(=O)CC(C)C)CCCCNC(=N)N)C\\\\C(\\\\[C@H]3C)=C\\\\C', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)C(C)C)CCC(=O)N', "
               "'Contains indole skeleton'), "
               "('CC(C)C[C@@H]1NC(=O)[C@H](NC(=O)[C@@H]2CCCN2C(=O)[C@@H](CC(O)=O)NC(=O)[C@@H](Cc2c[nH]c3ccccc23)NC1=O)c1cccs1', "
               "'Contains indole skeleton'), "
               "('O=C1N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)N[C@H](C(=O)NCC(=O)N[C@H](C(=O)N[C@H](C(=O)O)CC2=CC=CC=C2)CC(C)C)CC(=O)N)CCCCN)CO)CC(=O)NCC(=O)N[C@H](C(=O)NCC(N[C@H](C(N[C@H](C(N[C@H]1CC=3C4=C(C=CC=C4)NC3)=O)C)=O)CC(=O)N)=O)CC5=CC=C(O)C=C5', "
               "'Contains indole skeleton'), "
               "('C1CCC2=C(C1)C3=C4N2C(=O)C(=C(C4=CC=C3)O)CC5=CC=CC=C5', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C1C=CC2(C3(C=4NC=5C=C(O[C@@H]6O[C@@H]([C@@H](OC)[C@@H]([C@H]6O)O)CO)C=CC5C4CC3CCC2C1(C)C)C)C', "
               "'Contains indole skeleton'), "
               "('O=C(N1[C@H](CCC1)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC=2C=3C(NC2)=CC=CC3)CO', "
               "'Contains indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](NCC34CCN(CC4)CC5=CC=NC=C5)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(N1[C@@H](CCC1)C(=O)N[C@@H](CC=2C=3C(NC2)=CC=CC3)C(=O)N[C@@H]([C@H](O)C)C(O)=O)[C@@H](NC(=O)[C@@H](NC(=O)[C@@H](NC(=O)[C@@H](N)CC(C)C)C(C)C)C(C)C)CC4=CC=C(O)C=C4', "
               "'Contains indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@@H](N(CC34CCN(CC4)CC5CC5)C(=O)C6=CC=NC=C6)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('C1=CC=C2C(=C1)C3=CC=CC=C3N2CC(COC4=CC=CC5=C4N=CC=C5)O', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(N1[C@H](CCC1)C(O)=O)[C@@H](NC(=O)[C@H]2NCCC2)CC=3C=4C(NC3)=CC=CC4', "
               "'Contains indole skeleton'), "
               "('CC(=O)N1CCC2(CC1)CN([C@H](C3=C2C4=C(N3)C=C(C=C4)OC)CO)C(=O)C', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@H](C(=O)N[C@@H](CCC(=O)N)C(O)=O)CO)[C@@H](N)CC=1C=2C(NC1)=CC=CC2', "
               "'Contains indole skeleton'), "
               "('CN1CC2(CCN(CC2)S(=O)(=O)C3=CC=CC(=C3)F)C4=C([C@H]1CO)N(C5=C4C=CC(=C5)OC)C', "
               "'Contains N-substituted indole skeleton'), "
               "('S(CC[C@@H]1NC(=O)[C@@H](NC(=O)[C@@H](NC(=O)CNC(=O)[C@@H](NC(=O)C)CC2=CC=C(O)C=C2)CC(=O)N)[C@H](OC(=O)C[C@@H]3NC(=O)[C@H](CO)NC([C@H]4N(C([C@@H](NC([C@@H](NC1=O)CCCCNC(=O)CC[C@H](NC(=O)[C@H](NC([C@@H](NC3=O)CC=5C6=C(C=CC=C6)NC5)=O)CCC(=O)OC)C(=O)N[C@H](C(=O)O)CC7=CC=C(O)C=C7)=O)CC8=CC=C(O)C=C8)=O)CCC4)=O)C)C', "
               "'Contains indole skeleton'), "
               "('C\\\\C=C1\\\\CN2[C@H]3Cc4c([nH]c5ccccc45)[C@@H]2C[C@@H]1[C@H]3C=O', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC1=CC=CC=C1)C(O)=O)[C@@H](NC(=O)[C@@H](N)CCCCN)CC=2C=3C(NC2)=CC=CC3', "
               "'Contains indole skeleton'), "
               "('O=C1NCC=2C1=C3C(NC4=C3C=CC=C4)=C5N(NC)C6=C(C25)C=CC=C6', "
               "'Contains indole skeleton'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CCC(O)=O)CCC(=O)N', "
               "'Contains indole skeleton'), "
               "('C1(=CC(=CC=C1)C#N)S(=O)(=O)[N-]C2=CC=C(C=3C(=CNC23)C#N)C', "
               "'Contains indole skeleton'), "
               "('O=C1N([C@H](C(=O)N[C@@H](C(=O)N[C@@H](C)CC(O[C@@H]([C@@H](C(N[C@H]1[C@H](CC)C)=O)NC(=O)C(=NNC2=C(C(=O)OCCCCCCCCCC(C)C)C=C(O)C=C2)C)C3=CC=CC=C3)=O)CC=4C5=C(C(OC)=CC=C5)NC4)[C@@H](O)C6=CC=CC=C6)C', "
               "'Contains indole skeleton'), "
               "('O=C1N[C@H](C(=O)NC2=C1C=CC=C2)CC=3C4=C(C=CC=C4)N(C3)CC=C(C)C', "
               "'Contains N-substituted indole skeleton'), "
               "('CN[C@@H]1Cc2c[nH]c3cccc([C@H]1\\\\C=C(/C)C=O)c23', 'Contains "
               "indole skeleton'), "
               "('CN1C2=C(C=CC(=C2)OC)C3=C1[C@H](N(CC34CCN(CC4)C(=O)C5=CC6=C(C=C5)OCO6)C(=O)NC7=CC=C(C=C7)F)CO', "
               "'Contains N-substituted indole skeleton'), "
               "('O=C(NC(CCCCN)C(O)=O)C(NC(=O)C(N)CC=1C=2C(NC1)=CC=CC2)CCCCN', "
               "'Contains indole skeleton'), "
               "('[C@@H]1(N2C3=C(C(=NC=N3)N)N=C2)O[C@H](COP(OP(OCC(C)([C@H](C(NCCC(NCCSC([C@@H](CC=4C5=CC=CC=C5NC4)O)=O)=O)=O)O)C)(=O)[O-])(=O)[O-])[C@H]([C@H]1O)OP([O-])([O-])=O', "
               "'Contains indole skeleton'), "
               "('NC(Cc1c[nH]c2ccc(O)cc12)C(O)=O', 'Contains indole "
               "skeleton'), "
               "('O=C(NC=CC1=CC=C(O[C@@H]2O[C@@H]([C@H](NC(=O)C)[C@@H]([C@H]2O)O)C)C=C1)[C@@H](N[C@@H](C(=O)NCCC3=CC=CC=C3)CC(C)C)CC=4C5=C(C=CC=C5)N(C4)C', "
               "'Contains N-substituted indole skeleton'), "
               "('O(C=1C=2NC=3C(C2C=C(C1)C(OC)=O)=CC=CC3)C', 'Contains indole "
               "skeleton'), ('O1C(C=CC=2C=3NC=4C(C3C=C(C12)C=O)=CC=CC4)(C)C', "
               "'Contains indole skeleton'), "
               "('O=C1[C@H](N[C@H](C(=O)O)CC2=CC=C(O)C=C2)CCC(O)=NCCCC[C@@H]3N=C(O)[C@@H](N=C([C@H]([C@H](OC(C[C@H]4C(=N[C@H](C(=N[C@H]1CCC(=O)OC[C@H](N=C(O)[C@H]5N(C(=O)[C@@H](N=C3O)CC6=CC=C(O)C=C6)CCC5)C(O)=N4)O)CC=7C8=C(C=CC=C8)NC7)O)=O)C)N=C(O)[C@@H](N=C(O)[C@@H](N=C(O)C)CC9=CC=C(O)C=C9)CO)O)CC(C)C', "
               "'Contains indole skeleton'), "
               "('COC(=O)\\\\C=C/NC(=O)c1cc2c3cccc(O)c3[nH]c2c(n1)C(C)=O', "
               "'Contains indole skeleton'), "
               "('S=C1NCCC1C2NC(CC3=C2NC=4C3=CC=CC4)C(O)=O', 'Contains indole "
               "skeleton'), "
               "('O=C(NC(CC=1C=2C(NC1)=CC=CC2)C(O)=O)CCCCCCC/C=C/C\\\\C=C\\\\CCCCC', "
               "'Contains indole skeleton'), "
               "('S1S[C@@]23N([C@H]4N5C=6C=CC=CC6[C@]4(C2)N7C8=C(C=CC=C8)C(=C7)C[C@@]9(SC)N(C)C([C@](COC5)(SC)N(C)C9=O)=O)C([C@]1(N(C)C3=O)CO)=O', "
               "'Contains N-substituted indole skeleton'), "
               "('[H][C@@]12CC3=CNC4=CC=CC(=C34)C1=C[C@H](CN2C(C)(C)C)C(=O)N(CC)CC', "
               "'Contains indole skeleton'), "
               "('CCS(=O)(=O)N1CC2(C1)CN([C@H](C3=C2C4=C(N3)C=C(C=C4)OC)CO)C(=O)CC5=CC=CC=N5', "
               "'Contains indole skeleton'), "
               "('C12=CC=CC=C1NC3=C2[C@]4([C@H]([C@](CC[C@@]4(C3(C)C)[H])(C)C=C)[N+]#[C-])[H]', "
               "'Contains indole skeleton'), "
               "('[H][C@@]12Cc3c[nH]c4cccc(c34)[C@@]1([H])[C@@H](OC(C)=O)[C@@H](C)C[NH+]2C', "
               "'Contains indole skeleton')]\n"
               'False negatives: '
               "[('S1S[C@]23N([C@@H]4[C@H]5O[C@H]5[C@H](C=C4C2)OC(=O)C)C([C@]16N([C@@H]7[C@H]8O[C@H]8[C@H](C=C7C6)OC(=O)CCCCC)C3=O)=O', "
               "'No indole skeleton found'), "
               "('ClC1=C[C@H]2C(=O)[C@]([C@@]1(C=C)C)(N=C=S)C3=C4C(=CC=C3)NC([C@@H]4C2(C)C)=O', "
               "'No indole skeleton found'), "
               "('C1CCC(CC1)NC(=O)N2CCC3=CC=CC=C32', 'No indole skeleton "
               "found'), ('S(C(SC)=N\\\\C=C/1\\\\C=2C(N(OC)C1=O)=CC=CC2)C', "
               "'No indole skeleton found'), "
               "('S(=O)(=O)(C1=CC=C2C(=C1)C(C(N2CC3=CC=CC=C3)=O)=O)N4CC(CC(C4)C)C', "
               "'No indole skeleton found'), "
               "('C1=CC=C2C(=C1)C(=CC3=CN=CN3)C(=O)N2', 'No indole skeleton "
               "found'), ('C=1C2=C(C=C3C1NC(C3=O)=O)CCC2', 'No indole skeleton "
               "found'), ('O=C1N(CC=2C1=C(C(O)=C(O)C2O)C)CCC(=O)OC', 'No "
               "indole skeleton found'), "
               "('S1S[C@]23N([C@@H]4[C@@H](O)[C@@H](OC)CC([C@@H]4C2)=O)C([C@]15N([C@@H]6[C@@H](O)C=CC([C@H]6C5)=O)C3=O)=O', "
               "'No indole skeleton found'), "
               "('ClC1=C[C@H]2C(=O)[C@]([C@@]1(C=C)C)([N+]#[C-])C3=C4C(=CC=C3)N(C([C@@H]4C2(C)C)=O)C', "
               "'No indole skeleton found'), "
               "('O=C1N(CC2=C1C=C(O)C3=C2OC(C(O)C3)(CC/C=C(/CCC=C(C)C)\\\\C)C)CCCCC(=O)O', "
               "'No indole skeleton found'), "
               "('C1CN2C3=NC4=CC=CC=C4C3=NN=C2SC1', 'No indole skeleton "
               "found'), ('CC(COC)NC1=NNC(=C2C=NC3=CC=CC=C32)CS1', 'No indole "
               "skeleton found'), "
               "('COC(=O)[C@@H]1C[C@@]23CC[C@@]11N(C(=O)OC)c4ccccc4[C@]1(CCN2C(=O)C=C3)C(=O)OC', "
               "'No indole skeleton found'), "
               "('C[C@@H]1[C@H]([C@@H](O[C@]12C3=C(C=CC(=C3)I)N(C2=O)CC=C)CCO)[Si](C)(C)C4=CC=C(C=C4)OC', "
               "'No indole skeleton found'), "
               "('C1=CC2=C(C=C1Br)C(=CNNC(=O)C3=CC=NC=C3)C=N2', 'No indole "
               "skeleton found'), "
               "('CC1=C(NC(=C1C(=O)N2CCC[C@@H]2CN3CCCC3)C)C=C4C5=C(C=CC(=C5)S(=O)(=O)CC6=C(C=CC=C6Cl)Cl)NC4=O', "
               "'No indole skeleton found'), "
               "('CC1=CC=C(C=C1)C2=CSC(=C3C=NC4=CC=CC=C43)N2', 'No indole "
               "skeleton found'), "
               "('O=C(O)C1=CC2=C(N[C@@H](C2)[C@](O)(CCC(=C(C)C)C)C)C=C1', 'No "
               "indole skeleton found'), "
               "('C1=CC2=C(C=C1I)C(=CC3=CC(=C(C(=C3)Br)O)Br)C(=O)N2', 'No "
               "indole skeleton found'), "
               "('[H][C@@]12CCCN1C(=O)C1=C[C@]3([C@H](C=C(C)C)N1C2=O)C(=O)Nc1cc(OC)ccc31', "
               "'No indole skeleton found'), "
               "('CNC(=O)Oc1ccc2N(C)[C@H]3OCC[C@@]3(C)c2c1', 'No indole "
               "skeleton found'), "
               "('N1C(C(C=2C=CC=CC12)=CC=3NC(=C(C3C)CCC(=O)O)C)=O', 'No indole "
               "skeleton found'), ('O=C1N(C=2C=C(OC)C=CC2[C@]1(O)CC(=O)C)C', "
               "'No indole skeleton found'), "
               "('CC[C@H](C)C(=O)C(=O)N[C@H]1Cc2ccc(O)c(c2)-c2cccc3c2NC(=O)[C@@]3(O)[C@H](O)[C@H](NC(=O)[C@H](CC(N)=O)NC1=O)C(=O)N\\\\C=C/C', "
               "'No indole skeleton found'), "
               "('CC1CC2=C(N1S(=O)(=O)C)C=CC(=C2)C3=CSC(=N3)NC(=O)CC4=CC=CS4', "
               "'No indole skeleton found'), "
               "('C1=CC=C(C=C1)CN(CC2=CC=CC=C2)CC3=NNC(=C4C=C5C=CC=CC5=N4)S3', "
               "'No indole skeleton found'), ('O=C1CNc2ccccc12', 'No indole "
               "skeleton found'), ('CSc1cc2CCN(C(=O)Nc3cccnc3)c2cc1C(F)(F)F', "
               "'No indole skeleton found'), "
               "('CCCCN1C2=CC=CC=C2C(=C(C#N)C(=O)N)C1=O', 'No indole skeleton "
               "found'), "
               "('CC(\\\\C=C\\\\C=C1\\\\N(CCS(O)(=O)=O)c2ccc(cc2C1(C)C)S(O)(=O)=O)=C/C=C/C1=[N+](CCS([O-])(=O)=O)c2ccc(cc2C1(C)C)S(O)(=O)=O', "
               "'No indole skeleton found'), "
               "('CC1=CC(=C(O1)C)C(=O)OCC(=O)N2CCC3=CC=CC=C32', 'No indole "
               "skeleton found'), ('O=C1C(=O)C=C(NCCCO)C2=C1NC=C2CCO', 'No "
               "indole skeleton found'), "
               "('N1C(/C(/C=2C=CC=CC12)=C\\\\C=3NC(=CC3C)C)=O', 'No indole "
               "skeleton found'), "
               "('CN(C1=C(C=CC=N1)CNC2=NC(=NC=C2C(F)(F)F)NC3=CC4=C(C=C3)NC(=O)C4)S(=O)(=O)C', "
               "'No indole skeleton found'), "
               "('O=C1N(CC=2C1=C(C(O)=C(O)C2O)C)CCCCC(=O)OC', 'No indole "
               "skeleton found'), "
               "('S1S[C@]23N([C@@H]4[C@@H](O)[C@@H](O)CC([C@@H]4C2)=O)C([C@]15N([C@@H]6[C@@H](O)[C@@H](O)CC([C@@H]6C5)=O)C3=O)=O', "
               "'No indole skeleton found'), ('OC(=O)C1CC2=CC=CC=C2N1', 'No "
               "indole skeleton found'), "
               "('S(=O)(=O)(C1=CC=C2C(=C1)C(C(N2)=O)=O)N3CCN(CC3)C', 'No "
               "indole skeleton found'), ('C=1C=CC=C2NC(C(C12)CC(=O)O)=O', 'No "
               "indole skeleton found'), ('CN(C)CCN1CCC2=CC=CC=C21', 'No "
               "indole skeleton found'), "
               "('[K+].[K+].CN1c2ccc(NC(=O)CI)cc2C(C)(C)\\\\C1=C\\\\C=C\\\\C=C\\\\C1=[N+](CCS([O-])(=O)=O)c2ccc3c(cc(cc3c2C1(C)C)S([O-])(=O)=O)S([O-])(=O)=O', "
               "'No indole skeleton found'), "
               "('CCOC1=C(C=C(C=C1)C2CC3=C(C(=C(N3)C(=O)OC)C)C(=O)C2)OC', 'No "
               "indole skeleton found'), "
               "('O1N(CC[C@@]2([C@]1(N(C3=C2C=C(OC(=O)NCCCCCCC)C=C3)C)[H])C)C', "
               "'No indole skeleton found'), "
               "('O1N=C(C(OC)=C)CC2(C1NC=3C(=CC=CC23)/C=C/C(=C)C)O', 'No "
               "indole skeleton found'), "
               "('ClC1=C[C@H]2C(=O)[C@]([C@@]1(C=C)C)(N=C=S)C3=C4C(=CC=C3)N(C([C@@H]4C2(C)C)=O)C', "
               "'No indole skeleton found'), "
               "('CC1=CC=CC=C1CN2C3=CC=CC=C3C(=NNC(=S)NC4=CC=CC=C4F)C2=O', 'No "
               "indole skeleton found'), "
               "('COC(=O)c1ccc2\\\\C(=C(\\\\Nc3ccc(cc3)N(C)C(=O)CN3CCN(C)CC3)c3ccccc3)C(=O)Nc2c1', "
               "'No indole skeleton found'), "
               "('[H]C(=CC([H])=C1N(CCCCCC(=O)ON2C(=O)CCC2=O)c2ccc(cc2C1(C)C)S([O-])(=O)=O)C1=[N+](CCCCCC(=O)ON2C(=O)CCC2=O)c2ccc(cc2C1(C)C)S([O-])(=O)=O', "
               "'No indole skeleton found'), "
               "('S1SC23N(C4C(C2)=COC=CC4O)C(=O)C15N(C6C(C5)=COC=CC6OC(=O)C)C3=O', "
               "'No indole skeleton found'), "
               "('CCc1c(C)[nH]c2CCC(CN3CCOCC3)C(=O)c12', 'No indole skeleton "
               "found'), ('Clc1cc2NC(=O)Cc2cc1CCN1CCN(CC1)c1nsc2ccccc12', 'No "
               "indole skeleton found'), ('O=C/1NC=2C=CC=CC2\\\\C1=C\\\\C', "
               "'No indole skeleton found')]",
    'attempt': 1,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 213,
    'num_false_positives': 100,
    'num_true_negatives': 2890,
    'num_false_negatives': 15,
    'num_negatives': None,
    'precision': 0.6805111821086262,
    'recall': 0.9342105263157895,
    'f1': 0.787430683918669,
    'accuracy': 0.9642635177128651}