"""
Classifies: CHEBI:24669 hydroxy carboxylic acid
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_hydroxy_carboxylic_acid(smiles: str):
    """
    Determines if a molecule is a hydroxy carboxylic acid (carboxylic acid with at least one hydroxy group).

    Args:
        smiles (str): SMILES string of the molecule

    Returns:
        bool: True if molecule is a hydroxy carboxylic acid, False otherwise
        str: Reason for classification
    """
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return False, "Invalid SMILES string"

    # Check for carboxylic acid group
    carboxylic_acid_pattern = Chem.MolFromSmarts('C(=O)[OH]')
    if not mol.HasSubstructMatch(carboxylic_acid_pattern):
        # Check for ester pattern
        ester_pattern = Chem.MolFromSmarts('C(=O)O[C,c]')
        if mol.HasSubstructMatch(ester_pattern):
            # Check for hydroxy group
            hydroxy_pattern = Chem.MolFromSmarts('[CH,CH2,CH3,C][OH]')
            if mol.HasSubstructMatch(hydroxy_pattern):
                hydroxy_matches = len(mol.GetSubstructMatches(hydroxy_pattern))
                return True, f"Found ester group and {hydroxy_matches} hydroxy group(s)"
        return False, "No carboxylic acid group found"

    # Check for hydroxy group (excluding the OH from carboxylic acid)
    hydroxy_pattern = Chem.MolFromSmarts('[CH,CH2,CH3,C][OH]')
    if not mol.HasSubstructMatch(hydroxy_pattern):
        return False, "No hydroxy group found"

    # Count number of carboxylic acid and hydroxy groups
    carboxylic_acid_matches = len(mol.GetSubstructMatches(carboxylic_acid_pattern))
    hydroxy_matches = len(mol.GetSubstructMatches(hydroxy_pattern))

    return True, f"Found {carboxylic_acid_matches} carboxylic acid group(s) and {hydroxy_matches} hydroxy group(s)"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:24669',
                          'name': 'hydroxy carboxylic acid',
                          'definition': 'Any carboxylic acid with at least one '
                                        'hydroxy group.',
                          'parents': ['CHEBI:33575', 'CHEBI:33822']},
    'config': {   'llm_model_name': 'lbl/claude-sonnet',
                  'f1_threshold': 0.8,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0.7005649717514124 is too low.\n'
               'True positives: '
               "[('C(\\\\[C@H](CCCC(O)=O)O)=C\\\\C=C\\\\C=C\\\\[C@@H](C/C=C\\\\C=C\\\\C(CC)=O)O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C(CC(CCCCCCCCCCC(=O)O)O)(=O)O', 'Found 2 carboxylic acid "
               "group(s) and 3 hydroxy group(s)'), "
               "('C(=C\\\\CC1OC1CCCCCO)\\\\CCCCCCCC(=O)O', 'Found 1 carboxylic "
               "acid group(s) and 2 hydroxy group(s)'), "
               "('OC(C(O)=O)c1ccc(O)cc1', 'Found 1 carboxylic acid group(s) "
               "and 2 hydroxy group(s)'), ('C(CC(CC(=O)O)O)(=O)O', 'Found 2 "
               "carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('S([C@@H]([C@@H](O)CCCCC)/C=C/[C@@H](O)C/C=C\\\\C/C=C\\\\CCCC(O)=O)C[C@H](N)C(=O)NCC(O)=O', "
               "'Found 2 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('OC(CC)(CC)C(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), ('C1(C(C/C=C\\\\CCCCCO)O1)CCCCCCCC(=O)O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC1(C)CC[C@@]2(CC[C@]3(C)C(=CC[C@H]4[C@@]3(C)CC[C@H]3[C@](C)(CO)[C@@H](O)CC(=O)[C@]43C)[C@@H]2C1)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('O[C@@H]1C[C@@](O)(CC(=O)[C@H]1O)C(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 4 hydroxy group(s)'), "
               "('CC\\\\C=C/C[C@@H]1[C@@H](O)CC[C@@H]1CC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OCCC(O)C(O)=O', 'Found 1 carboxylic acid group(s) and 3 "
               "hydroxy group(s)'), ('O[C@H](Cc1ccc(O)cc1)C(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H]1C(C2[C@@](C3[C@](C4[C@@](C5[C@@](CC4)(CC[C@H]5C(C)=C)C(O)=O)(CC3)[H])(CC2)C)(CC1)C)(C)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C[C@](O)(CCO)CC(O)=O', 'Found 1 carboxylic acid group(s) and "
               "3 hydroxy group(s)'), "
               "('CC\\\\C=C/C\\\\C=C/C\\\\C=C/C=C/C(O)C\\\\C=C/C\\\\C=C/CCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C[C@@H](O)CCCCCC(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 2 hydroxy group(s)'), "
               "('CC(C)CCCC(C)CCCC(C)CCCC(C)C(O)C(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCC\\\\C=C/C[C@@H](O)\\\\C=C\\\\C=C/C\\\\C=C/CCCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCC\\\\C=C/C[C@@H](O)\\\\C=C\\\\C=C/CCC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C[C@@H]1O[C@@H](OCCCCCCC[C@@H](O)CC(O)=O)[C@H](O)C[C@H]1O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('CC\\\\C=C/C\\\\C=C/C\\\\C=C/C\\\\C=C/C\\\\C=C/C=C/C(O)CCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC(O)CCCCC(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('CCCCCCC\\\\C=C\\\\C(=O)N(O)CCCNC(=O)CC(O)(CC(=O)NCCCN(O)C(C)=O)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC(CC(O)=O)c1ccccc1', 'Found 1 carboxylic acid group(s) and "
               "2 hydroxy group(s)'), ('OCCCCCCCCCCCCCCCCCCCCCCCCCCCCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H]1C[C@@](O)(C[C@H](O)[C@H]1O)C(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 5 hydroxy group(s)'), "
               "('[H][C@@]12C[C@H](O)CC[C@]1(C)[C@@]1([H])C[C@H](O)[C@]3(C)[C@]([H])(CC[C@@]3([H])[C@]1([H])[C@H](O)C2)[C@H](C)CCC(O)C(C)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 5 hydroxy group(s)'), "
               "('OC(CC/C=C\\\\C/C=C\\\\C[C@@H](/C=C/C=C\\\\C=C\\\\[C@H](C/C=C\\\\CC)O)O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C(CC(C(CCCCCCCCCCCCCCCCCCCC1C(CCCCCCCCCCCCCC[C@@H](O)[C@H](C(=O)O)CCCCCCCCCCCCCCCCCCCCCCCC)C1)OC)C)CCCCCCCCCCCCCCCC', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC[C@H](C)C(=O)O[C@H]1C[C@@H](C)C=C2C=C[C@H](C)[C@H](CC[C@@H](O)C[C@@H](O)CC(O)=O)[C@@H]12', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('CC(C)(O)C(O)C(O)=O', 'Found 1 carboxylic acid group(s) and 3 "
               "hydroxy group(s)'), "
               "('C(=C/[C@]1([C@](C/C=C\\\\CCCCC)(O1)[H])[H])\\\\[C@H](C/C=C\\\\CCCC(=O)O)O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(CCC/C=C\\\\C/C=C\\\\C/C=C\\\\[C@H](/C=C\\\\CCCCC)O)O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@@H](CS)C(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('C[C@H](CCCC(C)(C)O)[C@H]1CC[C@@]2(C)[C@@H]3[C@H](C[C@@H]4[C@]5(C[C@@]35CC[C@]12C)CC[C@H](O)[C@@]4(C)C(O)=O)OC(C)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C1(CCCCCCCCCCC2C(CCCCCCCCCCCCCCCCCCC[C@@H](O)[C@H](C(O)=O)CCCCCCCCCCCCCCCCCCCCCC)C2)C(C1)CCCCCCCCCCCCCCCCCC', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCC\\\\C=C/C\\\\C=C/CC(O)\\\\C=C\\\\C=C/CCCC(O)=O', 'Found "
               "1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCCC(O)C(O)C\\\\C=C/CCCCCCCC(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 3 hydroxy group(s)'), "
               "('[C@@H]1(O)[C@@H](CO[C@@H]2[C@@H]([C@H]([C@@H]([C@H](O2)COC(CCCCCCC)=O)O)O)O)O[C@@H]([C@@H]([C@H]1O)O)O[C@@H](C(=O)O)CO', "
               "'Found 1 carboxylic acid group(s) and 8 hydroxy group(s)'), "
               "('OC(CCCCCC/C=C/C/C=C\\\\C/C=C\\\\CC)C(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C[C@@H]1O[C@@H](OCCCCCCCCCCCCCCCCCCC[C@@H](O)CC(O)=O)[C@H](O)C[C@H]1O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C(CCCCCCCCCCO)CC[C@H](CC(O)=O)O', 'Found 1 carboxylic acid "
               "group(s) and 3 hydroxy group(s)'), "
               "('[H][C@@]12CCC3=C(CC[C@@]4(C)[C@]5([H])C[C@](C)(CC[C@]5(C)CC[C@]34C)C(O)=O)[C@@]1(C)CC[C@H](O)C2(C)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC(C)c1nc(nc(-c2ccc(F)cc2)c1\\\\C=C\\\\[C@@H](O)C[C@@H](O)CC(O)=O)N(C)S(C)(=O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('[H][C@@]1(O[C@@H]2CC\\\\C(Cl)=C\\\\C\\\\C=C(C)/[C@@H](CC(=O)O[C@@H]1C2)OC(C)=O)[C@H](O)\\\\C=C(/C)CC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C\\\\C(CC[C@@H]1C(=C)C[C@@H](OC(=O)c2ccccc2)[C@@H]2[C@]1(C)CCC[C@@]2(C)C(O)=O)=C/CO', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCC(\\\\C=C\\\\C=C/C=C/C(C\\\\C=C/C=C/C(CCCCCC(=O)O)O)O)O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C(CCCC(CCCCCCO)=O)CCCCC(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), ('C(CCCCCCC=C(C(=O)O)O)CC', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H](C/C=C\\\\C/C=C\\\\CC)/C=C/C=C\\\\C/C=C\\\\CCCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCC(O)CCCCCCCCC(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 2 hydroxy group(s)'), "
               "('C[C@@H](O)CCCCCCCCC[C@@H](O)CC(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 3 hydroxy group(s)'), "
               "('[H]C(C=C)=C(O)C(O)=O', 'Found 1 carboxylic acid group(s) and "
               "2 hydroxy group(s)'), "
               "('C(=O)(/C=C/C1=CC=C(O)C(O)=C1)O[C@@]2(C[C@@H](O)[C@@H]([C@@H](C2)OC(=O)/C=C/C3=CC(=C(O)C=C3)OC)O)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('OC(C(O)=O)=C(C)C', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('C1=C(C(=C(C=C1/C=C/C(O[C@H]2[C@H](O)C[C@@](C[C@H]2O)(C(O)=O)O)=O)OC)O)OC', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C[C@@H](O)CCCCCCCCCCCCCC[C@@H](O)CC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C[C@@](O)(C(O)=O)c1ccccc1', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), ('CC(=O)CC(O)CC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC\\\\C(\\\\C=C\\\\[C@@H]1OC(=O)C=C[C@@H]1C)=C\\\\[C@H](C)C\\\\C=C\\\\C(\\\\C)=C\\\\[C@@H](C)C(=O)[C@@H](C)[C@H](O)[C@@H](C)C\\\\C(C)=C\\\\C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC(\\\\C=C\\\\[C@]1(O)[C@@]2(C)CO[C@]1(C)C[C@H](O)C2)=C\\\\C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('O=C(O)/C=C\\\\C=C\\\\C=C\\\\C[C@H](O)/C=C/C', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@@H](CCCCCCCC(O)=O)\\\\C=C\\\\C=C\\\\CCCCC', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CC[C@@]1(O)C[C@H](O[C@H]2C[C@@H]([C@H](O)[C@H](C)O2)N(C)C)c2c(O)c3C(=O)c4c(O)cccc4C(=O)c3cc2[C@H]1C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('OCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCCC(O)CC(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('CCCCC[C@@H](O)[C@H](O)\\\\C=C\\\\C=C\\\\C=C/C=C/[C@@H](O)CCCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C(CCCCCCCC[C@@H](CCCCCCCC)O)(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('CCCCCCCCCCCCCCCCCCCCCCCC[C@H]([C@H](O)CCCCCCCCCCCCCCCC[C@@H]1C[C@@H]1[C@H](C)CCCCCCCCCCCCCCCCCCC(=O)C(C)CCCCCCCCCCCCCCCCCC)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OCCCCCC(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('C(O)(=O)CCC[C@H]1[C@@H](/C=C/C=C/C=C\\\\C/C=C\\\\C=C\\\\[C@H](CC)O)O1', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('[H][C@@]12CC(C)(C)[C@@H](O)C[C@@]1([C@H](O)C[C@]1(C)C2=CC[C@]2([H])[C@@]3(C)CC[C@H](O)C(C)(C)[C@]3([H])CC[C@@]12C)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('OC(=O)c1cc(O)c2ccccc2c1O', 'Found 1 carboxylic acid group(s) "
               "and 1 hydroxy group(s)'), "
               "('OC(C(O)CC=CCC=CCCCCC)CC=CCCCC(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 3 hydroxy group(s)'), ('OCC(CC(O)=O)C(O)=O', "
               "'Found 2 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('OCC(O)=O', 'Found 1 carboxylic acid group(s) and 2 hydroxy "
               "group(s)'), ('OCCCCCCCCCCCCCCCCCCCCCCCCCCC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC(CCCCCCC(O)=O)CC(O)=O', 'Found 2 carboxylic acid group(s) "
               "and 3 hydroxy group(s)'), "
               "('C(CCC)C[C@@H]([C@H](C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCC(O)=O)O)O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('OC1CC(O)(CC(OC(=O)\\\\C=C\\\\c2ccc(O)c(O)c2)C1OC(=O)CCc1ccc(O)c(O)c1)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('CCCCCCCCCCCCCCCCCCCCCCCC[C@H]([C@H](O)CCCCCCCCCCCCCCCCC[C@H]1C[C@H]1CCCCCCCCCCCCCCCC[C@@H](OC)[C@H](C)CCCCCCCCCCCCCCCCCC)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H](CCCC(O)=O)CC\\\\C=C\\\\C=C\\\\[C@H](O)C/C=C\\\\CCCCC', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C[C@H](O)CC(O)=O', 'Found 1 carboxylic acid group(s) and 2 "
               "hydroxy group(s)'), "
               "('COc1cc(ccc1O)\\\\C=C\\\\C(=O)O[C@H]1[C@H](O)C[C@@](O)(C[C@H]1O)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C[C@@](O)(CCO)CC(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 3 hydroxy group(s)'), "
               "('O[C@@H](CC(O[C@@H](CC(O)=O)C)=O)C', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('OC(=O)C(O)(c1ccccc1)c1cccs1', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('[C@H]1(CCCCCCCC(=O)O)[C@@H](CCCCCCCCO)O1', 'Found 1 "
               "carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC[C@H](O)C(O)=O', 'Found 1 carboxylic acid group(s) and 3 "
               "hydroxy group(s)'), ('NC(CO)(CO)C(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 3 hydroxy group(s)'), ('NCC(O)CC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('C[C@@H](O)CCCCCCCC(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 2 hydroxy group(s)'), ('OCCCCCCCC[C@@H](O)CC(O)=O', 'Found "
               "1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('CCCCC\\\\C=C/C\\\\C=C/C\\\\C=C/[C@H](O)[C@H]1O[C@H]1CCCC(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('CCCCC\\\\C=C/C\\\\C=C/[C@H](O)[C@@H](O)CCCCCC(O)=O', 'Found "
               "1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C[C@@H]1O[C@@H](OCCCCCCCC[C@@H](O)CC(O)=O)[C@H](O)C[C@H]1OC(=O)c1c[nH]c2ccccc12', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C[C@@H](O)CCCC[C@@H](O)CC(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 3 hydroxy group(s)'), "
               "('CCCCCCCCCCCCCCCCCCCCCCCC[C@H]([C@H](O)CCCCCCCCCCCCCCCCC[C@@H]1C[C@@H]1CCCCCCCCCCCCCCCC[C@H](OC)[C@@H](C)CCCCCCCCCCCCCCCCCC)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H](CC/C=C\\\\C/C=C\\\\CCCC(O)=O)C/C=C\\\\CCCCC', 'Found "
               "1 carboxylic acid group(s) and 2 hydroxy group(s)')]\n"
               "False positives: [('NCCCC(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('O1C(C(O)C(O)C(O)C1OC2=CC(O)=C(C=C2)C(O)=O)C(O)=O', 'Found 2 "
               "carboxylic acid group(s) and 5 hydroxy group(s)'), "
               "('[H][C@]1(C=C(C)CC[C@H]1C(C)=C)c1c(O)cc(CCCCC)c(C(O)=O)c1O', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('OC(=O)C(=C(C)C)C', 'Found 1 carboxylic acid group(s) and 1 "
               "hydroxy group(s)'), "
               "('CC(O)C(O)=O.Cc1[nH]c2ccccc2c1CCNCc1ccc(\\\\C=C\\\\C(=O)NO)cc1', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O[C@H]1[C@@H]([C@@H](CCCCCCCCC(O)=O)C(=O)C1)/C=C/[C@@H](O)CCCCC', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C(CC(O)=O)(CN)C1=CC=CC=C1', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('OC(C(O)CCCCCCCC(O)=O)CCC(O)CCCCC', 'Found 1 carboxylic acid "
               "group(s) and 4 hydroxy group(s)'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC=3NC=NC3)CC(O)=O', "
               "'Found 2 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC(=O)\\\\C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCCC', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(N[C@@H](CCCN=C(N)N)C(O)=O)[C@@H](NC(=O)[C@@H](N)[C@H](O)C)CC(=O)N', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('S(CC[C@H](NC(=O)[C@@H](N)CCCN=C(N)N)C(=O)N[C@@H](CCCCN)C(O)=O)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('[H]C(CCCC(O)=O)=CCCC([H])=CC([H])=CCC([H])=CCCCCC', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C1C=C2[C@@]3(C(C(C(=O)CC3)(C)C)C[C@H]4[C@@]2(O4)[C@]5([C@]1(C(=C[C@H]5O)C(O)(CC(=O)CC(C(=O)O)C)C)C)C)C', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('O=C1C2=C(O[C@@]3(C[C@@H](O[C@@H]3C2)C=C(CCCC(C(=O)O)C)C)C)CC[C@H]1O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(O)[C@]1([C@@H](O)C(=CCC1)CO)CC(=O)C2=C(O)C=CC(=C2)O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('C[C@H]1CN(C(=O)C2=C(C=CC(=C2)NC(=O)C3CC3)O[C@H]1CN(C)CC4=CC=C(C=C4)C(=O)O)[C@H](C)CO', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(N[C@@H](CC(C)C)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(C)C)C(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('CC1=COC2=CC3=C(C=C12)C(=C(C(=O)O3)CCC(=O)O)C', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C1O[C@@H]([C@@H](NC(=O)[C@H](NC(=O)[C@H](NC(=O)CC(O)CCCCCCC)CC(C)C)CC(=O)O)C(=O)N[C@@H](C(=O)N[C@@H](C(=O)N[C@@H](C(=O)N[C@H](C(N[C@@H](C(N[C@H](C(N[C@H](C(N[C@H]1CC(=O)O)=O)[C@H](CC)C)=O)[C@H](CC)C)=O)CO)=O)CC(C)C)CO)CC(C)C)CC(C)C)C', "
               "'Found 2 carboxylic acid group(s) and 5 hydroxy group(s)'), "
               "('O([C@@H]1[C@H](O)[C@H](O[C@H]2[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]2CO)O[C@H]3[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]3CO[C@@H]4O[C@H]([C@@H](O)[C@@H](O)[C@@H]4O)C)O)O[C@@H]([C@H]1O)CO[C@H]5O[C@@H]([C@@H](O)[C@H](O)[C@@H]5O[C@@H]6O[C@@H]([C@@H](O[C@@H]7O[C@@H]([C@H](O)[C@H](O[C@]8(O[C@H]([C@H](NC(=O)C)[C@@H](O)C8)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]7O)CO)[C@H](O)[C@H]6NC(=O)C)CO)CO)[C@H]9O[C@@H]([C@@H](O)[C@H](O)[C@@H]9O[C@@H]%10O[C@@H]([C@@H](O[C@@H]%11O[C@@H]([C@H](O)[C@H](O)[C@H]%11O)CO[C@]%12(O[C@H]([C@H](NC(=O)CO)[C@@H](O)C%12)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@@H]%10NC(=O)C)CO)CO', "
               "'Found 2 carboxylic acid group(s) and 36 hydroxy group(s)'), "
               "('O=C(N[C@@H](C(C)C)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC=1NC=NC1)C(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('C1=CC(C=C(C1=O)CC(O)=O)=O', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('O=C(O[C@H](C(=O)O)CC1=CC=CC=C1)[C@@H]([C@@H](OC)[C@H]2N(C(=O)C[C@@H](OC)[C@@H](N(C(=O)[C@@H](NC(=O)[C@@H](N(C)C)C(C)C)C(C)C)C)[C@H](CC)C)CCC2)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('OC(=O)CC(=O)\\\\C=C\\\\C(O)=O', 'Found 2 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('P(OC[C@H](OC(=O)CCC(O)/C=C/C(O)=O)COC(=O)CCCCCCC/C=C\\\\CCCCCCCC)(OCCN)(O)=O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(N[C@@H](CCC(=O)N)C(O)=O)[C@@H](NC(=O)[C@@H](N)CO)CC(=O)N', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C1C2=C([C@@]3(C(=O)C[C@@H]([C@]3(C1)C)/C(=C/C(=O)C(O)C(C(=O)O)C)/C)C)[C@@H](O)CC4[C@@]2(CC[C@@H](C4(C)C)O)C', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('S(C(=O)/C=C/C=C/C)CC[C@H](NC(=O)C(O)/C=C/C(O)C)C(=O)O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('[H][C@]12SCC(CSc3nnnn3CCN(C)C)=C(N1C(=O)[C@@]2([H])NC(=O)Cc1csc(N)n1)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O([C@@H]1[C@H](O[C@]2(O[C@H]([C@H](NC(=O)C)[C@@H](O)C2)[C@H](O)[C@H](O)CO)C(O)=O)[C@@H](O)[C@@H](O[C@@H]1CO)O[C@H]3[C@H](O)[C@@H](O)C(O[C@@H]3CO)O)[C@@H]4O[C@@H]([C@H](O)[C@H](O[C@@H]5O[C@@H]([C@H](O)[C@H](O[C@]6(O[C@H]([C@H](NC(=O)C)[C@@H](O)C6)[C@H](O)[C@H](O[C@]7(O[C@H]([C@H](NC(=O)C)[C@@H](O)C7)[C@H](O)[C@H](O)CO)C(O)=O)CO)C(O)=O)[C@H]5O)CO)[C@H]4NC(=O)C)CO', "
               "'Found 3 carboxylic acid group(s) and 25 hydroxy group(s)'), "
               "('O1[C@H]([C@H](NC(=O)C)[C@@H](O)C[C@]1(OC[C@H]2OC(O)[C@H](NC(=O)C)[C@@H](O[C@H]3O[C@@H]([C@H](O)[C@H](O)[C@H]3O)CO)[C@H]2O)C(O)=O)[C@H](O)[C@H](O)CO', "
               "'Found 1 carboxylic acid group(s) and 11 hydroxy group(s)'), "
               "('O=C(N[C@@H](CO)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(C)C)[C@H](CC)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(O)/C(=C/C/C=C(/CCC=C(C)C)\\\\C)/CC(=O)C1=C(O)C=CC(=C1)O', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(NCC(O)=O)\\\\C=C\\\\C/C=C/CC', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('O1[C@@H](O[C@@H]2[C@H](O)[C@@H](O)[C@@H](O[C@@H]2CO)O[C@H]3[C@H](O)[C@@H](O)[C@@H](O[C@@H]3CO)O)[C@H](NC(=O)C)[C@@H](O[C@@H]4O[C@@H]([C@H](O)[C@H](O)[C@H]4O)CO)[C@@H](O)[C@H]1CO[C@]5(O[C@H]([C@H](NC(=O)C)[C@@H](O)C5)[C@H](O)[C@H](O)CO)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 17 hydroxy group(s)'), "
               "('O[C@H]1[C@]2([C@]3([C@@]([C@](CC3)([C@@H](CCC(O)=O)C)[H])(CC[C@@]2([C@@]4([C@]([C@@H]1O)(C[C@@H](O)CC4)[H])C)[H])C)[H])[H]', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('O=C(CCCC(O)=O)CC', 'Found 1 carboxylic acid group(s) and 1 "
               "hydroxy group(s)'), "
               "('O=C(OCC1OC(NC2=N[C@H](C(=O)O)[C@H](N2)[C@H](O)CN)C(NC(=O)C[C@@H](NC(=O)C)CCCN)C(C1O)O)N', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('C[C@@H]1O[C@@H](O[C@@H]2[C@@H](O)[C@H](C)O[C@@H](O[C@@H]3[C@@H](CO[C@@H]4OC[C@@H](O)[C@H](O)[C@H]4O)O[C@@H](O[C@H]4CC[C@@]5(C)[C@@H](CC[C@]6(C)[C@@H]5CC=C5[C@@H]7CC(C)(C)CC[C@@]7([C@H](O)C[C@@]65C)C(O)=O)C4(C)C)[C@H](O)[C@H]3O)[C@@H]2O)[C@H](O)[C@H](O)[C@H]1O', "
               "'Found 1 carboxylic acid group(s) and 12 hydroxy group(s)'), "
               "('OC(=O)CCc1cc2c(-c3cc(ccc3C(O)=O)C(O)=O)c3cc(CCC(O)=O)c(=O)cc3oc2cc1O', "
               "'Found 4 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('CC=CC1=CN=C2C(=C1)C(=O)N(C[C@H]([C@@H](O2)CN(C)CC3=CC=CC(=C3)C(=O)O)C)[C@@H](C)CO', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('[C@@H]1([C@@H]([C@H]([C@H](O[C@@H]2O[C@@H]([C@@H]([C@@H]([C@H]2O)O[C@@H]3O[C@@H]([C@@H]([C@@H]([C@H]3O)O[C@@H]4O[C@@H]([C@H]([C@@H]([C@H]4O)O)O[C@H]5O[C@@H]([C@H]([C@@H]([C@H]5NS(=O)(O)=O)OS(O)(=O)=O)O[C@@H]6O[C@@H]([C@@H]([C@H]([C@@H]6OS(=O)(=O)O)O)O[C@H]7O[C@@H]([C@H]([C@@H]([C@H]7N)O)O)CO)C(O)=O)CO)C(O)=O)O)CO)O)CO)CO1)O)O)*', "
               "'Found 2 carboxylic acid group(s) and 17 hydroxy group(s)'), "
               "('O[C@@H]([C@H](NC(=O)[C@@H](N)CC1=CC=CC=C1)C(=O)N[C@@H]([C@H](CC)C)C(O)=O)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O([C@H]1[C@@H](O)[C@H](O[C@@H]2O[C@@H]([C@@H](O[C@@H]3O[C@@H]([C@H](O)[C@H](O)[C@H]3O)CO)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O[C@@H]1O[C@H]4[C@H](O)[C@H](O[C@@H](O[C@H]5[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]5CO)O)[C@H]4O)CO[C@H]6O[C@@H]([C@@H](O)[C@H](O)[C@@H]6O[C@@H]7O[C@@H]([C@@H](O[C@@H]8O[C@@H]([C@H](O)[C@H](O)[C@H]8O)CO[C@]9(O[C@H]([C@H](NC(=O)C)[C@@H](O)C9)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]7NC(=O)C)CO)CO)CO)[C@@H]%10O[C@@H]([C@@H](O[C@@H]%11O[C@@H]([C@H](O)[C@H](O)[C@H]%11O)CO[C@]%12(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%12)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%10NC(=O)C)CO', "
               "'Found 2 carboxylic acid group(s) and 36 hydroxy group(s)'), "
               "('N[C@@H](CCCC(=O)S[*])C(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('S(CC[C@H](NC(=O)[C@@H](N)CCCCN)C(=O)N[C@@H]([C@H](O)C)C(O)=O)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(N[C@@H](CC1=CC=CC=C1)C(O)=O)[C@@H](NC(=O)[C@@H](N)C(C)C)C(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('CCCCCCC[C@@H](C\\\\C=C\\\\CCC(O)=O)OC', 'Found 1 carboxylic "
               "acid group(s) and 1 hydroxy group(s)'), "
               "('C([C@H](CC1=CC=C(C=C1)O)NC(C)=O)(=O)O', 'Found 1 carboxylic "
               "acid group(s) and 1 hydroxy group(s)'), "
               "('CC1(C)N[C@@H](CS1)C(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 1 hydroxy group(s)'), "
               "('O1[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@@H](O)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O[C@H]3O[C@@H]([C@@H](O)[C@H](O)[C@@H]3O[C@@H]4O[C@@H]([C@@H](O[C@@H]5O[C@@H]([C@H](O)[C@H](O)[C@H]5O)CO[C@]6(O[C@H]([C@H](NC(=O)C)[C@@H](O)C6)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]4NC(=O)C)CO)CO)[C@H](O)[C@@H]1O[C@H]7[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]7CO)O[C@H]8[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]8CO[C@@H]9O[C@H]([C@@H](O)[C@@H](O)[C@@H]9O)C)O)CO[C@H]%10O[C@@H]([C@@H](O)[C@H](O)[C@@H]%10O[C@@H]%11O[C@@H]([C@@H](O[C@@H]%12O[C@@H]([C@H](O)[C@H](O)[C@H]%12O)CO[C@]%13(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%13)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%11NC(=O)C)CO)CO', "
               "'Found 2 carboxylic acid group(s) and 37 hydroxy group(s)'), "
               "('O[C@@H](CCCCCCCC)C(O)=O', 'Found 1 carboxylic acid group(s) "
               "and 2 hydroxy group(s)'), "
               "('OCC=1C(C2(C(C(CCC2)(C)C)CC1)C)CCC(COC(=O)C)=CC(O)=O', 'Found "
               "1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(N[C@@H](CC=1NC=NC1)C(O)=O)[C@@H](NC(=O)[C@@H](N)C(C)C)C(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('OC=1C(=C(CCCCCCCCCCCCC)C=CC1)C(O)=O', 'Found 1 carboxylic "
               "acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(NCC(=O)N[C@@H](CC1=CC=CC=C1)C(O)=O)[C@@H](N)CC=2C=3C(NC2)=CC=CC3', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O([C@H]1[C@H](O)[C@H](O[C@@H](O[C@H]2[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]2CO)O[C@@H]([C@H](O)[C@@H](NC(=O)C)CO)[C@H](O)CO[C@@H]3O[C@H]([C@@H](O)[C@@H](O)[C@@H]3O)C)[C@H]1O)CO[C@H]4O[C@@H]([C@@H](O)[C@H](O)[C@@H]4O[C@@H]5O[C@@H]([C@@H](O)[C@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O[C@]7(O[C@H]([C@H](NC(=O)C)[C@@H](O)C7)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]6O)CO)[C@H]5NC(=O)C)CO)CO)[C@H]8O[C@@H]([C@@H](O)[C@H](O)[C@@H]8O[C@@H]9O[C@@H]([C@@H](O)[C@H](O[C@@H]%10O[C@@H]([C@H](O)[C@H](O)[C@H]%10O)CO)[C@H]9NC(=O)C)CO)CO', "
               "'Found 1 carboxylic acid group(s) and 32 hydroxy group(s)'), "
               "('S(CC[C@H](N)C(=O)NCC(=O)N[C@@H](CCC(=O)N)C(O)=O)C', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('C[C@H](C[C@H](N)C(O)=O)C(O)=O', 'Found 2 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('O=C1C([C@H]2[C@](C=3C([C@]4([C@]([C@@H]([C@H](C(=O)O)CCC(=C)C(C)C)[C@@H](C4)O)(C)CC3)C)=C[C@@H]2O)(C)CC1)(C)C', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('O=C1O[C@@H](CCCCCCCCCCCCC[C@@H](O)C)[C@@H](C1=C)C(=O)O', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC(CCCC(O)=O)C(O)=O', 'Found 2 carboxylic acid group(s) and "
               "3 hydroxy group(s)'), ('O(C=1C(OC)=C(O)C=C(C1)C=CC(O)=O)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('SC[C@H](NC(=O)[C@@H](N)CC=1NC=NC1)C(=O)NCC(O)=O', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('S(CC[C@H](NC(=O)[C@H]1NCCC1)C(=O)N[C@@H](CC=2NC=NC2)C(O)=O)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(O)[C@@H]1[C@H](O)[C@H](O)[C@H](NC(=O)C)NC1', 'Found 1 "
               "carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('S(C1=C2NC(=O)C=CC=CC=C[C@@H]([C@@H](O)CC(=O)C(C)=CC[C@@H](C=C[C@@H]([C@@H]([C@H](C=C(C(C=3C(C1=O)=C(C2=O)C=C(C)C3O)=O)C)C)O)C)O)C)CC(NC(=O)C)C(=O)O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('O([C@H]1[C@@H](O)[C@H](O[C@@H]2O[C@@H]([C@@H](O[C@@H]3O[C@@H]([C@H](O)[C@H](O)[C@H]3O)CO)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O[C@@H]1OC[C@H]4O[C@@H](O[C@H]5[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]5CO)O[C@H]6[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]6CO)O)[C@@H](O)[C@@H](O[C@H]7O[C@@H]([C@@H](O)[C@H](O)[C@@H]7O[C@@H]8O[C@@H]([C@@H](O[C@@H]9O[C@@H]([C@H](O)[C@H](O)[C@H]9O)CO)[C@H](O)[C@H]8NC(=O)C)CO)CO)[C@@H]4O)CO)[C@@H]%10O[C@@H]([C@@H](O[C@@H]%11O[C@@H]([C@H](O)[C@H](O)[C@@H]%11O)CO[C@]%12(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%12)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%10NC(=O)C)CO', "
               "'Found 1 carboxylic acid group(s) and 34 hydroxy group(s)'), "
               "('O=C(O)C(C=1C2=C(C=CC=C2)NC1)(C=3C4=C(C=CC=C4)NC3)CC(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('COC1=CC(=CC(OC)=C1OC)C(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('O=C(NC(CC(O)=O)C(O)=O)C(N)CC1=CC=CC=C1', 'Found 2 carboxylic "
               "acid group(s) and 2 hydroxy group(s)'), "
               "('O=C(N[C@@H](C)C(O)=O)[C@@H](NC(=O)[C@@H](N)CCC(O)=O)CCC(O)=O', "
               "'Found 3 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('O(C1CCN(CC1)CCCC(=O)C2=CC=C(C(C)(C)C(O)=O)C=C2)C(C3=CC=CC=C3)C4=CC=CC=C4', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C1N[C@H](C(=O)N[C@H](C(=O)O)CCCNC(=N)N)CC(=O)NCCCC[C@@H]2NC(=O)[C@@H](NC([C@H]([C@H](OC(C[C@H]3C(N[C@H](C(N[C@H]1CCC(=O)OC[C@H](NC(=O)[C@H]4C(C(=O)[C@@H](NC2=O)CC5=CC=CC=C5)CCC4)C(=O)N3)=O)CC(C)C)=O)=O)C)NC(=O)[C@@H](NC(=O)[C@H]6N(C(=O)[C@@H](N)C(C)C)CCC6)CC=7C8=C(C=CC=C8)NC7)=O)CC9=CC=C(O)C=C9', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O[C@@H](CCCCCC)CC#CCCCCCCCC(O)=O', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('CCCCCCCC\\\\C=C/CCCCCCCC(=O)OC[C@H](COP(O)(=O)OC[C@H](N)C(O)=O)OC(=O)CCCCCCC\\\\C=C/CCCCCC', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('C[C@@H]1O[C@@H](OCCCCCC\\\\C=C\\\\C(O)=O)[C@H](O)C[C@H]1O', "
               "'Found 1 carboxylic acid group(s) and 3 hydroxy group(s)'), "
               "('S(O[C@@H]1[C@@H](O)[C@H](O[C@H]2[C@H](O)[C@@H](NC(=O)C)[C@H](O[C@H]3[C@H](O)[C@H](O[C@@H]4O[C@@H]([C@@H](O[C@@H]5O[C@@H]([C@H](O)[C@H](O[C@]6(O[C@H]([C@H](NC(=O)C)[C@@H](O)C6)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]5O)CO)[C@H](O)[C@H]4NC(=O)C)CO)[C@@H](O[C@H]7[C@H](O)[C@H](O[C@@H](O[C@H]8[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]8CO)O[C@H]9[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]9CO[C@@H]%10O[C@H]([C@@H](O)[C@@H](O)[C@@H]%10O)C)O)[C@H]7O)CO[C@H]%11O[C@@H]([C@@H](O)[C@H](O)[C@@H]%11O[C@@H]%12O[C@@H]([C@@H](O[C@@H]%13O[C@@H]([C@H](O)[C@H](O[C@]%14(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%14)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]%13O)CO)[C@H](O)[C@H]%12NC(=O)C)CO)CO[C@@H]%15O[C@@H]([C@@H](O[C@@H]%16O[C@@H]([C@H](O)[C@H](O[C@]%17(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%17)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]%16O)CO)[C@H](O)[C@H]%15NC(=O)C)CO)O[C@@H]3CO)O[C@@H]2CO)O[C@@H]([C@@H]1O)CO)(O)(=O)=O', "
               "'Found 3 carboxylic acid group(s) and 48 hydroxy group(s)'), "
               "('O=C(N[C@@H](CC=1C=2C(NC1)=CC=CC2)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC(O)=O)CC=3NC=NC3', "
               "'Found 2 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('O([C@@H]1O[C@@H]([C@@H](O)[C@H](O[C@@H]2O[C@@H]([C@H](O)[C@H](O[C@]3(O[C@H]([C@H](NC(=O)C)[C@@H](O)C3)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]2O)CO)[C@@H]1NC(=O)C)CO[C@]4(O[C@H]([C@H](NC(=O)C)[C@@H](O)C4)[C@H](O)[C@H](O)CO)C(O)=O)[C@@H]5[C@@H](O[C@@H]6[C@H](O)[C@@H](O[C@@H]([C@H]6O)CO[C@H]7O[C@@H]([C@@H](O)[C@H](O)[C@@H]7O[C@@H]8O[C@@H]([C@@H](O[C@@H]9O[C@@H]([C@H](O)[C@H](O[C@]%10(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%10)[C@H](O)[C@H](O)CO)C(O)=O)[C@H]9O)CO)[C@H](O)[C@H]8NC(=O)C)CO)CO)O[C@H]%11[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]%11CO)O[C@H]%12[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]%12CO[C@@H]%13O[C@H]([C@@H](O)[C@@H](O)[C@@H]%13O)C)O)O[C@@H]([C@@H](O)[C@@H]5O)CO', "
               "'Found 3 carboxylic acid group(s) and 39 hydroxy group(s)'), "
               "('OC(=O)c1cccc2nc3ccccc3nc12', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('O=C(O)CCCCCCCC(O)C(O)CC(O)C(O)CC(O)C(O)C/C=C/CCCCCCC', "
               "'Found 1 carboxylic acid group(s) and 7 hydroxy group(s)'), "
               "('OC(=O)CCCCCCCCC/C=C\\\\CCCCC', 'Found 1 carboxylic acid "
               "group(s) and 1 hydroxy group(s)'), "
               "('[C@@H]1([C@@H]([C@H]([C@@H]([C@H](O1)CO)O)O)NC(=O)C)O[C@H]2[C@@H]([C@H](C(O[C@@H]2CO)O)NC(=O)C)O[C@@H](C(=O)N[C@H](C(=O)N[C@@H](C(=O)O)CCC(N[C@H](C(O)=O)CCCC(N)C(=O)O)=O)C)C', "
               "'Found 3 carboxylic acid group(s) and 8 hydroxy group(s)'), "
               "('CSCC[C@H](NC(=O)[C@@H](N)CC(N)=O)C(=O)N[C@@H](Cc1c[nH]c2ccccc12)C(=O)N[C@@H](CC(N)=O)C(O)=O', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C1[C@@]2(O)[C@H]3O[C@H]3[C@H]([C@@H]1[C@@H](O)[C@H](C2)NC(=O)/C=C/CCCCCCCCC(=O)O)O', "
               "'Found 1 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('OC(=O)CCCCCC/C=C\\\\C/C=C\\\\CCCCCC', 'Found 1 carboxylic "
               "acid group(s) and 1 hydroxy group(s)'), "
               "('CC[C@H](C)[C@H](N)C(=O)N[C@@H](C)C(=O)NC\\\\C=C\\\\[C@@H](Cc1ccccc1)C(=O)N[C@@H](CC[C@H](CN)O[C@@H]1O[C@H](CO)[C@H](O)[C@H](O)[C@H]1O)C(=O)NCC(=O)N[C@@H](CCC(O)=O)C(=O)N[C@@H](CCC(N)=O)C(=O)NCC(=O)N1CCC[C@H]1C(=O)N[C@@H](CCCCN)C(=O)NCC(=O)N[C@@H](CCC(O)=O)C(=O)N[C@@H]([C@@H](C)O)C(O)=O', "
               "'Found 3 carboxylic acid group(s) and 8 hydroxy group(s)'), "
               "('O=C(N[C@@H](CCC(=O)N)C(=O)N[C@@H]([C@H](O)C)C(O)=O)[C@@H](N)CC(C)C', "
               "'Found 1 carboxylic acid group(s) and 2 hydroxy group(s)'), "
               "('OC(=O)CCCC/C=C\\\\C/C=C\\\\CCCCCC(C)C', 'Found 1 carboxylic "
               "acid group(s) and 1 hydroxy group(s)'), "
               "('OC(=O)CCCCCCCCCC/C=C\\\\CO', 'Found 1 carboxylic acid "
               "group(s) and 2 hydroxy group(s)'), "
               "('O=C1OC(=C([C@H](CC=2O[C@@](/C=C/C=C/CC)(C)C(C2C(=O)O)=O)C)C3=C1C(=O)[C@](/C=C/C=C/CC)(C)O3)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(N[C@@H](CCCCN)C(=O)N[C@@H](CC1=CC=C(O)C=C1)C(O)=O)[C@H]2NCCC2', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O([C@H]1[C@H]([C@H](O[C@H]([C@@H]1O)O[C@@H]2[C@H](O[C@@H](OC[C@@H]([C@@H](/C=C/CCCCCCCCCCCCC)O)NC(=O)*)[C@@H]([C@H]2O)O)CO)CO)O)[C@@H]3[C@@H]([C@H]([C@H](O)[C@H](O3)CO)O[C@@H]4O[C@@H]([C@@H]([C@@H]([C@H]4O)O[C@]5(O[C@]([C@@H]([C@H](C5)O)NC(C)=O)([C@@H]([C@@H](CO)O)O)[H])C(O)=O)O[C@H]6[C@@H]([C@H]([C@@H](O)[C@H](O6)CO)O)NC(=O)C)CO)NC(C)=O', "
               "'Found 1 carboxylic acid group(s) and 19 hydroxy group(s)'), "
               "('O=C1OC(C(=O)N[C@H](C(=O)N[C@H](C(=O)O)[C@H](CC)C)C(C)C)C1C(C)C', "
               "'Found 1 carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('O=C(O)C1=C2N[C@H](C[C@@H](C2=CC=C1)OC)C', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('Cc1cc(O)c(O)c(Oc2cc(C)c(C(O)=O)c(O)c2)c1', 'Found 1 "
               "carboxylic acid group(s) and 1 hydroxy group(s)'), "
               "('OC(=O)\\\\C=C\\\\C(O)=O.[H]C(=O)Nc1cc(ccc1O)[C@@H](O)CN[C@H](C)Cc1ccc(OC)cc1.[H]C(=O)Nc1cc(ccc1O)[C@@H](O)CN[C@H](C)Cc1ccc(OC)cc1', "
               "'Found 2 carboxylic acid group(s) and 4 hydroxy group(s)'), "
               "('O([C@H]1O[C@@H]([C@@H](O[C@@H]2O[C@@H]([C@@H](O[C@@H]3O[C@@H]([C@H](O)[C@H](O)[C@H]3O)CO[C@]4(O[C@H]([C@H](NC(=O)C)[C@@H](O)C4)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]2NC(=O)C)CO)[C@H](O)[C@@H]1O[C@@H]5O[C@@H]([C@@H](O[C@@H]6O[C@@H]([C@H](O)[C@H](O)[C@H]6O)CO[C@]7(O[C@H]([C@H](NC(=O)C)[C@@H](O)C7)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@@H]5NC(=O)C)CO)CO)[C@H]8[C@H](O)[C@H](O[C@@H](O[C@H]9[C@H](O)[C@@H](NC(=O)C)[C@@H](O[C@@H]9CO)O[C@H]%10[C@H](O)[C@@H](NC(=O)C)C(O[C@@H]%10CO[C@@H]%11O[C@H]([C@@H](O)[C@@H](O)[C@@H]%11O)C)O)[C@H]8O)CO[C@H]%12O[C@@H]([C@@H](O)[C@H](O)[C@@H]%12O[C@@H]%13O[C@@H]([C@@H](O[C@@H]%14O[C@@H]([C@H](O)[C@H](O)[C@H]%14O)CO[C@]%15(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%15)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%13NC(=O)C)CO)CO[C@@H]%16O[C@@H]([C@@H](O[C@@H]%17O[C@@H]([C@H](O)[C@H](O)[C@H]%17O)CO[C@]%18(O[C@H]([C@H](NC(=O)C)[C@@H](O)C%18)[C@H](O)[C@H](O)CO)C(O)=O)[C@H](O)[C@H]%16NC(=O)C)CO', "
               "'Found 4 carboxylic acid group(s) and 53 hydroxy group(s)')]\n"
               "False negatives: [('ClC/C=C/C(=O)OC[C@@H](O)CC(=O)OC', 'No "
               "carboxylic acid group found'), ('O(CCCC)C(=O)C(C(O)C)=C', 'No "
               "carboxylic acid group found'), "
               "('CN1[C@H]2CC(CC1C3[C@@H]2O3)OC(=O)[C@H](CO)C4=CC=CC=C4', 'No "
               "carboxylic acid group found'), "
               "('CCCC[N+]1([C@@H]2CC(C[C@H]1[C@H]3[C@H]2O3)OC(=O)[C@H](CO)C4=CC=CC=C4)C', "
               "'No carboxylic acid group found'), "
               "('[C@@]1(C(OC)=O)(C[C@H]([C@H](O)[C@@H](C1)O)OC(=O)\\\\C=C\\\\C2=CC(=C(C=C2)O)O)O', "
               "'No carboxylic acid group found'), "
               "('C[N+]1(C)[C@@H]2[C@H]3[C@@H]([C@H]1C[C@H](C2)OC([C@H](CO)C4=CC=CC=C4)=O)O3', "
               "'No carboxylic acid group found')]",
    'attempt': 1,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 130,
    'num_false_positives': 100,
    'num_true_negatives': 254,
    'num_false_negatives': 0,
    'num_negatives': None,
    'precision': 0.5652173913043478,
    'recall': 1.0,
    'f1': 0.7222222222222222,
    'accuracy': 0.7933884297520661}