"""
Classifies: CHEBI:33566 catechols
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_catechols(smiles: str):
    """
    Determines if a molecule contains an o-diphenol (catechol) component.

    Args:
        smiles (str): SMILES string of the molecule

    Returns:
        bool: True if molecule contains a catechol component, False otherwise
        str: Reason for classification
    """
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return False, "Invalid SMILES string"

    # Find all rings in the molecule
    rings = mol.GetRingInfo().AtomRings()

    # Check if any ring has two adjacent hydroxyl groups
    for ring in rings:
        for i in range(len(ring)):
            atom1 = mol.GetAtomWithIdx(ring[i])
            atom2 = mol.GetAtomWithIdx(ring[(i + 1) % len(ring)])
            if atom1.GetAtomicNum() == 8 and atom2.GetAtomicNum() == 8:
                neighbors1 = [mol.GetAtomWithIdx(neighbor.GetIdx()) for neighbor in atom1.GetNeighbors()]
                neighbors2 = [mol.GetAtomWithIdx(neighbor.GetIdx()) for neighbor in atom2.GetNeighbors()]
                common_neighbor = list(set(neighbors1) & set(neighbors2))
                if common_neighbor and common_neighbor[0].GetAtomicNum() == 6:
                    smarts_pattern = Chem.MolFromSmarts('c1ccc(O)c(O)c1')
                    if mol.HasSubstructMatch(smarts_pattern):
                        return True, "Molecule contains a catechol (o-diphenol) ring system"

    # Check for isolated catechol fragments
    smarts_pattern = Chem.MolFromSmarts('c1ccc(O)c(O)c1')
    if mol.HasSubstructMatch(smarts_pattern):
        return True, "Molecule contains a catechol (o-diphenol) fragment"

    return False, "Molecule does not contain a catechol (o-diphenol) component"


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:33566',
                          'name': 'catechols',
                          'definition': 'Any compound containing an o-diphenol '
                                        'component.',
                          'parents': ['CHEBI:33570']},
    'config': {   'llm_model_name': 'claude-3-sonnet',
                  'f1_threshold': 0.0,
                  'max_attempts': 5,
                  'max_negative_to_test': None,
                  'max_positive_in_prompt': 50,
                  'max_negative_in_prompt': 20,
                  'max_instances_in_prompt': 100,
                  'test_proportion': 0.1},
    'message': '\n'
               'Attempt failed: F1 score of 0 is too low.\n'
               'True positives: []\n'
               'False positives: '
               "[('C[C@@H]1[C@]2([C@]34[C@](O[C@@H]1O)(O[C@](OO3)(CC[C@]4([C@@H](CC2)C)[H])C)[H])[H]', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCC[C@@H](/C=C/C=C\\\\C/C=C\\\\C=C\\\\[C@H](CCCCC)OO)O)(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CC\\\\C=C/C[C@H](OO)\\\\C=C\\\\C=C/CCCCCCCC(O)=O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@@]23C=C[C@]14C(=CC[C@]5([C@H]4CC[C@@H]5[C@@H](/C=C/[C@@H](C(C)C)C)C)C)[C@]2(CC[C@@H](C3)O)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C1=CC(=CC=C1C2=CC=C(C=C2)NN=C3C=CC(=O)C(=C3)C(=O)O)NN=C4C=CC(=O)C(=C4)COO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[H][C@@]12CC(=O)O[C@]1(C)C[C@](C)(CCCCCCCC\\\\C=C\\\\C=C\\\\c1ccccc1)OO2', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C1C([C@@]23OO[C@]4([C@]5([C@]([C@@H]([C@H](C(=O)O)CCC=C(C)C)[C@@H](C5)O)(C)CC=C4[C@@]2(C)CC1)C)C=C3)(C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('OC(=O)C(O)=O.C[C@@H]1CC[C@H]2[C@@H](C)[C@@H](OCCN)O[C@@H]3O[C@@]4(C)CC[C@@H]1[C@@]23OO4', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C1O[C@@H](C=C(CC=CC)C)CC=CC=CC[C@H](C)[C@@H](C([C@H](CC=C[C@H](C1)O[C@@H]2O[C@H](OC)[C@@H](O)[C@@H]([C@H]2O)OC)OC)=C)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/C\\\\C=C/C[C@@H](OO)\\\\C=C\\\\C=C/CCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(C([O-])=O)C/C=C\\\\C[C@@H](/C=C/C=C\\\\C\\\\C=C/C=C/[C@H](C/C=C\\\\CC)OO)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCC[C@@H](/C=C/C=C\\\\C/C=C\\\\C=C\\\\[C@H](CCCCC)OO)OO)([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/C[C@@H](OO)\\\\C=C\\\\C=C/CCCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CC/C=C\\\\C[C@H](OO)/C=C/C=C\\\\C=C\\\\[C@H](C/C=C\\\\C/C=C\\\\CC)OO)(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C([C@@H]([C@H](O)[C@@]1([C@@H](/C(=C/[C@@]2(O[C@H](C)[C@@]3([C@H]2O3)C)C)/C)[C@]4([C@@H]5OO[C@]([C@@H]14)(C=C5C)C)C)C)C)C(C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[H][C@]12CC(=O)O[C@@]1(C)C[C@](C)(CCCCCCCCCCCCc1ccccc1)OO2', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)C(CCCCCCC\\\\C=C\\\\C=C\\\\C(=O)N1CCCCC1)/C=C/CCCCC', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('SC[C@@]12OO[C@@H]([C@H]1[C@H](OC(=O)C)C)[C@@H](C=C2)O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCCCCCC(/C=C/C1C(CCCCC)O1)OO)(=O)O', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('P(=O)([O-])(OCC[NH3+])OC[C@@H](CO)OC(CCC/C=C\\\\C[C@H]1[C@@H]2C[C@H]([C@@H]1/C=C/[C@H](CCCCC)O)OO2)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('N(CCCl)(CCCl)[P@@]1(=O)N[C@H](OO)CCO1', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('C(=C/[C@H](C/C=C\\\\CCCCC)OO)\\\\C=C/C/C=C\\\\CCCC(=O)NCCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)C12C(C3C(C4C(C(CC4)C(CCCC(C)C)C)(CC3)C)CC1OO)(CCC(O)C2)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(C[C@H]1[C@@H]2C[C@H]([C@@H]1/C=C/[C@H](CCCCC)OO)OO2)CCCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1OC(CC1C(OO)/C=C/C(OO)CCCCC)/C=C/C=C/CCCC(OC)=O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('[H]C1(CC2(O)OOC3([H])CC4([H])CC(O)CCC4(C)C4([H])CCC1(C)C2C34)C(C)CCCC(C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCCCCC[C@@H](/C=C/C=C\\\\C=C\\\\[C@H](CC)OO)OO)(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(C/C=C\\\\C=C\\\\[C@H](CCCCC)OO)CCCCCCCC(=O)O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O=C1OO[C@@]2([C@H]([C@@H](O)C=C(CO)[C@@H]3[C@](CC2)(O)C(C3)(C)C)[C@@H]4C5=C(C1=C(O)C=C5C[C@@H](CCCCCC)O4)C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C(O)C1=CC=2OO[C@@](C2C=C1)(CCCC(O)(C)C)C', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O(O)[C@H](\\\\C=C/1\\\\[C@]2([C@@]([C@](CC2)([C@@H](CCCC(C)C)C)[H])(CCC1)C)[H])C=3C(CC[C@H](O)C3)=C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('OOC(=O)c1cccc(Cl)c1', 'Molecule contains a catechol "
               "(o-diphenol) fragment'), "
               "('CC\\\\C=C/C\\\\C=C/C\\\\C=C/CCCCCC[C@@H](OO)C([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)C(C/C=C\\\\CCCCCCCC(O)=O)/C=C/C=C/CC', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O(O)[C@]12C(=CC(C1)(C)C)[C@H]([C@@H](CO)CC[C@@H]2C)CO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C1C[C@]2(CC[C@@]1(CC(NCC(N)(C)C)=O)[H])OC3(OO2)C4CC5CC3CC(C5)C4.C(=C\\\\C(=O)O)\\\\C(=O)O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O[C@@H]1CC=2[C@@]([C@@]3([C@]([C@]4([C@@]([C@](CC4)([C@@H](CCC(OO)(C(C)C)C=C)C)[H])(CC3)C)[H])(CC2)[H])[H])(CC1)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[C@@H](C(O)=O)(*)OO', 'Molecule contains a catechol "
               "(o-diphenol) fragment'), "
               "('C(CCC(/C=C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCC(=O)[O-])OO)CC', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C1=C(C=C(C(=C1O)O)SC[C@@H](C(=O)O)N)CCOO', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('P(OC[C@H](OO)COC(=O)CCC/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCCC)(O)(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCCCCC[C@H](OO)\\\\C=C\\\\C=C/C\\\\C=C/CCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(=C/C=C/[C@H](CCCCC)OO)/C=C/[C@H](C/C=C\\\\CCCC([O-])=O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1OC(CC1C2OOC(C2)/C=C/C=C/CCCCC)C(OO)CCCC(OC)=O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@@H](C[C@@]1([C@H](O)CC)[H])/C=C/C=C\\\\CCCCCCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/C\\\\C=C/C=C/[C@@H](C\\\\C=C/CCCC(O)=O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[C@@H]1(C[C@@H](C2=C(C1)[C@](OOC2)(/C=C/3\\\\CCC[C@]4([C@]3(CC[C@]4([H])[C@](CCCC(C)C)([H])C)[H])C)[H])O)O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(C([O-])=O)CCCCC(/C=C/C=C\\\\C/C=C\\\\C=C\\\\C(C/C=C\\\\CC)OO)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C(O)[C@@H]([C@@H]1[C@@]2([C@@]([C@]34OO[C@@]5(C([C@@H](O)CC[C@@]5(C3=CC2)C)(C)C)C=C4)(C)C[C@H]1O)C)CCC(=C)C(C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('OC(CC/C=C\\\\CC/C=C/C=C\\\\C=C\\\\[C@H](C/C=C\\\\C/C=C\\\\CC)OO)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[H][C@@]1(C[C@@H]2OO[C@H]1C=C2C)[C@@H](C)CCC=C(C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)C(CCCCCCC)/C=C/C#CC#CC(O)C=C', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('P(=O)([O-])(OCC[N+](C)(C)C)OC[C@@H](CO)OC(CCC/C=C\\\\C[C@H]1[C@@H]2C[C@H]([C@@H]1/C=C/[C@H](CCCCC)OO)OO2)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@]23[C@@]([C@@]14C(=C5[C@@]([C@](CC5)([C@@H](CCCC(C)C)C)[H])(CC4)C)C=C2)(CC[C@H](O)C3)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('COC(=O)N[C@@H](C(Cc1c(C)nc2n(C)c3n(cnc3c(=O)n12)[C@@H]1O[C@H](CO)[C@@H](O)[C@H]1O)OO)C(=O)OC', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CC(=O)OC(C)(C)[C@H]1CC[C@@](C)(OO)C=C1', 'Molecule contains "
               "a catechol (o-diphenol) fragment'), "
               "('CCCCCCC\\\\C=C\\\\C(CCCCCCCC(O)=O)OO', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('O=C1O[C@@H](/C=C(\\\\C/C=C/C)/C)C[C@@H]2OO[C@@H](C[C@H](C)C=C([C@H](CC=C[C@H](C1)O[C@@H]3O[C@H](OC)[C@@H](O)[C@@H]([C@H]3O)OC)OC)C)C=C2', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[C@@H]1([C@@H]2C[C@H]([C@@H]1/C=C/[C@H](C/C=C\\\\CC)OO)OO2)C/C=C\\\\CCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/C\\\\C=C/C=C/[C@@H](C\\\\C=C/CCCC([O-])=O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCC)C[C@@H](/C=C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCC(N[C@H](C(O)=O)C)=O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)[C@H]1C=2[C@]3([C@@]([C@](CC3)([C@@H](CCCC(C)C)C)[H])(CCC2[C@@]4(C(=C1)C[C@@H](O)CC4)C)C)[H]', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('P(=O)(O)(OCCN)OC[C@@H](CO)OC(CCC/C=C\\\\C[C@H]1[C@@H]2C[C@H]([C@@H]1/C=C/[C@H](CCCCC)O)OO2)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@]2([C@@H]([C@@H]([C@@]1(C2)[H])CCC(O)=O)/C=C/[C@@H](OO)C/C=C\\\\C/C=C\\\\C/C=C\\\\CC)[H]', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1OC(CC1C2OOC(C2)/C=C/C=C/C/C=C\\\\C/C=C\\\\CC)C(OO)CCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CC\\\\C=C/C\\\\C=C/CC(OO)\\\\C=C\\\\C=C/C\\\\C=C/CCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C1C2=C(O[C@@]3(C[C@@H](O[C@@H]3C2)/C=C(/C/C=C/C(OO)(C)C)\\\\C)C)[C@@H](O)CC1', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C1C2=CC(=O)CC[C@@]2([C@H]3CC[C@]4([C@@](C3=C1)(OO)CC[C@@H]4[C@@H](/C=C/[C@@H](C(C)C)C)C)C)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC[C@H](OO)\\\\C=C\\\\C=C/CCCCCCCC(O)=O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('CC(C)[C@H]1CC[C@]2(C)C[C@@H]3OO[C@]12[C@@H](O)C=C3C=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCCCC/C=C/[C@@H](C/C=C\\\\C/C=C\\\\CC)OO)(=O)[O-]', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@@]23C=C[C@]14C(=CC[C@]5([C@H]4CC[C@@H]5[C@@H](/C=C/[C@@H](C(C)C)CC)C)C)[C@]2(CC[C@@H](C3)O)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@]2([C@@H]([C@H]([C@@]1(C2)[H])C/C=C\\\\CCCC(=O)NCCC)/C=C/[C@@H](O)CCCCC)[H]', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@H]2[C@H](O)CC[C@]3([C@@]1(C=CCC3(C)C)C)C2', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('C(C([O-])=O)C/C=C\\\\CC(/C=C/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CC)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1O[C@@]23C=C[C@]14C(CC[C@]5(C4CC[C@@H]5[C@@H](/C=C/[C@@H](C(C)C)C)C)C)[C@]2(CC[C@@H](C3)O)C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('[H][C@]12CC(=C)[C@H](CC[C@@](C)(OC(=O)CCC)[C@]([H])(O1)[C@]1([H])[C@]3([H])C[C@@]([H])(OC[C@@H]3C)[C@](C)(O)[C@]21[H])OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/C[C@@H](OO)\\\\C=C\\\\C=C/C\\\\C=C/CCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCC\\\\C=C/CC(OO)\\\\C=C\\\\CCCCCCC(O)=O', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('O(O)C(CCCC(O)=O)/C=C/C=C/C=C/C(OO)C/C=C\\\\CCCCC', 'Molecule "
               "contains a catechol (o-diphenol) fragment'), "
               "('C=1(C=CC(=CC1)O)C=2C=NC(=C(N2)CC=3C=CC=CC3)NC4(CC5=CC=C(C=C5)O)C(OO4)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(\\\\C=C/C=C\\\\C(C/C=C\\\\CCCCCC(O)=O)OO)=C/C(C/C=C\\\\CC)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O1OCC2=C(O)C=CC=C2C=CC1C(O)/C=C/C', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('[C@@H](CC)(/C=C/C=C\\\\C/C=C\\\\C/C=C\\\\C/C=C\\\\CCCC(=O)O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('N/C=C(\\\\C(OO)=O)/C', 'Molecule contains a catechol "
               "(o-diphenol) fragment'), "
               "('O1OC2(O[C@]3(OC(=O)[C@@H]([C@]4([C@@]13[C@]([C@@H](CC4)C)(CC2)[H])[H])C)[H])C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CCCCCCC\\\\C=C\\\\C(CCCCCCCC([O-])=O)OO', 'Molecule contains "
               "a catechol (o-diphenol) fragment'), "
               "('[H][C@]12C[C@@]1([H])[C@@]1(C)[C@@]([H])(CC3=C(CO)C(=O)O[C@@]33C4=C5[C@@](C)([C@@H](O)C(=O)C\\\\4=C(\\\\C)C(=O)OC)[C@]4([H])C[C@]4([H])[C@]5(C[C@@]13[H])OO)[C@]2(O)COC(=O)C(\\\\C)=C\\\\C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCCCCCC)C[C@H](C([O-])=O)OO', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('CC\\\\C=C/C\\\\C=C/C[C@H](OO)\\\\C=C\\\\C=C/C\\\\C=C/C\\\\C=C/CCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O(O)[C@H](\\\\C=C/1\\\\[C@]2([C@@]([C@](CC2)([C@H](C)/C=C/[C@@H](C(C)C)C)[H])(CCC1)C)[H])C=3C(CC[C@H](O)C3)=C', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCCCCCCCC)C[C@H](C([O-])=O)OO', 'Molecule contains a "
               "catechol (o-diphenol) fragment'), "
               "('CCCCC[C@H](OO)\\\\C=C\\\\C=C/C\\\\C=C/C\\\\C=C/CCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(CCCC\\\\C=C\\\\C(C/C=C\\\\CCCCC)OO)CCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('CC(C)C12CCC(C)(OO1)C=C2', 'Molecule contains a catechol "
               "(o-diphenol) fragment'), "
               "('C(\\\\CC)=C\\\\C/C=C\\\\C/C=C\\\\C=C\\\\C(C/C=C\\\\CCCC(=O)O)OO', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C(=C\\\\[C@@H](/C=C\\\\CCCCC)OO)\\\\CCCCCCCC(=O)O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('O=C(OOC(=O)c1ccccc1)c1ccccc1', 'Molecule contains a catechol "
               "(o-diphenol) fragment'), "
               "('C1[C@H]2[C@@H]([C@H]([C@@H]1OO2)/C=C/[C@H](CCCC(C)O)O)CCCCCCC([O-])=O', "
               "'Molecule contains a catechol (o-diphenol) fragment'), "
               "('C1[C@H]2[C@@H]([C@H]([C@@H]1OO2)/C=C/[C@H](CCCC(C)O)O)CCCCCCC(O)=O', "
               "'Molecule contains a catechol (o-diphenol) fragment')]\n"
               "False negatives: [('CC(=O)c1ccc(O)c(O)c1', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('OC[C@H]1O[C@@H](OC(=O)\\\\C=C\\\\c2ccc(O)c(O)c2)[C@H](O)[C@@H](O)[C@@H]1O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('Oc1ccc(cc1O)S(O)(=O)=O', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('Cc1cc(O)c(O)c(Oc2cc(C)cc(O)c2O)c1', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('COc1cc(C=CCO)cc(O)c1O', 'Molecule does not contain a "
               "catechol (o-diphenol) component'), ('Oc1cccc(F)c1O', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('O(C(CC1=CC(O)=C(O)C=C1)C(O)=O)C(=O)/C=C/C2=CC(O/C(=C\\\\C3=CC(O)=C(O)C=C3)/C(O)=O)=C(O)C=C2', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('C=1(C=CC(=CC1O)CCNC(CCCCCCC/C=C\\\\CCCCCCCC)=O)O', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('O=C1OC(=CC(C[C@@]1(O)C)=O)/C=C/C2=CC(O)=C(O)C=C2', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('CC(C)=CCC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\CC\\\\C(C)=C\\\\Cc1cccc(O)c1O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('O[C@@H]1[C@@H](COC(=O)\\\\C=C\\\\c2ccc(O)cc2)O[C@@H](OCCc2ccc(O)c(O)c2)[C@H](O)[C@H]1O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('OC1=C(O)C=CC=C1CCCCCCCCCCCCCCC2=C(O)C(O)=CC=C2', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('COc1cc(-c2ccc(O)cc2)c(OC)c(O)c1-c1ccc(O)c(O)c1', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('Cc1ccc(O)c(O)c1', 'Molecule does not contain a catechol "
               "(o-diphenol) component'), "
               "('C1(=C(C=C(C=C1)CNCCC2=CC=C(C=C2)O)O)O', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('[H][C@]1(O[C@H](CO)[C@@H](O)[C@H](O)[C@H]1O)O[C@H]1[C@H](O)[C@@H](CO)O[C@@H](OCCc2ccc(O)c(O)c2)[C@@H]1O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('Oc1ccc-2c(OCC(=O)Cc3cc(O)c(O)cc-23)c1', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('CC(=O)O[C@@H](CCCCc1ccc(O)cc1)CCc1ccc(O)c(O)c1', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('Oc1ccc(C=C)cc1O', 'Molecule does not contain a catechol "
               "(o-diphenol) component'), "
               "('O=C1OC(=CC(=C1C(C=2C(=O)OC(/C=C/C3=CC(O)=C(O)C=C3)=CC2O)CCC(=O)O)O)/C=C/C4=CC(O)=C(O)C=C4', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('OC(=O)CCc1ccc(O)c(O)c1', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), ('Oc1ccc(F)cc1O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('Oc1cc(Cl)cc(Cl)c1O', 'Molecule does not contain "
               "a catechol (o-diphenol) component'), "
               "('C1=C(C(C)C)C(=C(C2=C1C(C[C@@]3([C@@]2(CCCC3(C)C)C)[H])=O)O)O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('Cc1ccc(O)c(O)c1CCC1[C@@H]2CCC(O)[C@@]2(C)CCC1=O', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('Cc1cc2oc3cc(O)c(O)c(C(O)=O)c3c(=O)c2cn1', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('Cc1cc(O)c(Oc2cc(C)cc(O)c2O)c(O)c1', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('C=1(C=CC=CC1C(=O)/C=C/C2=CC(=C(C=C2)O)O)O', 'Molecule does "
               "not contain a catechol (o-diphenol) component'), "
               "('O=C1OC(=CC(=C1C2=C(C=3OC(=CC(C3)=O)C)C=C(O)C(=C2)O)O)/C=C/C4=CC(O)=C(O)C=C4', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('C=1(C=C(C(=CC1)O)OC)C(O)CN', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('C1CNCC(C2=C1C=C(C(=C2)O)O)C3=CC=CC=C3', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('O=C(NCCC(O)=O)C1=CC(O)=C(O)C=C1', 'Molecule does not contain "
               "a catechol (o-diphenol) component'), "
               "('Oc1ccc(\\\\C=C2\\\\C=C(OC2=O)c2ccc(O)c(O)c2)cc1O', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('COc1cc(-c2ccc(O)c(CC=C(C)C)c2)c(OC)c(O)c1-c1ccc(O)c(O)c1', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('O[C@@H]([C@@H](O)[C@@](O)(C(=O)/C=C/C1=CC(O)=C(O)C=C1)C(O)=O)[C@@](O)(C(=O)/C=C/C2=CC(O)=C(O)C=C2)C(O)=O', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), "
               "('O=C1OC[C@@H](C)[C@@H]([C@@H]1O)C2=C(O)C(O)=C(C)C=C2', "
               "'Molecule does not contain a catechol (o-diphenol) "
               "component'), ('Oc1cccc(-c2ccc(Cl)cc2)c1O', 'Molecule does not "
               "contain a catechol (o-diphenol) component'), "
               "('OC1=CC(\\\\C=C\\\\C2=CC=C(O)C(O)=C2)=CC(O)=C1', 'Molecule "
               "does not contain a catechol (o-diphenol) component'), "
               "('NCC(O)c1ccc(O)c(O)c1', 'Molecule does not contain a catechol "
               "(o-diphenol) component')]",
    'attempt': 2,
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 39,
    'num_false_positives': 100,
    'num_true_negatives': 1721,
    'num_false_negatives': 0,
    'num_negatives': None,
    'precision': 0.2805755395683453,
    'recall': 1.0,
    'f1': 0.4382022471910112,
    'accuracy': 0.946236559139785}