# AUTOGENERATED! DO NOT EDIT! File to edit: 03_hbn_puf.ipynb (unless otherwise specified).

__all__ = ['bool_diff', 'plot_lya', 'plot_mu', 'puf_statistics', 'bool_lyapunov', 'lya_fit', 'booleanize', 'analyze',
           'BooleanNetworkEnsemble']

# Cell
import warnings
with warnings.catch_warnings(): #ignore warnings
    warnings.simplefilter("ignore")
    import networkx as nx
    import numpy as np
    import sidis
    rng=sidis.RNG(0)
    import matplotlib.pyplot as plt
    import typing
    from typing import Optional, Tuple, Dict, Callable, Union, Mapping, Sequence, Iterable, Hashable, List, Any
    from collections import namedtuple
    import einops
    import numba
    from numba import njit
    from scipy import stats
    import scipy.optimize as sciopt
    import matplotlib as mpl
    from .graph_functions import *
    from .model_functions import *
    from .network_class import *

# Cell
def bool_diff(comparisons):
    '''
    Compare timeseries to the 0th entry
    '''
    res=[]
    base=comparisons[0]
    for i,x in enumerate(comparisons):
        res+=[x-base]

# Cell
def plot_lya(data,lstart=0,lend=15,g=0,show=True,err=True,lambda_xy=(10,-4.5),XY=(5,3),
             fname=None#r'paper/LyapunovExponent.pdf'
            ):
    mpl.rcParams['figure.figsize'] = [XY[0],XY[1]]
    T=5
    log_dist=data['log_dist'][...,0]
    err_dist=data['log_dist'][...,1]
    lya=data['lyapunov'][...,0]
    dlya=data['lyapunov'][...,1]
    for i in range(log_dist.shape[0]):
        avgln=log_dist[i].mean(axis=0)
        sigmaln=err_dist[i].mean(axis=0)
        xdata=np.arange(lstart,lend+1)
        ydata=np.asarray(avgln[lstart:lend+1])
        sdata=np.asarray(sigmaln[lstart:lend+1])
        m=lya[i].mean(axis=0)
        m_sigma=dlya[i].mean(axis=0)
        b=data['intercept'][i].mean(axis=0)

        if err:
            plt.errorbar(np.arange(len(avgln)), np.asarray(avgln), xerr=0, yerr=np.asarray(sigmaln),
                         fmt="k-", elinewidth=0.2, label='Model Data')
        if not err:
            plt.plot(np.arange(len(avgln)), np.asarray(avgln), color="black", label='Model Data')
        plt.plot(xdata,m*xdata+b,"m--",linewidth="2", label=('Linear Fit from t=0 to '+str(lend)))

        #plt.legend([r'y = '+str(np.around(m,decimals=3))+'t + '+str(np.around(b,decimals=3)),r'$<ln(d(t))>$'])
        plt.xlabel(r'Time ($\tau$)')
        plt.ylabel(r'$\langle\logD(t)\rangle$')
        #plt.title('Fit of Lyapunov Exponent '+ r'$\lambda$ = '+str(np.around(m,decimals=3))+\
        #    r'$\pm$'+str(np.around(m_sigma,decimals=3)))

    txt=r'$\langle\lambda_{max}\rangle$ = '+f'{np.mean(lya):.2f} $\pm$ {np.mean(dlya):.2f}'+r'$\tau$'
    plt.annotate(xy=lambda_xy, text=txt, color='purple')
    if fname:
        plt.savefig(fname=fname,bbox_inches='tight',dpi=600)
    plt.show()

# Cell
def plot_mu(data,X=7,Y=3,offset=-0.375,ncol=2,nrow=1,alpha=0.2,err_scale=0,
           inter_xy=(1.5, 0.3),intra_xy=(14, 0.1),
            fname=None#r'paper/MuPlot.pdf'
           ):

    muintercolor='blue'
    muintracolor='red'
    subfontsize=12

    import matplotlib as mpl
    mpl.rcParams['axes.titley'] = offset
    mpl.rcParams['axes.titley'] = offset
    #Unpack
    muinters=data['mu_inter'][...,0]
    muintras=data['mu_intra'][...,0]
    dinters=data['mu_inter'][...,1]*err_scale
    dintras=data['mu_intra'][...,1]*err_scale
    deltamus=data['mu_inter'][...,0]-data['mu_intra'][...,0]
    ddeltamus=np.sqrt(np.square(data['mu_inter'][...,1])+np.square(data['mu_inter'][...,1]))*err_scale
    maxmu=np.argmax(np.max(deltamus,axis=1))
    minmu=np.argmin(np.max(deltamus,axis=1))
    time_axis=np.arange(muinters.shape[-1])

    fig, (ax1, ax2) = plt.subplots(nrow, ncol, figsize=[X, Y], sharey=True)

    #ensemble
    for i in range(len(muinters)):
        ax1.errorbar(time_axis, muinters[i], dinters[i], color=muintercolor, alpha=alpha)
        ax1.errorbar(time_axis, muintras[i], dintras[i], color=muintracolor, alpha=alpha)
        #ax1.errorbar(time_axis, muinters[i], 0, color=muintercolor, alpha=alpha)
        #ax1.errorbar(time_axis, muintras[i], 0, color=muintracolor, alpha=alpha)

    #average
    ax1.errorbar(time_axis,np.mean(muinters,axis=0),np.mean(dinters,axis=0),color=muintercolor,linewidth=2)
    ax1.errorbar(time_axis,np.mean(muintras,axis=0),np.mean(dintras,axis=0),color=muintracolor,linewidth=2)

    #text
    #inter_x=np.argwhere(np.mean(muinters,axis=0))
    ax1.annotate(xy=inter_xy, text=r'$\mu_{inter}$', color=muintercolor)
    ax1.annotate(xy=intra_xy, text=r'$\mu_{intra}$', color=muintracolor)

    ax1.set_xlabel(r'Time ($\tau$)')
    ax1.set_ylabel(r'$\mu$')


    #sub figure label
    ax1.set_title('(a)', fontsize=subfontsize, loc='center')

    #ensemble
    for i in range(len(muinters)):
        ax2.errorbar(time_axis, deltamus[i], ddeltamus[i], color='purple', alpha=alpha)
        topt=np.ravel(np.argmax(deltamus[i]))

    #average
    ax2.errorbar(time_axis, np.mean(deltamus,axis=0), np.mean(ddeltamus,axis=0), color='purple', linewidth=2)

    #text
    ax2.annotate(xy=(time_axis[np.argmax(np.mean(deltamus,axis=0))],
                     np.max(deltamus+ddeltamus)*1.1), text=r'$t_{opt}$',ha='center')
    ax2.vlines(ymax=np.max(np.mean(deltamus,axis=0)), ymin=0, x=time_axis[np.argmax(np.mean(deltamus,axis=0))],
               linestyles='dashed', color='black')

    ax2.set_xlabel(r'Time ($\tau$)')
    ax2.set_yticks([0,0.1,0.2,0.3,0.4,0.5])
    ax2.set_ylabel(r'$\Delta\mu$')

    #subplot letter label
    ax2.set_title('(b)', fontsize=subfontsize, loc='center')

    if fname:
        plt.savefig(fname=fname,bbox_inches='tight',dpi=600)

    plt.show()

# Cell
def puf_statistics(responses : np.ndarray,
                   shape : str = 'challenge repeat time instance node'
                  ) -> Tuple[Union[np.ndarray,float,int]]:
    '''
    Given an array of `responses` of a given `shape`, calculate the inter and intra
    PUF statistics mu_inter and mu_intra and their standard deviations over time.
    Return these as arrays over time, as well as their maximum separation mu_opt
    and the time at which this occurs t_opt.
    `shape` must be a permutation of the words 'challenge repeat time instance node'
    and describes the input shape of the array.
    '''
    resp=einops.rearrange(responses,shape+
                          '-> challenge repeat time instance node').astype(float)
    n_distinct=resp.shape[0]
    n_repeat=resp.shape[1]
    measure_time=resp.shape[2]
    n_synth=resp.shape[3]
    n=resp.shape[4]
    #Number of pairwise combinations for inter and intra calculations.
    l_intra=n_repeat*(n_repeat-1)/2
    l_inter=n_synth*(n_synth-1)/2
    #Pairwise differences of timeseries from responses used for comparison.
    pdt_intra=np.zeros((n_distinct,n_synth,int(l_intra),measure_time))
    pdt_inter=np.zeros((n_distinct,n_repeat,int(l_inter),measure_time))
    #Loop over each pairwise combination and form fractional hamming distances at each time.
    for i in range(n_distinct):
        for g in range(n_synth):
            l=0
            for j in range(n_repeat):
                for k in range(j+1,n_repeat):
                    pdt_intra[i,g,l]=np.sum(abs(resp[i,j,:,g,:]-resp[i,k,:,g,:]),axis=-1)/n
                    l+=1
        for g in range(n_repeat):
            l=0
            for j in range(n_synth):
                for k in range(j+1,n_synth):
                    pdt_inter[i,g,l]=np.sum(abs(resp[i,g,:,j,:]-resp[i,g,:,k,:]),axis=-1)/n
                    l+=1
    #Compute means on a per-device and overall level.
    #Intra block below. Copies along axes for quick vector calculations.
    mu_intra_per_device=np.mean(pdt_intra,axis=(0,2))
    mu_intra_per_device_copy=np.repeat(\
        np.repeat(mu_intra_per_device[:,np.newaxis,:],\
            l_intra,axis=1)[np.newaxis,:],n_distinct,axis=0)
    sigma_intra_per_device= np.sqrt(np.mean((np.square(pdt_intra-mu_intra_per_device_copy)),axis=(0,2)))
    mu_intra=np.mean(mu_intra_per_device,axis=0)
    sigma_intra=np.mean(sigma_intra_per_device,axis=0)
    #Inter block below. Copies along axes for quick vector calculations.
    mu_inter_per_device=np.mean(pdt_inter,axis=(0,2))
    mu_inter_per_device_copy=np.repeat(np.repeat(\
        mu_inter_per_device[:,np.newaxis,:],l_inter,axis=1)[np.newaxis,:],\
            n_distinct,axis=0)
    sigma_inter_per_device= np.sqrt(np.mean((np.square(pdt_inter-mu_inter_per_device_copy)),axis=(0,2)))
    mu_inter=np.mean(mu_inter_per_device,axis=0)
    sigma_inter=np.mean(sigma_inter_per_device,axis=0)
    #Find optimum measurement time and save time series.
    t_opt=np.argmin(mu_intra-mu_inter)
    mu_opt=mu_inter[t_opt]-mu_intra[t_opt]
    return mu_inter,mu_intra,sigma_inter,sigma_intra,mu_opt,t_opt

# Cell
def bool_lyapunov(responses : np.ndarray,
                  window : int = 5,
                  shape : str = 'challenge repeat time node'
                 ) -> Tuple[np.ndarray]:
    """
    Boolean timeseries Lyapunov exponent calculator.
    Takes as input the time series of PUF responses
    and calculates the maximum Lyapunov exponent to determine if the system is chaotic.
    Args:
        responses: array of PUF responses for a single instance
        window: Temporal window length for comparison of Boolean distance between time series.
        shape: input shape of PUF responses for a single instance
    """
    T=window
    resp=einops.rearrange(responses,shape+
                          '-> challenge repeat time node').astype(float)
    nrp= resp.shape[1] #number of reps
    nch= resp.shape[0] #number of challenges
    measure_time=resp.shape[2]
    n=resp.shape[-1]
    #We loop over pairwise combinations of timeseries comparisons and compute the
    #average Boolean distance within the window length.
    clist=[]
    for c in range(nch):
        ilist=[]
        for i in range(1,nrp):
            tlist=[]
            for t in range(measure_time-T):
                d=np.sum(abs(resp[c,i,t:t+T]-resp[c,0,t:t+T]))/(n*T)
                if d!=0:
                    first_t=t
                    break
                else:
                    first_t=0
            for t in range(first_t,measure_time-T):
                d=np.sum(abs(resp[c,i,t:t+T]-resp[c,0,t:t+T]))/(n*T)
                tlist+=[d]
            if tlist!=[]:
                ilist+=[tlist]
        if ilist!=[]:
            clist+=[ilist]

    avgln=[0 for t in range(measure_time-T)]
    tcounterlist=[0 for t in range(measure_time-T)]
    longesttime=-1
    for c in range(nch):
        for i in range(len(clist[c])):
            for t in range(len(clist[c][i])):
                tcounterlist[t]+=1
                avgln[t]+=np.log(clist[c][i][t]) if clist[c][i][t]!=0 else np.log(0.01)
                longesttime=t if t>longesttime else longesttime

    avgln=avgln[:longesttime+1]
    tcounterlist=tcounterlist[:longesttime+1]

    for t in range(len(avgln)):
        avgln[t]=avgln[t]/tcounterlist[t]

    sigmaln=[0 for t in range(len(avgln))]
    for c in range(nch):
        for i in range(len(clist[c])):
            for t in range(len(clist[c][i])):
                xi=np.log(clist[c][i][t]) if clist[c][i][t]!=0 else np.log(0.01)
                sigmaln[t]+=(xi-avgln[t])**2

    for t in range(len(avgln)):
        sigmaln[t]=np.sqrt(sigmaln[t]/(tcounterlist[t]-1))

    return avgln,sigmaln

# Cell
def lya_fit(avgln,sigmaln,lstart=0,lend=5,intercept=False):
    """
    Fits average logarithm of boolean distances calculated in lya() function.
    Calculates resulting maximum lyapunov exponent.
    Args:
        lstart: Start of linear fit.
        lend: End of linear fit.
    """
    linearstart=lstart
    linearend=lend
    xdata=np.arange(linearstart,linearend+1)
    ydata=np.asarray(avgln[linearstart:linearend+1])
    sdata=np.asarray(sigmaln[linearstart:linearend+1])
    def lin(x,m,b):
        return m*x+b
    popt, pcov = sciopt.curve_fit(lin, xdata, ydata, sigma=sdata, absolute_sigma=True)
    m=popt[0]
    b=popt[1]
    p_sigma = np.sqrt(np.diag(pcov))
    m_sigma = p_sigma[0]
    residuals = ydata- lin(xdata, m,b)
    ss_res = np.sum(residuals**2)
    ss_tot = np.sum((ydata-np.mean(ydata))**2)
    r_squared = 1 - (ss_res / ss_tot)
    lya_b=b
    lya_max=m
    lya_max_err=m_sigma
    if not intercept:
        return lya_max,lya_max_err
    else:
        return lya_max,lya_max_err,lya_b

# Cell
@njit
def booleanize(vn, threshold=0.5):
    '''
    Convert the numpy array `vn` into a bitstream
    according to `threshold`; values of `vn>=threshold`
    will be set to `1`, and values of `vn<threshold`
    will be set to `0`. If `threshold` is not supplied,
    it defaults to halfway between the range of `vn`.
    '''
    if threshold is None:
        threshold=(np.max(vn)-np.min(vn))/2
    B=np.zeros(vn.shape).astype(vn.dtype)
    for s in np.ndindex(vn.shape):
        if vn[s]>=threshold:
            B[s]+=1.
    return B

# Cell
def analyze(responses,
           shape = 'design instance challenge repeat time node',
           window = 5
          ):
    '''
    Calculates the `puf_statistics` and `boolean_lyapunov` for an array of `responses`.
    '''
    responses = einops.rearrange(responses,
                                 shape+'->design instance challenge repeat time node')
    classes,instances,challenges,repeats,T,N=responses.shape
    boolean = booleanize(responses).astype(int)
    mu_inter=np.empty((classes,T,2))
    mu_intra=np.empty((classes,T,2))
    delta_mu=np.empty((classes,2))
    t_opt=np.empty((classes))

    for s in range(responses.shape[0]):
        muinter,muintra,dmuinter,dmuintra,muopt,topt=puf_statistics(
            responses=boolean[s],shape='instance challenge repeat time node')
        mu_inter[s,:,0]=muinter
        mu_inter[s,:,1]=dmuinter
        mu_intra[s,:,0]=muintra
        mu_intra[s,:,1]=dmuintra
        t_opt[s]=topt
        delta_mu[s,0]=muopt
        delta_mu[s,1]=np.sqrt((muintra[topt]*dmuinter[topt])**2+(muinter[topt]*dmuintra[topt])**2)


    log_dist=np.empty((classes,instances,T,2))
    lyapunov=np.empty((classes,instances,2))
    intercept=np.empty((classes,instances))

    for s in np.ndindex(responses.shape[:2]):
        avgln,sigmaln=bool_lyapunov(boolean[s],window,shape= 'challenge repeat time node')
        lya,dlya,b=lya_fit(avgln,sigmaln,lstart=0,lend=int(T/2),intercept=True)
        a=np.array([avgln,sigmaln])
        a=a.T
        log_dist[s][:len(a)]=a
        a=np.array([lya,dlya])
        a=a.T
        lyapunov[s]=a
        intercept[s]=b

    return dict(mu_inter=mu_inter,mu_intra=mu_intra,delta_mu=delta_mu,t_opt=t_opt,
               log_dist=log_dist,lyapunov=lyapunov,intercept=intercept)


# Cell
class BooleanNetworkEnsemble:
    def __init__(self,
                 classes=3,
                 instances=3,
                 challenges=3,
                 repeats=3,
                 scale=0.1,
                 variables=('a','tau','hold'),
                 steady=False,
                 override=False,
                 decimation=None,
                 g = (nx.random_regular_graph,3,256),
                 a = np.inf,
                 tau = (rng.normal,1,0.1),
                 f = NOT,
                 delay = 0.,
                 edge_replacements = None,
                 T = 25,
                 dt = 0.01,
                 noise = 0.,
                 view = 'out',
                 plot = False,
                 init = None,
                 hold = None,
                ):

        self.bn_kwargs=dict(g=g,a=a,tau=tau,f=f,delay=delay,edge_replacements=edge_replacements,
                           T=T,dt=dt,noise=noise,view=view,plot=plot,init=init,hold=hold)

        self.query_kwargs=dict(instances=instances,challenges=challenges,repeats=repeats,scale=scale,
                              variables=variables,steady=steady,override=override,decimation=decimation)

        self.classes=[]
        self.responses=[]

        self(classes,**self.bn_kwargs)
        self.query(**self.query_kwargs)


    def __call__(self,classes,**bn_kwargs):
        if isinstance(classes,int):
            for k,v in self.bn_kwargs.items():
                if k not in bn_kwargs:
                    bn_kwargs[k]=v
                if k=='g':
                    g=parse_kwargs(g=bn_kwargs['g'])['g']
                    bn_kwargs['g']=g
            for i in range(classes):
                self.classes+=[BooleanNetwork(**bn_kwargs)]
        else:
            self.classes+=[classes]

    @sidis.timer
    def query(self,**query_kwargs):
        for k,v in self.query_kwargs.items():
            if k not in query_kwargs:
                query_kwargs[k]=v
        if isinstance(self.responses,np.ndarray):
            self.responses=np.ndarray.tolist(self.responses)
        for b in self.classes:
            self.responses+=[b.query(**query_kwargs)[...,b.dynamical_nodes]]
        self.responses=np.array(self.responses)
        #self.__dict__.update(analyze(self.responses))
        self.data=analyze(self.responses)

    def __getitem__(self,i):
        return self.classes[i]

