from unittest import TestCase

from iota import Address, Bundle, BundleHash, BundleValidator, TransactionTrytes


class BundleValidatorTestCase(TestCase):
  """
  Generating validation data using the JS lib:

  .. code-block:: javascript

     var Bundle = require('./lib/crypto/bundle/bundle'),
         Converter = require('./lib/crypto/converter/converter'),
         IOTA = require('./lib/iota'),
         Signing = require('./lib/crypto/signing/signing'),
         Utils = require('./lib/utils/utils');

     // Set the seed that will be used to generate signing keys.
     // Skip this step if there are no inputs.
     var seed = 'SEED9GOES9HERE';

     // Specify constant timestamp value to use for transactions.
     var timestamp = 1482938294;

     // Specify (optional) tag to attach to transactions.
     // IMPORTANT:  This must be exactly 27 trytes long!
     var tag = 'TAG';
     tag += '9'.repeat(Math.max(27-tag.length, 0));

     // Define parameters we will use to generate inputs/signatures.
     var inputs = [
       {balance: 100, keyIndex: 0, securityLevel: 3},
       ...
     ];

     // Assemble the bundle.
     var bundle = new Bundle();
     var iota = new IOTA();

     // Add spend transaction(s) to the bundle.
     // See ``lib/crypto/bundle/bundle.js:Bundle.prototype.addEntry``
     bundle.addEntry(1, 'RECIPIENT9ADDY', 42, tag, timestamp);
     ...

     // Count the number of spend transactions.
     // We'll need this later.
     // Skip this step if there are no inputs.
     var currentIndex = bundle.bundle.length;

     // Add input transaction(s) to the bundle.
     // Skip this step if there are no inputs.
     for(var i=0; i<inputs.length; ++i) {
       bundle.addEntry(
         inputs[i].securityLevel,
         iota.api._newAddress(seed, inputs[i].keyIndex, inputs[i].securityLevel, false),
         -inputs[i].balance,
         tag,
         timestamp
       );
     }

     // Add change transaction (if applicable).
     // Skip this step if there are no inputs.
     bundle.addEntry(1, 'CHANGE9ADDY', 58, tag, timestamp);

     // Finalize the bundle.
     bundle.finalize();

     // Add message fragments to spend transaction(s), if desired.
     // IMPORTANT:  Each message fragment must be exactly 2187 trytes!
     // ALSO IMPORTANT:  If you don't want to add a message to any
     // transactions, invoke this method with an empty array instead.
     // See ``lib/crypto/bundle/bundle.js:Bundle.prototype.addTrytes``
     bundle.addTrytes(['HELLO9IOTA99999...']);

     // Add signatures.
     // Skip this step if there are no inputs.
     // See ``lib/api/api.js:api.prototype.prepareTransfers``
     var normalizedBundleHash = bundle.normalizedBundle(bundle.bundle[0].bundle);
     for(var i=0; i<inputs.length; ++i) {
       var signingKey = Signing.key(
         Converter.trits(seed),
         inputs[i].keyIndex,
         inputs[i].securityLevel
       );

       for(var j=0; j<inputs[i].securityLevel; ++j) {
         bundle.bundle[currentIndex].signatureMessageFragment = Converter.trytes(
           Signing.signatureFragment(
             normalizedBundleHash.slice(j*27, (j+1)*27),
             signingKey.slice(j*6561, (j+1)*6561)
           )
         );

         ++currentIndex;
       }
     }

     // Convert into tryte sequences, and output the finished product.
     // Phew!
     var transactionTrytes = bundle.bundle.map(Utils.transactionTrytes).reverse();
     console.log(transactionTrytes);

  Note: For extra realism, it is recommended that you also perform PoW,
  but it is not needed in order for :py:class:`BundleValidator` to
  consider the bundle to be valid.
  """

  def setUp(self):
    super(BundleValidatorTestCase, self).setUp()

    # Define a valid bundle that will serve as the happy path.
    # We will mangle it in various ways to trigger validation errors.
    self.bundle =\
      Bundle.from_tryte_strings([
        # Change Transaction, Part 1 of 1
        TransactionTrytes(b'999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999TESTVALUE9DONTUSEINPRODUCTION99999OGFZEFWIVOZCAYNFOQGRZ9AIRBMWRRFIPCVLEG9ULC9CLIDHE9999999999999999999999999PYOTA9UNIT9TESTS99999999999NYBKIVD99G99999999G99999999EIYHEUACZZAVPQFGIPKDUBFPJVOFTWHADBEXIXINYPAKPUMGZKEKVZQVQREIZHYPEBNZUAJQXHAGDYRG9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999'),

        # Input 2, Part 2 of 2
        TransactionTrytes(b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

        # Input 2, Part 1 of 2
        TransactionTrytes(b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

        # Just to be tricky, one of the inputs has a different
        # security level.
        # Input 1, Part 3 of 3
        TransactionTrytes(b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

        # Input 1, Part 2 of 3
        TransactionTrytes(b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

        # Input 1, Part 1 of 3
        TransactionTrytes(b'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'),

        # Just to be tricky, we will include a spend transaction with a
        # message that's too big to fit in a single fragment.
        # Spend Transaction, Part 2 of 2
        TransactionTrytes(b'DEAUCXCSCVCTCHDTCSCSAEACCWCTCEAHDTCBDGDXCCDBDEAKDPCGDEAIDBDQCTCPCFDPCQC9DTCSAJ9GAHCCDIDLAFDTCEAFDTCPC9D9DMDEABDCDHDEAVCCDXCBDVCEAHDCDEA9DXCZCTCEAXCHDQAGAEACDQCGDTCFDJDTCSCEANBTCTCDDEACCWCCDIDVCWCHDSAJ9GACCTC9D9DEAIDGDFAGAJ9GAKB9D9DEAFDXCVCWCHDQAGAEAGDPCXCSCEANBTCTCDDEACCWCCDIDVCWCHDSAEAGACCWCTCEAKBBDGDKDTCFDEAHDCDEAHDWCTCEAQBFDTCPCHDEA9CIDTCGDHDXCCDBDSASASAGAJ9GAHCTCGDIBGAJ9GAYBUCEAVBXCUCTCQAEAHDWCTCEADCBDXCJDTCFDGDTCEAPCBDSCEAOBJDTCFDMDHDWCXCBDVCSASASAGAEAGDPCXCSCEANBTCTCDDEACCWCCDIDVCWCHDSAJ9GAHCTCGDIBIBGAJ9GASBGDSASASAGAJ9GAHCTCGDIBFAGAJ9GAPBCDFDHDMDRAHDKDCDQAGAEAGDPCXCSCEANBTCTCDDEACCWCCDIDVCWCHDQAEAKDXCHDWCEAXCBDUCXCBDXCHDTCEAADPCYCTCGDHDMDEAPCBDSCEARCPC9DADSAJ9EAEAEAEAEAEAEAEA99999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999TESTVALUE9DONTUSEINPRODUCTION99999ZDDJMOGAXHRGS9XCNSKQQNVB9BSYFCVANLJM9CLRLJSQKFO999999999999999999999999999PYOTA9UNIT9TESTS99999999999NYBKIVD99A99999999G99999999EIYHEUACZZAVPQFGIPKDUBFPJVOFTWHADBEXIXINYPAKPUMGZKEKVZQVQREIZHYPEBNZUAJQXHAGDYRG9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999'),

        # Spend Transaction, Part 1 of 2
        TransactionTrytes(b'J9GAQBCDCDSCEAADCDFDBDXCBDVCQAGAEAGDPCXCSCEANBTCTCDDEACCWCCDIDVCWCHDEAPCHDEA9DPCGDHDSAJ9GAOBFDSASASAEAQBCDCDSCEAADCDFDBDXCBDVCQAEAYBEANBTCTCDDEACCWCCDIDVCWCHDQAGAEAGDPCXCSCEAVBCDCDBDEDIDPCKD9DEABDTCFDJDCDIDGD9DMDSAJ9EAEAGANBCDEAMDCDIDEAWCPCJDTCSASASAEATCFDQAEAHDWCPCHDEAXCGDSASASAGAJ9GASASASAEAPCBDEAPCBDGDKDTCFDEAUCCDFDEAMDCDIDIBGAEAXCBDHDTCFDFDIDDDHDTCSCEANBTCTCDDEACCWCCDIDVCWCHDEAADPCYCTCGDHDXCRCPC9D9DMDSAEAGAHCTCGDSAEASBEAWCPCJDTCSAGAJ9CCWCTCEAHDKDCDEAADTCBDEAGDWCXCJDTCFDTCSCEAKDXCHDWCEATCLDDDTCRCHDPCBDRCMDSAEACCWCTCXCFDEAKDPCXCHDXCBDVCEAWCPCSCEABDCDHDEAQCTCTCBDEAXCBDEAJDPCXCBDSAJ9GACCWCTCFDTCEAFDTCPC9D9DMDEAXCGDEACDBDTCIBGAEAQCFDTCPCHDWCTCSCEAZBWCCDIDRCWCVCSAJ9GACCWCTCFDTCEAFDTCPC9D9DMDEAXCGDEACDBDTCQAGAEARCCDBDUCXCFDADTCSCEANBTCTCDDEACCWCCDIDVCWCHDSAJ9GACCCDEAOBJDTCFDMDHDWCXCBDVCIBEACCCDEAHDWCTCEAVCFDTCPCHDEA9CIDTCGDHDXCCDBDEACDUCEAVBXCUCTCQAEAHDWCTCEADCBDXCJDTCFDGDTCEAPCBDSCEAOBJDTCFDMDHDWCXCBDVCIBGAJ9GAHCTCGDSAGAJ9LBCDHDWCEACDUCEAHDWCTCEAADTCBDEAWCPCSCEAQCTCTCBDEAHDFDPCXCBDTCSCEAUCCDFDEAHDWCXCGDEAADCDADTCBDHDEBEAHDWCTCXCFDEA9DXCJDTCGDEAWCPCSCEAQCTCTCBDEAPCJ9EAEADDFDTCDDPCFDPCHDXCCDBDEAUCCDFDEAXCHDEBEAHDWCTCMDEAWCPCSCEAQCTCTCBDEAGDTC9DTCRCHDTCSCEAPCHDEAQCXCFDHDWCEAPCGDEAHDWCCDGDTCEAKDWCCDEAKDCDID9DSCJ9EAEAKDXCHDBDTCGDGDEAHDWCTCEAPCBDGDKDTCFDEBEAQCIDHDEATCJDTCBDEAGDCDEAHDWCTCMDEAUCCDIDBDSCEAHDWCTCADGDTC9DJDTCGDEAVCPCGDDDXCBDVCEAPCBDSCEAGDEDIDXCFDADXCBDVCJ9EAEA9DXCZCTCEATCLDRCXCHDTCSCEARCWCXC9DSCFDTCBDSAJ9GAKBBDSCEAMDCDIDLAFDTCEAFDTCPCSCMDEAHDCDEAVCXCJDTCEAXCHDEAHDCDEAIDGDIBGAEAIDFDVCTCSCEAVBCDCDBDEDIDPCKD9DSAJ9GASBEAPCADSAGAJ9GAXBCDKDIBGAJ9GAXBCDKDQAGAEAGDPCXCSCEANBTCTCDDEACCWCCDIDVCWCHDSAJ9CCWCTCMDEAQCCDHDWCEA9DXCRCZCTCSCEAHDWCTCXCFDEASCFDMDEA9DXCDDGDSAJ9GACCWCCDIDVCWCEASBEASCCDBDLAHDEAHDWCXCBDZCQAGAEAPCSCSCTCSCEANBTCTCDDEACCWCCDIDVCWCHDQAEAGAHDWCPCHDEAMDCDIDLAFDTCEAVCCDXCBDVCEAHDCDEA9DXCZCTCEAXCHDSAGAJ9GANBCDTCGDBDLAHDEAADPCHDHDTCFDQAGAEAGDPCXCSCEAZBWCCDIDRCWCVCSAEAGAFCTCEAADIDGDHDEAZCBDCDKDEAXCHDFAEAXBCDKDFAGAJ9GAXBCDKDIBGAEATCBDEDIDXCFDTCSCEANBTCTCDDEACCWCCDIDVCWCHDSAJ9GAHCTCGDFAEAXBCDKDFAGAJ9GAKB9D9DEAFDXCVCWCHDQAGAEAGDPCXCSCEAHDWCTCEARCCDADDDIDHDTCFDEAPCBDSCEAGDTCHDHD9DTCSCEAXCBDHDCDEAGDXC9DTCBDRCTCEAPCVCPCXCBDSAJ9EAEACCWCTCEAHDKDCDEAADTCBTESTVALUE9DONTUSEINPRODUCTION99999ZDDJMOGAXHRGS9XCNSKQQNVB9BSYFCVANLJM9CLRLJSQKFOOB9999999999999999999999999PYOTA9UNIT9TESTS99999999999NYBKIVD99999999999G99999999EIYHEUACZZAVPQFGIPKDUBFPJVOFTWHADBEXIXINYPAKPUMGZKEKVZQVQREIZHYPEBNZUAJQXHAGDYRG9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999'),
      ])

  def test_pass_happy_path(self):
    """
    Bundle passes validation.
    """
    validator = BundleValidator(self.bundle)

    self.assertListEqual(validator.errors, [])
    self.assertTrue(validator.is_valid())

  def test_pass_empty(self):
    """
    Bundle has no transactions.
    """
    validator = BundleValidator(Bundle())

    self.assertListEqual(validator.errors, [])
    self.assertTrue(validator.is_valid())

  def test_fail_balance_positive(self):
    """
    The bundle balance is > 0.
    """
    self.bundle.transactions[0].value += 1

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Bundle has invalid balance (expected 0, actual 1).',
      ],
    )

  def test_fail_balance_negative(self):
    """
    The bundle balance is < 0.
    """
    self.bundle.transactions[3].value -= 1

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Bundle has invalid balance (expected 0, actual -1).',
      ],
    )

  def test_fail_bundle_hash_invalid(self):
    """
    One of the transactions has an invalid ``bundle_hash`` value.
    """
    self.bundle.transactions[3].bundle_hash =\
      BundleHash(
        b'NFDPEEZCWVYLKZGSLCQNOFUSENIXRHWWTZFBXMPS'
        b'QHEDFWZULBZFEOMNLRNIDQKDNNIELAOXOVMYEI9PG'
      )

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 3 has invalid bundle hash.',
      ],
    )

  def test_fail_current_index_invalid(self):
    """
    One of the transactions has an invalid ``current_index`` value.
    """
    self.bundle.transactions[3].current_index = 4

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 3 has invalid current index value '
        '(expected 3, actual 4).',
      ],
    )

  def test_fail_last_index_invalid(self):
    """
    One of the transactions has an invalid ``last_index`` value.
    """
    self.bundle.transactions[0].last_index = 2

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 0 has invalid last index value '
        '(expected 7, actual 2).'
      ],
    )

  def test_fail_missing_signature_fragment_underflow(self):
    """
    The last transaction in the bundle is an input, and its final
    signature fragment is missing.
    """
    # Adjust bundle balance, since we will also remove the change
    # transaction.
    self.bundle[0].value += self.bundle[-1].value

    # Remove the last input's second signature fragment, and the change
    # transaction.
    del self.bundle.transactions[-2:]
    for txn in self.bundle:
      txn.last_index -= 2

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 4 has invalid signature (using 2 fragments).',
      ],
    )

  def test_fail_signature_fragment_address_wrong(self):
    """
    One of the signature fragments for an input is associated with the
    wrong address.
    """
    self.bundle[5].address =\
      Address(
        b'QHEDFWZULBZFEOMNLRNIDQKDNNIELAOXOVMYEI9P'
        b'GNFDPEEZCWVYLKZGSLCQNOFUSENIXRHWWTZFBXMPS'
      )

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        # The bundle validator uses the address to link inputs
        # together, so if it encounters a different address, then it
        # assumes it has found a new input.
        'Transaction 4 has invalid signature (using 1 fragments).',
      ],
    )

  def test_fail_signature_fragment_value_wrong(self):
    """
    One of the second (or subsequent) fragments for an input has a
    nonzero balance.
    """
    # Don't forget to adjust the change transaction, in order to ensure
    # the bundle has a zero balance.
    self.bundle[5].value = -1
    self.bundle[-1].value += 1

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 5 has invalid value (expected 0, actual -1).',
      ],
    )

  def test_fail_signature_invalid(self):
    """
    One of the input signatures fails validation.
    """
    self.bundle[5].signature_message_fragment[:-1] = b'9'

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        # Transaction 5's fragment is invalid, but the validator has no
        # way to determine this, so it just assumes the entire input is
        # invalid (the first transaction for this input is at index 4).
        'Transaction 4 has invalid signature (using 3 fragments).',
      ],
    )

  def test_fail_multiple_errors(self):
    """
    The bundle has multiple problems.
    """
    del self.bundle.transactions[4]

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      # Note that there is no error about the missing signature
      # fragment for the first input.  The bundle fails some basic
      # consistency checks, so it doesn't even bother to validate
      # signatures.
      [
        'Transaction 0 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 1 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 2 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 3 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 4 has invalid current index value '
        '(expected 4, actual 5).',

        'Transaction 4 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 5 has invalid current index value '
        '(expected 5, actual 6).',

        'Transaction 5 has invalid last index value '
        '(expected 6, actual 7).',

        'Transaction 6 has invalid current index value '
        '(expected 6, actual 7).',

        'Transaction 6 has invalid last index value '
        '(expected 6, actual 7).',

        'Bundle has invalid balance '
        '(expected 0, actual 147).',
      ],
    )


class BundleValidatorMultisigTestCase(TestCase):
  """
  Tests how :py:class:`BundleValidator` handles a bundle with a
  multisig input.

  Generating validation data using the JS lib:
  See :py:class:`BundleValidatorTestCase` (code is exactly the same;
  there's just more input transactions in the bundle).
  """
  def setUp(self):
    super(BundleValidatorMultisigTestCase, self).setUp()

    # This is the result from ``examples/multisig.py``.
    self.bundle =\
      Bundle.from_tryte_strings([
        # Spend transaction.
        TransactionTrytes(b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

        # Input (requires 8 fragments to store the entire signature).
        TransactionTrytes(b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
        TransactionTrytes(b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
        TransactionTrytes(b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
        TransactionTrytes(b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
        TransactionTrytes(b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
        TransactionTrytes(b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
        TransactionTrytes(b'RJJBGKTVFJCBRJAPEKBJPINIILSXFJCVEKJNDMJUOGWI9URTAKQASSHFHDIQT99XYNUURBMVDDTTOZUYCEMDJZUKDEDBIZMSHLRAZULBBNPUESBVEW9FSQVFDGJPCO9EFWABKOCJBKCT9SLXDF9TNGHELJBBGVEDEDYFKXGXDFLFLDO9IMWNGLLYOPLIJSJCVYTBYI9CZXVPSZGKFRXV9VYAPKSGCPOINZMLGSRCRTCJCPNBZDDATOMHIJFZJUCULPCNYHJMOHQZBQ9MGUSHGEQDCUUXMQGMKTPSBMHNRUZVRQBZKFAKGRCASKGWNELYLJKCODBMKESYLBTGX9WKPP9MZLHFEQY9ECLDISRINJGIIBJIKHZQWBRWCZZILRUQXX9RSIBOWYCVIDAEUNZWYBSHA9PCNSSSALCO9YOXMKLN9JASZIKNSRVPOWBBIHPVFOMH9TOFSDZIWUYYOSLMFZTORE9BQKR9STQVHWOPPCAVNTBSKJMFQLEDTHYBTGZKZC9XIPZVUGOXWZOCC9UFIJCPWLDNSZFTUTYRNKX9PSXAWXSJRXDDCEZMTQDVXIARXUIWTIVCZMABKCZMYNTAPLXSKDYZUGIJZAJURUURLKVSSJK9RUVEDFLQRZWWYKRKPZNJESSYKTQOJMNUVZCUJZJUUBVPDIGBXIIBIXT9FSPUQGBRNWSOLPURUKUWIVZHCRMNSAWWVIEIOGAZZFVYKEDGDZXJDXWD9AHRWDFMVJADLPUWSJYXDXGJEDJWNH99AGNKKAXNA9SWPTZVNHUKEQKFHL9JGQWCXETW9OQFXWTCFOKRAL9NODZZJE9BPGRQQCQWLPPMGROLAYPAHWFBZB9OQ9XTKFTDFUMUSUMWJUYSETOJPDRGOMCQWEWBOMWMHPAPEYSDXEEONFIYILHXQGTHSLXLEGJIGKYKXMMWNLSKZUPSPLSCXLWMXQZWERPHMINLJDXAAOWDLCJCHOVYKOIPYPTOPXSGKHRAPBJISGTBEXPWDKHDILDE9PIDGYJYPDIQXBLSHKUWGUVFMI9PTPYQQSGSXWGKIHYMGQE9CNFTPDHYZRLCJPPRRQULAEB99EGPGW9SE9UUNRL9MUUCPCPGAYRUZJWEJKLWQOGAVCPJ9BIDGZPNZYTOIPPEVJSQWUAXLHFFPR9HDVQOXZ9YKIH9J9UOXVCNJLNN9AEUNBUOGZPAQKLQ9OYL9LKCBWYNPHXSZMCYYTPDIGDMZWVV9EJGYCTMMSWBGXNYZIEAMMSNFDKWECIKJOOAMJJSDNPHIRZRGTASURCPKHJYAJYVU9QFV9AQDDMQNWDUVIYLLJTYBUMPLPED9FCSKXCUIEEOSHJMGAXLNYKQJRDONXZUO9XAVVCZ9LR9ZWPGSCP9GCP9URGTFNGCBPEVSTAGUOSXHUTFSF9NNPMFFVHFOKIHJQDAHY9VHHHPVTRYLE9MIQZJJTIQFD9EHIBUIATEEZMKYIFZGLJXTBLJWZPKRVZP9KQBMKHJYJOMKJGZNH9RYMZWFCU9LONCFSLMMSKQZJHKJCC9BFPWVQNVPFTIBCBWNDTFRIABPMVXYJZDXDQCQBHMTXKVUVEJESWOQTPULVIBSSZBNMJNON9DSN9LAK9XBWVBNOP9LQJKTSWBXRLGVRKZSLEOKC9VISQNZLP9CCWTEUPLHIPQQFDURLQCJ9OGRBZDAZGJLFDODBPXXMYR9CMTMBINV9VQDV9ATGXNFG9GMAARROKQPNYNKIGDZLC9SZWEZHGRBSLRNDPFMYCTPCLUITXSNRCANWMZJPHCZFYJ9WQATOLZUGEULLVNGZLC9YMPABKNMMRDQVSYLKDHILZAGEAPD99WLASSJVTJXURZEIXFIMZJXPUSG9LEUSSHCSROLECNXDWKODEZYFAMWTQGRQFYCYVHQUUTWIVM9ZSRENZGHBCKJZDPMBWDFYHTVFTCBD99TFCLZUCIOWARRNMRLFBPOPUNLQVJND9ZCIVCFR9RRWDDPVSDRIVAAIVCKWSNZSUFCVREXHVUVGQEDZEVAWIIERGDFG9CLGNSHIRDSKXUQHHB9MNOFKLEJ9NIQVAXVXKOGZVNYPKMCPUOMUVMSJWTOIYBJEH9NBSBWMZDMBHWNBWDMVXZH9G9NHULNMLBAZIOKNQGTXVVXEZAJCVSGUAYAKTDTCCXUQGZGOJBPXFQSCMMASBHZMNT9GBCBAHUCETD9MKVQ9R9BTMJZGVWQYWLXVQL9EPJQSLYQSHKAOQKKLFWMGTGV9ZYNDWN9ZCWOCSWLY9999999999999999999999999KITTEHS99999999999999999999GZGFZXD99A99999999H99999999GYASNKZDVDMTPR9BJLHFMYBVXRRDCUQOPOOUAIUZG9MKZQPEEDLEQ9FNCQYKAUZWBHC9WQXOGX9YHVFPD999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999'),
        TransactionTrytes(b'HDWCPCBDLDEAUCIDFDEARCWCTCTCNDQCIDFDVCTCFDGD9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999TESTVALUE9DONTUSEINPRODUCTION99999NDGYBCQZJFGGWZ9GBQFKDOLWMVILARZRHJMSYFZETZTHTZROB9999999999999999999999999KITTEHS99999999999999999999GZGFZXD99999999999H99999999GYASNKZDVDMTPR9BJLHFMYBVXRRDCUQOPOOUAIUZG9MKZQPEEDLEQ9FNCQYKAUZWBHC9WQXOGX9YHVFPD999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999'),
      ])

  def test_pass_happy_path(self):
    """
    Bundle passes validation.
    """
    validator = BundleValidator(self.bundle)

    self.assertListEqual(validator.errors, [])
    self.assertTrue(validator.is_valid())

  def test_fail_signed_in_wrong_order(self):
    """
    The signatures were applied to the bundle in the wrong order.
    """
    # Swap the signatures from the first and second keys.
    # Note that the keys have security level of 3, so we need to swap
    # out a total of 6 signatures.
    sig_1_1 = self.bundle[1].signature_message_fragment
    sig_1_2 = self.bundle[2].signature_message_fragment
    sig_1_3 = self.bundle[3].signature_message_fragment

    sig_2_1 = self.bundle[4].signature_message_fragment
    sig_2_2 = self.bundle[5].signature_message_fragment
    sig_2_3 = self.bundle[6].signature_message_fragment

    self.bundle[1].signature_message_fragment = sig_2_1
    self.bundle[2].signature_message_fragment = sig_2_2
    self.bundle[3].signature_message_fragment = sig_2_3

    self.bundle[4].signature_message_fragment = sig_1_1
    self.bundle[5].signature_message_fragment = sig_1_2
    self.bundle[6].signature_message_fragment = sig_1_3

    validator = BundleValidator(self.bundle)

    self.assertFalse(validator.is_valid())

    self.assertListEqual(
      validator.errors,

      [
        'Transaction 1 has invalid signature (using 8 fragments).',
      ],
    )
