import warnings
from unittest import TestCase

from iota import Hash, TryteString
from iota.crypto import SeedWarning
from iota.crypto.signing import KeyGenerator, SignatureFragmentGenerator
from iota.crypto.types import PrivateKey


class KeyGeneratorTestCase(TestCase):
  """
  Generating validation data using the JS lib:

  .. code-block:: javascript

     // Specify parameters used to generate the private key.
     var seed = 'SEED9GOES9HERE';
     var keyIndex = 42;
     var securityLevel = 3;

     var Converter = require('./lib/crypto/converter/converter');
     var Signing = require('./lib/crypto/signing/signing');

     // Generate the key.
     var privateKey = Signing.key(Converter.trits(seed), keyIndex, securityLevel);

     // Output human-readable version.
     console.log(Converter.trytes(privateKey));
  """
  def test_get_keys_single(self):
    """
    Generating a single key.
    """
    with warnings.catch_warnings(record=True) as catched_warnings:

      # Cause all warnings to always be triggered
      warnings.simplefilter("always")

      kg = KeyGenerator(
        seed = b'TESTSEED9DONTUSEINPRODUCTION99999ZTRFNBTRBSDIHWKOWCFBOQYQTENWL',
      )

      # no warning must be raised, since seed has a valid length
      self.assertEqual(len(catched_warnings), 0)

      self.assertListEqual(
        kg.get_keys(start=0),

        # Note that the result is always a list, even when generating a
        # single key.
        [
          PrivateKey(
            b'RYTWXBSSDDMFTHADVDRQN9HVOABBDMJDGAN9HPFAUOBZRKIWSVHJOPPETPFEF9UM9V'
            b'ZETGGPMRJFPQAUWAVSHPZLGWNXLE9EKMGENDWUGRFBV9IVBAQZBVKF9GKTIAUFTSRZ'
            b'NKGVITUYZDFRUXUJSVS9TRPMXJYKNBSHGUSUKVFLDHPSWNINOQLYTYYUY9W9EFDEHB'
            b'CQZADZIIVSTOOQDVXKLCNHYZPUXTUUXTNAUDS9NZIVRHXNDMXNMOKHDEFGGLFMJIIT'
            b'MJ9MVDMFYJ9SY9IWSVRWITSWFMQJZIBETXXSXUDNSJQIJLMNXJGUUKBAII9YLSQVQO'
            b'SHZQJWPEJQTS9IJAEKPFLQYLUANQGJTWEVNXOFYHFUJCTSFT9GXVCAPHUMTTTVPXHE'
            b'KORVSSMVWLALNGUSFRZYEUPJRDZ9ESSZYEHGYSYDKLLPLKUPUIGOKVBI9YJTN9MLQC'
            b'HTLQYLLBXGKQM9DOIPOZEWOXJKHNASEUUCUVPCGL9HWCBQTTFIWUCMXVMEMVUKQWVU'
            b'ELWBMRQQGY9DO9UISPLULEKORKJVPLHUSDRNZJCNNUZEKMUDQJNRNNQWQOBHIYMEQD'
            b'LRHLNNSLRLWJEXRVUKXBLOKAXPKWVAGFUFFKCKCJAYYFNMVHLIMQABMM9O9BCANIZX'
            b'YXYVSLWJOQEDKF9TGQGVGSRBZZCXTGAMROYCW9WCRDDBWSSHK9UOGASQBZVGLHLNKR'
            b'EHXXS9UWCOQGSOGRYQIVX9EZQAP9KTRTOTTDCFRYHL9HGIJNRBADEGVVZGEQLKJNFN'
            b'RDHIVSJARABAWPRB9DGQHROWIQZCFQJMNIDNDZRFFSABYCHRKGZNAXPNTCKAQAHXWS'
            b'FHDZTSGZFOJQRSWDQOZELCYFHDVZADU9XCIPNKGAFQS9DFKYWJPQBRJK9MPXDTJOC9'
            b'URFTMQXDQLOFFLQCNMTZLROECGMXUCENMYHVIBJIO9DWOKRDPVVUUBZ9DYBKEYEFYO'
            b'SXAYBGJTEJKSBZJRPLZSTWG9YVVF9OWXIC9MNSDMACOVRHNSTTGARCCSDJAWSRWDKT'
            b'OCDLZUAWQANXXX9RAFLIQUISYWPMDKTZWUZOKOWDHFJ9N9EMQVA9DRZGAIAJTIPBNI'
            b'MTOL9JY9IIGYIXDIB9MZNTZYHJYMGAJEAOKZYYFAIVWQMQAJJJWECCJ9VZNMWNPMZA'
            b'VTDWEZBZSIGGDDPTIPKEASMWBHXNKTGKHNNBURXLCOBR9CYSXZXUYVOCLRJTXDQCZB'
            b'HKZ9TRWRHYZAMEHLKCAKUYXAGUUANNYTJKUEIENVYXBQLTKWYPDFCCSGQKWRYZDASY'
            b'DKUMZYXEWUUROUXUHEFMWVRDNRVZJTANTDZIZUXQMBCIVAF9JCYQZEVAAQKRZY9OYL'
            b'EURQGARRSDGSFEW9GIGFMJDPAY9NFXTQMXXTJUIMKKUE9BOOHUOHHENLI9DVYJJXLM'
            b'UKWKNWTBODTHZFBNQOVJIXOOMBULE9IGIITWQACCWZCOPDHNLHCKACNETJJNSOHQFO'
            b'99RLRZFLPGMBYZYHKSNKYSULKENZODTDKFKJEZNWK9IDIPQLYGSETEMLYFDQXDHVA9'
            b'XJUUCWIDUMU9AE9HGOWDIVALKUVAZV9TBEGYPDWQKQOH9SNKLBTGEQZAJZXBJUIDA9'
            b'YCBHONYIZTZWSAVXPGDDWN9PQWKXGGFUQSFUSPJIVLDDZUSZVRXMYZEBEXYFVLPZQQ'
            b'YAUZOYIUWGXNJEXCQT9GDFLFDCPZBSRHZSGYUHUBGQSSQEXB9PDPRQQ9YICHOWTJRW'
            b'XGCZRTXC9ERXFFHLDMMVKF9LULV9LDQPXJAHUKWZNINFHTCSYNAXFST9IFRAXPASIJ'
            b'9WJWWFO9SYIQZNDMVXG9JXONUDMVGSNRQIJKVEDCGHWKEDOGED9YPQIXBUCZJRSMRM'
            b'S9RJTFIYDIXAQMWNXGHHTCJLCSQNH9CNP9PJZCLIBQQBNMEY9PIJJLXPLREAHPKOZD'
            b'JMPVYNUK9EMRVEJVMCQPPW9HOUHQ9YNNBXGLDLUJQEQFXTDTKHDREEIHHKIWWSPKWY'
            b'QYXLGXUEFAIWSA9NSDXWPSKISBADFJHJCSCNRORALCQNMIJ9U9TNCHZOC9NARJWQPH'
            b'O9HALCYZMNPCIV9VROERYT9UCNLZGKJEORFT9AMEZCXOVUIXSEPEIBMUNPPDEWIAAK'
            b'GRYJANZDY'
          ),
        ],
      )

      self.assertListEqual(
        kg.get_keys(start=1),

        [
          PrivateKey(
            b'PUQKFGMMWIEFMSGJI9QGVRYOCAPKTRJAOJLGGLURVTWDOWOI9ZHKHSVC9GPKNPHDTD'
            b'MSMJNGNXDR9NDYDRMUM9CTOI9MCGMCRAARLPVUSJGXEYRRBUSUQLGMTDGGGWYASWTK'
            b'KJHHBZ9BIYVNKGHKCTXVVJLFSWXW9WVU9NWTKO99ZZP9FOMLULMJPZZEAJBZXMW9BD'
            b'IWBMZMAERMLVPW9ADZKWQLSSHCYZZQIGXDBRDEEHUBZV9FJBIOPPAAE9RMKMAQFHYW'
            b'ELQPUAFTMBBXFSPOIVJL9QDYZLDTPMLZQSEJHWEBAUTWAIFAUSGHZMYPSVBDTKQKWQ'
            b'UQNRIEMYMNS9AYZJGURKDWEZHVGBJIOJKHBVWXJVFUVBZKOMWIHNPKV9SWRHYNKN9T'
            b'WNDYETPNXYKRMRPLDXUMGQ9GNSF9KHFXVQVIDCQRJJL9DFZHOFFNMJDO9EEAYTLHUT'
            b'WHJWKKONESMCENCALGTLDZNBUWLBKMKEPXTTHLBQLTVDBXZNLFJRZM9MSGLUDYPBUD'
            b'ISGNOAXXZNPEQQXFDZAKEPHJZHGDTXDBHPTQLPCLBXDZERTCNJQYLSKSVOAXUBQNWK'
            b'GOIMHJIOBYPYUBANJTXBOGLSASYWMKV9JZTVNZVOZJCEOUVWXTWQPWNVIWUAB9IPGJ'
            b'KREMJCAEMOZJJXQGZTAUBITQBMCARFG9NWLTWOYCWSBKTKXXJRRIFFDSKF9AMJBZIM'
            b'FQXVRSSB9NXXLXORUHEKOWAQHZVFUETBTRCDQFFNGLTFKSQNOJWEXTOBWDGBMDSNLQ'
            b'BORKWPMUNGIKQQNPBCFRDZKCEJUKFWAKJXKVVTJMP9NGUAYVCIHKKYOEQBONDIOIOQ'
            b'AYYYGJDI9EIOIHFCBJTEUOHGASNXKVFR9QGWWQNQEWLNYLRBDLPBGSSEFGHIKLIYHR'
            b'KR9YHZTMWSOSMPIJKNKYG9TRXMKIYZLTCQOEXYWDUTRXLMRWYZJVH9LPTIDYINNPZF'
            b'TPCL9HRXJYQXEAXRXPN9RINJHBYJPHEFQJPDVHKJJKIAQWVNKRBD9WLVIDGKDXXGJF'
            b'EIZJTYIC9HGYCZH9MQROVDVDVWFULPNSHZCUOUVABLAYGFSOVTABUUWUY9ACVPAJQC'
            b'OJCXJOML9VJDSIHNVRR9RBNGCUAIVBWNGFDJWVWHJQFXMXYDWKOMFVPNBWJKIBXY9Z'
            b'LYSHEMWPCDXIVOUKKTHYBHBNI9YYLNGYYDOLUAKUKFRJZA9FYCFNTFTNBCXLMMCZIJ'
            b'OPJFIWYENQOHXWIAT9JODYDOEHESLGHZIMFGRKAVHZSZYLKTYSCHKQFSDTSHTOWHHQ'
            b'XGFKAYYCMPYTTJFQPLEYFMLAQBPYMFPYYLR9JQPUBMJUVDYSGUXVVLSRXJJEERZVSY'
            b'KDPCTOXQNMFWAXQXIGUQRDKUQD9XDTZHRZJGNQCNORSDHWSCZGFLZ9PCCUPSVQCYUT'
            b'9HGXHZHPNMSBFQIDEHLMWGVJAWUXSSV9VRRIBRBOTSWUAVMWSTBVXNCAPMCQRSTMBF'
            b'ZOCAYLBVSAXBTDIUYQMDDGT99YMUVTIHLGFUYJCYFJLLX9GBAGAVISAZ9QJSKNHXVX'
            b'YIVKOARIA9YFDGSMPPWPJFCHAYGDRJJ9GYKYAMFS99OMOTEBOGODKUITJWFOMRQNBS'
            b'9ZMCFIFJKISKKAJFPGVWWOERDSFCUEBSZPTKYMTTRAFLWYOZYJDNYIQWTBSLANDXQX'
            b'QZUYCMKQINKOAXGHVMEGWLWTJPKJUPWCXRPZVDMBRDSQLTKLNIDXINWCRYYNSLDT99'
            b'ORWVIFHY9DFXFXQM9LPRMFOVZJSMJAQAMRPAWNTGEA9VWBQTXJCGYZHPKWLL9VCVDQ'
            b'PUIKFLGK9WKMMUWQPCJCWDK99QLUYUPJIVPMNL9QZK9QDTGFFKMMAOSEPWKOGLIVYD'
            b'VKRAAGDFMEULWVCUPUV9ZOTSKORD9CTI9DFYBOJMEDOSOHNARELZNFPGXFLEPBLYQO'
            b'WMRNZLVDRNBWOTRDMHEURDWGQKCXZEX9UAWBSEKOVDG9CGXOZXMPYCUWQTKLKAYGQN'
            b'TPNUXAFOJTZRRPSAQ9HGOPFFVMYTEEFKOWZWVMIAYZULGWODOCOYHIJITYBDLQCTXZ'
            b'TJUIGGWPUHOWVQSDDNVWNUNGUPUPYBZMSZTYXLRYSGVLVMGERUNIPZDDFHLJCDIHVI'
            b'NETYGWNAY'
          ),
        ],
      )

      # You can request a key at any arbitrary index, and the result will
      # always be consistent (assuming the seed doesn't change).
      # Note: this can be a slow process, so we'll keep the numbers small
      # so that tests don't take too long.
      self.assertListEqual(
        kg.get_keys(start=13),

        [
          PrivateKey(
            b'TIUQQXOOYTKCESSBBGKKVJQPSJNVTAGCTAJVRJNXJUIHXWXQHSTACBRP9BJCSQD9CA'
            b'FNQNNFVEPMYFCPYKXRCNWZHYKYCRSFTAZTPGQXBIMGAZEXABLUJQKEQMKAOUXYNICZ'
            b'YWDIBLZVJYGUPET9BAV9PTM9TJLHTWLXV9NXMNDNDYHTLIVGZJSEHIBWOYIXABKP9O'
            b'APPGWTUNLKYDFNWSFDTGRDPTMABYOPCTZM9LQIESPNNVZVYGGCJOHC9NSXMX9ETPGH'
            b'NQACONLQJIHRNAKUYMLSGTEJUO9MZGEQZPQBTQXKWJDJLFSRBPERWKVHUZXANWMOKW'
            b'QPPGVFZRVUEOWDMZM9IOZUCOYTQIRZXWDYWAHGOATEW9AMZNSTZRZCQYS9MYREAFMT'
            b'BIQFUUZKBSWFSNZO9HDBVMNTXX9EAAVTIZSARDP9ZKCUBIRPPGWWXV9PDXLCSDKLWA'
            b'NVLTXPTEBUAHJJL9LESHCQQWZIYMRNCVYMDQUQFWNAGTCOQBNCLHZMJRNYWIZXDLFF'
            b'JVA9OTEVHOQZKUDHXLHLZLH9YZR9WBCMFOVRUCDBNVHRGMAYLIEKNDCSWMT9HXU9SD'
            b'LMLKOBEISWYQCSXYINQNNHSOGTZTZNUXNVHBBOUCFIKIICILJADHTDBJNTPORFBQGC'
            b'LJKOMVRRBBXZJLRVKBLELUDBQGFHOHRAXTHVUSJMZBO9KKXZZNNKJEN9GNYVBDFZDU'
            b'QNYTENTFYUSOGJOOJJFGAADFKNKMVWKUWCCTACYZZEYTIKBJP9SCUGTRYLYCBXCIQP'
            b'LQFKARVWU9VKUGUZYXZXCNSXGNZBXIZDDCSMTOHOCCI9GGXMNAYLVMITJXEN9MKUAP'
            b'NUBBSZCXNWN9DKMHROJQPNWEGHDPKPVNZLFMHXDUOTBVBGDLWGSODW9ACJJIJW9NXD'
            b'YOGFIPHPET9AJZXNKYWXKGRSINIODDS9IAT9LAEQWUODEBW9MQIDKZPIINGKUXZGKG'
            b'NXTPDMUOKEKKM9TQQJCGMOGXITJAVODWPFCJX9UEDVABYLUIUUUCAXUFWZCYWGWL9Z'
            b'FZP9YSD9RJCMZJQZEQJLAZNJPSPOPJCEPOJTFZOBTWWLHFAALRHISBVFUYAHNBMT9I'
            b'TDDFKREXTZUXHCOL9VDDVQAXKBKZSLVEREOOSPPZHIWH9AQHVNZEXVCWPQLVUGPVBZ'
            b'GIRYUMQXPJBGP9AUFCHZHTENRJAKJHLKORDSYZYISTFUPAZNIWUJUGEDFSI9XMZRLB'
            b'HCZSGOVXINMDJZYSPHBOVTTDCYONQKZ9UJHNJDWR9YT9MJNSFFUQDCLPOXFV9RDQSA'
            b'TGCCWAAZVVJSCZLZTPJXBFBUDLIBABLKLZFZUHGARDDFX9OZDXBZNSFEWYXNSLCMHF'
            b'ROWWJHK9IRLEZZEVFWJ9KCEWGVHXTSXWMJSWPH99BU9GVXOIHTFTOGFMGZVCYCPDHZ'
            b'OUUB9DEPJLJCFFKEGMBYLLSXEDGWNPMWCSQOAMDITYDHDPTDQRKMKPAJIRLMSAN9OW'
            b'DOYZQHJJSVVOSYIYCMYVXEAOOFNNRWKJWNYDEZOLZ9MMMSUIPWIYKWZVEFKSTARVVO'
            b'UONUBAXJEFKJRNLDBQTRYAWK9NZYFRBCYVPZOACJPZRSAFYQPA9KJGZMAKKPAAUYC9'
            b'NQGEPWKUVCHPSUNQDDMOIMKCLLBSACBXDVXUDWYOUCVOVZCNNRAURQFUPQGNB9CAXS'
            b'CPXPFPJENYWFBOJFFWZAUUTYRRQQMRBWUQOMJYPMONU9EYWJSGQQZHHZSHF9XQPIAD'
            b'AUHDKOIWJJGRLIPZQJ9LKPJRBHRTKUBWQHAOKZKT9GPDTSRXDRESZXMCMRNGUJOJPC'
            b'CSFMMPITLHU9FEBFMYLCBPCFEFXAZXYZRZFZ9KIQINS9GSN9BZLMMODAPNORGPMYNT'
            b'XXASTKLVIGMUGJWRSLMLIDQYHXNTBDZR9EQDBDUZVSIALOEHBBUS9QILWCBIYYUSHZ'
            b'RNICAYREZANRQWGGBJZCMSB9TRXXCUSPIIEJLKLEOMVCYHRNI9KEHPXAIUQIFUNSVK'
            b'JMEUMLBXSIHQUFK9QLA9UTBJZQGAQHWXVHWXYSEPCDO9M99HYRPAXFXHQENZHNBWXD'
            b'VN9KOMUINNCIMWKWI9QITAORLVJL9HRYSDNIQNAAQOQPSQNCQCGYAPIYSNOPHECZJY'
            b'VTVVMKOUZ'
          ),
        ],
      )

  def test_get_keys_long_seed(self):
    """
    Generating a PrivateKey from a seed longer than 1 hash.

    This catches a regression caused by
    :py:meth:`iota.crypto.pycurl.Curl.squeeze` processing the wrong
    number of trits.

    Note that seeds longer than 1 hash are not more secure.

    References:
      - https://forum.iota.org/t/why-arent-seeds-longer-than-81-trytes-more-secure/1278/4
    """
    with warnings.catch_warnings(record=True) as catched_warnings:

      # Cause seed related warnings to be triggered
      warnings.simplefilter("always", category=SeedWarning)

      kg = KeyGenerator(
        b'TESTSEED9DONTUSEINPRODUCTION99999ZTRFNBTRBSDIHWKOWCFBOQYQTENWLTG9S'
        b'IGVTKTOFGZAIMWHNQWWENEFBAYZXBYWK9QBIWKTMO9MFZIEQVJULQILER9GRDCBLEY'
        b'OPLCYJALVJESQMIEZOVOPYYAOLJMIUCGAJLIUFKHTIHZSEOYYLTPHKSURQSWPQEESV'
        b'99QM9DUSKSMLSCCDYMDAJIAPGJIHWBROISBLAA9GZFGPPRPHSTVNJMPUWGLTZEZEGQ'
        b'HIHMCRZILISRFGVOJMXOYRALR9ZOUAMQXGW9XPFID',
      )

      # check attributes of warning
      self.assertEqual(len(catched_warnings), 1)
      self.assertIs(catched_warnings[-1].category, SeedWarning)
      self.assertIn("inappropriate length", str(catched_warnings[-1].message))

      self.assertListEqual(
        kg.get_keys(start=0),

        [
          PrivateKey(
            b'BXBGJNNW9NXZYRHFVVXVVSRD9ECGQSXNWPLBDLMDYXWJJLVELFZZESFJYH9FCSTINR'
            b'CCJODSWMHFWVGB9HYNHATWHWOJKNVWEVXGZBSXSYVKMEDEZVCME9MQOCFFGVWCZHDT'
            b'9DTPOZYBEQQWCDVMJQC9BCTBJWIGW9B9PUHCBWOHN9OHYRMSBFCAPGOEC9GYIMRPNC'
            b'VCZYZLEFFJBGEERRKQHVNFMPPSWZAAXOVJXEEIZI9DNNDDONRNSHUUCNYWIIXIRWLR'
            b'AV9CYCNHCKSVWS9ERXCUQZLUAWEAU9IUGEQUCXDRTVWGPDRSOCSFKRZSSTKXXYWBBU'
            b'JMGIIQUWKXGHJBEUONHMESPAHCUGRDPDDRPQJYWPIFVYNNTXZAWBWQPOOOLAIVI9HY'
            b'UYIGXCUXDWTCL9YXUDXCEJRVBVXTMAFXHBXRKFXBYHMPZPKSVUKNSJXSCPTXTYITAT'
            b'CLUWNTGMBTMCCBOCINJSXSYDLJAXCUWQKGLXVQWNIXKHN9KZOHFLFURSHOXSTJCUUH'
            b'IOBEYWMWMNMFHEGKEINGIJFJOHAZAYQPIULXGNYZMTLJVUPWW9MAAWCUF9QAYATOPZ'
            b'MMKZRSVBWRSAWAOYRF9BVSWMNKAFEUHOTLFARLV9NDGLCXUIQUCQSXFHCAS9EWBYNO'
            b'VJ9HEKOACWDDQMPPIOE9UDLIGZJTBLWLWOMPAOCFDFJPKVGVEHEUKXFULTKRGMGKMZ'
            b'T9AFWJ9VDFJIBSODRRBFFXIMONSSCLPKD9JZFSCDBBQIMBUYMMVCZVRCZJBSCIBJMY'
            b'VNC99VUAWYBXDTPVBXJRZVOQQJFKHCGLTZPYSWMNBQUYHOXZXSYPMWBS9HCLPQGRVS'
            b'9TNGKSGRWHEM99YA9BYUBJKJFGSN9XKQXHOJHKEIVSGYCEYARXUVJKVYHRJHMGDNVS'
            b'NAXQDXYTYAME9BXJFZ9HEPVSYMSDAPIBJKPJFOAO9XQYWNFCFAMNOMOPIAHWOFSSFT'
            b'VQVETQUHLCNAKGXICWCNBHMMJARWUXBLWFOLDRDYDOZAMDKMMXLGNSLNKHHMVKZTQO'
            b'TJSZWAP9LUWVHGPUKWNUJANUZQGLMKDUCVSSBQMZCGZURESHMUWUEEPHSVRETSDCOW'
            b'VNKMRPUTLRQCRHKCZBDDFQBELWMVCMIUELCJMKTKHMXSKSQNCAQOIJLSGWVOLWQCLG'
            b'QBQKKMHRNJKJNWZMCNDNXAFXWDBSIDOMSJFUTQKAEZJNWYXPZDVXQQFDYPJXWCPCMW'
            b'TWBQVSXCXTUFKWQKTFEATWSDFI9ETMWQOC9PY9VOMWDFBCLBRDTDSX9ICUQDEVCFLA'
            b'UDNFCFJCWIGNGLAILHKFBG9NK9YXCIBZQGODDGZZRCMVIR9EXOYCCODKWASLVLBDQY'
            b'VQYRBNGVSOPKZEGLWHTPAGPQQHGHVCTVDVBAWLDHUPZISMSDVJOAEUGOHUOOMBQEB9'
            b'TSRMSWIAIGGYOFQAVHOKMAIDGFHZHQNBDILMKIFQPIWZRMKVOVSYDXTMIFMHGCZJZQ'
            b'9CCYINX9DDGZHYMXXFEZXDTLOEOSNVULRD9VNBLTGDAMQSQKOGDLLDYDNFEVMMWYXZ'
            b'IRJP9LMNAOEBZ9FXKLACMUPSIABFBDNCJTVCCDZGFQU9OKDUEDPCPCTHKAUZ9ZZXSG'
            b'XHLYMFOVZXVHXFVJNMRBJJTSJPPBLZWMSOWYTLXXXYEVBVQJHRCLROVNPSVFLPRIIG'
            b'GDGSPIHOEYNQELOYHXDRHXOWOMP9OPYCL9QUUOYBQQMDULTMZMOHZAELOUDVBVRGLD'
            b'IN9DVYJPKYWILMWAYCLOPCXVBPPIQDILYPRFJZLFFTWFAESFTPBRRLJPATSIWKXJFO'
            b'HV9BMDLLBRLCBABP9UZPFADGUKWBJZMHUN9URWOALIZIOLFTVNTNTHBHYMWJUU9MOB'
            b'9NAKVBME9TAKCTLCSOEOBLRQLWHUEBTGJKYXBRLOCUKEVLGWSGNPHAYYHDPISOMDOL'
            b'ALS9VODLNXXOBBCNDNHSBEVNRMLD9JPYUVOZVMORYRQJWXYBWSIQDYXOSAZABKYINW'
            b'ZZJTONDTMMUQ9OLCUKBVCJDRPFVGIBJKAWEUVSBTIJWEVQP9LIHLKGHJYLBYOMIDKS'
            b'BETUZYEFU9SNPHQDWUXKX9VRTTMWHZUFQMBRWQDVFZXVGBFEJRQVTFFTQBTIGVHBDZ'
            b'9JPZYQEPW'
          ),
        ],
      )

  def test_get_keys_multiple(self):
    """
    Generating multiple keys in one go.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999TPXGCGPRTMI9QQNCW9PKWTAAOPYHU',
    )

    self.assertListEqual(
      kg.get_keys(start=1, count=2),

      [
        PrivateKey(
          b'IQOTORFDCOZORDLUUQAXXNFCILODCMVOOEJEGUCZTSFMQONYDALBCAD9YETATQRRRF'
          b'AHUAHU9VARQZPFWVLRUPXXPGDTQJDVJBMUMOBXFMEKFNGOIKUMZBIGNJGLWCPPCHHX'
          b'AJAI9RMRFJICRXVTIYLQWGTNRMOIE9VMHYAJLQPPEKPS9RZZJSPTDRRHRUOYOWMFGN'
          b'OVMJDPDJHRGYYWPTIYCVNITYVMSHGC9NLAJWCZVEHJQIXDZBRPSZHC9JNTPTSJZAW9'
          b'9CIZLHIIDCONEDPUWBXVAQHRTICUQO9UQLFPLJUTIHYMIBRUZNCVSCZT9TNZQHCUEM'
          b'TTOUWELUXJCMFRSZVOPBNZR9AGEAKUIXGOZQDGJKPOEYKDZJHDJ9RUSMUGPFIEQGAH'
          b'FHMDQLDI9HHWPBJFERFQAOIDPNGVQTARVJH9TJRKQJWRECXIUITPWNQSMDPJJEOPIG'
          b'YJZTURMZDYFMZQJWJVEVWFZAHAGWSGLNUIEDFRRSXSA9ZGNYKXGCLKRSAUIUKYZTGC'
          b'B9RLGBPR9MIDZMLJHVGV9UIOZDQUEJEGXBSAOFZ9XGPHQNLNUEWOKHDSOWXDBIGQKA'
          b'JMXQKJZTFHK9SHX9CVNINETMGCKJNI9PNSF9CZKZARPR9CR9LBWZOZXDATXDXYNFEJ'
          b'PPOE9VGUQFAFZKNJLAETHOUKAXCRFUKB9FG9IWCEPWIUHZPSAO9TRNPSQUDOZKSHER'
          b'YZYYVVAWWDRUDAEJLHNCRFAMBSRZX9ISXVGXPWXC9CHNUKGHLYZHIV9HFXWOXBFZPL'
          b'XOMMSK9DSWPBJMKTLBMTVWNTWDUXSI9BHPBCSQJX9ZLFGZWIEQPFHYVEXSIEWIBEGI'
          b'EDXP9ZBUQLJAON99KRVLHLBTLSIAENF9WLI9LBIXIUHLI9ESVBSGSKLXMFUOGJTJZD'
          b'QLMZTFDDJDBHJJLKBQULQSSNIYIQG9TAFZXPZOUJ9MUEZLQAT9SKI9VPRZ9LBWUYYP'
          b'SZRLMLDITJOJEULG9BLDKKEKPXJKUIDORYDVNZFABCLINHNHZEZRYRTYCBNMMOZSNC'
          b'FYADBORKPLLDWW9LHUHHLFDRP9VTDLUO9ENPHJXAWJRQKVSUGFYDGWVGAPUFGYTQZF'
          b'WSZTILZDHKHWGCKGZXPZFW9OMMGCQOXHPDIQDSZCVKBFZKEBUNYEBUIZSZWXMCSXPG'
          b'FYFDXFDZBQCNISTV9M9NLICOLTZQJELMBOOHYZ9WC9UJYMIGDSOVQXFFWBVMIXJWBM'
          b'KQQBCXGREPAQIWPJXHIKWYYO9LVDOSOFJXB9FFDZRWJACEPVSUN9YNMJJQTYIIENLB'
          b'IGIRJKBWSWJFRHQRUCCDZY99BXGJIQOHMEPPNHPVQFFMYUZXWRCBOOGKAADLXZIEEU'
          b'NYKQURKPAIBBYNFHJEOWX9SGRKSYINGKNORUGNKQMZUDBUJGWHALUYXII9XNKHVPYK'
          b'YMDHDZPWWWKYZESNZDXMDFHYJCXZMGQVTEIVOQTHVMDVUMQMMRSVWLFDGYEOJJX9DB'
          b'CHSGMOWHZOLDNBSWCUJR9PGOIRRTSFDVZWTELIICRQFLPNFZMMQVLYNWKAMWDRTYLZ'
          b'9OEJGIVLQQTFNOWKUQFDCWVATINKDZROEENXEAMOOLKNOCLGANKKNZKUKZGWSI9JW9'
          b'GUOARMFPHJPLSEXWXQATKBOTWMNVITSX9MWBYOGIIGMLJQXDYEWTJFPSUSDMMSXCOE'
          b'ULGOHEJYCSAOOTHDPTOYXTORYOLOVFMHYNPVHYQTGTIMSRDEEFB9VSBADXDNDQYRAA'
          b'SWQZGPCMPXH9YRRWOOKOFGIUOLNNEVCZOTXZPBQNPPLTCOXLMCXWSHNYYJEM9BTXNJ'
          b'YQM9DDLSTLZUBLAUOWDWKL9OSTWVJZPIDTOBQZKASTTUNWU9JECOZQXM9NXUOGKYKI'
          b'OVSBDDPKIQUBBCCBVD9AXCAEMSND9DDYOESN9SBGIVTYGOTLVUCIDRFLXOZWIYBSWH'
          b'XPUSURKBXNBFYHCWNINLPYFIYPIPPVCNSVTQAQ9EGQLICGTHVCLGWHJVL9NBRNJTWN'
          b'LCMXXQCBDVFLHONYRUBXYYAV9FSSLWBUIZCJMTJQCODHDZUAGZNFNCZPQRW9PRKDZR'
          b'V9IRFJAEDODDCNHZFBJPI9MBYAVNENYZOHOZCQBIVDSBKYQOBRZZYPGYSZCCGMAEXY'
          b'ZSGENLRSA'
        ),

        PrivateKey(
          b'RRVJULJGXMDUR9DEGFRBXWHHTSGMTZZYDQRWJKHV99XAWYSUEFLXUACXKSKALMYLWB'
          b'AMWVRINPNDGXGODEFFGYJYJELKVCJYPFXOIAI9SUVCRLUGXCMDTEHHJUWGYLDSVQSY'
          b'XTEVSAXHPLZQHDDUPRAHRWGNCTDIFDQVLWVZLNWZYAEXWVIPKUDZSPLYCFMCDKDSLH'
          b'CTZDNQYTRLGMFGR9EJISUOAWLEW9FWFBYCFBYGX9DBLUXOAIXEDTGLSXAH9NFEPVNN'
          b'NTUXSCSFYKXJDIMYVGU9BKLAIHYPYKE9KL9ALNGNEMUZEAQCICHTWDVLZSZDRWPXFW'
          b'VGQTZISZCSSKQYZBLUSCLMALMBHAQUEHPIFTXZSM9IFEEHKRTJVCXRGCL9EOKEHDKN'
          b'GFNKQNBECQ9SBFCAZQHFPM9AUURHVLDGRCHQYHTSE9SV99ECSZPUFSDWPXGVQZYAYA'
          b'WPJRKOOD9WGXLAQFGZLXUKEDSWRAPVFIBPPIPYHAKVJWTCD9YFDFZYHUNNEL9LILSG'
          b'WKPBRP9MXK9HUPSKECPWLGMFFCCUVWRXNFYSIZAZOEHBLMTE9EDOCSHHDGYEYEWYCC'
          b'UJQMXLLUSRXUVJKVAHPO9CAQEMCSIYQMSUKBIOGUDDJJIOUMB9TGZDDDXSBK9KIFGF'
          b'9FYOXWHRWI99M9NYISSTMINBJDVCFEZMGNNW9EYWPYKWQWJTFVPQVP9BWJ9VTTZNZI'
          b'MQBBDFAICJESMEGIJWEHHWBPLCPZRIDZJAZTEHRI9ELFMRASOAMMYUYWYNF9URSBWR'
          b'YCUKWYXTGXYMW9VCPAGPNEDQCZWMWOCLUKQPTIQKXPYMQBFWMGLKYHXPFAIGYDGKSI'
          b'PZ9EHSHWBHO9M9KDQDFOVKXOXDNSXHJMXGGMGTQPVGRYPHCDHMYWSGXQJKAOZVNFEV'
          b'ZNDNRAAAUPQHECGNSUHMDNBTBHXVEIOCTJH9CYIPJSLPWFRAKONPLALPXDOEOQTIZF'
          b'TPUAM9GLUEPURGKDJAEMTKUJFXFAYZVVVAFILXQLWXVFUHHLCVOKIJCCII9ONRWOZE'
          b'HAEGRHMTE9LPBNJPDJD9K9GNTYANA9SNHNWSAHHVCMNCWOQSJICTGCVEWOBHGTYZJX'
          b'WOLCABCFJTSXQUMGRZSGNELULCLEUYKUKBAEEXZXPLISBBCHJRDKAOBPL9T9RRZTLS'
          b'FZWOXYBRWFDWSNDRNPU9KETYVYBLYPGJUPRFKJ9WKOVGYBQVSSMXQCDRSCWSYPZBLP'
          b'NSG9JMVTEPWHTYOSQPTVRHVDAFVV9L9LEGWEGJKOZWDSPNL9CHGIIVMFGVOIFEZSXY'
          b'GCMWOBZLTHLBQ9KZQHQXSUGGDKGIRNFBMPRYWSQXFIMUGXEBSXVGRLX9XHLXEMBKET'
          b'MIVQIQWAIFAFVJPYVDUNTHLDCULGSXAMJPQKATIFDUDXNGJZNDFMLBBDJBFNGGGMVS'
          b'YPDSQCGZUZDQHLUZEWWTTHYITIRVSNPQBAEEALYTOXZTZMSJAGZXUWNWJ9UTMIDYBD'
          b'NDBX9RBTHURTCYOHWGGNAPMDDHYNGVLTKX99QKBJECELQSALNYOPFBLFLKXVTTACFB'
          b'NPFRDVHZRX9FPOUQMGFZZSKSTLGOTFUGLEX9GBAHXUPZUBSXBKQU9RLZHTIWHIVRMQ'
          b'LSKLSIYQIQSFEGLZQSLSCSUMDC9ERMHQCUEJCJD9SJDLNCHKEVCVERUMLPZQPLDDRE'
          b'XCRTCVJXPFFJZQPMKYXYQZFNU99SYX9DKSLLUTVNOMNXWFOGCHFZUVXAFDOUXEBZND'
          b'OOWUKMWNJAPIPLB9WUNJIPLMX9PTGLGDWLHSVRVUIJGNXEXKITXU9JDFSLWFZQUXRD'
          b'QXEILJVAJZIUSHYXZYJNEEAQEIZFSXKOKLQT9OUOZEEFFNEJUPCUTSYBLEXCEEOJAT'
          b'LTNKAYIYIQRUSWQNJELMNRR9GOXJZCLBOALYIIHZIWQLP9MQVJNRHHJIJQZRSAHRPH'
          b'ONWCGMZXGC9BGMBTYKSUXIMJIHXLGVQLWGCMGTOFADWBXNVUEJNAVBGIMVVZJTC9OS'
          b'PPBYQBPPHGP9KMTKHSPAVRWDQLPRRWAYDZWEHMPWDSIFVJSBCLDZNGWSZZD99LPTJC'
          b'BUBSKAWUTBXBMXEJWCLFNTQ9MYVTLOTNRFZSVVMQWOPZDSCRVN9WNVIVIXLBBXS9SQ'
          b'JMZHGZVHX'
        ),
      ],
    )

  def test_get_keys_error_start_too_small(self):
    """
    Providing a negative ``start`` value to ``get_keys``.

    :py:class:`KeyGenerator` can potentially generate an infinite
    number of keys, so there is no "end" to offset against.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=-1)

  def test_get_keys_error_count_too_small(self):
    """
    Providing a ``count`` value less than 1 to ``get_keys``.

    :py:class:`KeyGenerator` can potentially generate an infinite
    number of keys, so there is no "end" to offset against.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=42, count=0)

  def test_get_keys_error_step_zero(self):
    """
    Providing a ``step`` value of 0 to ``get_keys``.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=42, step=0)

  def test_get_keys_step_negative(self):
    """
    Providing a negative ``step`` value to ``get_keys``.

    This is probably a weird use case, but what the heck.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999JKOJGZVGHCUXXGFZEMMGDSGWDCKJX',
    )

    self.assertListEqual(
      kg.get_keys(start=1, count=2, step=-1),

      # This is the same as ``kg.get_keys(start=0, count=2)``, except
      # the order is reversed.
      [
        PrivateKey(
          b'FZHJJLLYUDCKQDIYNIZFTCGRT9YLLDNWGRZSDXILJPGF9OVZLEOSTZVPVSSMXDJJZP'
          b'T9BGACTLDAXVRBC9RNLK9KSYZK9WRNGMIQRYWSIJUSQNYUE9LLHRGVGHABQZDHTJAI'
          b'GJCHXQTUXYTCTN9GQSCLTRUOIHOAQZRPV9BWUNGNXAGCPX9DVBJFAQZCKREACZWRRH'
          b'UPQRJARSUFPHHYZGHDDPNMLIIWOPRVYQFH9XTBVVLOMLDWTNSLSNRTSLMHVQCFXTXF'
          b'YOVYOPZ9SCRUVVNYXKFELIDGUXFNGLKIHGKPV9FKFJAYKWUDEKNCMDIKXXTARRVJZE'
          b'VUJTZGJVIYRQKHCZFXZZCLBNLBFZRNBMWNU9O99OETP9UFCJLAMYXJTRXUDIZGXJBY'
          b'TUDURKYYZBYOPMCA9OTZDH9KBRFWNXXJMTZJXQUWTSGWGDMDTVGNRTEOUYNEKZIUKH'
          b'RKPBNYPPOTWMLWXJROQUSWYDMAVGJFWJEWRFNSFSIFEGWWEBVRRVEZZNXQ9YBXFPPY'
          b'JZNDEUMFBIXZZIB9UMYFDADLQKSYXWUWXBBRJAYLREVVMKHNQCTEUCIGTZSHXCRGYK'
          b'XVCZTKGORGFVFCJXVDKZJOPQLUQMANRKBHKFOVAIFLCREDUOAXB9D9PPFDOYPAIPMB'
          b'VXMQTJZDMKDKEMIJC9FFNHUQQYGBKJU9PVYRMVZMYYMGSKTPFDFCACEDBFIYDXEDKY'
          b'OKCASQFEOSHGXTFIRXIDWS9WHBER9YP9MGGBYWQW9FTDCLNYYF9KTSUGYQYYAKTFAZ'
          b'KSONKKWLVQIQGSSKBWCEVTFNGJRFGLUNBYAHXRVF9SIWM9JHHHVMOTSFWMGMJJLYEB'
          b'QJUEUNSNVDEDATVFQUYRTJBQDQPEPOWWXLXMNYBU9PFNALVPTSXLM9OPYKGQGSFXBN'
          b'SXJIEVQ9XUQOTDGFUGYIRPZMJIUORIPMIKCFPXLJLWBVRDUWSJY9JVZHYDPTLTSABP'
          b'QVZ9MTR9AACCKIM9VJGXPKUANNTCTAGCWMN9QPW9PCZKWC9SSEJFVAFIDBMWU9C9ZI'
          b'QOFQZQZERTIRYDYRRXOZTBBPKTBPKGBFZ9FCHSQURZSUHFQK9YKTFWYWHQXYRPACPS'
          b'QT9ZSDVTVEC9AIXSDOZUEDDOGB99RPABOUWPPSIQXVLPLXOZG9SNIPGGTFQDSNNYBR'
          b'JDOEURXDZX9DWQDBMVUZDWGQMVXBNATQPJWGWYIZTZAKPHDJAGK9VL9FWOBECFOIBN'
          b'URTZUQXYUJEXEWUVCLNDIVMLQBNXVAMGIUWKJYJZODLJWHGJFSVGAPOPUHIVLYDNLK'
          b'NBLPVCUQQY9DHI99MBPLEHNBWTSVOQGKBNZRRFHCDJRC9FFOPHKVUAIGWPU9X9RJPW'
          b'9XCICKYARHPXMNSX9QGWJYG9VKXYOIDEAZHMIYBNRDUJBSL9BOZXCYWGIQU9AOPHRC'
          b'CYPR9XLBTKMND9YTDVNCYXBDHPRZAM9ZYCQIV9KLVEVFSZEWWQIJAJIHCEHCRDPGMF'
          b'RSVYFJTDTULEXTFMM9HOAEHBSOJWLAUH9SWLQZRTCKLWRVLSMXDGPZDELJSSBFKNTI'
          b'RPVKVHPZKOXYWLBJJTRLOLLFIXVCYXIAFOBABDWLPKRPMOEJULMQDZQGHPXNXWDKJA'
          b'KAJHAAA9YQCFMQGDJHOYGCAXMXYFJMUKYJGMEDVGOVQQZBEYTFCZDNSXCXGMCWSWMN'
          b'TSTEQGICEDWUQMFRQTNAFPQMXQLJ9LLRQTDJFSWACVRCTYTMXLWBUFLXYHCXEZQTI9'
          b'RHKLJBXYWSNJOREXBWEWSSPXWASQYIFJLTXIKE9WHURZZORBUI9EFLNHNRAHMTDXMU'
          b'CV9SPLMTDBVGZQ9NRNQANLGCIVHQO9FIEIOTFQUVUXXZTVQ9JFIJMVHHSUWOW9UPVA'
          b'OITNCUDJRXKYWNPTDRNOPAYEENJATXNRYPGVAVVAXLOBIDZNZXWREYRPWZDCUVCNNU'
          b'SKYMFJRNXWSBOKWEBDBRFMHYKLCTVSOMIXAADVSGWNDHB9LEIXOERNWDRFYUAA9FLZ'
          b'RLJUDOEUAKULOD9OFGSPDPMAIGLJQRNDKGRUXUJZPXDPFWGCGEZIEMLVCQUWXMQDYN'
          b'NYBHAHAQLISVSEXWETWMDTQKCL9LABMHQDOGKJKKEOBCJRCKGVIYYILOZLWIMQYYWV'
          b'DCBXIIVVD'
        ),

        PrivateKey(
          b'KHYPJPEEDAHDQDEXMZIBFQNFXYQHTLMZFZZMYMS9PNSTBNMDLNJ9GTKBEXMWULOBJN'
          b'LEWOQZESAYA9JGZARRGRTYKEEDALDRH9BFHSPFSFQGUAMBLNWKOZYJHRGAAJYWUDOA'
          b'MRRPMEVTOUXDGGZIBMBMZSCTBOWTFWBOFGSEKG9SDZLMLW9YCJIWZIPUVIYMMAAPJU'
          b'P9XRCHZIIQVF9UUAGPZI9RALRLHLLNDURDVAEVPKGTFQWVTEFHQAWTKTQTVECN9IBX'
          b'DHTDGNDSEIUOXRNFZLNYEDIPKHSVUNPQVIDHSLVYKLMDVVRRTIJPATKZZKODHSQFCO'
          b'GOJDQE9PZJMEDSKYI9SAUNINEDCYFMXSEADCPINSSTNXM9WU9XASQYHUQWVKAXYQZV'
          b'MCRCRSWBCBTZTGZDKRWLMPKXHYUH9IQSRULRNMAGCWG9YSOIKVHHH9MNAVLXJLWIOY'
          b'DOZEQRSM99ZDXBGELXLGZZXBXHUIIGGQPUFYQBQUYMH9RWBJCZJTM9RSVOWDDYLRHS'
          b'LRUGSOMFUNROBLEPGIMCYYNLALKDMPRWUARWZYC9IDJRENFKVMTLZQAOZDMOXNZ9YR'
          b'CTIZYGMLMKJZMSDGHUFDZVFEVLRXLPMUABUYQWECOPJBZITECXJFUXEKFDMQTE9RAK'
          b'MAJTFFJCSUMTVCXXUFGYGQXNFUG9EGZPVHXV9PEABWKXXXIHBDNETVOVOVPIXAI9UV'
          b'UNAGRSC9EKXILF9KLZPDCZOCLKFZQXXQLENUTVYSORFDWYUIDMRTYLT9TOYPKKIXQL'
          b'YQVDIRILI9SFKXAOPXRHHZBS9OOOMSCAYIYDRNUWZFQPIR9WZ99X9OD9ZJCVDIEYUJ'
          b'QTHRARYNQXQGIGZGQOQPZYUYPFNYMIXCDVLDZFBBBUTFTFJGFAHYBCFPCXGJITGCFU'
          b'QAOSAFHILVAZHT9LIDZVKTVTHVUHHMUJFTJHHMFZNEPNPWYZCRJSGSIIQSXUDBOHO9'
          b'EMAORRBNKVYLNMPZKGEBQRQMWTUAN9HLKKTIWFXG9VMOT9TGZEFRTODIILAW9FDLKF'
          b'WAAKT9FTMLMKTGOQFIDEMMPIPHXJPOOVKNBUXEZKHISSXWGWTQDGJWLTNIFILMAMRY'
          b'YNYQQBAADIODPB9EM9EGFGZXKGZXUVXYKGYFJZXIAOYODYBCIXOEWPGCGRZPFKQVWX'
          b'WNSSCOPBXQCUTI9H9MJXJQVKBGCZEG9UZNGAFQKEUSPXWSQSKAXRANIIUCAVSCDJYM'
          b'OEAPRIYJQTQFMW9ZEWRIANKGHDAPCPPOPECAGGZSQXTGNQBNBFCHK9VLLCNURNVVN9'
          b'ITLGSTZKCKZLUKQMU9NORQAGARMSSCWJWOBSBBCYVLGOJQQCGSXTNNAFWPATOSMHKW'
          b'KAZVOGGHOZVRPLLTKCSSUCNPALFFPJGP9MSOSSVMCQ9WRZPEEWUEY9IAGRTKYGAMWV'
          b'BGWFQXBCFMFOWRZFOASBVDFNFNICAZZJJOFJHNINGQIWPVHCTXSYJIRCMDI9TKRVKQ'
          b'KOHWOFVLPCIDPJMYJIBWBWCWNRGUDD9VAQJQFKSGFTWMYQTAUXNYCXQBRDZJFITLLX'
          b'JHBYA9LOZTFIYMIFVEVAUROMZVTDTGUYMCXXNKDSJKTULMIGDKITLHCYYAHGGFHBUI'
          b'ZFNQ9JFMMXFOJUCTLJKLOJZLSSTCJSKDJHGMBJMHMGZEKAAEYY9NUB9PXLULFSAKSJ'
          b'K9UBHSAEQVSYZ9SUQPJQLJQECEURDUFNERRYVWAIUZLNFFPXOWEIOPQDVOKMACRLJZ'
          b'NTY9CRCTNPLLYISOCDPYXVYQJNCWAZPLQBNEATUBEPLHPPG9RYTXLGFSEVRUOVOBUY'
          b'CYUEOXSKPSIKBGWCQQCJZRWTXQOWKHPFFZEDXKT9ZONNWYVEHLO9YBPDUQIWPBQXYS'
          b'QHWIUSLCWZTMKUCYGIGPEOVZSNXOIDCOZWEAEPGCIIDNUTIHVKDWVXXI9LKCWKNKTW'
          b'LARZPOBKWKKZGZUWAIODFDDMWNQQEQ9XCTXYWVXLSQPVWYKWGBKOWSXGTFTCVDZUYS'
          b'EGXOEEZGTRRAFDLDOI9IIKLJMYDYPNYZFGMVHNAZJBGWMAPWVV9ZBGTANAPDYCKUSD'
          b'OJOYOFRUROMAXAOA9JUTX9TRQZ9TBTDEMAWQDV9UXAVB9WLKOS9VLEEGRYU9FLWUSA'
          b'DVNPBIFLB'
        )
      ],
    )

  def test_iterations(self):
    """
    Using more iterations to generate longer, more secure keys.
    """
    kg = KeyGenerator(
      # Using the same seed as the previous test, just to make sure the
      # key generator doesn't cheat.
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999TPXGCGPRTMI9QQNCW9PKWTAAOPYHU',
    )

    self.assertListEqual(
      kg.get_keys(start=1, iterations=2),

      [
        # 2 iterations = key is twice as long!
        PrivateKey(
          b'IQOTORFDCOZORDLUUQAXXNFCILODCMVOOEJEGUCZTSFMQONYDALBCAD9YETATQRRRF'
          b'AHUAHU9VARQZPFWVLRUPXXPGDTQJDVJBMUMOBXFMEKFNGOIKUMZBIGNJGLWCPPCHHX'
          b'AJAI9RMRFJICRXVTIYLQWGTNRMOIE9VMHYAJLQPPEKPS9RZZJSPTDRRHRUOYOWMFGN'
          b'OVMJDPDJHRGYYWPTIYCVNITYVMSHGC9NLAJWCZVEHJQIXDZBRPSZHC9JNTPTSJZAW9'
          b'9CIZLHIIDCONEDPUWBXVAQHRTICUQO9UQLFPLJUTIHYMIBRUZNCVSCZT9TNZQHCUEM'
          b'TTOUWELUXJCMFRSZVOPBNZR9AGEAKUIXGOZQDGJKPOEYKDZJHDJ9RUSMUGPFIEQGAH'
          b'FHMDQLDI9HHWPBJFERFQAOIDPNGVQTARVJH9TJRKQJWRECXIUITPWNQSMDPJJEOPIG'
          b'YJZTURMZDYFMZQJWJVEVWFZAHAGWSGLNUIEDFRRSXSA9ZGNYKXGCLKRSAUIUKYZTGC'
          b'B9RLGBPR9MIDZMLJHVGV9UIOZDQUEJEGXBSAOFZ9XGPHQNLNUEWOKHDSOWXDBIGQKA'
          b'JMXQKJZTFHK9SHX9CVNINETMGCKJNI9PNSF9CZKZARPR9CR9LBWZOZXDATXDXYNFEJ'
          b'PPOE9VGUQFAFZKNJLAETHOUKAXCRFUKB9FG9IWCEPWIUHZPSAO9TRNPSQUDOZKSHER'
          b'YZYYVVAWWDRUDAEJLHNCRFAMBSRZX9ISXVGXPWXC9CHNUKGHLYZHIV9HFXWOXBFZPL'
          b'XOMMSK9DSWPBJMKTLBMTVWNTWDUXSI9BHPBCSQJX9ZLFGZWIEQPFHYVEXSIEWIBEGI'
          b'EDXP9ZBUQLJAON99KRVLHLBTLSIAENF9WLI9LBIXIUHLI9ESVBSGSKLXMFUOGJTJZD'
          b'QLMZTFDDJDBHJJLKBQULQSSNIYIQG9TAFZXPZOUJ9MUEZLQAT9SKI9VPRZ9LBWUYYP'
          b'SZRLMLDITJOJEULG9BLDKKEKPXJKUIDORYDVNZFABCLINHNHZEZRYRTYCBNMMOZSNC'
          b'FYADBORKPLLDWW9LHUHHLFDRP9VTDLUO9ENPHJXAWJRQKVSUGFYDGWVGAPUFGYTQZF'
          b'WSZTILZDHKHWGCKGZXPZFW9OMMGCQOXHPDIQDSZCVKBFZKEBUNYEBUIZSZWXMCSXPG'
          b'FYFDXFDZBQCNISTV9M9NLICOLTZQJELMBOOHYZ9WC9UJYMIGDSOVQXFFWBVMIXJWBM'
          b'KQQBCXGREPAQIWPJXHIKWYYO9LVDOSOFJXB9FFDZRWJACEPVSUN9YNMJJQTYIIENLB'
          b'IGIRJKBWSWJFRHQRUCCDZY99BXGJIQOHMEPPNHPVQFFMYUZXWRCBOOGKAADLXZIEEU'
          b'NYKQURKPAIBBYNFHJEOWX9SGRKSYINGKNORUGNKQMZUDBUJGWHALUYXII9XNKHVPYK'
          b'YMDHDZPWWWKYZESNZDXMDFHYJCXZMGQVTEIVOQTHVMDVUMQMMRSVWLFDGYEOJJX9DB'
          b'CHSGMOWHZOLDNBSWCUJR9PGOIRRTSFDVZWTELIICRQFLPNFZMMQVLYNWKAMWDRTYLZ'
          b'9OEJGIVLQQTFNOWKUQFDCWVATINKDZROEENXEAMOOLKNOCLGANKKNZKUKZGWSI9JW9'
          b'GUOARMFPHJPLSEXWXQATKBOTWMNVITSX9MWBYOGIIGMLJQXDYEWTJFPSUSDMMSXCOE'
          b'ULGOHEJYCSAOOTHDPTOYXTORYOLOVFMHYNPVHYQTGTIMSRDEEFB9VSBADXDNDQYRAA'
          b'SWQZGPCMPXH9YRRWOOKOFGIUOLNNEVCZOTXZPBQNPPLTCOXLMCXWSHNYYJEM9BTXNJ'
          b'YQM9DDLSTLZUBLAUOWDWKL9OSTWVJZPIDTOBQZKASTTUNWU9JECOZQXM9NXUOGKYKI'
          b'OVSBDDPKIQUBBCCBVD9AXCAEMSND9DDYOESN9SBGIVTYGOTLVUCIDRFLXOZWIYBSWH'
          b'XPUSURKBXNBFYHCWNINLPYFIYPIPPVCNSVTQAQ9EGQLICGTHVCLGWHJVL9NBRNJTWN'
          b'LCMXXQCBDVFLHONYRUBXYYAV9FSSLWBUIZCJMTJQCODHDZUAGZNFNCZPQRW9PRKDZR'
          b'V9IRFJAEDODDCNHZFBJPI9MBYAVNENYZOHOZCQBIVDSBKYQOBRZZYPGYSZCCGMAEXY'
          b'ZSGENLRSAHJLPRRZXCTAXMYUCDVEFB9J9FMTXZSVXMNBTWIGQZZFZZQJAKWMQGODXV'
          b'HJWNQVMPLTJ9OXYKPWPVVTCCCHSKSDZKACZXZHOUHHRSQHVBZXMOTPHXZLJSVXX9OG'
          b'WYJNOEZAWKVSAVSEOSBJKCBNMG9EHFEBKDWLFLXNVWUCEYBQFKLOFLSQKMQIZCZZMI'
          b'KJBCNKMYMKTWABX9URWXDRINXJQA9UWFBKQZION9JSHWQBOLEEWVXDODFJEMBVGLMM'
          b'ZGIOSWOAEBNLL9KLINAZIW9FAYSRSJUGFSDHENJOSYOBTFPWTGFHNSDNDHGDMU9EAP'
          b'UVDQYXTECZPUYHIVAXCNOTUNKTZMZZRXQUFSMEKJNDLP9BKWLVWMGVTJTGLEPECPOB'
          b'LHTGAFZMUBTSBEBTNXYJRMRVWBJLSZBCECAXZVAPMGSKGQUDOKEXN9JBWDVDRRIXNV'
          b'NJQSLNXKNHFKTUDFTNUQWIWQYZQGS9RABWASMDHUCGDOYTSDUNPLSMNAGKYEQZTGAN'
          b'CMJFVKBJMMHOVV9FTMRMTFMBEBNMBWXRRGBVEBZXSNTZRGIAQPBNGAMQEOEMCUYSLT'
          b'M9DKTSKCWDHTBDEQH9EVWYSDJXRVVEWARWXHKHZQCOIZWJ9JCYWOLUBPJQGCFECNXS'
          b'QTVVLMQXRT9UEYVPYHXSZBWISVKFZXTFVQTNYSWBVKDADATCKDMUJLPPTWQXRFWKDM'
          b'OGQDIWRKEZLYZEACTGPQFDCIXRBBUKVHQVRDHVLICJZCWIORJWPMRIFLBZBLXKXIKF'
          b'PDNLIBCTCYYNYDVRILCJZXHBDC9RKRNRHAJCUOLPTSTQIWTYARFGXMBLRSOXDFQQKV'
          b'RLWNEVHZGKIVKEBQSZBPRAWAYOOXIGEPOBZVQ9CJHYHIGV9VMHNNRVANNMKTKMLZLB'
          b'WFGGSOFUIGLJUEQUSBYCWNIRUMICHACAWIOSRKXIEEFOIP9RF9RDSAYHWJOSUQGEIW'
          b'WTYRBJOPOPJZWZF9GZWUKMBMKUMSJI9UCPQFXNWCVLGL9UPZPLG9UDOWLRRZUMHOME'
          b'MRNTBBGU9UCDKJOUEITNRXIMHEII9EYUADPLRTFTUANAXYWVMBLGCEUAWHAHUWJXML'
          b'QWYJXQYNCXL9VVDTOKLMYV9OUWNDTCQFEDELEETNWNU9XXRBZGMYKOPBXZEZRKEJUZ'
          b'BAMUQDQKQRQDRWP9RXDTNQGPHDJDBVKBZ9GCOCRCKXCFSRIGRYEXGYOMKZOSQQGMUU'
          b'QGYBSUDUPFARYRP9JCQAHMIJUZCUTNTDGBVYKZHKIGXLIL99TVWNKFHBJHDPSPWIVG'
          b'ZYQYAOAWTPMUALUZQKCJBPBXPMNYJJIPCNXCDFYV9QQBJFVGIKIZWPIZCAOKNVSI9X'
          b'SUWZEHPORPLKNYOYTMHSIZAQ9ZLXJHWBHWIA9ACGLOB9UZJAFEYNS9GLOMNHJAHYJ9'
          b'EBCHWVYKZDPC9G9XKBNSUBRLCVORIPAZZQFMDNUDPRVXTUKHXBKGKSYVZBNXRASYNC'
          b'URANZDYESQDIYSGCDI9HEIWARKXSRCAQO9MZYIMRKZBZHYWMHNWADLRFAPWSUOSRXD'
          b'JYYMYTZXEOPVBJXHEW9VZQIXTPRIQWGEENDHTMOHKBEOWMFO9XSGY9AIOUBOYBIFBD'
          b'EQBRRCKJCJSWAKKZKBUQTAOKFSEHLJPHPTQGIAKDOCVJRUESHWSVXJVQVB9WIRMZUF'
          b'PDTJVTVSEGMSGDNFOGGOVNUELOYGP9AAEKL9ZGKFCXFGFKXHX9SLQBCWVXTAAV9HUP'
          b'BLETXNLIYWKMAKXQTFYEADOYUGFGFLYKPDMEAXXXRQRBIJJUKBDLU9KAZPWCGWCW9K'
          b'IHVSMMNHKDK9AMDDCLU9OYAXQONZWHDEWENLLGZGYXZJEIXHMAJKINTFXPVCMGPLYR'
          b'OGSQVIOXKJOCZGEPGLP9MIBVQXBBZQPU9L9QZOYQPPAGDJTCLDUELXZGQSQFKOQOFE'
          b'HKJEMLCJ9ZNWARUEAFEDKQBQ9NKT9UHASO9JGXW9UBZGX9IEZFCYCWBINHW9DDWBUK'
          b'QZY9FIKKWVRAMGMSZDVUCSZKMPWHTGQUEPANTUWIYCKHENOG9OQKINYIPM99MAXPW9'
          b'UB9BQWUXZKLJOSPEMGALNNLJDFYRZCSETCELPMEOGYBAXJ99NSTIHZOILSAUXUZSTV'
          b'BPNVFNHJMFIJNWJDPZ'
        ),
      ],
    )

  def test_error_iterations_zero(self):
    """
    Attempting to generate a key with a number of iterations < 1.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=0, iterations=0)

  def test_generator(self):
    """
    Creating a generator.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999IPKZWMLYYOLWBJGINLSO9EEYQMCUJ',
    )

    iterator = kg.create_iterator()

    self.assertEqual(
      next(iterator),

      PrivateKey(
        b'CXNFADEBFFAAJSOODA9ZOBXJPCWKKJCYOW9ORHEELC9ITPHLLDUFILASQMXOBQXGQD'
        b'ZVTZOSJYIUIQBIDDDJEREMHXXTOV9O9EOMQDVYZIFQDCJTJTOAGDKCZNUENSOBWQWI'
        b'DHFU9MIIJMAEGZQNXRAQVLGWCKRKWDDHLHUIDSDCPZURBPBIYUEUXWZVZS9TCBBMLY'
        b'YLSOVLTHTBTNAURBYJFMJSDTYRK9GEFJLDHQUSIHSSLKBCJ9CGAJMCACGPVXPYBLBL'
        b'WECXXIHMVXMVWLBVNEYQHFNPUYMLLCWIHYGJHGDGHQYLYOJSSKFHOLMURNDZSYVVOQ'
        b'DKAFOHL9KVV9DMJPKFVHSDNMWULHSQIPSZJFLBQCCPIVHIYDZ9DSXCKFL9WHNGIABK'
        b'BZLOIYRCXUCPNBYIZBZNJE9CQLAJTAOHBQGMS9NKHEON9PJTOZNVHGBUZHOWEKYNSB'
        b'B9GGAPRUSNPOLIYDYJHDAWCBCQVWECDFSTMWQRJFWMHJPNVKPKTIQCRXQXRKPZVTQK'
        b'BMECURCKREMQIV9MNZIRYZR9DLEPRYLTCDNNHADYK9GZJKZDODIREMY9QOEZTUXYYE'
        b'ACHTMWRLLCHT9WMVZAG9YXQZBNMSXTVDXH9FDWQMUVADNQUJHMOVSZORILIYAETLQD'
        b'WGZRBSUUCAZMKKVHDLYFJDYLRNRFMVRKISPRAQRFY9TDJBHODVIPKQJVFEKHHUHENN'
        b'TIACCEQJLGDORLETYMMDOHQYETGLKCOHZVUZYAOD9IUJHSXOZGPUXUNWQINBOJJQEA'
        b'ZHJAJ9KZREYMNBGXVZTEXYSPVAJKKMHQOMLITYJKPEXGPREMJSUTVAUTZQQHPCDHTK'
        b'VVYUFZVJMXJBLHUWRTMUCNHRMG9VWXNTDDFFVMTNFSI9INCTSATR99UEDOUUNXAQAA'
        b'VPAFVFDFLWMLIWWWOWRDJOVECEVVJWOFVMRJKWZGBAKDLVAWGGLLYBQNRGVIWOZNMM'
        b'XWYNGFZ9VTCBCQQMBOUVXHXXVBWMUVGMQQJQZPDAPJHSYKKUM9TFDHBGIHAVXUWJPH'
        b'OAKIQFMJPDZNRYORGCSWQLMKSHBAZJEJTZATSTEOVBDWDKFBQAOIIHLGBKME9AWTIK'
        b'DHTICZRPFJRAFPWYHWFFKYWSJDVQ9UBUBPUFRHFVDNYTMRFKKYVIFKGXTYWCJXI9SK'
        b'WISZJGZLPMYZRCZPUWYKT9BSCXDUWHAXAZSGZTUBERVDMIBFQLJSLQBLXXSKJVRXEC'
        b'LYPQNBBTRKAXPJSCGZA9ZXUZRUYKZWUSTNGMEKGXYADAHBGLGWMFICHFHZOOR9DIWK'
        b'TPNCZQUCTTHVLNCVYMNLFSHUBHRKPRNLGEJJUYSNFKAIOHOJRPKRNFRMWZFUYOEKTY'
        b'EYMBNPVYQXNTWRJYESMC9FSXCCMWV9KAQSCXBXYORCABDYTBIMKQRNAJTERZODHRTB'
        b'JSIHLDIWNZMBQEENNURMRYLFP9RLARPZWSAPZGCIAMNGQMATVSKLOXUPGEGTOXSJKZ'
        b'9OYAILOLPCYZNCBRKP9JXCSHZAUQVRLIKJ9JAHDXQ9E9QQHSGGDGLESCQ9NWSLEOAC'
        b'JQOXSMJBQLSRXFPLNL9EZBRLCRMBHNUMONIYK9NFOXQLUKHZHPIAKARXJHZUMRZHQP'
        b'JLDSSAGFBMYONHBXZNVWZHXT9G9XWZSMOWCOXXLMOJCJBZDZ9CDXWRTVZZSNVSIYSJ'
        b'ZCCTHPM9SHZ9VRXKYAHQZYQ9UINIFJRQNDDCDSZNSTLAPDGEJJEAKCPTDDVPLVZYJA'
        b'TGTNTGRIUKJXPOPISTGQINGBFKZ9MBWZJSPORCOTYSKLCQYWGNNYBCPHXYOPECWVFJ'
        b'MITWHHHXELMJPYDOLJGDVPOMCRXINDPCSHFY9IUOUAIKWOCFHTYBHIPGVDSZKOOTBC'
        b'9DVKMCLODRCDZBYC9APPPQKUFXV9OZGKYRZFMYHMTZTPKBWQCEGRPKEZKEJOXQQIQT'
        b'BZPBBPNQJBIJCVY9GDUFHEVPCHYNTIZUZPEWJYNILZJG9F9AEDLIFTXNKFIRHYDRPO'
        b'QDHI9GWBGGD9KNSZRK9NUCHDBRKXPMMQPVWJFJTRDIINUNKNGEZFENEJO9Q9JCFJOT'
        b'WXCDFMCZZZSYHXASPPSJXHZZVYZIFFNTVVPNPFADKY9EPAKECAVQYCPS9FCBWXLKBQ'
        b'NG9IQVLTC'
      ),
    )

    self.assertEqual(
      next(iterator),

      PrivateKey(
        b'EZQRJTGHMGDBQDEHEDGJZA9VGEIXKXBAJWMUHLJORWMFFZCDVKDWPZT9YMPJVDHWBC'
        b'FRSENRXLLBYTRPCHLFE9MDXWCTZIGOSHLQAOPZDFDSWXXK9LOX99DMMOO9LNWRDACR'
        b'KVXRWJTPASBPOBZKL9OGTDTVAZZEJCAYMQFYLZSUFKDFTTDBWMDAAITSVVHPJLGTLS'
        b'K9XWHDBZTT9BSVXJCYY9IJ9MIMUJIARPJONSFRTIIFKQXUBAHRRF9OHSYYXAILQDWD'
        b'HJGKYLRZSDTSKPXFMWIAVXMZXCSRFTTT9QCHW99ALGWWAGXCOIOLKVUFWDAZZPSWHB'
        b'RFEAAG9FVMFXHPZCPREPUVZYZVVMLJWBRXBCQCOLHDVOBJYCEZDQODXKTTDSRADBDB'
        b'KATXM9GGAWJBGKCQZNIPI9CCIMNNNCNLIBFU9R9TPUYZRLUMDSBGAKBFJGITVORILR'
        b'WJARQSTEJBVDBDQAYAXBSFGYIWHEQXCPMZGWXDXDDXPTYXMGHRPYMZDABDYCQGUBTA'
        b'YXQNOXWAKYEHBGDZCVRVYMSCNMHBWHPBDISCZRWNDETGITXBHXXOEOIKVLNW9IELBK'
        b'NHE9RQESYEATQALWGDEWIZENQODBCVPS9TPUWAQLVOIDFMNPGIDERXST9MVUEYJGVS'
        b'JYRF9ETXQNXAKVQKUXMVSSECZPXXE9MIOMYDXIZZXNCYVWBBWPFPNHLLXN9MAXXVRX'
        b'XRDILJGWHXEPCDT9FYFMONVFIWIPV9HYLCCFEESXSITIFIRKDNLBCONFNXWQCLIZQO'
        b'YOPVOHEDD9NAFHDESZCBMENLFXALDOUHCVAFHBIAICRDNWDYYUMTIPSCXVVKBGCNUN'
        b'WRBEVBAHREFJDQMXRMXEBKCXMVUGAOE9BALTFLFGNOYGARDFIKZ999SGEAT9DDJMGW'
        b'GUVDPUMOUEGEFKOHTOYAJ9BYWLXSHNFUEVJN9KETAIFQQZAACSKTLWIBQGFYR9CCZR'
        b'HNWNJEEXGAQRROW9WODHBICDIIGEZSMEZHCJBNVRKQKOT9YICETQUNWOJMQNNLYQSC'
        b'QDHJQBDMFIRQAVOAKAOAJUVKYZN9BCOUCGOIXGNMSCSNUQV9POFLLESIFFCAAVNQGU'
        b'ZSVBLRWLVUCCNAKOVWSYUPEQUNRABATEREEYOQQVEWJJHQZDSMSEYSKHCWYYPRNV9P'
        b'U9IKVIUWVCNNWEEVHNAMWYHFYMYIZVAOROXENARJZSIUONONKLBEIOPIAJCTNPDHCX'
        b'BHVWQHF9GZSRYJIASXNNEGWIEZUMJZXXATZXABYXOCSSMCZUVOWQSMMEDJALVS9WLO'
        b'TNSQRUKYQOIQCVZOZVY9OFKUZYOGTMUJMTMZZPMTROLQITPIRLCKKZJFBFAXFDGMQY'
        b'CEENIRTFGKKSLLYFSESSXZXUVMSRWARCV9AFKWZVTXTAWUMIKTXNAWLYMSSLCTPYWP'
        b'FUKHUXXRCXXSPGS9SVQIGKHUFRZQKWXFPQGTTFTLSPFPEGWIBXLFUOLZG9GMVWIUDT'
        b'9VASDDSEXVLKRVZQ9BXC9EOECROTWUSPVG9WPPPFDBSLGTYXVAXNOYRAVBHZJBBKED'
        b'NBAKIVTPRSNUHAIAHKCOYTTELLMN9KBBLESBWQRUGTQ9CXSH9RFY9MECJVBSAPPFCN'
        b'GRS9GPWEHQNNCYQWTYVLVOEIVKEBYGSXQWHAKUZKUKJNIIKEGEDZTWZLGTREKZVPKT'
        b'PES9CNXFULSOQXQQUPSRIRZTWAOGAIBRFNWYILQXXKZPZSRXELFNSKSHFCWWQDFXKB'
        b'GCETSHDKLIGABYCMWKHOHDBSBLSNVOCGEXZZIALZQFYBKWDAGEAMALALDELFTGMYJQ'
        b'NNDAMH9UAGIYPAYTATXMSBXUBQRJUEXNBENKDTMUBACEXMGJJZUJIPGQMUNPOEFTDY'
        b'FSPFWWK9GYUHMLGFYTDVBUSRMGQ9QXUZILIFLMOD9HT9HZXHOLYBHYIM9HZJ9CJBGM'
        b'XQJAZLCCKY9XAZIK9VGZEGIZVEEWWYUVRPXQUWXHIKRADCZ9VV9AMESCGRHRRKVZMM'
        b'GIDVWDLYACPXWUDYMSSKKGLAOSNWUUOJGSMBHGLBGPNDSGINIFODHCGGGZXYYGUZHI'
        b'YMIVWIBDE9FZFKRHMVYJBJSDVNQ9HWLEMMIJLAQNOHXK9WQELXWXIFHCNSR9QSFQOV'
        b'H9L9OHTND'
      ),
    )

  def test_generator_with_offset(self):
    """
    Creating a generator that starts at an offset greater than 0.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999FFRFYAMRNWLGSGZNYUJNEBNWJQNYF',
    )

    iterator = kg.create_iterator(start=3, step=2)

    self.assertEqual(
      next(iterator),

      PrivateKey(
        b'DEILIWDSOTXXVPJ9GKCY9EKLJAAVNL9TZQUYGXSVFEMWZGPKUWFQUQR9JL9SSSHFYZ'
        b'FPZKACLPLH9CHQYQBXDTTJWXOGL9AXPRSKAETKENMJDTRGCLDKDISPRYVDKPBIKESW'
        b'THBRYKOZUWFAKSJNEDJYAUKFGTURZWMSEIPIZPYDQHMXNTA9WL9KLFKRXCDCOHORXS'
        b'ISJWLPDSUTVNYDENFLCKHGKJLFJEIXUUBESWXTRTWZAWDCJCORDDEWCZUPTAKKFQOX'
        b'S9WNRNUQMMLRYJIRJWGNYFPIFMG9WO9EYBFZJLNGLASAE9SBXAI9LNVFVUWXPXQXVJ'
        b'EBVLDI9PLQTWKXIATCVUHMCXGGVOAD9HLEIYWOLYVDSY9DVHBUVXQJBAUOUPFVGWMP'
        b'HNLIIHKTDASXQVCM9NLBBGFJL9BYHYAYHNSXYWISFRCHELLUKYENACVAQRDBCXOZNF'
        b'THJ9WJZXSNZBV9ZNUGCFIHJWTSCGRYATQOHWBVSONUOJKOL9RZNTCKPDBQMXBHUEKT'
        b'GR9QBFFGPWFKJWSTFIORNLYHLIVBCPMEOGARGYZHNMBPMLQPPHFKFRAQYIPIVVEIXN'
        b'9LGTM9LMTEOGTJPYAAXTJUFYCBUJCYNLQPTVZXMMMVUZCWVFYNZHQAAF9IIHV9ERSK'
        b'K9YGICMF9YIXWVFLEBIKGI9NNI9JFXWMXTTSOAXCVYNOIFUOULT9MLAVE9HOYVXHMW'
        b'GKB9RFIDVWXLYXOHUGCNJNFDKWTXBNVBWEYDRBXDTQNLEBCDLSDNEHMKFCSEUBBBBC'
        b'WKUHOMDFSMZGWTZOOCQPURKCAWT9ZBFNWTZQNFWBWTOXXTKKJDQWAGBALOVEJWTPAW'
        b'QQRKPGGWGOKYOEUHNVHOPECDTFRBBDGJDFPYU9PPE9QTOFGWABAPNIXIJYJJAYKNGO'
        b'DFRGJX9N9WRPPPKJGZOVRBYZVYCBMMCKSXUCDPWROYUJFXXCSDRGRU9FFEMCSHITWR'
        b'URJINOEOTPBRNMKKCOGMWKLFJFNOZUTCMNKESNJOEVYOPMKODCXSETWUHOUWVSDHWY'
        b'YPUJAYNAUQGZISKRRSVLPQCYTUIEUZSCKVRADGHM9KPPX9XUVW9VGILBDKLCMCLYEL'
        b'YTPPXQKGNEI9TXUGOCYKEKMCOFBRXNDRZRDPPEBLATOPGTUHMGTNEOB9BEHPIZQGOA'
        b'JFCC9VABHCZLHQNCRKILEDZULPCNRNY9RMKVVARIJJEWCJFYYPWEEPCTMZBHAEAUIB'
        b'MXJMTLITMKRYNYHOYTFEANRUOSHWCHSUULJXRMYEBWMGNMJSYOKMTLIOIWFGQ9GXLD'
        b'CFEFBKIVKXYCRHMAHTP99EPUSLPXAMYHCFRSVHWSOEHJVJLOCXOSDKSPYWG9XMCPPZ'
        b'NAVULZWTECRGGVWGHFMRWPAOKLXOQMSGEFSVTCWITKGXTWLGXNS9WDXECMMMNFMJXG'
        b'KVLXR9G9HOHYHSM9IE9DQOUYAREBHFWIITROKMQPVFHVQZMWUUFEJRZTADTBRENMDM'
        b'PXMYZSIVTLYHDDLWWNWCWGGYAW9TOQUTRENBGAJCMISIWEPCBFUHPXIDKUVMCYSNBY'
        b'AXESQQPUSNKPXLLAUQT9NDYV9YMIXZAKOFCBLAFQFJ9VRXAXJQLZWXVAYGTRIVNDVS'
        b'FCJOKWPRKRFX9XJFFAAZINQLTOHEOTINY9XRAQGGCENHSVGZKB9GHJEEHRUSBLZHEK'
        b'U9BOBGYRCRKEPYADDVNKYKJWMFHLYUG9WLTDYELCSLDWFPXQUWVHULQFPUUHL9NWRZ'
        b'9U9EPZHCPRBPT9BFA9JGVKKKMAMQHMMZHLOAIW9YLUVFT9LPG9MFDTAQBVZWZNW9EU'
        b'9LNBZIQWOZAHXRWJIVRCMULGTOMGHWG9DWYOBZHBVD9CRABMXCASGRBGXSNRTOFZSP'
        b'ZAGJWHZIFZMZWQYDATWGFWEKYYXJCZ9JFJOZKDDHHQVXYSSTYXNHB9NPNOKMZ9RNLX'
        b'AOL9IADMCIIAATLNEZCNCBCUULZTDIWOIZUHLVQWECIYCDATIOUWOMOPBMOTBYRZJN'
        b'KIPZMGAJEWXVHROJPHYJNJJVNEMD9WXNSQNARYGEYKQMXUTEWPUGJ9XAEJXWXHOIYM'
        b'BNJBIYRWBTLVVZTCBJFEQVZIWN9TFFYDLEODFTXECOYHZGIKNS9ESHQGJNXMHPSJMB'
        b'IYUAZJFTW'
      ),
    )

    self.assertEqual(
      next(iterator),

      PrivateKey(
        b'ULABUJASASWFHYTHWDANSXEISTMZXPFTAGDYNUWNBZOBNSMR9XZBWMIDVAHUHYIXPT'
        b'STHTXDVEFUBOGCAFTSOFPOYLEAFSYAPHAFPRTWQCDAGF9EHTOGWZHTDRNACYTBIBPU'
        b'UBCBIDXQZFFJR9HOEXAGRGLZS9ZJZBQOPXIRWPQY9BNHOZNTKTKQDGGTXBBAZCYGWP'
        b'YN9ZRSI9ORTBXYTW9KKIMILXBPSLRXCIGIJESSXOGDOYXEBEVIAQOVHCJBLJFSSPPO'
        b'BVFIGVMJUULVVCBZIFMSVIEQLFSMUIPMGWTXDTCFDQMKNHIHQIMQODZUGWDBNMVOWX'
        b'CTIASMPCLQGUDXHHMKRIOWZOWXNVTFTEQITHYIBWUJFW9DYRBOUC9OIYAKIHLQNAWD'
        b'KMFNZJMLCMKPENGHGOBZABAUYVPPKWLMPYXQYXPNFQTQAWISIJAXWAESRBMCQHWR9D'
        b'CFNGRRLMBCTRARQRBMWO9O9W9IDLPNNRESBVADMJOFBJGFJMMRXXCVUGPBJPMJA99R'
        b'BMIBXEFKQH9QLVPBRXBDPKCTOMAWFIKOAELUTBXPBFRATSLFTCR9ADKGAOEBZAKINX'
        b'WUUJRWNHDTNZGHOUUDXFKHNBYIFAJXGHRVOSVSAPIZ9EDOCSJWMLHD9JEVBTD9SNDZ'
        b'GJPUTWBLCHFLDOWDOLUKRRRRSAVEWOPYWIYNVTR9IXVCGOXWAZPEMI9MGHQFOJWTRL'
        b'SXXTLEKGIOGCVQCGKUQHZYOVLTMPHLYEXX9RDUHXECNZWZPYGQPRGHKQJHTINBCFGN'
        b'BEYFLKZIFKTNFUMIMBGV9DWGKEUUQXNCGEVRQVQWKSEVVKQP9YDXWVAQP9BUYCZZWK'
        b'PQEDFAJETIC9DTXHDMYZLHKXXWVXTUHEDTDDFAJSEUAABJCTOPWCACXDQBBQMWILHE'
        b'9HRNZKOSQAFXWJQJRBNDZEYCLIMFWETVERUBAXVZHRXYNEKA9VFQGCVOCVMHPQZAHT'
        b'9JJFSXKL9JTIHIRAJZFNGQZSOCNQIFVUSEAEQNUXDVNAEVAHGNUTCIPDYMWMCPYUNV'
        b'CVJPNUGGDPONVXMJENLDPHTWNSXYAHCEZMOZXIRUFESZPGWOB9HSSGBDZTOAKLTXHS'
        b'YYQ9YGEOMU9WEMAGTI9BXMKSRACJZAMFQWMZIXQOLRI9LKCJMTJNZYPDSN9LHW9LQC'
        b'CKWVUXRGRDNN9AGDKFRVJVOUBGWBMMOQVQDNEDVFCFRMZ9KZZWLKFICUGBHJAKJMEI'
        b'OCDQPVPWNJLRKQTLPSXSGNMGRJMJFNLTHRZERCRDBYEYZZGGCUMETWFFWUHZFGU9UX'
        b'B9OIYAHLHGHUYJNBYTBOSEMMIWNYDTY9HIUSCYELVWUBJJKCNUTQOMNRCXLNVWWUIB'
        b'NCQJUVJIIAOTJXCGAQI9A9GDQXHFZYJKQFQWCXOGOPCJNXUFGGRYPBCOWGPADFJKTS'
        b'EKBWDWCTYLACMGWOAJZAAYLXCLYDSGOBXFG9JJNPDUENUZWHEEPSGLVXATRHORZIFR'
        b'EPKMQGGTDQZPKUVNLXQMZIYMQJ9DQ9YVHCMWQGNQNASBBMSWTGSRGTDLWQSYFKTEVW'
        b'9SMMIZLOUOP9RJELWSDSKKBGJQ99TBXCJEJXGXAEQY9MWOWMYIJWTAFRXPHRKGICXM'
        b'DTRNTYPPT9ASUSVAORPJZHIKMPCXDZCHNUKVCWVPBMKFXFPZUDDQTPVJDKRLHOXSLJ'
        b'AOUFUZJTMWPRRGBENPEVMNAURDJXJCTDWKJTTIH9VRBEGTCEHAQRTHQPWCGBHF9RNW'
        b'NJXHLDPHLVLDWJOPAVTDNGA9BWQASZHWGETWUOWFTXRNOBCLFZGIQOHVZ9CRBNGSXN'
        b'DDQI9KIARJNDOFYOGDNQDDWXAALKLPMH9SKIKF9VKWODV9ZBUMKCJNGQXIGQOUQWDB'
        b'ZUBDPMXZS9LSE9PRRDSLEHPYFTEWHAABELHKOPYEWBMAVAWOGWEXVWG9QPZAUHHSJX'
        b'KNTWVKNW9LBWOSBEGPOJPYSSII9CXQNZQQARJBCPGBALDHYPWVPGIYIAHQAXOGKBVT'
        b'IPK9EHLLRGKGYYEIPGCODOOGEMSIJGPXXIDNPOCCGE9TKGEXMTKPWLUDCWMZO9YBQX'
        b'CBH9R9CYYSNQWQFMADEKZVLKS9BDSZZBHSLGCNKZSCNQGJQCYTVJEOYL9KQ9S9GPSE'
        b'LWDIKF9MD'
      ),
    )

  def test_generator_with_security_level(self):
    """
    Creating a generator that uses higher security level in order to
    create longer keys.
    """
    kg = KeyGenerator(
      # Using the same seed as the previous test, just to make sure the
      # key generator doesn't cheat.
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999FFRFYAMRNWLGSGZNYUJNEBNWJQNYF',
    )

    iterator = kg.create_iterator(start=3, security_level=2)

    self.assertEqual(
      next(iterator),

      PrivateKey(
        b'DEILIWDSOTXXVPJ9GKCY9EKLJAAVNL9TZQUYGXSVFEMWZGPKUWFQUQR9JL9SSSHFYZ'
        b'FPZKACLPLH9CHQYQBXDTTJWXOGL9AXPRSKAETKENMJDTRGCLDKDISPRYVDKPBIKESW'
        b'THBRYKOZUWFAKSJNEDJYAUKFGTURZWMSEIPIZPYDQHMXNTA9WL9KLFKRXCDCOHORXS'
        b'ISJWLPDSUTVNYDENFLCKHGKJLFJEIXUUBESWXTRTWZAWDCJCORDDEWCZUPTAKKFQOX'
        b'S9WNRNUQMMLRYJIRJWGNYFPIFMG9WO9EYBFZJLNGLASAE9SBXAI9LNVFVUWXPXQXVJ'
        b'EBVLDI9PLQTWKXIATCVUHMCXGGVOAD9HLEIYWOLYVDSY9DVHBUVXQJBAUOUPFVGWMP'
        b'HNLIIHKTDASXQVCM9NLBBGFJL9BYHYAYHNSXYWISFRCHELLUKYENACVAQRDBCXOZNF'
        b'THJ9WJZXSNZBV9ZNUGCFIHJWTSCGRYATQOHWBVSONUOJKOL9RZNTCKPDBQMXBHUEKT'
        b'GR9QBFFGPWFKJWSTFIORNLYHLIVBCPMEOGARGYZHNMBPMLQPPHFKFRAQYIPIVVEIXN'
        b'9LGTM9LMTEOGTJPYAAXTJUFYCBUJCYNLQPTVZXMMMVUZCWVFYNZHQAAF9IIHV9ERSK'
        b'K9YGICMF9YIXWVFLEBIKGI9NNI9JFXWMXTTSOAXCVYNOIFUOULT9MLAVE9HOYVXHMW'
        b'GKB9RFIDVWXLYXOHUGCNJNFDKWTXBNVBWEYDRBXDTQNLEBCDLSDNEHMKFCSEUBBBBC'
        b'WKUHOMDFSMZGWTZOOCQPURKCAWT9ZBFNWTZQNFWBWTOXXTKKJDQWAGBALOVEJWTPAW'
        b'QQRKPGGWGOKYOEUHNVHOPECDTFRBBDGJDFPYU9PPE9QTOFGWABAPNIXIJYJJAYKNGO'
        b'DFRGJX9N9WRPPPKJGZOVRBYZVYCBMMCKSXUCDPWROYUJFXXCSDRGRU9FFEMCSHITWR'
        b'URJINOEOTPBRNMKKCOGMWKLFJFNOZUTCMNKESNJOEVYOPMKODCXSETWUHOUWVSDHWY'
        b'YPUJAYNAUQGZISKRRSVLPQCYTUIEUZSCKVRADGHM9KPPX9XUVW9VGILBDKLCMCLYEL'
        b'YTPPXQKGNEI9TXUGOCYKEKMCOFBRXNDRZRDPPEBLATOPGTUHMGTNEOB9BEHPIZQGOA'
        b'JFCC9VABHCZLHQNCRKILEDZULPCNRNY9RMKVVARIJJEWCJFYYPWEEPCTMZBHAEAUIB'
        b'MXJMTLITMKRYNYHOYTFEANRUOSHWCHSUULJXRMYEBWMGNMJSYOKMTLIOIWFGQ9GXLD'
        b'CFEFBKIVKXYCRHMAHTP99EPUSLPXAMYHCFRSVHWSOEHJVJLOCXOSDKSPYWG9XMCPPZ'
        b'NAVULZWTECRGGVWGHFMRWPAOKLXOQMSGEFSVTCWITKGXTWLGXNS9WDXECMMMNFMJXG'
        b'KVLXR9G9HOHYHSM9IE9DQOUYAREBHFWIITROKMQPVFHVQZMWUUFEJRZTADTBRENMDM'
        b'PXMYZSIVTLYHDDLWWNWCWGGYAW9TOQUTRENBGAJCMISIWEPCBFUHPXIDKUVMCYSNBY'
        b'AXESQQPUSNKPXLLAUQT9NDYV9YMIXZAKOFCBLAFQFJ9VRXAXJQLZWXVAYGTRIVNDVS'
        b'FCJOKWPRKRFX9XJFFAAZINQLTOHEOTINY9XRAQGGCENHSVGZKB9GHJEEHRUSBLZHEK'
        b'U9BOBGYRCRKEPYADDVNKYKJWMFHLYUG9WLTDYELCSLDWFPXQUWVHULQFPUUHL9NWRZ'
        b'9U9EPZHCPRBPT9BFA9JGVKKKMAMQHMMZHLOAIW9YLUVFT9LPG9MFDTAQBVZWZNW9EU'
        b'9LNBZIQWOZAHXRWJIVRCMULGTOMGHWG9DWYOBZHBVD9CRABMXCASGRBGXSNRTOFZSP'
        b'ZAGJWHZIFZMZWQYDATWGFWEKYYXJCZ9JFJOZKDDHHQVXYSSTYXNHB9NPNOKMZ9RNLX'
        b'AOL9IADMCIIAATLNEZCNCBCUULZTDIWOIZUHLVQWECIYCDATIOUWOMOPBMOTBYRZJN'
        b'KIPZMGAJEWXVHROJPHYJNJJVNEMD9WXNSQNARYGEYKQMXUTEWPUGJ9XAEJXWXHOIYM'
        b'BNJBIYRWBTLVVZTCBJFEQVZIWN9TFFYDLEODFTXECOYHZGIKNS9ESHQGJNXMHPSJMB'
        b'IYUAZJFTWOJWZMOTAQWGNABYNTAZMVWYRQXWUNCFVAEJPXEJAQVUEJVLSQGZHFOL9X'
        b'ZIIMCILT9CDUBH9AY9SSWMEBKXQTQJXYYUMLKVVWPEBMLXEXREVNGOVQTSLKAOCJQR'
        b'FMFLCSLIUESVWEKZDYZAXTUGVV9NFVC9GEUBCN9ZNCBRTIIGIVJ9WNCTEGTYYZVNLB'
        b'ISHOKQBAVETUJSQMGZZUMBJTHBBIBOMWJIDKKPARGLOLJMNQUISYDDCUJIZNRHAYIZ'
        b'MAXWJTSEWHFBPAIE9ELXXRJHATEMDPWJRGAZ9KMEVLBUIHRYE9LWZRPN9JNKSROTJK'
        b'TKCIOXISKVLPHEMXMDAUYZTTMDPBMRWYOJEMST9CARUGWYBSPZFIFXPXOVMVSWIYBK'
        b'HARCVHLAE9JEKZIVUWRKYXDODGSZDBHL9GPBLUJSHMNHLSRCRHWDRWTYAUXVPGVAFV'
        b'PQEKRBPJGTIMRTEOGI9KGQCVVDCEJDPFXUDXMKW9ZJKSHDUHEWUHREKDQYCFNDNXAQ'
        b'QNSOJVTVEDZ9ZMGV9CHFRVMRBDBPAJ9IXPAU99MZRCMWSUYXZOQTBPKYVIAT9YNHSZ'
        b'RGMMTWJITPBWTQWCNACHGNSXGAZH99TLWNRAWHKRGOQREXZFADLOIBIQZVTQUW9IWL'
        b'YCHSUOFZQYPFNIMMJSGNO9ICPCPGW9DNRNPENRKAYTAIINUDWYN9UBQKQEOEYAXSBA'
        b'QPRD9UEKNLWYPOVWWRMEZZOKJAJJWKIILAXLTXTGSIUVFMDGJ9OGCOHPFEJCNBZFFA'
        b'MIBSVRMOCSDRJVCLDIXFKMWFRYCBTGBZULULAEUNJEKSOB9FRXMYWXDIDAQ9CBGCCG'
        b'NXJTUAVQOYFPJVKB9CPTXRKVKPJNNUVTHEWD9GCOEXBUKKRTMUJKUKVCDBSMHI99AF'
        b'LAAGIUUI9LMYLPP9BIQDOGLBOTNEOW9NBSEKJHFDHDVUMHKQELRIZKFWBZSXCJGODU'
        b'RVMSIWLHUUKRDLCWVEJDMPZGOSJONZLDMNFLAJEK9EWSIHKXEFQVTDFCTITJREYRPB'
        b'HYTUJYALFYQLEITWYNVWJIXNSV99WKUVVBJQLIQQCWIXLCAMMBMUMUURBUJOYOLEBG'
        b'XMZAYLNBMWVYUWOR9PBCDQVID9CELEMIFQCZMMTDYLXZWXNXBBIDLURKCZDWDFXPVT'
        b'SQKEGPQUXFYDINSSKFZKJHULVRGK9ICCHSIDCMQU9JDMQSMRO9DKAWGATCNSAS9OSU'
        b'KD9VCRVTHASMAQOBV9RVGOQJADJVLFQYNOALAXNBANF9WTPLUPYPIGOFBYTZZIUKLU'
        b'KUQKMY9BWIDJXQEFBUOHFB9JNOMEMUPYQMNDDCMLVSQVTPFCRE9HCCQFSFMQMIKEIZ'
        b'EGVQYHXMF9LVNWQLYYCLOPQDEQLDQMCPEAYFESVW9MKHHJVBTFLXMZGTBAIKOZOZEC'
        b'MVWLZJNTZOHMBSDUPZCUIIQHRDZMSMIGCTHHEDBJQDWKTFFWZBOQTRVCULTLUCE9TS'
        b'WXVNXIUEUBHCDOCEVIJELBYMTJKBODXYYOFPXWJE9CEGPGREWHHIANYDGGWCAAOPIC'
        b'FWZWBWNFIUKBIAHLRINVFKMH9AHKLDOYOQWAVZ9I99DUDGKLFPEVMAPNYISYCEPDOR'
        b'VEUZYFU9BQRFXYLPD99MBIMNNVTGIBVNQRQIYEKKZADTUEYCIUTIAVPQNXODQIZSPV'
        b'DNUMQEOYQUVUHSSXCSMPIHJEKWECOKOETBJAQWR9RSVFSWIXZJIEUOCBNCGAFGJHCM'
        b'ZLTEQRS9AINIZHZVDDKOYDKHKABCTXXLKGXY9SHTKJZ9AZLAZZWLFSNIRJQKX9RHWX'
        b'ZCKUQUMPUXRGRJEMVA9RNGUWQMTCUIZPJZKWPFEIYXAKGLAVZ9YCWZDWWPLJIAUZWF'
        b'BVKYBEJIIMXWSSSWFXTDXTAHRGSKOIFXMJUUOYXMAZKWVYLCRUBUJFDIPDBDVMUBBZ'
        b'KUXYHBWMAGOUAZOIPUPVUFWTMTIROZKAMBRHOMZRMAPGQNGCCLDYKAOJAFGTERDJZS'
        b'YVCJSMVZO9QPE9CAOTVJABMIGRZIXUCAMCENUISOIFSAAMEIHK9WNXVJRRQAATXXDQ'
        b'EM9ZCCETCAUBBYEWWWXWBVVHTHGNMGESVVWMUDBHFYZQPTKSCJFKGWIY9LCJGUDHRS'
        b'HVJXLYSLYSNSUCDGID'
      ),
    )


class SignatureFragmentGeneratorTestCase(TestCase):
  """
  Generating values for this test case using the JS lib:

  .. code-block:: javascript

     var privateKeyTrytes = '...';
     var bundleHashTrytes = '...';

     var Bundle = require('./lib/crypto/bundle/bundle')
     var Converter = require('./lib/crypto/converter/converter');
     var Signing = require('./lib/crypto/signing/signing');

     var normalizedBundleHashTrits = Bundle.prototype.normalizedBundle(bundleHashTrytes);

     for(var i = 0; i < (privateKeyTrytes.length/2187); i++) {
       var normalizedBundleFragment = normalizedBundleHashTrits.slice(i*27, (i+1)*27);
       var keyFragment = Converter.trits(privateKeyTrytes.slice(i*2187, (i+1)*2187));
       var signatureFragment = Signing.signatureFragment(normalizedBundleFragment, keyFragment);
       console.log(Converter.trytes(signatureFragment));
     }
  """
  def test_single_fragment(self):
    """
    Creating signature fragments from a PrivateKey exactly 1 fragment
    long.
    """
    generator =\
      SignatureFragmentGenerator(
        private_key =
          PrivateKey(
            b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
          ),

        hash_ =
          Hash(
            b'RBUEILZHMYBLXFXBZLASTGECTXQDWPXKNXCYXBQHASRRRLXBDUFYZNPLWKKQYAQYOVKWFKSJT9OXIUOWC'
          ),
      )

    self.assertEqual(
      next(generator),

      TryteString(
        b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
      ),
    )

    # A generator can only generate one signature fragment per fragment
    # in its private key.
    with self.assertRaises(StopIteration):
      next(generator)

  def test_multiple_fragments(self):
    """
    Creating signature fragments from a PrivateKey longer than 1
    fragment.
    """
    generator =\
      SignatureFragmentGenerator(
        private_key =
          PrivateKey(
            b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
            b'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',
          ),

        hash_ =
          Hash(
            b'UZOWRZHEJG9SZKIADSTRZLJSTKMJVPRLFKIMJCLYDKCZAQMMQYCQGTSESVDYSSWMEORCHQUSWMSCVPTXX',
          ),
      )

    self.assertEqual(
      next(generator),

      TryteString(
        b'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',
      ),
    )

    self.assertEqual(
      next(generator),

      TryteString(
        b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
      ),
    )

    # A generator can only generate one signature fragment per fragment
    # in its private key.
    with self.assertRaises(StopIteration):
      next(generator)
