"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainNames = props.recordNames || [props.zone.zoneName];
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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