#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# fortney2k7.py - Waqas Bhatti (wbhatti@astro.princeton.edu) - 04/2016
# MIT License - data from tables in Fortney et al. 2007, ApJ, 2659, 1661

'''
This contains data from Fortney et al. 2007 on planet compositions, masses and
radii. Also contains functions that return numpy arrays from these data based on
specified input.

Requires numpy.

'''

from numpy import nan, array

# planet radii as a function of mass (first key) for different compositions
ICEROCKPLANETRADII = {
    0.010:{
        'ice':               0.38,
        '0.50ice+0.50rock':  0.33,
        'rock':              0.25,
        '0.67rock0.33iron':  0.24,
        '0.50rock0.50iron':  0.23,
        'iron':              0.19,
    },
    0.032:{
        'ice':               0.55,
        '0.50ice+0.50rock':  0.48,
        'rock':              0.37,
        '0.67rock0.33iron':  0.34,
        '0.50rock0.50iron':  0.33,
        'iron':              0.27,
    },
    0.1:{
        'ice':               0.79,
        '0.50ice+0.50rock':  0.69,
        'rock':              0.54,
        '0.67rock0.33iron':  0.50,
        '0.50rock0.50iron':  0.48,
        'iron':              0.39,
    },
    0.32:{
        'ice':               1.12,
        '0.50ice+0.50rock':  0.97,
        'rock':              0.77,
        '0.67rock0.33iron':  0.71,
        '0.50rock0.50iron':  0.68,
        'iron':              0.55,
    },
    1.0:{
        'ice':               1.55,
        '0.50ice+0.50rock':  1.36,
        'rock':              1.08,
        '0.67rock0.33iron':  1.00,
        '0.50rock0.50iron':  0.95,
        'iron':              0.77,
    },
    3.16:{
        'ice':               2.12,
        '0.50ice+0.50rock':  1.85,
        'rock':              1.48,
        '0.67rock0.33iron':  1.36,
        '0.50rock0.50iron':  1.30,
        'iron':              1.04,
    },
    10.0:{
        'ice':               2.87,
        '0.50ice+0.50rock':  2.48,
        'rock':              1.97,
        '0.67rock0.33iron':  1.80,
        '0.50rock0.50iron':  1.71,
        'iron':              1.36,
    },
    31.6:{
        'ice':               3.74,
        '0.50ice+0.50rock':  3.23,
        'rock':              2.54,
        '0.67rock0.33iron':  2.31,
        '0.50rock0.50iron':  2.19,
        'iron':              1.72,
    },
    100.0:{
        'ice':               4.68,
        '0.50ice+0.50rock':  4.03,
        'rock':              3.14,
        '0.67rock0.33iron':  2.84,
        '0.50rock0.50iron':  2.69,
        'iron':              2.09,
    },
    316.0:{
        'ice':               5.43,
        '0.50ice+0.50rock':  4.67,
        'rock':              3.64,
        '0.67rock0.33iron':  3.29,
        '0.50rock0.50iron':  3.12,
        'iron':              2.42,
    }
}


# planet masses and radii for different semimajor axes and core masses
# first key is orbit distance in AU, second key is core mass in Mearth
MASSESRADII_0_3GYR = {
    0.02:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[nan,2.326,1.883,1.656,1.455,
                            1.378,1.342,1.327,1.308,1.311,
                            1.315,1.284]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[1.102,1.388,1.465,1.422,1.349,1.325,1.311,
                             1.306,1.295,1.304,1.310,1.281]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.493,0.945,1.133,1.220,1.253,]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.801,1.030,1.144,1.193,1.226,
                             1.245,1.276,1.292,1.270]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.669,0.939,1.055,1.128,
                              1.187,1.242,1.270,1.256]}
    },
    0.045:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[2.795,1.522,1.345,1.255,1.240,1.228,1.212,
                            1.206,1.199,1.210,1.203,1.170]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.801,1.012,1.091,1.124,1.168,1.185,1.185,
                             1.188,1.188,1.204,1.199,1.168]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.447,0.793,0.968,1.071,1.124,1.147,
                             1.161,1.173,1.195,1.193,1.164]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.719,0.921,1.033,1.084,1.119,
                             1.148,1.179,1.183,1.157]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.627,0.863,0.968,1.036,
                              1.101,1.148,1.163,1.146]}
    },
    0.1:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.595,1.395,1.270,1.197,1.202,1.198,
                            1.187,1.182,1.178,1.189,1.178,1.144]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.755,0.956,1.035,1.084,1.134,1.157,
                             1.160,1.164,1.168,1.183,1.174,1.142]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.438,0.767,0.938,1.042,1.099,1.123,
                             1.138,1.153,1.174,1.169,1.138]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.702,0.899,1.011,1.063,
                             1.098,1.129,1.158,1.159,1.132]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.618,0.847,0.950,
                              1.018,1.084,1.128,1.140,1.121]}
    },
    1.0:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.504,1.325,1.222,1.169,1.182,1.182,
                            1.173,1.169,1.168,1.179,1.169,1.136]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.727,0.921,1.004,1.063,1.116,1.141,1.146,
                             1.152,1.158,1.173,1.165,1.134]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.433,0.754,0.923,1.027,1.085,1.110,
                             1.127,1.143,1.164,1.159,1.130]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.693,0.888,0.999,1.051,
                             1.087,1.120,1.149,1.149,1.124]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.613,0.839,0.941,1.009,
                              1.075,1.119,1.131,1.113]}
    },
    9.5:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[0.929,0.951,0.983,1.020,1.070,1.106,
                            1.127,1.146,1.167,1.169,1.156,1.130]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.565,0.733,0.847,0.939,1.016,1.072,1.104,
                             1.131,1.157,1.163,1.152,1.127]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.394,0.664,0.826,0.942,1.024,
                             1.073,1.146,1.142,1.153,1.146,1.124]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.635,0.823,0.951,1.020,
                             1.072,1.119,1.137,1.137,1.118]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.587,0.810,0.920,0.999,
                              1.072,1.107,1.119,1.107]}
    }
}


MASSESRADII_1_0GYR = {
    0.02:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[nan,1.770,1.539,1.387,1.309,1.281,1.258,
                            1.248,1.235,1.244,1.240,1.199]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.909,1.150,1.221,1.211,1.228,1.234,
                             1.229,1.229,1.224,1.237,1.235,1.197]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.461,0.838,1.022,1.121,1.169,
                             1.189,1.200,1.206,1.228,1.229,1.192]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.746,0.958,1.072,1.122,
                             1.156,1.180,1.211,1.218,1.186]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.640,0.888,0.997,1.068,
                              1.130,1.179,1.198,1.173]}
    },
    0.045:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.490,1.271,1.183,1.144,1.163,
                            1.167,1.160,1.157,1.156,1.164,1.149,1.107]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.698,0.888,0.975,1.043,1.099,1.127,
                             1.134,1.140,1.147,1.158,1.145,1.105]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.426,0.739,0.908,1.012,1.072,
                             1.099,1.115,1.132,1.149,1.140,1.101]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.684,0.877,0.988,1.041,
                             1.077,1.109,1.134,1.130,1.095]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.607,0.831,0.932,
                              0.999,1.065,1.105,1.111,1.084]}
    },
    0.1:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.298,1.197,1.127,1.105,1.133,1.143,
                            1.139,1.138,1.139,1.147,1.130,1.087]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.665,0.847,0.934,1.012,1.072,1.105,
                             1.114,1.122,1.130,1.141,1.126,1.085]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.420,0.719,0.883,0.989,1.051,
                             1.080,1.097,1.116,1.132,1.121,1.081]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.670,0.859,0.970,1.023,
                             1.059,1.094,1.117,1.111,1.076]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.600,0.818,0.918,0.984,
                              1.050,1.088,1.093,1.065]}
    },
    1.0:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.229,1.148,1.095,1.086,1.118,
                            1.130,1.128,1.127,1.130,1.137,1.121,1.079]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.646,0.823,0.915,0.996,1.058,1.092,
                             1.103,1.111,1.121,1.131,1.117,1.077]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.416,0.709,0.871,0.977,1.040,
                             1.069,1.087,1.107,1.123,1.112,1.073]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.663,0.850,0.961,1.014,1.050,
                             1.085,1.108,1.102,1.068]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.595,0.811,0.910,
                              0.976,1.042,1.080,1.085,1.057]}
    },
    9.5:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[0.857,0.877,0.910,0.955,1.003,
                            1.044,1.068,1.089,1.113,1.119,1.109,1.074]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.532,0.683,0.791,0.882,0.955,
                             1.013,1.047,1.075,1.104,1.113,1.105,1.072]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.386,0.631,0.780,0.888,
                             0.970,1.018,1.089,1.090,1.105,1.100,1.069]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.610,0.784,0.904,
                             0.970,1.021,1.068,1.090,1.091,1.063]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.570,0.775,0.878,
                              0.954,1.026,1.063,1.074,1.053]}
    }
}


MASSESRADII_4_5GYR = {
    0.02:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[nan,1.355,1.252,1.183,1.190,1.189,
                            1.179,1.174,1.170,1.178,1.164,1.118]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.726,0.934,1.019,1.072,1.123,
                             1.148,1.153,1.157,1.160,1.172,1.160,1.116]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.430,0.756,0.928,1.032,1.091,
                             1.116,1.131,1.145,1.163,1.155,1.112]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.695,0.891,1.004,1.056,
                             1.091,1.121,1.148,1.144,1.106]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.613,0.841,0.944,1.011,
                              1.076,1.118,1.125,1.095]}
    },
    0.045:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                                1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.103,1.065,1.038,1.049,1.086,
                            1.105,1.107,1.108,1.113,1.118,1.099,1.053]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.599,0.775,0.878,0.964,1.029,
                             1.069,1.083,1.092,1.104,1.112,1.095,1.050]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.403,0.686,0.846,0.952,1.019,
                             1.050,1.069,1.090,1.104,1.090,1.047]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.648,0.831,0.942,0.996,1.033,
                             1.068,1.090,1.081,1.042]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.587,0.798,0.896,
                              0.961,1.026,1.062,1.063,1.032]}
    },
    0.1:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.068,1.027,1.005,1.024,1.062,1.085,
                            1.090,1.092,1.099,1.104,1.084,1.038]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.592,0.755,0.858,0.942,1.008,
                             1.051,1.067,1.077,1.090,1.098,1.080,1.036]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.404,0.675,0.829,0.934,1.002,
                             1.034,1.054,1.077,1.090,1.075,1.033]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.639,0.817,0.928,0.982,
                             1.019,1.055,1.076,1.066,1.027]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.582,0.788,0.884,
                              0.949,1.014,1.049,1.049,1.018]}
    },
    1.0:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[1.014,0.993,0.983,1.011,1.050,1.074,
                            1.081,1.084,1.091,1.096,1.075,1.030]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.576,0.738,0.845,0.931,0.997,1.041,
                             1.058,1.068,1.082,1.090,1.072,1.028]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.400,0.666,0.820,0.924,
                             0.993,1.026,1.046,1.069,1.082,1.067,1.025]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.633,0.810,0.920,0.974,
                             1.011,1.048,1.068,1.058,1.020]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.578,0.782,0.878,
                              0.942,1.007,1.041,1.041,1.010]}
    },
    9.5:{
        0:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                          1.0,1.46,2.44,4.1,6.8,11.3],
           'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
           'radiusjupiter':[0.798,0.827,0.866,0.913,0.957,
                            0.994,1.019,1.037,1.056,1.062,1.055,1.023]},
        10:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[0.508,0.653,0.759,0.844,0.911,0.966,
                             0.999,1.024,1.048,1.057,1.052,1.021]},
        25:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,0.378,0.611,0.750,0.849,0.926,
                             0.972,1.037,1.035,1.050,1.047,1.018]},
        50:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                           1.0,1.46,2.44,4.1,6.8,11.3],
            'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
            'radiusjupiter':[nan,nan,nan,0.594,0.754,0.865,
                             0.926,0.973,1.015,1.037,1.039,1.013]},
        100:{'massjupiter':[0.052,0.087,0.15,0.24,0.41,0.68,
                            1.0,1.46,2.44,4.1,6.8,11.3],
             'massearth':[17,28,46,77,129,215,318,464,774,1292,2154,3594],
             'radiusjupiter':[nan,nan,nan,nan,0.558,0.746,0.842,
                              0.911,0.976,1.012,1.023,1.004]}
    }
}


def massradius(age, planetdist, coremass,
               mass='massjupiter',
               radius='radiusjupiter'):
    '''This function gets the Fortney mass-radius relation for planets.

    Parameters
    ----------

    age : float
        This should be one of: 0.3, 1.0, 4.5 [in Gyr].

    planetdist : float
        This should be one of: 0.02, 0.045, 0.1, 1.0, 9.5 [in AU]

    coremass : int
        This should be one of: 0, 10, 25, 50, 100 [in Mearth]

    mass : {'massjupiter','massearth'}
        Sets the mass units.

    radius : str
        Sets the radius units. Only 'radiusjupiter' is used for now.

    Returns
    -------

    dict
        A dict of the following form is returned::

            {'mass': an array containing the masses to plot),
             'radius': an array containing the radii to plot}

        These can be passed to a plotting routine to make mass-radius plot for
        the specified age, planet-star distance, and core-mass.

    '''

    MR = {0.3:MASSESRADII_0_3GYR,
          1.0:MASSESRADII_1_0GYR,
          4.5:MASSESRADII_4_5GYR}

    if age not in MR:
        print('given age not in Fortney 2007, returning...')
        return

    massradius = MR[age]

    if (planetdist in massradius) and (coremass in massradius[planetdist]):

        print('getting % Gyr M-R for planet dist %s AU, '
              'core mass %s Mearth...' % (age, planetdist, coremass))

        massradrelation = massradius[planetdist][coremass]

        outdict = {'mass':array(massradrelation[mass]),
                   'radius':array(massradrelation[radius])}

        return outdict
