"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ColumnEncoding = exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable import/no-extraneous-dependencies */
const cdk = require("aws-cdk-lib/core");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction || (exports.TableAction = TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = this.configureTableColumns(props.tableColumns);
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        const useColumnIds = !!cdk.FeatureFlags.of(this).isEnabled(cx_api_1.REDSHIFT_COLUMN_ID);
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
                tableComment: props.tableComment,
                useColumnIds,
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            (0, util_1.getDistKeyColumn)(columns);
        }
        catch {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = (0, util_1.getDistKeyColumn)(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = (0, util_1.getSortKeyColumns)(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = (0, util_1.getSortKeyColumns)(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
    configureTableColumns(columns) {
        const newColumns = [...columns];
        const columnIds = new Set();
        for (let i = 0; i < columns.length; i++) {
            const column = newColumns[i];
            if (column.id) {
                if (columnIds.has(column.id)) {
                    throw new Error(`Column id '${column.id}' is not unique.`);
                }
                columnIds.add(column.id);
            }
            else {
                if (columnIds.has(column.name)) {
                    throw new Error(`Column name '${column.name}' is not unique amongst the column ids.`);
                }
                newColumns[i] = { ...column, id: column.name };
                columnIds.add(column.name);
            }
        }
        return newColumns;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.134.0-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle || (exports.TableDistStyle = TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle || (exports.TableSortStyle = TableSortStyle = {}));
/**
 * The compression encoding of a column.
 *
 * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Compression_encodings.html
 */
var ColumnEncoding;
(function (ColumnEncoding) {
    /**
     * Amazon Redshift assigns an optimal encoding based on the column data.
     * This is the default.
     */
    ColumnEncoding["AUTO"] = "AUTO";
    /**
     * The column is not compressed.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Raw_encoding.html
     */
    ColumnEncoding["RAW"] = "RAW";
    /**
     * The column is compressed using the AZ64 algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/az64-encoding.html
     */
    ColumnEncoding["AZ64"] = "AZ64";
    /**
     * The column is compressed using a separate dictionary for each block column value on disk.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Byte_dictionary_encoding.html
     */
    ColumnEncoding["BYTEDICT"] = "BYTEDICT";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 1-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA"] = "DELTA";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 2-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA32K"] = "DELTA32K";
    /**
     * The column is compressed using the LZO algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/lzo-encoding.html
     */
    ColumnEncoding["LZO"] = "LZO";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 1 byte.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY8"] = "MOSTLY8";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 2 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY16"] = "MOSTLY16";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 4 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY32"] = "MOSTLY32";
    /**
     * The column is compressed by recording the number of occurrences of each value in the column.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Runlength_encoding.html
     */
    ColumnEncoding["RUNLENGTH"] = "RUNLENGTH";
    /**
     * The column is compressed by recording the first 245 unique words and then using a 1-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT255"] = "TEXT255";
    /**
     * The column is compressed by recording the first 32K unique words and then using a 2-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT32K"] = "TEXT32K";
    /**
     * The column is compressed using the ZSTD algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/zstd-encoding.html
     */
    ColumnEncoding["ZSTD"] = "ZSTD";
})(ColumnEncoding || (exports.ColumnEncoding = ColumnEncoding = {}));
//# sourceMappingURL=data:application/json;base64,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