"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTablePrivileges = void 0;
const cdk = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const table_1 = require("../table");
const database_query_1 = require("./database-query");
const handler_name_1 = require("./database-query-provider/handler-name");
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
class UserTablePrivileges extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.privileges = props.privileges ?? [];
        new database_query_1.DatabaseQuery(this, 'Resource', {
            ...props,
            handler: handler_name_1.HandlerName.UserTablePrivileges,
            properties: {
                username: props.user.username,
                tablePrivileges: cdk.Lazy.any({
                    produce: () => {
                        const reducedPrivileges = this.privileges.reduce((privileges, { table, actions }) => {
                            const tableId = table.node.id;
                            if (!(tableId in privileges)) {
                                privileges[tableId] = {
                                    tableName: table.tableName,
                                    actions: [],
                                };
                            }
                            actions = actions.concat(privileges[tableId].actions);
                            if (actions.includes(table_1.TableAction.ALL)) {
                                actions = [table_1.TableAction.ALL];
                            }
                            if (actions.includes(table_1.TableAction.UPDATE) || actions.includes(table_1.TableAction.DELETE)) {
                                actions.push(table_1.TableAction.SELECT);
                            }
                            privileges[tableId] = {
                                tableName: table.tableName,
                                actions: Array.from(new Set(actions)),
                            };
                            return privileges;
                        }, {});
                        const serializedPrivileges = Object.entries(reducedPrivileges).map(([tableId, config]) => ({
                            tableId,
                            tableName: config.tableName,
                            actions: config.actions.map(action => table_1.TableAction[action]),
                        }));
                        return serializedPrivileges;
                    },
                }),
            },
        });
    }
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table, ...actions) {
        this.privileges.push({ table, actions });
    }
}
exports.UserTablePrivileges = UserTablePrivileges;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJpdmlsZWdlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInByaXZpbGVnZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQXdDO0FBQ3hDLDJDQUF1QztBQUV2QyxvQ0FBK0M7QUFFL0MscURBQWlEO0FBQ2pELHlFQUFxRTtBQW1DckU7Ozs7Ozs7R0FPRztBQUNILE1BQWEsbUJBQW9CLFNBQVEsc0JBQVM7SUFHaEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUErQjtRQUN2RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7UUFFekMsSUFBSSw4QkFBYSxDQUFrQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ25FLEdBQUcsS0FBSztZQUNSLE9BQU8sRUFBRSwwQkFBVyxDQUFDLG1CQUFtQjtZQUN4QyxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUTtnQkFDN0IsZUFBZSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO29CQUM1QixPQUFPLEVBQUUsR0FBRyxFQUFFO3dCQUNaLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsRUFBRTs0QkFDbEYsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7NEJBQzlCLElBQUksQ0FBQyxDQUFDLE9BQU8sSUFBSSxVQUFVLENBQUMsRUFBRSxDQUFDO2dDQUM3QixVQUFVLENBQUMsT0FBTyxDQUFDLEdBQUc7b0NBQ3BCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztvQ0FDMUIsT0FBTyxFQUFFLEVBQUU7aUNBQ1osQ0FBQzs0QkFDSixDQUFDOzRCQUNELE9BQU8sR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQzs0QkFDdEQsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLG1CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztnQ0FDdEMsT0FBTyxHQUFHLENBQUMsbUJBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQzs0QkFDOUIsQ0FBQzs0QkFDRCxJQUFJLE9BQU8sQ0FBQyxRQUFRLENBQUMsbUJBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLG1CQUFXLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQztnQ0FDakYsT0FBTyxDQUFDLElBQUksQ0FBQyxtQkFBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDOzRCQUNuQyxDQUFDOzRCQUNELFVBQVUsQ0FBQyxPQUFPLENBQUMsR0FBRztnQ0FDcEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dDQUMxQixPQUFPLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQzs2QkFDdEMsQ0FBQzs0QkFDRixPQUFPLFVBQVUsQ0FBQzt3QkFDcEIsQ0FBQyxFQUFFLEVBQXNFLENBQUMsQ0FBQzt3QkFDM0UsTUFBTSxvQkFBb0IsR0FBK0IsTUFBTSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDOzRCQUNySCxPQUFPOzRCQUNQLFNBQVMsRUFBRSxNQUFNLENBQUMsU0FBUzs0QkFDM0IsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsbUJBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQzt5QkFDM0QsQ0FBQyxDQUFDLENBQUM7d0JBQ0osT0FBTyxvQkFBb0IsQ0FBQztvQkFDOUIsQ0FBQztpQkFDRixDQUFRO2FBQ1Y7U0FDRixDQUFDLENBQUM7S0FDSjtJQUVEOztPQUVHO0lBQ0gsYUFBYSxDQUFDLEtBQWEsRUFBRSxHQUFHLE9BQXNCO1FBQ3BELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7S0FDMUM7Q0FDRjtBQXRERCxrREFzREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWIvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IERhdGFiYXNlT3B0aW9ucyB9IGZyb20gJy4uL2RhdGFiYXNlLW9wdGlvbnMnO1xuaW1wb3J0IHsgSVRhYmxlLCBUYWJsZUFjdGlvbiB9IGZyb20gJy4uL3RhYmxlJztcbmltcG9ydCB7IElVc2VyIH0gZnJvbSAnLi4vdXNlcic7XG5pbXBvcnQgeyBEYXRhYmFzZVF1ZXJ5IH0gZnJvbSAnLi9kYXRhYmFzZS1xdWVyeSc7XG5pbXBvcnQgeyBIYW5kbGVyTmFtZSB9IGZyb20gJy4vZGF0YWJhc2UtcXVlcnktcHJvdmlkZXIvaGFuZGxlci1uYW1lJztcbmltcG9ydCB7IFRhYmxlUHJpdmlsZWdlIGFzIFNlcmlhbGl6ZWRUYWJsZVByaXZpbGVnZSwgVXNlclRhYmxlUHJpdmlsZWdlc0hhbmRsZXJQcm9wcyB9IGZyb20gJy4vaGFuZGxlci1wcm9wcyc7XG5cbi8qKlxuICogVGhlIFJlZHNoaWZ0IHRhYmxlIGFuZCBhY3Rpb24gdGhhdCBtYWtlIHVwIGEgcHJpdmlsZWdlIHRoYXQgY2FuIGJlIGdyYW50ZWQgdG8gYSBSZWRzaGlmdCB1c2VyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlUHJpdmlsZWdlIHtcbiAgLyoqXG4gICAqIFRoZSB0YWJsZSBvbiB3aGljaCBwcml2aWxlZ2VzIHdpbGwgYmUgZ3JhbnRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHRhYmxlOiBJVGFibGU7XG5cbiAgLyoqXG4gICAqIFRoZSBhY3Rpb25zIHRoYXQgd2lsbCBiZSBncmFudGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgYWN0aW9uczogVGFibGVBY3Rpb25bXTtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBzcGVjaWZ5aW5nIHByaXZpbGVnZXMgZ3JhbnRlZCB0byBhIFJlZHNoaWZ0IHVzZXIgb24gUmVkc2hpZnQgdGFibGVzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJUYWJsZVByaXZpbGVnZXNQcm9wcyBleHRlbmRzIERhdGFiYXNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgdXNlciB0byB3aGljaCBwcml2aWxlZ2VzIHdpbGwgYmUgZ3JhbnRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHVzZXI6IElVc2VyO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJpdmlsZWdlcyB0byBiZSBncmFudGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBbXSAtIHVzZSBgYWRkUHJpdmlsZWdlc2AgdG8gZ3JhbnQgcHJpdmlsZWdlcyBhZnRlciBjb25zdHJ1Y3Rpb25cbiAgICovXG4gIHJlYWRvbmx5IHByaXZpbGVnZXM/OiBUYWJsZVByaXZpbGVnZVtdO1xufVxuXG4vKipcbiAqIFByaXZpbGVnZXMgZ3JhbnRlZCB0byBhIFJlZHNoaWZ0IHVzZXIgb24gUmVkc2hpZnQgdGFibGVzLlxuICpcbiAqIFRoaXMgY29uc3RydWN0IGlzIGxvY2F0ZWQgaW4gdGhlIGBwcml2YXRlYCBkaXJlY3RvcnkgdG8gZW5zdXJlIHRoYXQgaXQgaXMgbm90IGV4cG9ydGVkIGZvciBkaXJlY3QgcHVibGljIHVzZS4gVGhpc1xuICogbWVhbnMgdGhhdCB1c2VyIHByaXZpbGVnZXMgbXVzdCBiZSBtYW5hZ2VkIHRocm91Z2ggdGhlIGBUYWJsZS5ncmFudGAgbWV0aG9kIG9yIHRoZSBgVXNlci5hZGRUYWJsZVByaXZpbGVnZXNgXG4gKiBtZXRob2QuIFRodXMsIGVhY2ggYFVzZXJgIHdpbGwgaGF2ZSBhdCBtb3N0IG9uZSBgVXNlclRhYmxlUHJpdmlsZWdlc2AgY29uc3RydWN0IHRvIG1hbmFnZSBpdHMgcHJpdmlsZWdlcy4gRm9yIGRldGFpbHNcbiAqIG9uIHdoeSB0aGlzIGlzIGEgR29vZCBUaGluZywgc2VlIHRoZSBSRUFETUUsIHVuZGVyIFwiR3JhbnRpbmcgUHJpdmlsZWdlc1wiLlxuICovXG5leHBvcnQgY2xhc3MgVXNlclRhYmxlUHJpdmlsZWdlcyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHByaXZhdGUgcHJpdmlsZWdlczogVGFibGVQcml2aWxlZ2VbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVXNlclRhYmxlUHJpdmlsZWdlc1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMucHJpdmlsZWdlcyA9IHByb3BzLnByaXZpbGVnZXMgPz8gW107XG5cbiAgICBuZXcgRGF0YWJhc2VRdWVyeTxVc2VyVGFibGVQcml2aWxlZ2VzSGFuZGxlclByb3BzPih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIGhhbmRsZXI6IEhhbmRsZXJOYW1lLlVzZXJUYWJsZVByaXZpbGVnZXMsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHVzZXJuYW1lOiBwcm9wcy51c2VyLnVzZXJuYW1lLFxuICAgICAgICB0YWJsZVByaXZpbGVnZXM6IGNkay5MYXp5LmFueSh7XG4gICAgICAgICAgcHJvZHVjZTogKCkgPT4ge1xuICAgICAgICAgICAgY29uc3QgcmVkdWNlZFByaXZpbGVnZXMgPSB0aGlzLnByaXZpbGVnZXMucmVkdWNlKChwcml2aWxlZ2VzLCB7IHRhYmxlLCBhY3Rpb25zIH0pID0+IHtcbiAgICAgICAgICAgICAgY29uc3QgdGFibGVJZCA9IHRhYmxlLm5vZGUuaWQ7XG4gICAgICAgICAgICAgIGlmICghKHRhYmxlSWQgaW4gcHJpdmlsZWdlcykpIHtcbiAgICAgICAgICAgICAgICBwcml2aWxlZ2VzW3RhYmxlSWRdID0ge1xuICAgICAgICAgICAgICAgICAgdGFibGVOYW1lOiB0YWJsZS50YWJsZU5hbWUsXG4gICAgICAgICAgICAgICAgICBhY3Rpb25zOiBbXSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIGFjdGlvbnMgPSBhY3Rpb25zLmNvbmNhdChwcml2aWxlZ2VzW3RhYmxlSWRdLmFjdGlvbnMpO1xuICAgICAgICAgICAgICBpZiAoYWN0aW9ucy5pbmNsdWRlcyhUYWJsZUFjdGlvbi5BTEwpKSB7XG4gICAgICAgICAgICAgICAgYWN0aW9ucyA9IFtUYWJsZUFjdGlvbi5BTExdO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIGlmIChhY3Rpb25zLmluY2x1ZGVzKFRhYmxlQWN0aW9uLlVQREFURSkgfHwgYWN0aW9ucy5pbmNsdWRlcyhUYWJsZUFjdGlvbi5ERUxFVEUpKSB7XG4gICAgICAgICAgICAgICAgYWN0aW9ucy5wdXNoKFRhYmxlQWN0aW9uLlNFTEVDVCk7XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgcHJpdmlsZWdlc1t0YWJsZUlkXSA9IHtcbiAgICAgICAgICAgICAgICB0YWJsZU5hbWU6IHRhYmxlLnRhYmxlTmFtZSxcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBBcnJheS5mcm9tKG5ldyBTZXQoYWN0aW9ucykpLFxuICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICByZXR1cm4gcHJpdmlsZWdlcztcbiAgICAgICAgICAgIH0sIHt9IGFzIHsgW2tleTogc3RyaW5nXTogeyB0YWJsZU5hbWU6IHN0cmluZzsgYWN0aW9uczogVGFibGVBY3Rpb25bXSB9IH0pO1xuICAgICAgICAgICAgY29uc3Qgc2VyaWFsaXplZFByaXZpbGVnZXM6IFNlcmlhbGl6ZWRUYWJsZVByaXZpbGVnZVtdID0gT2JqZWN0LmVudHJpZXMocmVkdWNlZFByaXZpbGVnZXMpLm1hcCgoW3RhYmxlSWQsIGNvbmZpZ10pID0+ICh7XG4gICAgICAgICAgICAgIHRhYmxlSWQsXG4gICAgICAgICAgICAgIHRhYmxlTmFtZTogY29uZmlnLnRhYmxlTmFtZSxcbiAgICAgICAgICAgICAgYWN0aW9uczogY29uZmlnLmFjdGlvbnMubWFwKGFjdGlvbiA9PiBUYWJsZUFjdGlvblthY3Rpb25dKSxcbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgICAgIHJldHVybiBzZXJpYWxpemVkUHJpdmlsZWdlcztcbiAgICAgICAgICB9LFxuICAgICAgICB9KSBhcyBhbnksXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoaXMgdXNlciBhZGRpdGlvbmFsIHByaXZpbGVnZXMuXG4gICAqL1xuICBhZGRQcml2aWxlZ2VzKHRhYmxlOiBJVGFibGUsIC4uLmFjdGlvbnM6IFRhYmxlQWN0aW9uW10pOiB2b2lkIHtcbiAgICB0aGlzLnByaXZpbGVnZXMucHVzaCh7IHRhYmxlLCBhY3Rpb25zIH0pO1xuICB9XG59XG4iXX0=