"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const redshift_data_1 = require("./redshift-data");
const types_1 = require("./types");
const util_1 = require("./util");
async function handler(props, event) {
    const tableNamePrefix = props.tableName.prefix;
    const getTableNameSuffix = (generateSuffix) => generateSuffix === 'true' ? `${event.StackId.substring(event.StackId.length - 12)}` : '';
    const tableColumns = props.tableColumns;
    const tableAndClusterProps = props;
    const useColumnIds = props.useColumnIds;
    let tableName = tableNamePrefix + getTableNameSuffix(props.tableName.generateSuffix);
    if (event.RequestType === 'Create') {
        tableName = await createTable(tableNamePrefix, getTableNameSuffix(props.tableName.generateSuffix), tableColumns, tableAndClusterProps);
        return { PhysicalResourceId: (0, util_1.makePhysicalId)(tableNamePrefix, tableAndClusterProps, event.StackId.substring(event.StackId.length - 12)) };
    }
    else if (event.RequestType === 'Delete') {
        await dropTable(event.PhysicalResourceId.includes(event.StackId.substring(event.StackId.length - 12)) ? tableName : event.PhysicalResourceId, tableAndClusterProps);
        return;
    }
    else if (event.RequestType === 'Update') {
        const isTableV2 = event.PhysicalResourceId.includes(event.StackId.substring(event.StackId.length - 12));
        const oldTableName = event.OldResourceProperties.tableName.prefix + getTableNameSuffix(event.OldResourceProperties.tableName.generateSuffix);
        tableName = await updateTable(isTableV2 ? oldTableName : event.PhysicalResourceId, tableNamePrefix, getTableNameSuffix(props.tableName.generateSuffix), tableColumns, useColumnIds, tableAndClusterProps, event.OldResourceProperties, isTableV2);
        return { PhysicalResourceId: event.PhysicalResourceId };
    }
    else {
        /* eslint-disable-next-line dot-notation */
        throw new Error(`Unrecognized event type: ${event['RequestType']}`);
    }
}
exports.handler = handler;
async function createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps) {
    const tableName = tableNamePrefix + tableNameSuffix;
    const tableColumnsString = tableColumns.map(column => `${column.name} ${column.dataType}${getEncodingColumnString(column)}`).join();
    let statement = `CREATE TABLE ${tableName} (${tableColumnsString})`;
    if (tableAndClusterProps.distStyle) {
        statement += ` DISTSTYLE ${tableAndClusterProps.distStyle}`;
    }
    const distKeyColumn = (0, util_1.getDistKeyColumn)(tableColumns);
    if (distKeyColumn) {
        statement += ` DISTKEY(${distKeyColumn.name})`;
    }
    const sortKeyColumns = (0, util_1.getSortKeyColumns)(tableColumns);
    if (sortKeyColumns.length > 0) {
        const sortKeyColumnsString = getSortKeyColumnsString(sortKeyColumns);
        statement += ` ${tableAndClusterProps.sortStyle} SORTKEY(${sortKeyColumnsString})`;
    }
    await (0, redshift_data_1.executeStatement)(statement, tableAndClusterProps);
    for (const column of tableColumns) {
        if (column.comment) {
            await (0, redshift_data_1.executeStatement)(`COMMENT ON COLUMN ${tableName}.${column.name} IS '${column.comment}'`, tableAndClusterProps);
        }
    }
    if (tableAndClusterProps.tableComment) {
        await (0, redshift_data_1.executeStatement)(`COMMENT ON TABLE ${tableName} IS '${tableAndClusterProps.tableComment}'`, tableAndClusterProps);
    }
    return tableName;
}
async function dropTable(tableName, clusterProps) {
    await (0, redshift_data_1.executeStatement)(`DROP TABLE ${tableName}`, clusterProps);
}
async function updateTable(tableName, tableNamePrefix, tableNameSuffix, tableColumns, useColumnIds, tableAndClusterProps, oldResourceProperties, isTableV2) {
    const alterationStatements = [];
    const newTableName = tableNamePrefix + tableNameSuffix;
    const oldClusterProps = oldResourceProperties;
    if (tableAndClusterProps.clusterName !== oldClusterProps.clusterName || tableAndClusterProps.databaseName !== oldClusterProps.databaseName) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    const oldTableColumns = oldResourceProperties.tableColumns;
    const columnDeletions = oldTableColumns.filter(oldColumn => (tableColumns.every(column => {
        if (useColumnIds) {
            return oldColumn.id ? oldColumn.id !== column.id : oldColumn.name !== column.name;
        }
        return oldColumn.name !== column.name;
    })));
    if (columnDeletions.length > 0) {
        alterationStatements.push(...columnDeletions.map(column => `ALTER TABLE ${tableName} DROP COLUMN ${column.name}`));
    }
    const columnAdditions = tableColumns.filter(column => {
        return !oldTableColumns.some(oldColumn => {
            if (useColumnIds) {
                return oldColumn.id ? oldColumn.id === column.id : oldColumn.name === column.name;
            }
            return oldColumn.name === column.name;
        });
    }).map(column => `ADD ${column.name} ${column.dataType}`);
    if (columnAdditions.length > 0) {
        alterationStatements.push(...columnAdditions.map(addition => `ALTER TABLE ${tableName} ${addition}`));
    }
    const columnEncoding = tableColumns.filter(column => {
        return oldTableColumns.some(oldColumn => column.name === oldColumn.name && column.encoding !== oldColumn.encoding);
    }).map(column => `ALTER COLUMN ${column.name} ENCODE ${column.encoding || 'AUTO'}`);
    if (columnEncoding.length > 0) {
        alterationStatements.push(`ALTER TABLE ${tableName} ${columnEncoding.join(', ')}`);
    }
    const columnComments = tableColumns.filter(column => {
        return oldTableColumns.some(oldColumn => column.name === oldColumn.name && column.comment !== oldColumn.comment);
    }).map(column => `COMMENT ON COLUMN ${tableName}.${column.name} IS ${column.comment ? `'${column.comment}'` : 'NULL'}`);
    if (columnComments.length > 0) {
        alterationStatements.push(...columnComments);
    }
    if (useColumnIds) {
        const columnNameUpdates = tableColumns.reduce((updates, column) => {
            const oldColumn = oldTableColumns.find(oldCol => oldCol.id && oldCol.id === column.id);
            if (oldColumn && oldColumn.name !== column.name) {
                updates[oldColumn.name] = column.name;
            }
            return updates;
        }, {});
        if (Object.keys(columnNameUpdates).length > 0) {
            alterationStatements.push(...Object.entries(columnNameUpdates).map(([oldName, newName]) => (`ALTER TABLE ${tableName} RENAME COLUMN ${oldName} TO ${newName}`)));
        }
    }
    const oldDistStyle = oldResourceProperties.distStyle;
    if ((!oldDistStyle && tableAndClusterProps.distStyle) ||
        (oldDistStyle && !tableAndClusterProps.distStyle)) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    else if (oldDistStyle !== tableAndClusterProps.distStyle) {
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTSTYLE ${tableAndClusterProps.distStyle}`);
    }
    const oldDistKey = (0, util_1.getDistKeyColumn)(oldTableColumns)?.name;
    const newDistKey = (0, util_1.getDistKeyColumn)(tableColumns)?.name;
    if (!oldDistKey && newDistKey) {
        // Table has no existing distribution key, add a new one
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTSTYLE KEY DISTKEY ${newDistKey}`);
    }
    else if (oldDistKey && !newDistKey) {
        // Table has a distribution key, remove and set to AUTO
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTSTYLE AUTO`);
    }
    else if (oldDistKey !== newDistKey) {
        // Table has an existing distribution key, change it
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTKEY ${newDistKey}`);
    }
    const oldSortKeyColumns = (0, util_1.getSortKeyColumns)(oldTableColumns);
    const newSortKeyColumns = (0, util_1.getSortKeyColumns)(tableColumns);
    const oldSortStyle = oldResourceProperties.sortStyle;
    const newSortStyle = tableAndClusterProps.sortStyle;
    if ((oldSortStyle === newSortStyle && !(0, util_1.areColumnsEqual)(oldSortKeyColumns, newSortKeyColumns))
        || (oldSortStyle !== newSortStyle)) {
        switch (newSortStyle) {
            case types_1.TableSortStyle.INTERLEAVED:
                // INTERLEAVED sort key addition requires replacement.
                // https://docs.aws.amazon.com/redshift/latest/dg/r_ALTER_TABLE.html
                return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
            case types_1.TableSortStyle.COMPOUND: {
                const sortKeyColumnsString = getSortKeyColumnsString(newSortKeyColumns);
                alterationStatements.push(`ALTER TABLE ${tableName} ALTER ${newSortStyle} SORTKEY(${sortKeyColumnsString})`);
                break;
            }
            case types_1.TableSortStyle.AUTO: {
                alterationStatements.push(`ALTER TABLE ${tableName} ALTER SORTKEY ${newSortStyle}`);
                break;
            }
        }
    }
    const oldComment = oldResourceProperties.tableComment;
    const newComment = tableAndClusterProps.tableComment;
    if (oldComment !== newComment) {
        alterationStatements.push(`COMMENT ON TABLE ${tableName} IS ${newComment ? `'${newComment}'` : 'NULL'}`);
    }
    await Promise.all(alterationStatements.map(statement => (0, redshift_data_1.executeStatement)(statement, tableAndClusterProps)));
    if (isTableV2) {
        const oldTableNamePrefix = oldResourceProperties.tableName.prefix;
        if (tableNamePrefix !== oldTableNamePrefix) {
            await (0, redshift_data_1.executeStatement)(`ALTER TABLE ${tableName} RENAME TO ${newTableName}`, tableAndClusterProps);
            return tableNamePrefix + tableNameSuffix;
        }
    }
    return tableName;
}
function getSortKeyColumnsString(sortKeyColumns) {
    return sortKeyColumns.map(column => column.name).join();
}
function getEncodingColumnString(column) {
    if (column.encoding) {
        return ` ENCODE ${column.encoding}`;
    }
    return '';
}
//# sourceMappingURL=data:application/json;base64,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