"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const redshift_data_1 = require("./redshift-data");
const util_1 = require("./util");
async function handler(props, event) {
    const username = props.username;
    const tablePrivileges = props.tablePrivileges;
    const clusterProps = props;
    if (event.RequestType === 'Create') {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { PhysicalResourceId: (0, util_1.makePhysicalId)(username, clusterProps, event.RequestId) };
    }
    else if (event.RequestType === 'Delete') {
        await revokePrivileges(username, tablePrivileges, clusterProps);
        return;
    }
    else if (event.RequestType === 'Update') {
        const { replace } = await updatePrivileges(username, tablePrivileges, clusterProps, event.OldResourceProperties);
        const physicalId = replace ? (0, util_1.makePhysicalId)(username, clusterProps, event.RequestId) : event.PhysicalResourceId;
        return { PhysicalResourceId: physicalId };
    }
    else {
        /* eslint-disable-next-line dot-notation */
        throw new Error(`Unrecognized event type: ${event['RequestType']}`);
    }
}
exports.handler = handler;
async function revokePrivileges(username, tablePrivileges, clusterProps) {
    await Promise.all(tablePrivileges.map(({ tableName, actions }) => {
        return (0, redshift_data_1.executeStatement)(`REVOKE ${actions.join(', ')} ON ${tableName} FROM ${username}`, clusterProps);
    }));
}
async function grantPrivileges(username, tablePrivileges, clusterProps) {
    await Promise.all(tablePrivileges.map(({ tableName, actions }) => {
        return (0, redshift_data_1.executeStatement)(`GRANT ${actions.join(', ')} ON ${tableName} TO ${username}`, clusterProps);
    }));
}
async function updatePrivileges(username, tablePrivileges, clusterProps, oldResourceProperties) {
    const oldClusterProps = oldResourceProperties;
    if (clusterProps.clusterName !== oldClusterProps.clusterName || clusterProps.databaseName !== oldClusterProps.databaseName) {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { replace: true };
    }
    const oldUsername = oldResourceProperties.username;
    if (oldUsername !== username) {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { replace: true };
    }
    const oldTablePrivileges = oldResourceProperties.tablePrivileges;
    const tablesToRevoke = oldTablePrivileges.filter(({ tableId, actions }) => (tablePrivileges.find(({ tableId: otherTableId, actions: otherActions }) => (tableId === otherTableId && actions.some(action => !otherActions.includes(action))))));
    if (tablesToRevoke.length > 0) {
        await revokePrivileges(username, tablesToRevoke, clusterProps);
    }
    const tablesToGrant = tablePrivileges.filter(({ tableId, tableName, actions }) => {
        const tableAdded = !oldTablePrivileges.find(({ tableId: otherTableId, tableName: otherTableName }) => (tableId === otherTableId && tableName === otherTableName));
        const actionsAdded = oldTablePrivileges.find(({ tableId: otherTableId, actions: otherActions }) => (tableId === otherTableId && otherActions.some(action => !actions.includes(action))));
        return tableAdded || actionsAdded;
    });
    if (tablesToGrant.length > 0) {
        await grantPrivileges(username, tablesToGrant, clusterProps);
    }
    return { replace: false };
}
//# sourceMappingURL=data:application/json;base64,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