"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core_1 = require("aws-cdk-lib/core");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Node Types to use in the cluster
 * used for defining `ClusterProps.nodeType`.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.xlplus
     */
    NodeType["RA3_XLPLUS"] = "ra3.xlplus";
    /**
     * ra3.4xlarge
     */
    NodeType["RA3_4XLARGE"] = "ra3.4xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType || (exports.NodeType = NodeType = {}));
/**
 * What cluster type to use.
 * Used by `ClusterProps.clusterType`
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the `ClusterProps.numberOfNodes` parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using `ClusterProps.numberOfNodes` parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType || (exports.ClusterType = ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.parameterGroup = props.parameterGroup;
        this.roles = props?.roles ? [...props.roles] : [];
        const removalPolicy = props.removalPolicy ?? core_1.RemovalPolicy.RETAIN;
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.ClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: removalPolicy,
        });
        const securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingProperties) {
            loggingProperties = {
                bucketName: props.loggingProperties.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingProperties.loggingKeyPrefix,
            };
            props.loggingProperties.loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                actions: [
                    's3:GetBucketAcl',
                    's3:PutObject',
                ],
                resources: [
                    props.loggingProperties.loggingBucket.arnForObjects('*'),
                    props.loggingProperties.loggingBucket.bucketArn,
                ],
                principals: [
                    new iam.ServicePrincipal('redshift.amazonaws.com'),
                ],
            }));
        }
        this.cluster = new aws_redshift_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.clusterSubnetGroupName,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin (unsafeUnwrap here is safe)
            masterUsername: secret?.secretValueFromJson('username').unsafeUnwrap() ?? props.masterUser.masterUsername,
            masterUserPassword: secret?.secretValueFromJson('password').unsafeUnwrap()
                ?? props.masterUser.masterPassword?.unsafeUnwrap()
                ?? 'default',
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: core_1.Lazy.list({ produce: () => this.roles.map(role => role.roleArn) }, { omitEmpty: true }),
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: props.publiclyAccessible || false,
            // Encryption
            kmsKeyId: props.encryptionKey?.keyId,
            encrypted: props.encrypted ?? true,
            classic: props.classicResizing,
            elasticIp: props.elasticIp,
            enhancedVpcRouting: props.enhancedVpcRouting,
        });
        this.cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = this.cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(this.cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
        if (props.rebootForParameterChanges) {
            this.enableRebootForParameterChanges();
        }
        // Add default role if specified and also available in the roles list
        if (props.defaultRole) {
            if (props.roles?.some(x => x === props.defaultRole)) {
                this.addDefaultIamRole(props.defaultRole);
            }
            else {
                throw new Error('Default role must be included in role list.');
            }
        }
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            if (core_1.Token.isUnresolved(numberOfNodes)) {
                return numberOfNodes;
            }
            const nodeCount = numberOfNodes ?? 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
    /**
     * Adds a parameter to the Clusters' parameter group
     *
     * @param name the parameter name
     * @param value the parameter name
     */
    addToParameterGroup(name, value) {
        if (!this.parameterGroup) {
            const param = {};
            param[name] = value;
            this.parameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: this.cluster.clusterIdentifier ? `Parameter Group for the ${this.cluster.clusterIdentifier} Redshift cluster` : 'Cluster parameter group for family redshift-1.0',
                parameters: param,
            });
            this.cluster.clusterParameterGroupName = this.parameterGroup.clusterParameterGroupName;
        }
        else if (this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup) {
            this.parameterGroup.addParameter(name, value);
        }
        else {
            throw new Error('Cannot add a parameter to an imported parameter group.');
        }
    }
    /**
     * Enables automatic cluster rebooting when changes to the cluster's parameter group require a restart to apply.
     */
    enableRebootForParameterChanges() {
        if (this.node.tryFindChild('RedshiftClusterRebooterCustomResource')) {
            return;
        }
        const rebootFunction = new lambda.SingletonFunction(this, 'RedshiftClusterRebooterFunction', {
            uuid: '511e207f-13df-4b8b-b632-c32b30b65ac2',
            runtime: lambda.Runtime.NODEJS_18_X,
            code: lambda.Code.fromAsset(path.join(__dirname, '..', 'custom-resource-handlers', 'dist', 'aws-redshift-alpha', 'cluster-parameter-change-reboot-handler')),
            handler: 'index.handler',
            timeout: core_1.Duration.seconds(900),
        });
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:DescribeClusters'],
            resources: ['*'],
        }));
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:RebootCluster'],
            resources: [
                core_1.Stack.of(this).formatArn({
                    service: 'redshift',
                    resource: 'cluster',
                    resourceName: this.clusterName,
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        }));
        const provider = new custom_resources_1.Provider(this, 'ResourceProvider', {
            onEventHandler: rebootFunction,
        });
        const customResource = new core_1.CustomResource(this, 'RedshiftClusterRebooterCustomResource', {
            resourceType: 'Custom::RedshiftClusterRebooter',
            serviceToken: provider.serviceToken,
            properties: {
                ClusterId: this.clusterName,
                ParameterGroupName: core_1.Lazy.string({
                    produce: () => {
                        if (!this.parameterGroup) {
                            throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                        }
                        return this.parameterGroup.clusterParameterGroupName;
                    },
                }),
                ParametersString: core_1.Lazy.string({
                    produce: () => {
                        if (!(this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup)) {
                            throw new Error('Cannot enable reboot for parameter changes when using an imported parameter group.');
                        }
                        return JSON.stringify(this.parameterGroup.parameters);
                    },
                }),
            },
        });
        core_1.Lazy.any({
            produce: () => {
                if (!this.parameterGroup) {
                    throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                }
                customResource.node.addDependency(this, this.parameterGroup);
            },
        });
    }
    /**
     * Adds default IAM role to cluster. The default IAM role must be already associated to the cluster to be added as the default role.
     *
     * @param defaultIamRole the IAM role to be set as the default role
     */
    addDefaultIamRole(defaultIamRole) {
        // Get list of IAM roles attached to cluster
        const clusterRoleList = this.roles ?? [];
        // Check to see if default role is included in list of cluster IAM roles
        var roleAlreadyOnCluster = false;
        for (var i = 0; i < clusterRoleList.length; i++) {
            if (clusterRoleList[i] === defaultIamRole) {
                roleAlreadyOnCluster = true;
                break;
            }
        }
        if (!roleAlreadyOnCluster) {
            throw new Error('Default role must be associated to the Redshift cluster to be set as the default role.');
        }
        // On UPDATE or CREATE define the default IAM role. On DELETE, remove the default IAM role
        const defaultRoleCustomResource = new custom_resources_1.AwsCustomResource(this, 'default-role', {
            onUpdate: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: defaultIamRole.roleArn,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            onDelete: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: '',
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            installLatestAwsSdk: false,
        });
        defaultIamRole.grantPassRole(defaultRoleCustomResource.grantPrincipal);
    }
    /**
     * Adds a role to the cluster
     *
     * @param role the role to add
     */
    addIamRole(role) {
        const clusterRoleList = this.roles;
        if (clusterRoleList.includes(role)) {
            throw new Error(`Role '${role.roleArn}' is already attached to the cluster`);
        }
        clusterRoleList.push(role);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Cluster", version: "2.134.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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