"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
// tslint:disable-next-line:max-line-length
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'gateway endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3,
                    },
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.s3',
                        ],
                    ],
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A',
                },
                RouteTableIds: [
                    { Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1' },
                    { Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B' },
                    { Ref: 'VpcNetworkPublicSubnet1RouteTable25CCC53F' },
                    { Ref: 'VpcNetworkPublicSubnet2RouteTableE5F348DF' },
                ],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'routing on private and public subnets'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3,
                        subnets: [
                            {
                                subnetType: lib_1.SubnetType.PUBLIC,
                            },
                            {
                                subnetType: lib_1.SubnetType.PRIVATE,
                            },
                        ],
                    },
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.s3',
                        ],
                    ],
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A',
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPublicSubnet1RouteTable25CCC53F',
                    },
                    {
                        Ref: 'VpcNetworkPublicSubnet2RouteTableE5F348DF',
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1',
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B',
                    },
                ],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'add statements to policy'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3,
            });
            // WHEN
            endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                principals: [new aws_iam_1.AnyPrincipal()],
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*'],
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                's3:GetObject',
                                's3:ListBucket',
                            ],
                            Effect: 'Allow',
                            Principal: '*',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'throws when adding a statement without a principal'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3,
            });
            // THEN
            test.throws(() => endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*'],
            })), /`Principal`/);
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const ep = lib_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack2, 'ImportedEndpoint', 'endpoint-id');
            // THEN
            test.deepEqual(ep.vpcEndpointId, 'endpoint-id');
            test.done();
        },
        'works with an imported vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                privateSubnetRouteTableIds: ['rt1', 'rt2', 'rt3'],
                availabilityZones: ['a', 'b', 'c'],
            });
            // THEN
            vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: { 'Fn::Join': ['', ['com.amazonaws.', { Ref: 'AWS::Region' }, '.s3']] },
                VpcId: 'id',
                RouteTableIds: ['rt1', 'rt2', 'rt3'],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'throws with an imported vpc without route tables ids'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                availabilityZones: ['a', 'b', 'c'],
            });
            test.throws(() => vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 }), /route table/);
            test.done();
        },
    },
    'interface endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.ecr.dkr',
                        ],
                    ],
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A',
                },
                PrivateDnsEnabled: true,
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'VpcNetworkEcrDockerSecurityGroup7C91D347',
                            'GroupId',
                        ],
                    },
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1Subnet07BA143B',
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2Subnet5E4189D6',
                    },
                ],
                VpcEndpointType: 'Interface',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'VpcNetwork/EcrDocker/SecurityGroup',
                VpcId: {
                    Ref: 'VpcNetworkB258E83A',
                },
            }));
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const importedEndpoint = lib_1.InterfaceVpcEndpoint.fromInterfaceVpcEndpointAttributes(stack2, 'ImportedEndpoint', {
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack2, 'SG', 'security-group-id')],
                vpcEndpointId: 'vpc-endpoint-id',
                port: 80,
            });
            importedEndpoint.connections.allowDefaultPortFromAnyIpv4();
            // THEN
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                GroupId: 'security-group-id',
            }));
            test.deepEqual(importedEndpoint.vpcEndpointId, 'vpc-endpoint-id');
            test.done();
        },
        'with existing security groups'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER,
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'existing-id')],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                SecurityGroupIds: ['existing-id'],
            }));
            test.done();
        },
        'with existing security groups for efs'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('Efs', {
                service: lib_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'existing-id')],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                SecurityGroupIds: ['existing-id'],
            }));
            test.done();
        },
        'security group has ingress by default'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('SecretsManagerEndpoint', {
                service: lib_1.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupIngress: [
                    {
                        CidrIp: { 'Fn::GetAtt': ['VpcNetworkB258E83A', 'CidrBlock'] },
                        FromPort: 443,
                        IpProtocol: 'tcp',
                        ToPort: 443,
                    },
                ],
            }));
            test.done();
        },
        'non-AWS service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                PrivateDnsEnabled: false,
            }));
            test.done();
        },
        'marketplace partner service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: { name: 'com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns',
                    port: 443,
                    privateDnsDefault: true },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns',
                PrivateDnsEnabled: true,
            }));
            test.done();
        },
        'test endpoint service context azs discovered'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
            // Setup context for stack AZs
            stack.node.setContext(core_1.ContextProvider.getKey(stack, {
                provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            }).key, ['us-east-1a', 'us-east-1b', 'us-east-1c']);
            // Setup context for endpoint service AZs
            stack.node.setContext(core_1.ContextProvider.getKey(stack, {
                provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
                props: {
                    serviceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                },
            }).key, ['us-east-1a', 'us-east-1c']);
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: {
                    name: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                    port: 443
                },
                lookupSupportedAzs: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
        'endpoint service setup with stack AZ context but no endpoint context'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
            // Setup context for stack AZs
            stack.node.setContext(core_1.ContextProvider.getKey(stack, {
                provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            }).key, ['us-east-1a', 'us-east-1b', 'us-east-1c']);
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: {
                    name: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                    port: 443
                },
                lookupSupportedAzs: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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