from io import StringIO
from copy import deepcopy
from datetime import datetime as dt
from datetime import time as t

import pandas as pd
import requests
import re
import numpy as np
import time



class OptionChain:

    chain_url ='https://api.truedata.in/getOptionChain?'
    close_time = t(15, 40)

    def __init__(self,
                 TD_OBJ ,
                 symbol ,
                 expiry ,
                 chain_length ,
                 future_price,
                 bid_ask ,
                 market_close ):
        self.TD_OBJ = TD_OBJ
        self.login_id = TD_OBJ.login_id
        self.password = TD_OBJ.password
        self.symbol = symbol
        self.expiry = expiry
        self.chain_length = chain_length
        self.future_price = future_price
        self.strike_step = self.get_strike_step()
        self.option_symbols = self.get_option_symbols()
        self.chain_dataframe =  self.init_dataframe()
        self.chain_status = False
        self.bid_ask = bid_ask
        self.market_close = market_close

    def get_strike_step(self ):
        expiry = self.expiry.strftime('%Y%m%d')
        chain_link = f'{OptionChain.chain_url}user={self.login_id}&password={self.password}&symbol={self.symbol}&expiry={expiry}&csv=true'
        chain = requests.get(chain_link).text
        df = pd.read_csv(StringIO(chain) , header = None)
        df[1] = df[1].apply(lambda x : re.findall('\D+' , x)[0])
        df = df[df[1] == self.symbol]
        strikes = np.sort(df[6].unique())
        return strikes[1] - strikes[0]

    def get_option_symbols(self ):
        expiry = self.expiry.strftime('%y%m%d')
        future_atm = round(self.future_price / self.strike_step ) * self.strike_step
        start_strike = future_atm - self.strike_step * int(self.chain_length / 2 )
        end_strike = future_atm + self.strike_step * int( self.chain_length / 2)
        symbols = list(map(lambda x: f'{self.symbol}{expiry}{x}CE', range( start_strike , end_strike  , self.strike_step )))
        symbols.extend(list(map(lambda x: f'{self.symbol}{expiry}{x}PE', range( start_strike , end_strike  , self.strike_step ))))
        return symbols

    def init_dataframe(self):
        df = pd.DataFrame(columns=['symbols' , 'strike' , 'type' , 'ltp' , 'ltt' , 'oi' , 'oi_change_perc' , 'bid' , 'bid_qty' , 'ask' , 'ask_qty'])
        df.symbols = self.option_symbols
        df.strike = df.symbols.apply(lambda x: str(re.findall('\d+' , x )[0])[6:])
        df.type = df.symbols.apply(lambda x: re.findall('\D+' , x )[-1] )
        df.set_index('symbols' , inplace=True)
        df.sort_values(by=['strike' , 'type'] , inplace=True)
        return df

    def update_chain(self ,td_obj , symb_ids ):
        self.chain_status = True
        prev_tick_obj = {}
        time.sleep(1)
        for symb_id in symb_ids:
            prev_tick_obj[symb_id] = deepcopy(td_obj.live_data[symb_id])
            try:
                oi_change_perc = round((td_obj.touchline_data[symb_id].oi - td_obj.touchline_data[symb_id].prev_oi)*100 / (td_obj.touchline_data[symb_id].prev_oi) , 2 )
            except ZeroDivisionError:
                oi_change_perc = 0
            self.chain_dataframe.loc[td_obj.touchline_data[symb_id].symbol , ['ltp' , 'ltt'  , 'oi' , 'oi_change_perc' , 'bid' , 'bid_qty' , 'ask' , 'ask_qty' ]] = [
                td_obj.touchline_data[symb_id].ltp, td_obj.touchline_data[symb_id].timestamp ,td_obj.touchline_data[symb_id].oi, oi_change_perc  ,
                td_obj.touchline_data[symb_id].best_bid_price,td_obj.touchline_data[symb_id].best_bid_qty, td_obj.touchline_data[symb_id].best_ask_price,
                td_obj.touchline_data[symb_id].best_ask_qty]

        while self.chain_status:
            for symb_id in symb_ids:
                if not td_obj.live_data[symb_id] == prev_tick_obj[symb_id]:
                    try:
                        bid, bid_qnty = td_obj.live_data[symb_id].best_bid_price , td_obj.live_data[symb_id].best_bid_qty
                        ask , ask_qnty = td_obj.live_data[symb_id].best_ask_price, td_obj.live_data[symb_id].best_ask_qty
                    except Exception as e:
                        bid, bid_qnty, ask, ask_qnty = 0, 0, 0, 0
                    self.chain_dataframe.loc[td_obj.live_data[symb_id].symbol , ['ltt'  , 'oi' , 'oi_change_perc' , 'bid' , 'bid_qty' , 'ask' , 'ask_qty' ]] = [
                        td_obj.live_data[symb_id].timestamp ,td_obj.live_data[symb_id].oi, round(td_obj.live_data[symb_id].to_dict()['oi_change_perc'] ,2 ) ,
                        bid, bid_qnty, ask, ask_qnty]
                    if 'ltp' in td_obj.live_data[symb_id].__dict__.keys():
                        self.chain_dataframe.loc[td_obj.live_data[symb_id].symbol , ['ltp'] ] = td_obj.live_data[symb_id].ltp
                    else:
                        self.chain_dataframe.loc[td_obj.live_data[symb_id].symbol , ['ltp'] ] = td_obj.live_data[symb_id].close
                    prev_tick_obj[symb_id] = deepcopy(td_obj.live_data[symb_id])
            time.sleep(0.005)
            if self.market_close and dt.now().time() > OptionChain.close_time :
                break
        # print('exited chain thread')

    def get_option_chain(self):
        df = self.chain_dataframe
        if not self.bid_ask:
            df = df.drop(['bid' , 'bid_qty' , 'ask' , 'ask_qty'] , axis= 1  )
        return df

    def stop_option_chain(self):
        self.chain_status = False
        self.TD_OBJ.stop_live_data(self.option_symbols)

