"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log('Create new ActionSecret with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log(`Update ActionSecret ${secretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onDelete = async (event, octokit) => {
    const secretName = event.ResourceProperties.repositorySecretName;
    console.log('Delete ActionSecret ' + secretName);
    await deleteRepoSecret(event, octokit);
    return { PhysicalResourceId: secretName };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const secretId = event.ResourceProperties.sourceSecretArn;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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