"""
Tests for module facet.validation
"""
import warnings
from typing import List

import numpy as np
import numpy.typing as npt
import pytest
from sklearn import datasets, svm, tree
from sklearn.model_selection import GridSearchCV

from sklearndf import __sklearn_0_22__, __sklearn_version__

from facet.validation import BootstrapCV, StratifiedBootstrapCV


def test_bootstrap_cv_init() -> None:
    # filter out warnings triggered by sk-learn/numpy

    warnings.filterwarnings("ignore", message="numpy.dtype size changed")
    warnings.filterwarnings("ignore", message="numpy.ufunc size changed")

    # check erroneous inputs
    #   - n_splits = 0
    with pytest.raises(expected_exception=ValueError):
        BootstrapCV(n_splits=0)

    #   - n_splits < 0
    with pytest.raises(expected_exception=ValueError):
        BootstrapCV(n_splits=-1)


def test_get_train_test_splits_as_indices() -> None:

    n_test_splits = 200
    test_x = np.arange(0, 1000, 1)

    my_cv = BootstrapCV(n_splits=n_test_splits, random_state=42)

    def _generate_splits() -> List[npt.NDArray[np.int_]]:
        return [test_split for _, test_split in my_cv.split(X=test_x)]

    test_splits: List[npt.NDArray[np.int_]] = _generate_splits()

    # assert we get right amount of splits

    assert len(test_splits) == n_test_splits

    # check average ratio of test/train

    average_test_size = (
        sum(len(test_split) for test_split in test_splits) / n_test_splits
    )

    assert 0.35 < average_test_size / len(test_x) < 0.37

    # check that re-generating the split yields the same result

    test_splits_2: List[npt.NDArray[np.int_]] = _generate_splits()

    assert len(test_splits) == len(
        test_splits_2
    ), "the number of splits should be stable"

    for f1, f2 in zip(test_splits, test_splits_2):
        assert np.array_equal(f1, f2), "split indices should be stable"


def test_bootstrap_cv_with_sk_learn() -> None:
    # filter out warnings triggered by sk-learn/numpy

    warnings.filterwarnings("ignore", message="numpy.dtype size changed")
    warnings.filterwarnings("ignore", message="numpy.ufunc size changed")

    # load example data

    iris = datasets.load_iris()

    # define a yield-engine circular CV:

    my_cv = BootstrapCV(n_splits=50)

    # define parameters and pipeline

    parameters = {"kernel": ("linear", "rbf"), "C": [1, 10]}
    svc = svm.SVC(gamma="scale")

    # use the defined my_cv bootstrap CV within GridSearchCV

    if __sklearn_version__ < __sklearn_0_22__:
        # noinspection PyArgumentList
        clf = GridSearchCV(svc, parameters, cv=my_cv, iid=False)
    else:
        clf = GridSearchCV(svc, parameters, cv=my_cv)
    clf.fit(iris.data, iris.target)

    # test if the number of received splits is correct

    assert (
        clf.n_splits_ == 50
    ), "50 splits should have been generated by the bootstrap CV"

    assert clf.best_score_ > 0.85, "Expected a minimum score of 0.85"

    # define new parameters and a different pipeline
    # use the defined my_cv circular CV again within GridSeachCV

    parameters = {
        "criterion": ("gini", "entropy"),
        "max_features": ["sqrt", "auto", "log2"],
    }
    cl2 = GridSearchCV(tree.DecisionTreeClassifier(), parameters, cv=my_cv)
    cl2.fit(iris.data, iris.target)

    assert cl2.best_score_ > 0.85, "Expected a minimum score of 0.85"


def test_stratified_bootstrap_cv() -> None:
    n_test_splits = 200
    n_samples = 1000
    group_0_size = 100
    test_x = np.arange(n_samples).reshape(-1, 1)
    test_groups = np.hstack(
        [np.zeros(group_0_size), np.ones(len(test_x) - group_0_size)]
    )

    my_cv = StratifiedBootstrapCV(n_splits=n_test_splits, random_state=42)

    with pytest.raises(
        ValueError,
        match=r"^no target variable specified in arg y as labels for stratification$",
    ):
        next(my_cv.split(X=test_x))

    with pytest.raises(
        ValueError,
        match=r"^no target variable specified in arg y as labels for stratification$",
    ):
        next(my_cv.split(X=test_x, groups=test_groups))

    with warnings.catch_warnings():
        warnings.simplefilter("error")
        warnings.simplefilter("default", category=DeprecationWarning)
        train1, test1 = next(my_cv.split(X=test_x, y=test_groups))

    with pytest.warns(UserWarning, match=r"^ignoring arg groups=array\(\["):
        train2, test2 = next(my_cv.split(X=test_x, y=test_groups, groups=test_groups))

    assert np.array_equal(train1, train2)
    assert np.array_equal(test1, test2)

    def _generate_splits() -> List[npt.NDArray[np.int_]]:
        return [test_split for _, test_split in my_cv.split(X=test_x, y=test_groups)]

    test_splits: List[npt.NDArray[np.int_]] = _generate_splits()

    # assert we get right amount of splits

    assert len(test_splits) == n_test_splits

    # check average ratio of test/train for group 0

    n_test_size_in_range = sum(
        0.33 < (test_split < group_0_size).sum() / group_0_size < 0.4
        for test_split in test_splits
    )

    assert (
        n_test_size_in_range / n_test_splits > 2 / 3
    ), "2/3 of test sizes are within target range"

    # check total average ratio of test/train

    average_test_size = (
        sum(len(test_split) for test_split in test_splits) / n_test_splits
    )

    assert 0.35 < average_test_size / n_samples < 0.37

    # check that re-generating the split yields the same result

    test_splits_2: List[npt.NDArray[np.int_]] = _generate_splits()

    assert len(test_splits) == len(
        test_splits_2
    ), "the number of splits should be stable"

    for f1, f2 in zip(test_splits, test_splits_2):
        assert np.array_equal(f1, f2), "split indices should be stable"
