"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        this.hasSpotCapacity = false;
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        this.vpc = (_c = props.vpc) !== null && _c !== void 0 ? _c : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.external) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.internal) {
                this.hasInternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        if (this.hasInternalLoadBalancer) {
            this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
                vpc: this.vpc,
                internetFacing: false,
            });
        }
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        if (props.spot == true)
            this.hasSpotCapacity = true;
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
                circuitBreaker: props.circuitBreaker != false ? {
                    rollback: true,
                } : undefined,
            });
            this.service.push(svc);
            /**
             * determine if we have spot capacity in this cluster
             * scenario 1: FARGATE_SPOT with weight > 0
             * scenario 2: FARGATE_SPOT with base > 0
             * scenario 3: props.spot = true
             */
            (_e = t.capacityProviderStrategy) === null || _e === void 0 ? void 0 : _e.forEach(s => {
                if (s.capacityProvider == 'FARGATE_SPOT' && ((s.weight && s.weight > 0)
                    || (s.base && s.base > 0))) {
                    this.hasSpotCapacity = true;
                }
            });
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_g = (_f = t.scalingPolicy) === null || _f === void 0 ? void 0 : _f.maxCapacity) !== null && _g !== void 0 ? _g : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_j = (_h = t.scalingPolicy) === null || _h === void 0 ? void 0 : _h.targetCpuUtilization) !== null && _j !== void 0 ? _j : 50,
            });
            if (t.external) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.external.port}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.external.port,
                    protocol: t.external.certificate ? elbv2.ApplicationProtocol.HTTPS : elbv2.ApplicationProtocol.HTTP,
                    certificates: t.external.certificate,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_l = (_k = t.scalingPolicy) === null || _k === void 0 ? void 0 : _k.requestPerTarget) !== null && _l !== void 0 ? _l : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            if (t.internal) {
                const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the internal ALB
                new elbv2.ApplicationListener(this, `IntAlbListener${t.internal.port}`, {
                    loadBalancer: this.internalAlb,
                    open: true,
                    port: t.internal.port,
                    protocol: t.internal.certificate ? elbv2.ApplicationProtocol.HTTPS : elbv2.ApplicationProtocol.HTTP,
                    certificates: t.internal.certificate,
                    defaultTargetGroups: [inttg],
                });
                // extra scaling policy
                scaling.scaleOnRequestCount('RequestScaling2', {
                    requestsPerTarget: (_o = (_m = t.scalingPolicy) === null || _m === void 0 ? void 0 : _m.requestPerTarget) !== null && _o !== void 0 ? _o : 1000,
                    targetGroup: inttg,
                });
                inttg.addTarget(svc);
            }
        });
        // ensure the dependency
        const cp = this.node.tryFindChild('Cluster');
        this.service.forEach(s => {
            s.node.addDependency(cp);
        });
        // Route53
        const zoneName = (_e = (_d = props.route53Ops) === null || _d === void 0 ? void 0 : _d.zoneName) !== null && _e !== void 0 ? _e : 'svc.local';
        const externalAlbRecordName = (_g = (_f = props.route53Ops) === null || _f === void 0 ? void 0 : _f.externalAlbRecordName) !== null && _g !== void 0 ? _g : 'external';
        const internalAlbRecordName = (_j = (_h = props.route53Ops) === null || _h === void 0 ? void 0 : _h.internalAlbRecordName) !== null && _j !== void 0 ? _j : 'internal';
        // spot termination handler by default
        if (this.hasSpotCapacity && props.spotTerminationHandler !== false) {
            this.createSpotTerminationHandler(cluster);
        }
        if (this.enableLoadBalancerAlias) {
            const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
                zoneName,
                vpc: this.vpc,
            });
            if (this.hasInternalLoadBalancer) {
                new route53.ARecord(this, 'InternalAlbAlias', {
                    zone,
                    recordName: internalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new route53.ARecord(this, 'ExternalAlbAlias', {
                    zone,
                    recordName: externalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
            }
        }
        else {
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
            }
        }
    }
    createSpotTerminationHandler(cluster) {
        // create the handler
        const handler = new lambda.DockerImageFunction(this, 'SpotTermHandler', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/spot-term-handler')),
            timeout: cdk.Duration.seconds(20),
        });
        // create event rule
        const rule = new events.Rule(this, 'OnTaskStateChangeEvent', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [cluster.clusterArn],
                },
            },
        });
        rule.addTarget(new event_targets.LambdaFunction(handler));
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'elasticloadbalancing:DeregisterTargets',
                'ec2:DescribeSubnets',
            ],
            resources: ['*'],
        }));
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.2.20" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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