"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack-tiny', { env });
const zoneName = stack.stackName;
const internalAlbRecordName = 'internal';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
    ],
    route53Ops: {
        zoneName,
    },
});
//# sourceMappingURL=data:application/json;base64,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