"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        WAREHOUSE_SVC_URL: `${internalALBEndpoint}:9092/api.php`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
    environment: {
        serviceName: 'warehouse',
        versionNum: '0.1',
    },
});
// laravel-nginx-php-fpm service
const laravelNginxPhpFpmTask = new ecs.FargateTaskDefinition(stack, 'laravelNginxPhpFpmTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelNginxPhpFpmTask.addContainer('laravel-nginx-php-fpm', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-nginx-php-fpm')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-bitnami service
const laravelBitnamiTask = new ecs.FargateTaskDefinition(stack, 'laravelBitnamiTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelBitnamiTask.addContainer('laravel-bitnami', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-bitnami')),
    portMappings: [
        { containerPort: 3000 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// java spring boot service
const javaTask = new ecs.FargateTaskDefinition(stack, 'javaTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
javaTask.addContainer('java', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/java-spring-boot')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
// apache-php service
const apachePHP = new ecs.FargateTaskDefinition(stack, 'apachephpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
apachePHP.addContainer('apachephp', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/apache-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// python-flask service
const flaskTask = new ecs.FargateTaskDefinition(stack, 'flaskTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
flaskTask.addContainer('flask', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/python-flask')),
    portMappings: [
        { containerPort: 80 },
    ],
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
        // The nginx service(external only)
        {
            task: nginxTask,
            desiredCount: 1,
            external: { port: 9091 },
        },
        // The nginx-php-fpm service(external/internal)
        {
            task: phpTask,
            desiredCount: 1,
            internal: { port: 9092 },
            external: { port: 9092 },
        },
        // The NuxtJS service(external/internal)
        {
            task: nuxtTask,
            desiredCount: 1,
            internal: { port: 9093 },
            external: { port: 9093 },
        },
        // The node service(external/internal)
        {
            task: nodeTask,
            desiredCount: 1,
            internal: { port: 9094 },
            external: { port: 9094 },
        },
        // The laravel-nginx-php-fpm service(external/internal)
        {
            task: laravelNginxPhpFpmTask,
            desiredCount: 1,
            internal: { port: 9095 },
            external: { port: 9095 },
        },
        // The laravel-bitnami service(external/internal)
        {
            task: laravelBitnamiTask,
            desiredCount: 1,
            internal: { port: 9096 },
            external: { port: 9096 },
        },
        // java spring boot service(external/internal)
        {
            task: javaTask,
            desiredCount: 1,
            internal: { port: 9097 },
            external: { port: 9097 },
            healthCheck: { path: '/hello-world' },
        },
        // apache-php(external/internal)
        {
            task: apachePHP,
            desiredCount: 1,
            internal: { port: 9098 },
            external: { port: 9098 },
        },
        // python-flask(external/internal)
        {
            task: flaskTask,
            desiredCount: 1,
            internal: { port: 9099 },
            external: { port: 9099 },
        },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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