import glob
import os
import subprocess
from collections import OrderedDict

import numpy as np
import pandas as pd
import torch
import torch.nn as nn

# import tensorflow as tf
from sklearn.model_selection import train_test_split

from snorkel.classification.data import DictDataset, DictDataLoader


def load_spam_dataset(load_train_labels: bool = False, split_dev: bool = True):
    if os.path.basename(os.getcwd()) == "snorkel-tutorials":
        os.chdir("spam")
    try:
        subprocess.run(["bash", "download_data.sh"], check=True, stderr=subprocess.PIPE)
    except subprocess.CalledProcessError as e:
        print(e.stderr.decode())
        raise e
    filenames = sorted(glob.glob("data/Youtube*.csv"))

    dfs = []
    for i, filename in enumerate(filenames, start=1):
        df = pd.read_csv(filename)
        # Lowercase column names
        df.columns = map(str.lower, df.columns)
        # Remove comment_id field
        df = df.drop("comment_id", axis=1)
        # Add field indicating source video
        df["video"] = [i] * len(df)
        # Rename fields
        df = df.rename(columns={"class": "label", "content": "text"})
        # Shuffle order
        df = df.sample(frac=1, random_state=123).reset_index(drop=True)
        dfs.append(df)

    #     return X
    df_train = pd.concat(dfs[:4])

    #     if split_dev:
    #         df_dev = df_train.sample(100, random_state=123)

    if not load_train_labels:
        df_train["label"] = np.ones(len(df_train["label"])) * -1
    return df_train


#     df_valid_test = dfs[4]
#     df_valid, df_test = train_test_split(
#         df_valid_test, test_size=250, random_state=123, stratify=df_valid_test.label
#     )

#     if split_dev:
#         return df_train, df_dev, df_valid, df_test
#     else:
#         return df_train, df_valid, df_test


def get_keras_logreg(input_dim, output_dim=2):
    model = tf.keras.Sequential()
    if output_dim == 1:
        loss = "binary_crossentropy"
        activation = tf.nn.sigmoid
    else:
        loss = "categorical_crossentropy"
        activation = tf.nn.softmax
    dense = tf.keras.layers.Dense(
        units=output_dim,
        input_dim=input_dim,
        activation=activation,
        kernel_regularizer=tf.keras.regularizers.l2(0.001),
    )
    model.add(dense)
    opt = tf.keras.optimizers.Adam(lr=0.01)
    model.compile(optimizer=opt, loss=loss, metrics=["accuracy"])
    return model


def get_keras_lstm(num_buckets, embed_dim=16, rnn_state_size=64):
    lstm_model = tf.keras.Sequential()
    lstm_model.add(tf.keras.layers.Embedding(num_buckets, embed_dim))
    lstm_model.add(tf.keras.layers.LSTM(rnn_state_size, activation=tf.nn.relu))
    lstm_model.add(tf.keras.layers.Dense(1, activation=tf.nn.sigmoid))
    lstm_model.compile("Adagrad", "binary_crossentropy", metrics=["accuracy"])
    return lstm_model


def get_keras_early_stopping(patience=10):
    return tf.keras.callbacks.EarlyStopping(
        monitor="val_acc", patience=10, verbose=1, restore_best_weights=True
    )


def map_pad_or_truncate(string, max_length=30, num_buckets=30000):
    """Tokenize text, pad or truncate to get max_length, and hash tokens."""
    ids = tf.keras.preprocessing.text.hashing_trick(
        string, n=num_buckets, hash_function="md5"
    )
    return ids[:max_length] + [0] * (max_length - len(ids))


def featurize_df_tokens(df):
    return np.array(list(map(map_pad_or_truncate, df.text)))


def preview_tfs(df, tfs):
    transformed_examples = []
    for f in tfs:
        for i, row in df.sample(frac=1, random_state=2).iterrows():
            transformed_or_none = f(row)
            # If TF returned a transformed example, record it in dict and move to next TF.
            if transformed_or_none is not None:
                transformed_examples.append(
                    OrderedDict(
                        {
                            "TF Name": f.name,
                            "Original Text": row.text,
                            "Transformed Text": transformed_or_none.text,
                        }
                    )
                )
                break
    return pd.DataFrame(transformed_examples)


def df_to_features(vectorizer, df, split):
    """Convert pandas DataFrame containing spam data to bag-of-words PyTorch features."""
    words = [row.text for i, row in df.iterrows()]

    if split == "train":
        feats = vectorizer.fit_transform(words)
    else:
        feats = vectorizer.transform(words)
    X = feats.todense()
    Y = df["label"].values
    return X, Y


def create_dict_dataloader(X, Y, split, **kwargs):
    """Create a DictDataLoader for bag-of-words features."""
    ds = DictDataset.from_tensors(torch.FloatTensor(X), torch.LongTensor(Y), split)
    return DictDataLoader(ds, **kwargs)


def get_pytorch_mlp(hidden_dim, num_layers):
    layers = []
    for _ in range(num_layers):
        layers.extend([nn.Linear(hidden_dim, hidden_dim), nn.ReLU()])
    return nn.Sequential(*layers)


def get_various_data(
    X, Y, X_feats, temp_len, validation_size=100, test_size=200, L_size=100, U_size=None
):
    if U_size == None:
        U_size = X.size - L_size - validation_size - test_size
    index = np.arange(X.size)
    index = np.random.RandomState(seed=42).permutation(index)
    X = X[index]
    Y = Y[index]
    X_feats = X_feats[index]

    X_V = X[-validation_size:]
    Y_V = Y[-validation_size:]
    X_feats_V = X_feats[-validation_size:]
    R_V = np.zeros((validation_size, temp_len))

    X_T = X[-(validation_size + test_size) : -validation_size]
    Y_T = Y[-(validation_size + test_size) : -validation_size]
    X_feats_T = X_feats[-(validation_size + test_size) : -validation_size]
    R_T = np.zeros((test_size, temp_len))

    X_L = X[-(validation_size + test_size + L_size) : -(validation_size + test_size)]
    Y_L = Y[-(validation_size + test_size + L_size) : -(validation_size + test_size)]
    X_feats_L = X_feats[
        -(validation_size + test_size + L_size) : -(validation_size + test_size)
    ]
    R_L = np.zeros((L_size, temp_len))

    # X_U = X[:-(validation_size+test_size+L_size)]
    X_U = X[:U_size]
    X_feats_U = X_feats[:U_size]
    # Y_U = Y[:-(validation_size+test_size+L_size)]
    R_U = np.zeros((U_size, temp_len))

    return (
        X_V,
        Y_V,
        X_feats_V,
        R_V,
        X_T,
        Y_T,
        X_feats_T,
        R_T,
        X_L,
        Y_L,
        X_feats_L,
        R_L,
        X_U,
        X_feats_U,
        R_U,
    )


def get_test_U_V_data(
    X, X_feats, Y, temp_len, test_size=100, validation_size=100, U_size=None
):
    if U_size == None:
        U_size = X.size - test_size - validation_size
    index = np.arange(X.size)
    index = np.random.RandomState(seed=42).permutation(index)
    X = X[index]
    Y = Y[index]
    X_feats = X_feats[index]

    X_V = X[-validation_size:]
    Y_V = Y[-validation_size:]
    X_feats_V = X_feats[-validation_size:]

    X_T = X[-(validation_size + test_size) : -validation_size]
    Y_T = Y[-(validation_size + test_size) : -validation_size]
    X_feats_T = X_feats[-(validation_size + test_size) : -validation_size]
    R_T = np.zeros((test_size, temp_len))

    # X_U = X[:-(validation_size+test_size+L_size)]
    X_U = X[:U_size]
    X_feats_U = X_feats[:U_size]
    Y_U = Y[:U_size]
    R_U = np.zeros((U_size, temp_len))

    return X_T, Y_T, R_T, X_U, R_U, X_feats_T, Y_U, X_feats_U, X_V, Y_V, X_feats_V
