"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const kinesis = require("aws-cdk-lib/aws-kinesis");
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const pattern = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {});
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.kinesisStream !== null);
    expect(pattern.cloudwatchAlarms !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        kinesisStreamProps: {
            shardCount: 1,
            streamName: 'my-stream'
        },
        putRecordRequestTemplate: `{ "Data": "$util.base64Encode($input.json('$.foo'))", "PartitionKey": "$input.path('$.bar')" }`,
        putRecordRequestModel: { schema: {} },
        putRecordsRequestTemplate: `{ "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.foo)", "PartitionKey": "$elem.bar"}#if($foreach.hasNext),#end #end ] }`,
        putRecordsRequestModel: { schema: {} }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        Name: 'my-stream'
    });
    // Test for Cloudwatch Alarms
    expect(stack).toCountResources('AWS::CloudWatch::Alarm', 2);
});
// --------------------------------------------------------------
// Test deployment w/ existing stream without default cloudwatch alarms
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const construct = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {},
        existingStreamObj: new kinesis.Stream(stack, 'ExistingStream', {
            shardCount: 5,
            retentionPeriod: aws_cdk_lib_1.Duration.days(4)
        }),
        createCloudWatchAlarms: false
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 96
    });
    expect(construct.cloudwatchAlarms == null);
    // Since createCloudWatchAlars is set to false, no Alarm should exist
    expect(stack).not.toHaveResource('AWS::CloudWatch::Alarm');
});
test('Construct accepts additional PutRecord request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        additionalPutRecordRequestTemplates: {
            'text/plain': 'custom-template'
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'text/plain': 'custom-template'
            }
        }
    });
});
test('Construct accepts additional PutRecords request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        additionalPutRecordsRequestTemplates: {
            'text/plain': 'custom-template'
        }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'text/plain': 'custom-template'
            }
        }
    });
});
test('Construct uses default integration responses', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {});
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
});
test('Construct uses custom putRecordIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        putRecordIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom putRecordsIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        putRecordsIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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