"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToIotProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        this.addResourceMethod(namedShadowResource, props, 'things/{thingName}/shadow?name={shadowName}', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        if (props.apiGatewayCreateApiKey === true) {
            // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
            defaults.addCfnSuppressRules(apiMethod, [
                {
                    id: "W59",
                    reason: "When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE",
                },
            ]);
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "1.142.0" };
//# sourceMappingURL=data:application/json;base64,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