# 評価ガイド - 自動品質保証

このガイドでは、Refinireの組み込み評価システムを網羅的に説明し、構造化されたフィードバックとスコアリングによる自動品質評価とコンテンツ改善を可能にします。

## 概要

Refinireの評価システムは以下を提供します：
- **100点満点のスコアリング** による一貫した品質測定
- **構造化されたコメントリスト** による実行可能なフィードバック
- **自動再生成** による品質閾値未達成時の対応
- **カスタマイズ可能な評価基準** による異なるコンテンツタイプへの対応
- **改善プロンプト付き複数回リトライ** による継続的品質向上

## 目次

1. [基本評価設定](#基本評価設定)
2. [評価出力形式](#評価出力形式)
3. [効果的な評価指示の作成](#効果的な評価指示の作成)
4. [高度な評価パターン](#高度な評価パターン)
5. [ドメイン固有評価例](#ドメイン固有評価例)
6. [トラブルシューティングとベストプラクティス](#トラブルシューティングとベストプラクティス)

## 基本評価設定

### 標準評価パターン

```python
from refinire import RefinireAgent

agent = RefinireAgent(
    name="content_evaluator",
    generation_instructions="包括的で構造化されたコンテンツを作成してください",
    evaluation_instructions="""以下の基準でコンテンツ品質を0-100点で評価してください：
    - 明確性と読みやすさ（0-25点）
    - 正確性と事実の正しさ（0-25点）
    - 構成と組織化（0-25点）
    - 魅力的な文体とエンゲージメント（0-25点）
    
    評価結果は以下の形式で提供してください：
    スコア: [0-100]
    コメント:
    - [強みに関する具体的なフィードバック]
    - [改善点]
    - [向上のための提案]""",
    threshold=85.0,  # 85点未満で再生成
    max_retries=3,
    model="gpt-4o-mini"
)

result = agent.run("Pythonデコレータのガイドを書いて")
print(f"品質スコア: {result.evaluation_score}")
print(f"評価フィードバック: {result.evaluation_result}")
```

### コンテキスト付き評価

```python
from refinire import Context

ctx = Context()
result = agent.run("機械学習の概念を説明して", ctx)

# 詳細な評価結果へのアクセス
eval_result = ctx.evaluation_result
print(f"スコア: {eval_result['score']}")
print(f"合格: {eval_result['passed']}")
print(f"フィードバック: {eval_result['feedback']}")
```

## 評価出力形式

### 期待される評価応答構造

評価システムは、この**正確な形式**での応答を期待します：

```
スコア: 87
コメント:
- 技術的正確性が優れており、主要概念の包括的なカバレッジが素晴らしい
- 明確で論理的な構成により、概念が段階的に構築されている
- 抽象概念を説明する実践的な例の使用が効果的
- 複雑なワークフローには視覚的な図解があるとより良い
- 一般的な問題のトラブルシューティングセクションの追加を検討
- 一部の段落は読みやすさのために短縮できる
```

### 構造化された評価コンポーネント

1. **スコア行**: 「スコア:」で始まり、0-100の整数が続く
2. **コメントセクション**: 「コメント:」が単独の行で始まる
3. **コメントリスト**: 各コメントは「- 」で始まる箇条書き
4. **コメントカテゴリ**:
   - **強み**: コンテンツが優れている点
   - **弱点**: 改善が必要な領域
   - **提案**: 具体的で実行可能な推奨事項

### 評価結果へのアクセス

```python
# 評価付きエージェント実行後
result = agent.run("技術文書を作成して")

# 基本スコアアクセス
score = result.evaluation_score  # 0-100の整数

# 詳細評価データ（Context使用時）
ctx = Context()
result = agent.run("技術文書を作成して", ctx)

eval_data = ctx.evaluation_result
score = eval_data["score"]           # 87
passed = eval_data["passed"]         # 閾値に基づくTrue/False
feedback = eval_data["feedback"]     # 完全な評価テキスト
comments = eval_data["comments"]     # 解析されたコメントリスト
```

## 効果的な評価指示の作成

### 包括的評価テンプレート

```python
evaluation_template = """{content_type}の品質を0-100点で評価してください：

主要基準（合計80点）：
- {criterion_1}（0-20点）: {criterion_1_description}
- {criterion_2}（0-20点）: {criterion_2_description}
- {criterion_3}（0-20点）: {criterion_3_description}
- {criterion_4}（0-20点）: {criterion_4_description}

ボーナス基準（合計20点）：
- 革新性と創造性（0-10点）
- 実用性と適用可能性（0-10点）

評価形式：
スコア: [0-100]
コメント:
- [具体例を含む2-3の主要な強みを特定]
- [具体的な提案を含む2-3の改善点を指摘]
- [1-2の向上推奨事項を提供]

スコアリングガイドライン：
- 90-100点: 卓越した品質、最小限の改善が必要
- 80-89点: 高品質、軽微な改善が可能
- 70-79点: 良質、中程度の改善が必要
- 60-69点: 許容可能な品質、大幅な改善が必要
- 60点未満: 低品質、大規模な修正が必要"""
```

### ドメイン固有評価指示

**技術文書:**
```python
tech_doc_evaluation = """技術文書の品質を評価してください（0-100点）：

技術的正確性（0-30点）：
- コード例と技術詳細の事実的正確性
- 最新の情報とベストプラクティス
- 技術用語の適切な使用

明確性と使いやすさ（0-30点）：
- 対象読者にアクセス可能な明確な説明
- 論理的な流れと段階的複雑性
- 例と図解の効果的な使用

完全性（0-25点）：
- トピックの包括的なカバレッジ
- 前提条件とセットアップ指示の包含
- エラーハンドリングとトラブルシューティングガイダンス

フォーマットと構造（0-15点）：
- 一貫したフォーマットとスタイル
- 見出し、コードブロック、リストの適切な使用
- 簡単なナビゲーションと参照

評価は以下の形式で提供してください：
スコア: [0-100]
コメント:
- [技術的正確性と明確性の強み]
- [改善が必要な領域]
- [具体的な向上提案]"""
```

**創作文章:**
```python
creative_writing_evaluation = """創作文章の品質を評価してください（0-100点）：

ストーリーテリング（0-30点）：
- 魅力的な物語構造とペーシング
- キャラクター開発と対話の質
- プロットの一貫性とエンゲージメント

言語とスタイル（0-25点）：
- 文章の質と語彙選択
- 声の一貫性と適切なトーン
- 文学的技法とスタイル要素

独創性（0-25点）：
- ユニークな概念と創造的要素
- 新鮮な視点や革新的アプローチ
- 陳腐な表現や使い古されたテーマの回避

技術的工芸（0-20点）：
- 文法、スペル、句読点
- 文構造のバリエーション
- 適切なフォーマットと表現

評価は以下の形式で提供してください：
スコア: [0-100]
コメント:
- [創造的強みと魅力的要素]
- [技術的または物語的改善領域]
- [向上のための提案]"""
```

## 高度な評価パターン

### 多面的評価

```python
multi_aspect_agent = RefinireAgent(
    name="comprehensive_evaluator",
    generation_instructions="技術的正確性とアクセシビリティのバランスを取ったコンテンツを作成してください",
    evaluation_instructions="""多次元評価を実施してください（0-100点）：

技術的次元（0-40点）：
- 正確性と精密性（0-20点）
- 深さと包括性（0-20点）

コミュニケーション次元（0-40点）：
- 明確性とアクセシビリティ（0-20点）
- 構造と組織化（0-20点）

実用的次元（0-20点）：
- 行動可能性と有用性（0-10点）
- 実世界への適用可能性（0-10点）

各次元について以下を提供してください：
1. 次元スコアと理由
2. 具体的な強みと弱み
3. 改善推奨事項

最終評価形式：
スコア: [0-100]
コメント:
- 技術的: [40点中のスコア] - [分析]
- コミュニケーション: [40点中のスコア] - [分析]
- 実用的: [20点中のスコア] - [分析]
- 全体的提案: [向上推奨事項]""",
    threshold=80.0,
    max_retries=2,
    model="gpt-4o-mini"
)
```

### 段階的改善評価

```python
improvement_agent = RefinireAgent(
    name="improvement_evaluator",
    generation_instructions="継続的改善を伴う高品質コンテンツを作成してください",
    evaluation_instructions="""改善重視でコンテンツを評価してください（0-100点）：

現在の品質評価：
- コンテンツの強み（うまくいっている点）
- コンテンツの弱み（改善が必要な点）
- 全体的な一貫性と効果性

改善ロードマップ：
- 優先度1: 最も重要な改善が必要
- 優先度2: 中程度の向上が可能
- 優先度3: あったら良い最適化

再生成ガイダンス：
スコアが85点未満の場合、次回の試行に向けた具体的指示を提供：
- 維持: [保持すべき要素のリスト]
- 改善: [必要な具体的変更のリスト]
- 追加: [含めるべき不足要素のリスト]

スコア: [0-100]
コメント:
- 現在の強み: [維持すべきもの]
- 重要な改善: [優先度1項目]
- 向上機会: [優先度2-3項目]
- 次回反復焦点: [再生成の具体的ガイダンス]""",
    threshold=85.0,
    max_retries=3,
    model="gpt-4o-mini"
)
```

## ドメイン固有評価例

### コードレビュー評価

```python
code_review_agent = RefinireAgent(
    name="code_reviewer",
    generation_instructions="クリーンで効率的、よく文書化されたコードを書いてください",
    evaluation_instructions="""コード品質を評価してください（0-100点）：

機能性（0-25点）：
- 正確性とロジック
- エッジケースの処理
- パフォーマンスの考慮

コード品質（0-25点）：
- 読みやすさと明確性
- 命名規則
- コード組織

ベストプラクティス（0-25点）：
- デザインパターンの使用
- エラーハンドリング
- セキュリティ考慮事項

文書化（0-25点）：
- コメントとdocstring
- READMEとセットアップ指示
- 使用例

スコア: [0-100]
コメント:
- 機能的強み: [うまく動作する点]
- コード品質観察: [構造とスタイルのフィードバック]
- ベストプラクティス評価: [標準への準拠]
- 文書化レビュー: [明確性と完全性]
- 改善優先度: [具体的な向上領域]""",
    threshold=80.0,
    model="gpt-4o-mini"
)
```

### マーケティングコンテンツ評価

```python
marketing_agent = RefinireAgent(
    name="marketing_evaluator",
    generation_instructions="魅力的で説得力のあるマーケティングコンテンツを作成してください",
    evaluation_instructions="""マーケティングコンテンツの効果を評価してください（0-100点）：

メッセージの明確性（0-25点）：
- 明確な価値提案
- ターゲット読者との整合性
- 主要利益の伝達

説得力（0-25点）：
- 感情的アピールとエンゲージメント
- 信頼性と信頼構築
- 行動喚起の効果性

ブランド整合性（0-25点）：
- 一貫したブランドボイスとトーン
- ブランド価値の反映
- 市場ポジショニングの正確性

実用的効果性（0-25点）：
- 行動可能な次のステップ
- コンバージョン可能性
- 測定可能な目標

スコア: [0-100]
コメント:
- メッセージの強み: [明確な価値と利益]
- 説得要素: [感情的・論理的アピール]
- ブランド一貫性: [ボイスとポジショニング整合性]
- 効果性要因: [コンバージョンと行動可能性]
- 最適化提案: [具体的改善領域]""",
    threshold=85.0,
    model="gpt-4o-mini"
)
```

## トラブルシューティングとベストプラクティス

### 一般的な評価問題

**1. 一貫性のないスコアリング:**
```python
# 問題: 曖昧な評価基準
evaluation_instructions="品質を0-100で評価してください"

# 解決策: 具体的で詳細な基準
evaluation_instructions="""品質を評価してください（0-100点）：
- 正確性（0-30点）: 事実の正確性と精密性
- 明確性（0-30点）: 明確なコミュニケーションと理解
- 完全性（0-20点）: 包括的なカバレッジ
- スタイル（0-20点）: 適切なトーンと表現

スコア: [0-100]
コメント: [詳細なフィードバック]"""
```

**2. 不適切な評価形式:**
```python
# 評価が必要な形式に従うことを確保
def validate_evaluation_format(evaluation_text):
    """評価が必要な形式に従っているかを検証"""
    lines = evaluation_text.strip().split('\n')
    
    # スコア行をチェック
    score_line = next((line for line in lines if line.startswith('スコア:')), None)
    if not score_line:
        return False, "「スコア:」行が見つかりません"
    
    # コメントセクションをチェック
    comments_line = next((line for line in lines if line.startswith('コメント:')), None)
    if not comments_line:
        return False, "「コメント:」セクションが見つかりません"
    
    return True, "有効な形式です"
```

**3. 閾値調整:**
```python
# 保守的な閾値から始めて調整
initial_threshold = 70.0  # 初期テスト用に低め
production_threshold = 85.0  # 本番品質用に高め

# 再生成率を監視
regeneration_rate = (attempts - 1) / attempts
if regeneration_rate > 0.5:  # 50%超の再生成
    # 閾値を下げるか指示を改善することを検討
    pass
```

### ベストプラクティス

**1. 評価指示設計:**
- 具体的で測定可能な基準を使用
- 透明性のためのポイント配分を含める
- 明確な形式要件を提供
- 良い/悪いパフォーマンスの例を含める

**2. 閾値管理:**
- 開発中は低い閾値から始める
- 本番システム向けに段階的に増加
- 再生成率を監視して適宜調整
- 異なるコンテンツタイプに異なる閾値を検討

**3. コンテンツ品質改善:**
- 評価フィードバックを使用して生成指示を改善
- 一般的な失敗パターンを分析
- 生成と評価プロンプトの両方を反復
- 複雑なコンテンツには多段階評価を検討

**4. 監視と分析:**
```python
# 評価メトリクスを追跡
evaluation_metrics = {
    "平均スコア": sum(scores) / len(scores),
    "合格率": len([s for s in scores if s >= threshold]) / len(scores),
    "再生成率": regenerations / total_attempts,
    "スコア分布": {
        "90-100": len([s for s in scores if s >= 90]),
        "80-89": len([s for s in scores if 80 <= s < 90]),
        "70-79": len([s for s in scores if 70 <= s < 80]),
        "70未満": len([s for s in scores if s < 70])
    }
}
```

## まとめ

Refinireの評価システムは以下により一貫した高品質コンテンツ生成を可能にします：
- **構造化された100点スコアリング** による客観的品質測定
- **詳細なコメントリスト** による実行可能な改善ガイダンス
- **自動再生成** による継続的品質改善
- **柔軟な評価基準** による任意のドメインやコンテンツタイプへの適応

このガイドのパターンと例に従うことで、特定の用途に対して一貫した高品質出力を保証する堅牢な評価システムを作成できます。

**次のステップ**: [高度な機能](advanced_features_ja.md)を探索して、評価とストリーミング、コンテキスト管理、ワークフローオーケストレーションの組み合わせについて学びましょう。