"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.LogDriver = void 0;
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * (experimental) The log driver to use for the container.
 *
 * @experimental
 */
var LogDriver;
(function (LogDriver) {
    LogDriver["AWSLOGS"] = "awslogs";
    LogDriver["FLUENTD"] = "fluentd";
    LogDriver["GELF"] = "gelf";
    LogDriver["JOURNALD"] = "journald";
    LogDriver["LOGENTRIES"] = "logentries";
    LogDriver["JSON_FILE"] = "json-file";
    LogDriver["SPLUNK"] = "splunk";
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * (experimental) Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 *
 * @experimental
 */
class JobDefinition extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * (experimental) Imports an existing batch job definition by its amazon resource name.
     *
     * @experimental
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Imports an existing batch job definition by its name.
     *
     * If name is specified without a revision then the latest active revision is used.
     *
     * @experimental
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = core_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            sep: '/',
            resourceName: jobDefinitionName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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