# Copyright (C) 2024 European Union (Joint Research Centre)
#
# Licensed under the EUPL, Version 1.2 or - as soon they will be approved by
# the European Commission - subsequent versions of the EUPL (the "Licence");
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
#   https://joinup.ec.europa.eu/software/page/eupl
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the Licence is distributed on an "AS IS" basis,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the Licence for the specific language governing permissions and
# limitations under the Licence.

import numpy as np
import xarray as xr


def make_ts(dates, break_idx=-1, intercept=0.7, amplitude=0.15, magnitude=0.25,
            recovery_time=1095, sigma_noise=0.02, n_outlier=3,
            outlier_value=-0.1, n_nan=3):
    """Simulate a harmonic time-series with optional breakpoint, noise and outliers

    The time-series is generated by adding;
    - an intercept/trend component which varies depending on the phase of the time-series
    (stable, recovery)
    - An annual seasonal component
    - Random noise drawn from a normal distribution (white noise)
    Optional outliers are then added to randomly chosen observation as well as ``np.Nan`` values.
    Note that the seasonal cycles simulation approach used here is rather simplistic,
    using a sinusoidal model and therefore assuming symetrical and regular behaviour
    around the peak of the simulated variable. Actual vegetation signal is often more
    asymetrical and irregular.

    Args:
        dates (array-like): List or array of dates (numpy.datetime64)
        break_idx (int): Breakpoint index in the date array provided. Defaults to
            ``-1``, corresponding to a stable time-series
        intercept (float): Intercept of the time-series
        amplitude (float): Amplitude of the harmonic model (note that at every point
            of the time-series, the actual model amplitude is multiplied by the intercept
        magnitude (float): Break magnitude (always a drop in y value)
        recovery_time (int): Time (in days) to recover the initial intersect value
            following a break
        sigma_noise (float): Sigma value of the normal distribution (mean = 0) from which
            noise values are drawn
        n_outlier (int): Number of outliers randomly assigned to observations of the
            time-series
        outlier_value (float): Value to assign to outliers
        n_nan (int): Number of ``np.nan`` (no data) assigned to observations of the
            time-series

    Example:
        >>> from nrt.data import simulate
        >>> import numpy as np
        >>> import matplotlib.pyplot as plt

        >>> dates = np.arange('2018-01-01', '2022-06-15', dtype='datetime64[W]')
        >>> ts = simulate.make_ts(dates=dates, break_idx=30)

        >>> plt.plot(dates, ts)
        >>> plt.show()

    Returns:
        np.ndarray: Array of simulated values of same size as ``dates``
    """
    dates = dates.astype('datetime64[D]')
    timestamps = dates.astype(int)
    ydays = (dates - dates.astype('datetime64[Y]')).astype(int) + 1
    y = np.empty_like(dates, dtype=np.float64)
    # Intercept array
    y[:] = intercept
    # Build trend segment if break
    if break_idx != -1:
        # Segment bounds
        segment_start_y = intercept - magnitude
        segment_start_timestamp = timestamps[break_idx]
        segment_end_timestamp = segment_start_timestamp + recovery_time
        segment_end_idx = np.abs(segment_end_timestamp - timestamps).argmin()
        # Compute y values
        recovery_rate = magnitude / recovery_time
        days_since_break = timestamps - segment_start_timestamp
        trend_segment = (recovery_rate * days_since_break + segment_start_y)[break_idx + 1:segment_end_idx + 1]
        # include into y
        y[break_idx + 1:segment_end_idx + 1] = trend_segment
    # Seasonality
    amplitude_values = amplitude * y
    season = amplitude * np.sin(2 * np.pi * timestamps / 365.25 - 2)
    # noise and outliers
    noise = np.random.normal(0, sigma_noise, dates.size)
    # Combine the 3 (trend, season, noise) components
    ts = y + season + noise
    # Add optional outliers and Nans
    outliers_idx = np.random.choice(np.arange(0, dates.size), size=n_outlier, replace=False)
    nan_idx = np.random.choice(np.arange(0, dates.size), size=n_nan)
    ts[outliers_idx] = outlier_value
    ts[nan_idx] = np.nan
    return ts


def make_cube_parameters(shape=(100,100),
                         break_idx_interval=(0,100),
                         intercept_interval=(0.6, 0.8),
                         amplitude_interval=(0.12, 0.2),
                         magnitude_interval=(0.2, 0.3),
                         recovery_time_interval=(800,1400),
                         sigma_noise_interval=(0.02, 0.04),
                         n_outliers_interval=(0,5),
                         n_nan_interval=(0,5),
                         unstable_proportion=0.5):
    """Create ``xarray.Dataset`` of paramters for generation of synthetic data cube

    Prepares the main input required by the the ``make_cube`` function. This
    intermediary step eases the creation of multiple synthetic DataArrays sharing
    similar characteristics (e.g. to simulate multispectral data)

    Args:
        shape (tuple): A size two integer tuple giving the x,y size of the Dataset to be
            generated
        break_idx_interval (tuple): A tuple of two integers indicating the interval
            from which the breakpoint position in the time-series is drawn. Generate
            array of random values passed to the ``break_idx` argument of ``make_ts``.
            Similarly to python ranges, upper bound value is excluded from the resulting
            array. To produce a zero filled array ``(0,1)`` can therefore be used
            TODO: add a default to allow breakpoint at any location (conflict with Nan that indicate no break)
        intercept_interval (tuple): A tuple of two floats providing the interval
            from which intercept is drawn. Generate array of random values passed
            to the ``intercept`` argument of ``make_ts``
        amplitude_interval (tuple): A tuple of two floats indicating the interval
            from which the seasonal amplitude parameter is drawn. Generate array
            of random values passed to the ``amplitude`` argument of ``make_ts``
        magnitude_interval (tuple): A tuple of two floats indicating the interval
            from which the breakpoint magnitude parameter is drawn. Generate array
            of random values passed to the ``magnitude`` argument of ``make_ts``
        recovery_time_interval (tuple): A tuple of two integers indicating the interval
            from which the recovery time parameter (in days) is drawn. Generate array
            of random values passed to the ``recovery_time` argument of ``make_ts``
        sigma_noise_interval (tuple): A tuple of two floats indicating the interval
            from which the white noise level is drawn. Generate array of random
            values passed to the ``sigma_noise` argument of ``make_ts``
        n_outliers_interval (tuple): A tuple of two integers indicating the interval
            from which the number of outliers is drawn. Generate array
            of random values passed to the ``n_outliers` argument of ``make_ts``
        n_nan_interval (tuple): A tuple of two integers indicating the interval
            from which the number of no-data observations is drawn. Generate array
            of random values passed to the ``n_nan` argument of ``make_ts``
        unstable_proportion (float): Proportion of time-series containing a breakpoint.
            The other time-series are stable.

    Returns:
        xarray.Dataset: Dataset with arrays of parameters required for the generation
            of synthetic time-series using the spatialized version of ``make_ts``
            (see ``make_cube``)

    Examples:
        >>> import time
        >>> import numpy as np
        >>> import xarray as xr
        >>> from nrt.data import simulate
        >>> import matplotlib.pyplot as plt
        >>> params_nir = simulate.make_cube_parameters(shape=(20,20),
        ...                                            n_outliers_interval=(0,1),
        ...                                            n_nan_interval=(0,1),
        ...                                            break_idx_interval=(50,100))
        >>> params_red = params_nir.copy(deep=True)
        >>> # create parameters for red, green, blue cubes by slightly adjusting intercept,
        >>> # magnitude and amplitude parameters
        >>> params_red['intercept'].data = np.random.uniform(0.09, 0.12, size=(20,20))
        >>> params_red['magnitude'].data = np.random.uniform(-0.1, -0.03, size=(20,20))
        >>> params_red['amplitude'].data = np.random.uniform(0.03, 0.07, size=(20,20))
        >>> params_green = params_nir.copy(deep=True)
        >>> params_green['intercept'].data = np.random.uniform(0.12, 0.20, size=(20,20))
        >>> params_green['magnitude'].data = np.random.uniform(0.05, 0.1, size=(20,20))
        >>> params_green['amplitude'].data = np.random.uniform(0.05, 0.08, size=(20,20))
        >>> params_blue = params_nir.copy(deep=True)
        >>> params_blue['intercept'].data = np.random.uniform(0.08, 0.13, size=(20,20))
        >>> params_blue['magnitude'].data = np.random.uniform(-0.01, 0.01, size=(20,20))
        >>> params_blue['amplitude'].data = np.random.uniform(0.02, 0.04, size=(20,20))
        >>> dates = np.arange('2018-01-01', '2022-06-15', dtype='datetime64[W]')
        >>> # Create cubes (DataArrays) and merge them into a sligle Dataset
        >>> nir = data.make_cube(dates, name='nir', params_ds=params_nir)
        >>> red = data.make_cube(dates, name='red', params_ds=params_red)
        >>> green = data.make_cube(dates, name='green', params_ds=params_green)
        >>> blue = data.make_cube(dates, name='blue', params_ds=params_blue)
        >>> cube = xr.merge([blue, green, red, nir]).to_array()
        >>> # PLot one ts
        >>> cube.isel(x=5, y=5).plot(row='variable')
        >>> plt.show()
    """
    intercept = np.random.uniform(*intercept_interval, size=shape)
    amplitude = np.random.uniform(*amplitude_interval, size=shape)
    magnitude = np.random.uniform(*magnitude_interval, size=shape)
    recovery_time = np.random.randint(*recovery_time_interval, size=shape)
    sigma_noise = np.random.uniform(*sigma_noise_interval, size=shape)
    n_outlier = np.random.randint(*n_outliers_interval, size=shape)
    n_nan = np.random.randint(*n_nan_interval, size=shape)
    break_idx = np.random.randint(*break_idx_interval, size=shape)
    # Make a proportion of these cells stable
    size = np.multiply(*shape)
    stable_size = size - round(unstable_proportion * size)
    break_idx.ravel()[np.random.choice(size, stable_size, replace=False)] = -1
    # Build Dataset of parameters
    params = xr.Dataset(data_vars={'intercept': (['y', 'x'], intercept),
                                   'amplitude': (['y', 'x'], amplitude),
                                   'magnitude': (['y', 'x'], magnitude),
                                   'recovery_time': (['y', 'x'], recovery_time),
                                   'sigma_noise': (['y', 'x'], sigma_noise),
                                   'n_outlier': (['y', 'x'], n_outlier),
                                   'n_nan': (['y', 'x'], n_nan),
                                   'break_idx': (['y', 'x'], break_idx)},
                        coords={'y': np.arange(shape[0]),
                                'x': np.arange(shape[1])})
    return params


def make_cube(dates, params_ds, outlier_value=0.1, name='ndvi'):
    """Generate a cube of synthetic time-series

    See ``make_ts`` for more details on how every single time-series is generated

    Args:
        dates (array-like): List or array of dates (numpy.datetime64)
        params_ds (xarray.Dataset): Dataset containing arrays of time-series generation
            parameters. See ``make_cube_parameters`` for a helper to generate such Dataset.
            Spatial dimensions of the params_ds Dataset are used for the generated cube
        outlier_value (float): Value to assign to outliers
        name (str): Name of the generated variable in the DataArray

    Return:
        xarray.DataArray: Cube of synthetic time-series generated using the paramters
            provided via ``param_ds`` Dataset.

    Example:
        >>> import time
        >>> import numpy as np
        >>> from nrt.data import simulate
        >>> import matplotlib.pyplot as plt
        >>> dates = np.arange('2018-01-01', '2022-06-15', dtype='datetime64[W]')
        >>> params_ds = simulate.make_cube_parameters(shape=(100,100),
        ...                                           n_outliers_interval=(0,5),
        ...                                           n_nan_interval=(0,7),
        ...                                           break_idx_interval=(100,dates.size - 20))
        >>> cube = simulate.make_cube(dates=dates, params_ds=params_ds)
        >>> # PLot one ts
        >>> cube.isel(x=5, y=5).plot()
        >>> plt.show()
    """
    nrows, ncols = params_ds.intercept.data.shape
    # Vectorize function
    make_ts_v = np.vectorize(make_ts, signature='(n),(),(),(),(),(),(),(),(),()->(n)')
    # Create output array
    out = make_ts_v(dates=dates,
                    break_idx=params_ds.break_idx.data,
                    intercept=params_ds.intercept.data,
                    amplitude=params_ds.amplitude.data,
                    magnitude=params_ds.magnitude.data,
                    recovery_time=params_ds.recovery_time.data,
                    sigma_noise=params_ds.sigma_noise.data,
                    n_outlier=params_ds.n_outlier.data,
                    outlier_value=outlier_value,
                    n_nan=params_ds.n_nan.data)
    # Build xarray dataset
    xr_cube = xr.DataArray(data=np.moveaxis(out, -1, 0),
                           coords={'time': dates,
                                   'y': np.arange(nrows),
                                   'x': np.arange(ncols)},
                          name=name)
    return xr_cube

if __name__ == "__main__":
    import doctest
    doctest.testmod()

