"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenerateCSV = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_ = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Generates a output based on Textract Forms and Queries.
 * Supported output_types: "LINES" | "CSV"
 *
 * Input: "Payload"."textract_result"."TextractOutputJsonPath"
 * Output: "TextractOutputCSVPath" TODO: rename
 *
 *
 * Output as LINES
 * Example (Python)
 * ```python
 *         generate_text = tcdk.TextractGenerateCSV(
            self,
            "GenerateText",
            csv_s3_output_bucket=document_bucket.bucket_name,
            csv_s3_output_prefix=s3_txt_output_prefix,
            output_type='LINES',
            lambda_log_level="DEBUG",
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            input=sfn.TaskInput.from_object({
                "Token":
                sfn.JsonPath.task_token,
                "ExecutionId":
                sfn.JsonPath.string_at('$$.Execution.Id'),
                "Payload":
                sfn.JsonPath.entire_payload,
            }),
            result_path="$.txt_output_location")
   ```
 */
class TextractGenerateCSV extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var lambdaMemoryMB = 1048;
        if (typeof (props.lambdaMemoryMB != 0) && props.lambdaMemoryMB) {
            lambdaMemoryMB = props.lambdaMemoryMB;
        }
        var outputType = props.outputType === undefined ? 'CSV' : props.outputType;
        var lambdaLogLevel = 'INFO';
        if (typeof (props.lambdaLogLevel) != 'undefined' && !props.lambdaMemoryMB) {
            lambdaLogLevel = props.lambdaLogLevel;
        }
        var lambdaTimeout = 900;
        if (typeof (props.lambdaTimeout != 0) && props.lambdaTimeout) {
            lambdaTimeout = props.lambdaTimeout;
        }
        const csvGeneratorFunction = new lambda_.DockerImageFunction(this, 'TextractCSVGenerator', {
            code: lambda_.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/generatecsv/')),
            memorySize: lambdaMemoryMB,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                CSV_S3_OUTPUT_BUCKET: props.csvS3OutputBucket,
                CSV_S3_OUTPUT_PREFIX: props.csvS3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                OUTPUT_TYPE: outputType,
            },
        });
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:PutObject', 's3:Get*', 's3:List*'], resources: ['*'] }));
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.generateCSVLambda = csvGeneratorFunction;
        this.generateCSVLogGroup = csvGeneratorFunction.logGroup;
        const csvGeneratorLambdaInvoke = new tasks.LambdaInvoke(this, 'csvGeneratorInvoke', {
            lambdaFunction: csvGeneratorFunction,
        });
        csvGeneratorLambdaInvoke.addCatch(new sfn.Fail(this, 'csvGenerationFailure'), {
            errors: [sfn.Errors.ALL],
        });
        const workflow_chain = sfn.Chain.start(csvGeneratorLambdaInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            tracingEnabled: true,
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
           * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
           *
           * the scoped access policy should be generated accordingly.
           *
           * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
           */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenerateCSV = TextractGenerateCSV;
_a = JSII_RTTI_SYMBOL_1;
TextractGenerateCSV[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenerateCSV", version: "0.0.5" };
TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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