"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractA2ISfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls and A2I endpoint arn with a task_token and waits for the A2I job to finish in order to continue the workflow.
 *
 * Basic implementation
 *
 * Input: "Payload"."a2iInputPath"
 * Output:
 * ```json
 * {
        'humanLoopStatus': human_loop_status,
        'humanLoopResultPath': human_loop_result,
        'humanLoopCreationTime': human_loop_creation_time,
    }
    ```
 *
   ```python
   textract_a2i_task = tcdk.TextractA2ISfnTask(
        self,
        "TextractA2I",
        a2i_flow_definition_arn=
        "arn:aws:sagemaker:us-east-1:913165245630:flow-definition/textract-classifiction",
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.a2i_result")
  ```
 */
class TextractA2ISfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractA2ISfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        const taskTokenTable = new dynamodb.Table(this, 'A2ITaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            timeToLiveAttribute: 'ttltimestamp',
        });
        this.taskTokenTableName = taskTokenTable.tableName;
        const startA2IFunction = new lambda.DockerImageFunction(this, 'startA2ICall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/a2i_preprocess/')),
            memorySize: 128,
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                A2I_FLOW_DEFINITION_ARN: props.a2iFlowDefinitionARN,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        const startA2ITask = new tasks.LambdaInvoke(this, 'StartA2ITask', {
            lambdaFunction: startA2IFunction,
        });
        const postA2IFunction = new lambda.DockerImageFunction(this, 'endA2ICall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/a2i_postprocess/')),
            memorySize: 128,
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        const rule = new events.Rule(this, 'EventBridgeRule', {
            enabled: true,
            eventPattern: {
                source: ['aws.sagemaker'],
                detailType: ['SageMaker A2I HumanLoop Status Change'],
                detail: {
                    humanLoopStatus: ['Completed', 'Failed'],
                    flowDefinitionArn: [props.a2iFlowDefinitionARN],
                },
            },
        });
        rule.addTarget(new targets.LambdaFunction(postA2IFunction));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'sagemaker:StartHumanLoop',
            ],
            resources: [props.a2iFlowDefinitionARN],
        }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:PutObject', 's3:Get*', 's3:List*'], resources: ['*'] }));
        postA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        postA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        const workflow_chain = sfn.Chain.start(startA2ITask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(2),
            tracingEnabled: true,
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractA2ISfnTask = TextractA2ISfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractA2ISfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractA2ISfnTask", version: "0.0.5" };
TextractA2ISfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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