import * as iam from 'aws-cdk-lib/aws-iam';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import * as sfn from 'aws-cdk-lib/aws-stepfunctions';
import { Construct } from 'constructs';
export declare function validatePatternSupported(integrationPattern: sfn.IntegrationPattern, supportedPatterns: sfn.IntegrationPattern[]): void;
export declare function integrationResourceArn(service: string, api: string, integrationPattern?: sfn.IntegrationPattern): string;
export interface SpacySfnTaskProps extends sfn.TaskStateBaseProps {
    /** log level for Lambda function, supports DEBUG|INFO|WARNING|ERROR|FATAL */
    readonly lambdaLogLevel?: 'DEBUG' | 'INFO' | 'WARNING' | 'ERROR' | 'FATAL';
    /** how long can we wait for the process (default is 48 hours (60*48=2880)) */
    readonly textractStateMachineTimeoutMinutes?: number;
    /** timeout for Lambda function calling Spacy NLP, default is 900 seconds */
    readonly spacyLambdaTimeout?: number;
    /** memorySize for Lambda function calling Spacy NLP, default is 4096 MB */
    readonly spacyLambdaMemorySize?: number;
    /** ECR Container URI for Spacy classification */
    readonly spacyImageEcrRepository?: string;
    /**
         * The JSON input for the execution, same as that of StartExecution.
         *
         * @see https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html
         *
         * @default - The state input (JSON path '$')
         */
    readonly input?: sfn.TaskInput;
    /**
            * The name of the execution, same as that of StartExecution.
            *
            * @see https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html
            *
            * @default - None
            */
    readonly name?: string;
    /**
            * Pass the execution ID from the context object to the execution input.
            * This allows the Step Functions UI to link child executions from parent executions, making it easier to trace execution flow across state machines.
            *
            * If you set this property to `true`, the `input` property must be an object (provided by `sfn.TaskInput.fromObject`) or omitted entirely.
            *
            * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-nested-workflows.html#nested-execution-startid
            *
            * @default - false
            */
    readonly associateWithParent?: boolean;
}
/**
 * Deploys a Lambda Container with a Spacy NLP model to call textcat
 *
 * Input: "textract_result"."txt_output_location"
 * Output:  { "documentType": "AWS_PAYSTUBS" } (example will be at "classification"."documentType")
 *
 * Example (Python)
 * ```python
    spacy_classification_task = tcdk.SpacySfnTask(
        self,
        "Classification",
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.classification")
    ```
 */
export declare class SpacySfnTask extends sfn.TaskStateBase {
    private readonly props;
    private static readonly SUPPORTED_INTEGRATION_PATTERNS;
    protected readonly taskMetrics?: sfn.TaskMetricsConfig;
    protected readonly taskPolicies?: iam.PolicyStatement[];
    private readonly integrationPattern;
    stateMachine: sfn.IStateMachine;
    spacySyncLambdaLogGroup: ILogGroup;
    version: string;
    spacyCallFunction: lambda.IFunction;
    constructor(scope: Construct, id: string, props: SpacySfnTaskProps);
    /**
         * @internal
         */
    protected _renderTask(): any;
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    private createScopedAccessPolicy;
}
