"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CSVToAuroraTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const rds = require("aws-cdk-lib/aws-rds");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const rdsServerlessInit_1 = require("./rdsServerlessInit");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * CSVToAuroraTask is a demo construct to show import into a serverless Aurora DB.
 * At the moment it also creates the Aurora Serverless RDS DB, initializes a table structure the matches the output of the GenerateCSV construct.
 * The Step Functions flow expect a pointer to a CSV at "csv_output_location"."TextractOutputCSVPath" and uses that to execute a batch insert statement command.
 *
 * Example:
 * ```python
*  csv_to_aurora_task = tcdk.CSVToAuroraTask(
        self,
        "CsvToAurora",
        vpc=vpc,
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload
        }),
        result_path="$.textract_result")
  ```
 *
 * Input: "csv_output_location"."TextractOutputCSVPath"
 * Output: CSV in Aurora Serverless DB, table name 'textractcsvimport"
 */
class CSVToAuroraTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, CSVToAuroraTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 600 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 256 : props.lambdaMemory;
        var csvToAuroraMaxRetries = props.csvToAuroraMaxRetries === undefined ? 100 : props.csvToAuroraMaxRetries;
        var csvToAuroraBackoffRate = props.csvToAuroraBackoffRate === undefined ? 1.1 : props.csvToAuroraBackoffRate;
        var csvToAuroraInterval = props.csvToAuroraInterval === undefined ? 1 : props.csvToAuroraInterval;
        const lambdaSG = new ec2.SecurityGroup(this, 'LambdaSG', { allowAllOutbound: true, vpc: props.vpc });
        const auroraSg = new ec2.SecurityGroup(this, 'Aurora', { allowAllOutbound: true, vpc: props.vpc });
        auroraSg.addIngressRule(auroraSg, ec2.Port.tcp(5432), 'fromSameSG');
        auroraSg.addIngressRule(auroraSg, ec2.Port.tcp(443), 'fromSameSG');
        auroraSg.addIngressRule(lambdaSG, ec2.Port.tcp(5432), 'LambdaIngreess');
        auroraSg.addIngressRule(lambdaSG, ec2.Port.tcp(443), 'LambdaIngreess');
        // AURORA
        this.dbCluster = new rds.ServerlessCluster(this, id + 'AuroraPSQL', {
            engine: rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-postgresql10'),
            vpc: props.vpc,
            securityGroups: [auroraSg],
            enableDataApi: true,
        });
        const rdsServerlessInit = new rdsServerlessInit_1.RdsServerlessInit(this, 'RdsServerlessInit', {
            dbClusterSecretARN: this.dbCluster.secret.secretArn,
            dbClusterARN: this.dbCluster.clusterArn,
        });
        rdsServerlessInit.node.addDependency(this.dbCluster);
        // LAMBDA PUT ON Cluster
        this.csvToAuroraFunction = new lambda.DockerImageFunction(this, 'CSVToAuroraFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/csv_to_aurora/')),
            memorySize: lambdaMemory,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            securityGroups: [lambdaSG],
            vpc: props.vpc,
            environment: {
                SECRET_ARN: this.dbCluster.secret.secretArn,
                CLUSTER_ARN: this.dbCluster.clusterArn,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.csvToAuroraLambdaLogGroup = this.csvToAuroraFunction.logGroup;
        const csvToAuroraTask = new tasks.LambdaInvoke(this, 'TextractSyncCallTask', { lambdaFunction: this.csvToAuroraFunction });
        csvToAuroraTask.addRetry({
            maxAttempts: csvToAuroraMaxRetries,
            backoffRate: csvToAuroraBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(csvToAuroraInterval),
            errors: ['StatementTimeoutException', 'InternalServerErrorException', 'ServiceUnavailableError', 'Lambda.Unknown'],
        });
        // WORKFLOW
        const workflow_chain = sfn.Chain.start(csvToAuroraTask);
        this.stateMachine = new sfn.StateMachine(this, id + '-SFN', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'secretsmanager:GetSecretValue',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonRDSDataFullAccess'));
        // START DASHBOARD
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractIDPCSVToAurora';
        const csvToAuroraNumberRowsInsertedFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberOfRowsFilter`, {
            logGroup: this.csvToAuroraFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberOfRows',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'numberOfRows')
                .whereString('message', '=', 'csv_to_aurora_insert_rows:'),
            metricValue: '$numberOfRows',
        });
        this.csvToAuroraNumberRowsInsertedMetric = csvToAuroraNumberRowsInsertedFilter.metric({ statistic: 'sum' });
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.CSVToAuroraTask = CSVToAuroraTask;
_a = JSII_RTTI_SYMBOL_1;
CSVToAuroraTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.CSVToAuroraTask", version: "0.0.5" };
CSVToAuroraTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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