"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericAsyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
var textractSync_1 = require("./textractSync");
Object.defineProperty(exports, "TextractGenericSyncSfnTask", { enumerable: true, get: function () { return textractSync_1.TextractGenericSyncSfnTask; } });
var textractDecider_1 = require("./textractDecider");
Object.defineProperty(exports, "TextractPOCDecider", { enumerable: true, get: function () { return textractDecider_1.TextractPOCDecider; } });
var textractClassificationConfigurator_1 = require("./textractClassificationConfigurator");
Object.defineProperty(exports, "TextractClassificationConfigurator", { enumerable: true, get: function () { return textractClassificationConfigurator_1.TextractClassificationConfigurator; } });
var textractOutputConfigToJSON_1 = require("./textractOutputConfigToJSON");
Object.defineProperty(exports, "TextractAsyncToJSON", { enumerable: true, get: function () { return textractOutputConfigToJSON_1.TextractAsyncToJSON; } });
var textractGenerateCSV_1 = require("./textractGenerateCSV");
Object.defineProperty(exports, "TextractGenerateCSV", { enumerable: true, get: function () { return textractGenerateCSV_1.TextractGenerateCSV; } });
var textractA2I_1 = require("./textractA2I");
Object.defineProperty(exports, "TextractA2ISfnTask", { enumerable: true, get: function () { return textractA2I_1.TextractA2ISfnTask; } });
var rdsCSVToAurora_1 = require("./rdsCSVToAurora");
Object.defineProperty(exports, "CSVToAuroraTask", { enumerable: true, get: function () { return rdsCSVToAurora_1.CSVToAuroraTask; } });
var comprehendClassification_1 = require("./comprehendClassification");
Object.defineProperty(exports, "ComprehendGenericSyncSfnTask", { enumerable: true, get: function () { return comprehendClassification_1.ComprehendGenericSyncSfnTask; } });
var spacyClassification_1 = require("./spacyClassification");
Object.defineProperty(exports, "SpacySfnTask", { enumerable: true, get: function () { return spacyClassification_1.SpacySfnTask; } });
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls Textract asynchronous API.
 * Supports the Textract APIs:  'GENERIC' |  'EXPENSE'
 * When GENERIC is called with features in the manifest definition, will call the AnalzyeDocument API.
 * Takes the configuration from "Payload"."manifest"
 * Will retry on recoverable errors based on textractAsyncCallMaxRetries
 * errors for retry: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
 *
 * Internally calls Start* calls with OutputConfig and SNSNotification.
 * Another Lambda functions waits for SNS Notification event and notifies the Step Function flow with the task token.
 *
 * Input: "Payload"."manifest"
 * Output: potentially paginated Textract JSON Schema output at "TextractTempOutputJsonPath" (with the example will be at: "textract_result"."TextractTempOutputJsonPath")
 *
 * Works together with TextractAsyncToJSON, which takes the s3_output_bucket/s3_temp_output_prefix location as input
 *
 * Example (Python)
 * ```python
    textract_async_task = tcdk.TextractGenericAsyncSfnTask(
        self,
        "TextractAsync",
        s3_output_bucket=s3_output_bucket,
        s3_temp_output_prefix=s3_temp_output_prefix,
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.textract_result")
    ```

 */
class TextractGenericAsyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        const taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            timeToLiveAttribute: 'ttltimestamp',
        });
        this.taskTokenTableName = taskTokenTable.tableName;
        this.textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
            assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
        });
        this.textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        this.textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            environment: {
                NOTIFICATION_SNS: this.textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: this.textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractAsyncCallTask = new tasks.LambdaInvoke(this, 'TextractAsyncCallTask', { lambdaFunction: this.textractAsyncCallFunction });
        textractAsyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Start*',
                'textract:Get*',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        this.startTextractLambdaLogGroup = this.textractAsyncCallFunction.logGroup;
        this.textractAsyncReceiveSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.textractAsyncReceiveSNSFunction));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:GetItem'], resources: ['*'] }));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:Put*', 's3:List*'],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/${props.s3TempOutputPrefix}/*`],
        }));
        this.receiveStartSNSLambdaLogGroup = this.textractAsyncReceiveSNSFunction.logGroup;
        const workflow_chain = sfn.Chain.start(textractAsyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractConstructGenericAsync';
        // OPERATIONAL
        const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Duration',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                .whereString('message', '=', `textract_async_${textractAPI}_job_duration_in_ms:`),
            metricValue: '$durationMs',
        });
        this.asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
        const asyncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '$pages',
        });
        this.asyncNumberPagesMetric = asyncNumberPagesMetricFilter.metric({ statistic: 'sum' });
        // TODO: expose all filters as properties of Construct, so users can build their own metrics
        const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsFinished',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '1',
        });
        this.asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
        const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsStarted',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                .whereString('message', '=', `textract_async_${textractAPI}_job_started`),
            metricValue: '1',
        });
        this.asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
        const asyncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_send_to_process:`),
            metricValue: '$pages',
        });
        this.asyncNumberPagesSendMetric = asyncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
        // OPERATIONAL STOP
        // CALCUATED OPERATIONAL METRICS
        const pagesPerSecond = new cloudwatch.MathExpression({
            expression: 'pages / (duration / 1000)',
            usingMetrics: {
                pages: this.asyncNumberPagesMetric,
                duration: this.asyncDurationMetric,
            },
        });
        const openJobs = new cloudwatch.MathExpression({
            expression: 'startedJobs - finishedJobs',
            usingMetrics: {
                startedJobs: this.asyncJobStartedMetric,
                finishedJobs: this.asyncJobFinshedMetric,
            },
        });
        // CALCUATED OPERATIONAL METRICS STOP
        const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Errors',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
            metricValue: '1',
        });
        const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'LimitExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
            metricValue: '1',
        });
        const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ThrottlingException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
            metricValue: '1',
        });
        const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ProvisionedThroughputExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
            metricValue: '1',
        });
        const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'InternalServerError',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
            metricValue: '1',
        });
        const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
        const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
        const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
        const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
        const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
        const textractStartDocumentTextThrottle = new cloudwatch.Metric({
            namespace: 'AWS/Textract',
            metricName: 'ThrottledCount',
            dimensionsMap: {
                Operation: 'StartDocumentTextDetection',
            },
            statistic: 'sum',
        });
        // DASHBOARD LAMBDA
        const dashboardWidth = 24;
        // const widgetStandardHeight=9;
        // DASHBOARD
        new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
            end: 'end',
            periodOverride: cloudwatch.PeriodOverride.AUTO,
            start: 'start',
            widgets: [
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [this.asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [this.asyncJobFinshedMetric], right: [this.asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncCallFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Errors', left: [this.textractAsyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Invocations', left: [this.textractAsyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Throttles', left: [this.textractAsyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncReceiveSNSFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Errors', left: [this.textractAsyncReceiveSNSFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Invocations', left: [this.textractAsyncReceiveSNSFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Throttles', left: [this.textractAsyncReceiveSNSFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
            ],
        });
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericAsyncSfnTask = TextractGenericAsyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericAsyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericAsyncSfnTask", version: "0.0.5" };
TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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