# -*- coding: utf-8 -*-

# Copyright (c) Muhammet Emin TURGUT 2020
# For license see LICENSE
from tkinter import *
from tkinter import ttk


class ScrollableNotebook(ttk.Frame):
    def __init__(self, parent, wheelscroll=False, tabmenu=False, *args, **kwargs):
        ttk.Frame.__init__(self, parent, *args)
        self.xLocation = 0
        self.notebookContent = ttk.Notebook(self, **kwargs)
        self.notebookContent.pack(fill="both", expand=True)
        self.notebookTab = ttk.Notebook(self, **kwargs)
        self.notebookTab.bind("<<NotebookTabChanged>>", self._tabChanger)
        if wheelscroll == True:
            self.notebookTab.bind("<MouseWheel>", self._wheelscroll)
        slideFrame = ttk.Frame(self)
        slideFrame.place(relx=1.0, x=0, y=1, anchor=NE)
        self.menuSpace = 30
        if tabmenu == True:
            self.menuSpace = 50
            # bottomTab = ttk.Label(slideFrame, text="\u2630")
            bottomTab = ttk.Label(slideFrame, text=".....")
            bottomTab.bind("<ButtonPress-1>", self._bottomMenu)
            bottomTab.pack(side=RIGHT)
        # leftArrow = ttk.Label(slideFrame, text=" \u276E")
        leftArrow = ttk.Label(slideFrame, text=" <")
        leftArrow.bind("<ButtonPress-1>", self._leftSlideStart)
        leftArrow.bind("<ButtonRelease-1>", self._slideStop)
        leftArrow.pack(side=LEFT)
        # rightArrow = ttk.Label(slideFrame, text=" \u276F")
        rightArrow = ttk.Label(slideFrame, text=" >")
        rightArrow.bind("<ButtonPress-1>", self._rightSlideStart)
        rightArrow.bind("<ButtonRelease-1>", self._slideStop)
        rightArrow.pack(side=RIGHT)
        self.notebookContent.bind("<Configure>", self._resetSlide)
        self.contentsManaged = []

    def _wheelscroll(self, event):
        if event.delta > 0:
            self._leftSlide(event)
        else:
            self._rightSlide(event)

    def _bottomMenu(self, event):
        tabListMenu = Menu(self, tearoff=0)
        for tab in self.notebookTab.tabs():
            tabListMenu.add_command(
                label=self.notebookTab.tab(tab, option="text"),
                command=lambda temp=tab: self.select(temp),
            )
        try:
            tabListMenu.tk_popup(event.x_root, event.y_root)
        finally:
            tabListMenu.grab_release()

    def _tabChanger(self, event):
        try:
            self.notebookContent.select(self.notebookTab.index("current"))
        except:
            pass

    def _rightSlideStart(self, event=None):
        if self._rightSlide(event):
            self.timer = self.after(100, self._rightSlideStart)

    def _rightSlide(self, event):
        if (
            self.notebookTab.winfo_width()
            > self.notebookContent.winfo_width() - self.menuSpace
        ):
            if (
                self.notebookContent.winfo_width()
                - (self.notebookTab.winfo_width() + self.notebookTab.winfo_x())
            ) <= self.menuSpace + 5:
                self.xLocation -= 20
                self.notebookTab.place(x=self.xLocation, y=0)
                return True
        return False

    def _leftSlideStart(self, event=None):
        if self._leftSlide(event):
            self.timer = self.after(100, self._leftSlideStart)

    def _leftSlide(self, event):
        if not self.notebookTab.winfo_x() == 0:
            self.xLocation += 20
            self.notebookTab.place(x=self.xLocation, y=0)
            return True
        return False

    def _slideStop(self, event):
        if self.timer != None:
            self.after_cancel(self.timer)
            self.timer = None

    def _resetSlide(self, event=None):
        self.notebookTab.place(x=0, y=0)
        self.xLocation = 0

    def add(self, frame, **kwargs):
        if len(self.notebookTab.winfo_children()) != 0:
            self.notebookContent.add(frame, text="", state="hidden")
        else:
            self.notebookContent.add(frame, text="")
        self.notebookTab.add(ttk.Frame(self.notebookTab), **kwargs)
        self.contentsManaged.append(frame)

    def forget(self, tab_id):
        index = self.notebookTab.index(tab_id)
        self.notebookContent.forget(self.__ContentTabID(tab_id))
        self.notebookTab.forget(tab_id)
        self.contentsManaged[index].destroy()
        self.contentsManaged.pop(index)

    def hide(self, tab_id):
        self.notebookContent.hide(self.__ContentTabID(tab_id))
        self.notebookTab.hide(tab_id)

    def identify(self, x, y):
        return self.notebookTab.identify(x, y)

    def index(self, tab_id):
        return self.notebookTab.index(tab_id)

    def __ContentTabID(self, tab_id):
        return self.notebookContent.tabs()[self.notebookTab.tabs().index(tab_id)]

    def insert(self, pos, frame, **kwargs):
        self.notebookContent.insert(pos, frame, **kwargs)
        self.notebookTab.insert(pos, frame, **kwargs)

    def select(self, tab_id):
        ##        self.notebookContent.select(self.__ContentTabID(tab_id))
        self.notebookTab.select(tab_id)

    def tab(self, tab_id, option=None, **kwargs):
        kwargs_Content = kwargs.copy()
        kwargs_Content["text"] = ""  # important
        self.notebookContent.tab(
            self.__ContentTabID(tab_id), option=None, **kwargs_Content
        )
        return self.notebookTab.tab(tab_id, option=None, **kwargs)

    def tabs(self):
        ##        return self.notebookContent.tabs()
        return self.notebookTab.tabs()

    def enable_traversal(self):
        self.notebookContent.enable_traversal()
        self.notebookTab.enable_traversal()
