from starkcore.utils.checks import check_datetime, check_date
from ..__pixkey import _resource as _pixkey_resource
from starkcore.utils.resource import Resource
from starkcore.utils.api import from_api_json
from ...utils import rest


class Log(Resource):
    """# PixKey.Log object
    Every time a PixKey entity is modified, a corresponding PixKey.Log
    is generated for the entity. This log is never generated by the user.
    ## Attributes:
    - id [string]: unique id returned when the log is created. ex: "5656565656565656"
    - key [PixKey]: PixKey entity to which the log refers to.
    - type [string]: type of the PixKey event which triggered the log creation. ex: "created", "registered", "updated", "failed", "canceling", "canceled".
    - errors [list of strings]: list of errors linked to this PixKey event
    - created [datetime.datetime]: creation datetime for the log. ex: datetime.datetime(2020, 3, 10, 10, 30, 0, 0)
    """
    def __init__(self, id, key, type, errors, created):
        Resource.__init__(self, id=id)

        self.key = from_api_json(_pixkey_resource, key)
        self.type = type
        self.errors = errors
        self.created = check_datetime(created)


_resource = {"class": Log, "name": "PixKeyLog"}


def get(id, user=None):
    """# Retrieve a specific PixKey.Log
    Receive a single PixKey.Log object previously created by the Stark Infra API by its id
    ## Parameters (required):
    - id [string]: object unique id. ex: "5656565656565656"
    ## Parameters (optional):
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call
    ## Return:
    - PixKey.Log object with updated attributes
    """
    return rest.get_id(resource=_resource, id=id, user=user)


def query(ids=None, limit=None, after=None, before=None, types=None, key_ids=None, user=None):
    """# Retrieve PixKey.Logs
    Receive a generator of PixKey.Log objects previously created in the Stark Infra API
    ## Parameters (optional):
    - ids [list of strings, default None]: Log ids to filter PixKey Logs. ex: ["5656565656565656"]
    - limit [integer, default None]: maximum number of objects to be retrieved. Unlimited if None. ex: 35
    - after [datetime.date or string, default None]: date filter for objects created after specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None]: date filter for objects created before a specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter retrieved objects by types. ex: ["created", "registered", "updated", "failed", "canceling", "canceled"]
    - key_ids [list of strings, default None]: list of PixKey IDs to filter retrieved objects. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call
    ## Return:
    - generator of PixKey.Log objects with updated attributes
    """
    return rest.get_stream(
        resource=_resource,
        ids=ids,
        limit=limit,
        after=check_date(after),
        before=check_date(before),
        types=types,
        key_ids=key_ids,
        user=user,
    )


def page(cursor=None, ids=None, limit=None, after=None, before=None, types=None, key_ids=None, user=None):
    """# Retrieve paged PixKey.Logs
    Receive a list of up to 100 PixKey.Log objects previously created in the Stark Infra API and the cursor to the next page.
    Use this function instead of query if you want to manually page your keys.
    ## Parameters (optional):
    - cursor [string, default None]: cursor returned on the previous page function call
    - ids [list of strings, default None]: Log ids to filter PixKey Logs. ex: ["5656565656565656"]
    - limit [integer, default 100]: maximum number of objects to be retrieved. Max = 100. ex: 35
    - after [datetime.date or string, default None]: date filter for objects created after a specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None]: date filter for objects created before a specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter retrieved objects by types. ex: ["created", "registered", "updated", "failed", "canceling", "canceled"]
    - key_ids [list of strings, default None]: list of PixKey IDs to filter retrieved objects. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call
    ## Return:
    - list of PixKey.Log objects with updated attributes
    - cursor to retrieve the next page of PixKey.Log objects
    """
    return rest.get_page(
        resource=_resource,
        cursor=cursor,
        ids=ids,
        limit=limit,
        after=check_date(after),
        before=check_date(before),
        types=types,
        key_ids=key_ids,
        user=user,
    )
